/******************************************************************** 
   Copyright (C) 2000 Bassoukos Tassos <abas@aix.meng.auth.gr>
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later
   version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*********************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif
#include <gnome.h>

#include "tasks.h"
#include "protocol.h"
#include "connection.h"
#include "messages.h"
#include "transaction.h"
#include "guiprefs.h"
#include "news.h"
#include "guiutils.h"
#include "privs.h"
#include "pixmap.h"

#define NEWS "News"

typedef struct {
  Connection *c;
  char *news;
  Transaction *trans;
  NotebookPage *notebook_page;
  GtkWidget *news_widget;
  GtkWidget *post_frame;
  GtkWidget *post_text_widget;
} NewsData;


static void news_do_refresh(GtkButton *b,gpointer p);

static void news_post_cancel(GtkButton *b,gpointer p){
  NewsData *n=(NewsData *)p;
  if(n->post_frame!=NULL)
    gtk_widget_destroy(n->post_frame);
  n->post_frame=NULL;
  n->post_text_widget=NULL;
}

static void news_post_send(GtkButton *b,NewsData *n){
  Transaction *t;
  HLObject *o;
  char *text;

  if(n->post_frame==NULL){
    news_post_cancel(b,n);
    return;
  }
  text=gtk_editable_get_chars(GTK_EDITABLE(n->post_text_widget),0,-1);
  news_post_cancel(b,n);
  t=transaction_new(n->c,HLCT_POSTNEWS);
  transaction_add_task(t,_("Posting News..."),HL_STOCK_PIXMAP_NEWS_POST,0.5);
  transaction_set_error_check(t,strdup(_("Could not post news:")));
  o=create_string(HLO_MESSAGE,text);
  string_unix_to_net(o);
  message_add_object(t->request,o);
  transaction_start(t);
}

static void news_post_gui(GtkButton *b,gpointer p){
  NewsData *n=(NewsData *)p;
  GtkWidget *d,*text,*sl;
  if(n==NULL || n->post_frame!=NULL)
    return;
  d=gnome_dialog_new(_("Post a Message"),GNOME_STOCK_BUTTON_CANCEL,GNOME_STOCK_BUTTON_OK,NULL);
  text=gtk_text_new(NULL,NULL);
  gtk_text_set_editable(GTK_TEXT(text),TRUE);
  guiprefs_widget_set_font(text,&news_font);
  sl=gtk_scrolled_window_new(NULL,NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sl),
				 GTK_POLICY_NEVER,
				 GTK_POLICY_ALWAYS);
  gtk_window_set_policy(GTK_WINDOW(d), FALSE, TRUE, FALSE);
  gtk_container_add(GTK_CONTAINER(sl),text);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(d)->vbox),sl,TRUE,TRUE,0);
  gnome_dialog_button_connect(GNOME_DIALOG(d),0,
			      GTK_SIGNAL_FUNC(news_post_cancel),
			      (gpointer)n);
  gnome_dialog_button_connect(GNOME_DIALOG(d),1,
			      GTK_SIGNAL_FUNC(news_post_send),
			      (gpointer)n);
  if(n->c->gui->main_window!=NULL)
    gnome_dialog_set_parent(GNOME_DIALOG(d),GTK_WINDOW(n->c->gui->main_window));
  gtk_signal_connect(GTK_OBJECT(d),"destroy",
		     GTK_SIGNAL_FUNC(news_post_cancel),(gpointer)n);
  guiprefs_add_window(GTK_WINDOW(d),"Main/News/post_dialog_size");
  gtk_widget_show_all(d);
  n->post_text_widget=text;
  n->post_frame=d;
}

static void news_gui_destroy(NewsData *n){
  if(n->post_frame!=NULL)
    gtk_widget_destroy(n->post_frame);
  if(n->notebook_page!=NULL)
    gutils_nbpage_destroy(n->notebook_page);
  n->post_frame=NULL;
  n->notebook_page=NULL;
  n->news_widget=NULL;
  n->post_text_widget=NULL;
}

static void close_news(GtkButton *b,NewsData *n){
  news_gui_destroy(n);
}

static void destroy_news(Connection *c,NewsData *n){
  news_gui_destroy(n);
  if(n->news!=NULL)
    free(n->news);
  free(n);
  gtk_signal_disconnect_by_data(GTK_OBJECT(c),n);
  connection_set_data(c,NEWS,NULL);
}

static GnomeUIInfo news_toolbar[] = {
  GNOMEUIINFO_ITEM_STOCK("Close",N_("Close news"),
			 close_news,
			 HL_STOCK_PIXMAP_CLOSE_PAGE),
  GNOMEUIINFO_ITEM_STOCK("Refresh",N_("Reload news"),
			 news_do_refresh,
			 HL_STOCK_PIXMAP_REFRESH),
  GNOMEUIINFO_SEPARATOR,
  GNOMEUIINFO_ITEM_STOCK(N_("Post"),N_("Post a Message"),
			 news_post_gui,
			 HL_STOCK_PIXMAP_NEWS_POST),
  GNOMEUIINFO_END
};

static void update_news_gui(NewsData *n){
  if(n->news_widget==NULL)
    return;
  gtk_text_freeze(GTK_TEXT(n->news_widget));
  gtk_editable_delete_text(GTK_EDITABLE(n->news_widget),0,-1);
  if(n->news!=NULL)
    gtk_text_insert(GTK_TEXT(n->news_widget),NULL,NULL,NULL,n->news,-1);
  gtk_text_thaw(GTK_TEXT(n->news_widget));
}

static void create_news_gui(NewsData *n){
  GtkWidget *text,*sw;
  if(n->notebook_page!=NULL)
    return;

  n->notebook_page=gutils_nbpage_new("Main",news_toolbar,n->c,_("News"),"news",n);
  text=gtk_text_new(NULL,NULL);
  sw=gtk_scrolled_window_new(NULL,NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
				 GTK_POLICY_NEVER,
				 GTK_POLICY_ALWAYS);
  gtk_container_add(GTK_CONTAINER(sw),text);
  gtk_text_set_editable(GTK_TEXT(text),FALSE);
  gtk_text_set_word_wrap(GTK_TEXT(text),TRUE);
  gutils_nbpage_set_main(n->notebook_page,sw);
  n->news_widget=text;
}

static void news_handle_reply(Transaction *t,Message *m,NewsData *n){
  HLObject *o;
  char *s;
  
  n->trans=NULL;
  o=message_find_object(m,HLO_MESSAGE);
  if(o==NULL){
    s=_("This server has no news.");
  } else {
    string_net_to_unix(o);
    s=strdup(o->data.string);
    if(n->news!=NULL)
      free(n->news);
    n->news=s;
  }
  create_news_gui(n);
  update_news_gui(n);
}

static void send_news_refresh(NewsData *n){
  Transaction *t;
		       
  if(n->trans!=NULL)
    return;
  n->trans=t=transaction_new(n->c,HLCT_GETNEWS);
  transaction_add_task(t,_("Retrieving news..."),HL_STOCK_PIXMAP_NEWS,0.1);
  transaction_set_error_check(t,strdup(_("Could not refresh news:")));
  gtk_signal_connect(GTK_OBJECT(t),"response-received",
		     GTK_SIGNAL_FUNC(news_handle_reply),n);
  transaction_start(t);
}

static void news_do_refresh(GtkButton *b,gpointer p){
  send_news_refresh(p);
}

static void init_news(Connection *c){
  NewsData *n=(NewsData *)malloc(sizeof(NewsData));
  
  n->c=c;
  n->news=NULL;
  n->trans=NULL;
  n->notebook_page=NULL;
  n->news_widget=NULL;
  n->post_frame=NULL;
  n->post_text_widget=NULL;
  n->trans=NULL;
  connection_set_data(c,NEWS,n);
  gtk_signal_connect(GTK_OBJECT(c),"destroy",GTK_SIGNAL_FUNC(destroy_news),n);
}

void want_news(Connection *c){
  NewsData *n=(NewsData *)connection_get_data(c,NEWS);
  if(n==NULL){
    init_news(c);
    n=(NewsData *)connection_get_data(c,NEWS);
    send_news_refresh(n);
  }
  if(n->notebook_page==NULL){
    create_news_gui(n);
    update_news_gui(n);
  }
  gutils_nbpage_to_front(n->notebook_page);
}

void check_news(Connection *c){
  if(auto_open_news==TRUE)
    want_news(c);
}

void handle_server_new_post(Connection *c,Message *m,gpointer data){
  HLObject *o;
  char *s;
  NewsData *n=(NewsData *)connection_get_data(c,NEWS);

  o=message_find_object(m,HLO_MESSAGE);
  if(o==NULL || n==NULL){
    s=strdup(_("New News Post without text....\n"));
  } else {
    string_net_to_unix(o);
    s=g_strconcat(o->data.string,n->news,NULL);
  }
  free(n->news);
  n->news=s;
  message_unref(m);
  update_news_gui(n);
}

