//
// "$Id: export.cxx,v 1.17 2003/09/05 12:47:07 easysw Exp $"
//
// Export methods for flPhoto.
//
// Copyright 2002-2003 by Michael Sweet.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// Contents:
//
//   flphoto::export_album_cb() - Export the current album.
//   flphoto::export_album_cb() - Export the current album.
//   export_copy()              - Copy a file verbatim.
//   export_footer()            - Write a HTML footer.
//   export_header()            - Write a HTML header.
//   export_jpeg()              - Write an image file in JPEG format.
//   export_string()            - Write a string to a HTML file.
//   export_dir()               - Create the remote directory.
//   export_file()              - Copy a local file to a remote file.
//

#include "flphoto.h"
#include "i18n.h"
#include "flstring.h"
#include "http.h"
#include <stdlib.h>
#include <sys/stat.h>

#if defined(WIN32) && !defined(__CYGWIN__)
#  include <direct.h>
#  include <io.h>
#  define fl_mkdir(p)	mkdir(p)
#else
#  include <unistd.h>
#  define fl_mkdir(p)	mkdir(p, 0777)
#endif // WIN32 && !__CYGWIN__

#include <errno.h>

#ifdef HAVE_LIBJPEG
// Some releases of the Cygwin JPEG libraries don't have a correctly
// updated header file for the INT32 data type; the following define
// from Shane Hill seems to be a usable workaround...
#  if defined(WIN32) && defined(__CYGWIN__)
#    define XMD_H
#  endif // WIN32 && __CYGWIN__
extern "C" {
#  include <jpeglib.h>
#  include "transupp.h"
};
#endif // HAVE_LIBJPEG



//
// Local functions...
//

static int		export_copy(const char *src, const char *dst);
#ifdef USE_MKCOL
static HTTPStatus	export_dir(HTTP *http, const char *username,
			           const char *password, const char *url);
#endif // USE_MKCOL
static HTTPStatus	export_file(HTTP *http, const char *username,
			            const char *password, const char *filename,
				    const char *url);
static void		export_footer(FILE *fp, const char *footer);
static FILE		*export_header(const char *filename, const char *header,
			               const char *comment);
static int		export_jpeg(const char *filename, Fl_Shared_Image *img,
			            int w, int h, int quality);
static void		export_string(FILE *fp, const char *s, const char *el = 0);


//
// 'flphoto::export_album_cb()' - Export the current album.
//

void
flphoto::export_album_cb()
{
  int	val;
  char	s[1024];


  // Get the defaults and put them in the dialog...
  prefs.get("export_path", s, "", sizeof(s));
  export_path_field_->value(s);
  prefs.get("export_jpeg_size", val, 640);
  export_imagesize_value_->value(val);
  prefs.get("export_jpeg_columns", val, 1);
  export_imagecols_value_->value(val);
  prefs.get("export_jpeg_rows", val, 1);
  export_imagerows_value_->value(val);
  prefs.get("export_jpeg_quality", val, 75);
  export_imagequal_value_->value(val);
  prefs.get("export_thumb_size", val, 120);
  export_thumbsize_value_->value(val);
  prefs.get("export_thumb_columns", val, 5);
  export_thumbcols_value_->value(val);
  prefs.get("export_thumb_rows", val, 3);
  export_thumbrows_value_->value(val);
  prefs.get("export_thumb_quality", val, 50);
  export_thumbqual_value_->value(val);
  prefs.get("export_header", s, "", sizeof(s));
  export_header_field_->value(s);
  prefs.get("export_footer", s, "", sizeof(s));
  export_footer_field_->value(s);
  prefs.get("export_original", val, 1);
  export_original_button_->value(val);

  export_progress_->hide();
  export_group_->activate();

  export_window_->hotspot(export_window_);
  export_window_->show();
}


//
// 'flphoto::export_album_cb()' - Export the current album.
//

void
flphoto::export_ok_cb()
{
  int			i, j;
  int			delay;
  int			w, h;		// Resized width/height
  const char		*path,
			*header,
			*footer;
  char			dstdir[1024],
			url[1024],
			filename[1024],
			thumbname[1024],
			imagename[1024],
			label[1024];
  int			thumbsize,
			thumbquality,
			thumbpage,
			thumbpages,
			thumbcol,
			thumbcols,
			thumbrow,
			thumbrows;
  FILE			*thumbfile;
  int			imagesize,
			imagequality,
			imagepage,
			imagepages,
			imagecol,
			imagecols,
			imagerow,
			imagerows;
  int			originals;
  FILE			*imagefile;
  FILE			*slidefile;
  Fl_Shared_Image	*img;
  Fl_Image_Browser::ITEM *item;
  HTTP			*http;
  HTTPStatus		status;


  // Check that the user has chosen an export path...
  path = export_path_field_->value();
  if (!path || !*path)
  {
    fl_alert(_("Please choose an export directory or URL."));
    return;
  }

  if (strncmp(path, "http:", 5) == 0 ||
      strncmp(path, "https:", 5) == 0 ||
      strncmp(path, "ftp:", 4) == 0)
  {
    // Write the files to a temporary directory...
    prefs.getUserdataPath(dstdir, sizeof(dstdir));
  }
  else
  {
    // Write the files to the named directory...
    strlcpy(dstdir, path, sizeof(dstdir));
    if (access(path, 0))
    {
      if (fl_ask(_("Directory %s does not exist.\n\nCreate directory?"), path))
      {
        if (fl_mkdir(path))
	{
	  fl_alert(_("Unable to create directory %s:\n\n%s"), path,
	           strerror(errno));
	  return;
	}
      }
      else
        return;
    }
  }

  // Save options as defaults...
  prefs.set("export_path", export_path_field_->value());
  prefs.set("export_jpeg_size",
            imagesize = (int)export_imagesize_value_->value());
  prefs.set("export_jpeg_columns",
            imagecols = (int)export_imagecols_value_->value());
  prefs.set("export_jpeg_rows",
            imagerows = (int)export_imagerows_value_->value());
  prefs.set("export_jpeg_quality",
            imagequality = (int)export_imagequal_value_->value());
  prefs.set("export_thumb_size",
            thumbsize = (int)export_thumbsize_value_->value());
  prefs.set("export_thumb_columns",
            thumbcols = (int)export_thumbcols_value_->value());
  prefs.set("export_thumb_rows",
            thumbrows = (int)export_thumbrows_value_->value());
  prefs.set("export_thumb_quality",
            thumbquality = (int)export_thumbqual_value_->value());
  prefs.set("export_header",
            header = export_header_field_->value());
  prefs.set("export_footer",
            footer = export_footer_field_->value());
  prefs.set("export_original",
            originals = export_original_button_->value());

  if (!strncmp(path, "ftp:", 4))
  {
    fl_alert(_("Sorry, FTP export not currently supported."));
    export_window_->hide();
    return;
  }

  export_group_->deactivate();
  export_window_->cursor(FL_CURSOR_WAIT);
  export_progress_->show();

  thumbpage  = 0;
  thumbpages = (browser_->count() + thumbcols * thumbrows - 1) /
               (thumbcols * thumbrows);
  thumbcol   = 0;
  thumbrow   = 0;
  thumbfile  = NULL;

  imagepage  = 0;
  imagepages = (browser_->count() + imagecols * imagerows - 1) /
               (imagecols * imagerows);
  imagecol   = 0;
  imagerow   = 0;
  imagefile  = NULL;

  prefs.get("slideshow_delay", delay, 5);

  for (i = 0; i < browser_->count(); i ++)
  {
    // Load the current image...
    item = browser_->value(i);
    img  = browser_->load_item(i);

    if (!img)
      continue;

    // Update progress...
    snprintf(label, sizeof(label), _("Generating %s..."), item->label);
    export_progress_->label(label);
    export_progress_->value(100 * i / browser_->count());
    Fl::check();

    // Open files as needed..
    if (thumbcol == 0 && thumbrow == 0)
    {
      // Open the thumbnail index...
      export_footer(thumbfile, footer);

      if (thumbpage == 0)
        snprintf(thumbname, sizeof(thumbname), "%s/index.html", dstdir);
      else
        snprintf(thumbname, sizeof(thumbname), "%s/ind%05d.html", dstdir,
	         thumbpage);

      thumbfile = export_header(thumbname, header, album_comment_);

      fprintf(thumbfile, "<TR><TD COLSPAN=\"%d\" ALIGN=\"CENTER\">", thumbcols);

      fprintf(thumbfile, "<A HREF=\"slideshow.html\">%s</A><BR>\n",
              _("Show Slideshow"));

      if (thumbpages > 1)
      {
	if (thumbpage > 1)
	  fprintf(thumbfile, "<A HREF=\"ind%05d.html\">&lt;&lt;</A>", thumbpage - 1);
	else if (thumbpage == 1)
	  fputs("<A HREF=\"index.html\">&lt;&lt;</A>", thumbfile);

        for (j = 0; j < thumbpages; j ++)
	  if (j != thumbpage)
	  {
	    if (j == 0)
	      fputs(" <A HREF=\"index.html\">1</A>", thumbfile);
            else
	      fprintf(thumbfile, " <A HREF=\"ind%05d.html\">%d</A>", j, j + 1);
	  }
	  else
	    fprintf(thumbfile, " <B>%d</B>", j + 1);

	if ((thumbpage + 1) < thumbpages)
	  fprintf(thumbfile, " <A HREF=\"ind%05d.html\">&gt;&gt;</A>", thumbpage + 1);
      }

      fputs("</TD></TR>\n", thumbfile);

      thumbpage ++;
    }

    if (imagecol == 0 && imagerow == 0)
    {
      // Open the image index...
      export_footer(imagefile, footer);

      snprintf(imagename, sizeof(imagename), "%s/img%05d.html", dstdir,
               imagepage);

      imagefile = export_header(imagename, header, album_comment_);

      fprintf(imagefile, "<TR><TD COLSPAN=\"%d\" ALIGN=\"CENTER\">"
                         "<A HREF=\"%s\">%s</A></TD></TR>\n",
	      imagecols, strrchr(thumbname, '/') + 1,
	      _("Back to Thumbnails"));

      if (imagepages > 1)
      {
        fprintf(imagefile, "<TR><TD COLSPAN=\"%d\" ALIGN=\"CENTER\">",
	        imagecols);

	if (imagepage > 0)
	  fprintf(imagefile, "<A HREF=\"img%05d.html\">&lt;&lt;</A>",
	          imagepage - 1);

        for (j = -10; j <= 10; j ++)
	  if ((imagepage + j) < 0 || (imagepage + j) >= imagepages)
	    continue;
	  else if (j != 0)
	    fprintf(imagefile, " <A HREF=\"img%05d.html\">%d</A>",
	            imagepage + j, imagepage + j + 1);
	  else
	    fprintf(imagefile, " <B>%d</B>", imagepage + j + 1);

	if ((imagepage + 1) < imagepages)
	  fprintf(imagefile, " <A HREF=\"img%05d.html\">&gt;&gt;</A>",
	          imagepage + 1);

        fputs("</TD></TR>\n", imagefile);
      }

      imagepage ++;
    }

    // Write the thumbnail...
    w = thumbsize;
    h = w * img->h() / img->w();
    if (h > thumbsize)
    {
      h = thumbsize;
      w = h * img->w() / img->h();
    }

    if (thumbcol == 0)
      fputs("<TR>", thumbfile);

    fprintf(thumbfile, "<TD ALIGN=\"CENTER\"><A HREF=\"%s#img%05d\">"
                       "<IMG SRC=\"thm%05d.jpg\" WIDTH=\"%d\" HEIGHT=\"%d\" "
		       "ALT=\"",
            strrchr(imagename, '/') + 1, i, i, w, h);
    export_string(thumbfile, item->comments);
    fprintf(thumbfile, "\"/><BR/>%s</A></TD>\n", item->label);

    thumbcol ++;
    if (thumbcol >= thumbcols)
    {
      fputs("</TR>\n", thumbfile);
      thumbcol = 0;
      thumbrow ++;
    }

    if (thumbrow >= thumbrows)
      thumbrow = 0;

    snprintf(filename, sizeof(filename), "%s/thm%05d.jpg", dstdir, i);
    export_jpeg(filename, item->image, w, h, thumbquality);

    // Write the image...
    w = imagesize;
    h = w * img->h() / img->w();
    if (h > imagesize)
    {
      h = imagesize;
      w = h * img->w() / img->h();
    }

    if (imagecol == 0)
      fputs("<TR>", imagefile);

    fputs("<TD ALIGN=\"CENTER\">", imagefile);
    if (originals)
      fprintf(imagefile, "<A HREF=\"org%05d%s\">",
              i, fl_filename_ext(item->filename));
    fprintf(imagefile, "<IMG SRC=\"img%05d.jpg\" "
                       "WIDTH=\"%d\" HEIGHT=\"%d\" ALT=\"",
            i, w, h);
    export_string(imagefile, item->comments);
    fputs("\"/><BR/>", imagefile);
    export_string(imagefile, item->label);
    if (item->comments)
    {
      fputs("<BR/>", imagefile);
      export_string(imagefile, item->comments);
    }
    if (originals)
      fputs("</A>", imagefile);
    fputs("</TD>\n", imagefile);

    imagecol ++;
    if (imagecol >= imagecols)
    {
      fputs("</TR>\n", imagefile);
      imagecol = 0;
      imagerow ++;
    }

    if (imagerow >= imagerows)
      imagerow = 0;

    snprintf(filename, sizeof(filename), "%s/img%05d.jpg", dstdir, i);
    export_jpeg(filename, item->image, w, h, imagequality);

    // Copy the image as needed...
    if (originals && strncmp(path, "http:", 5) && strncmp(path, "https:", 6))
    {
      snprintf(filename, sizeof(filename), "%s/org%05d%s", dstdir, i,
               fl_filename_ext(item->filename));
      export_copy(item->filename, filename);
    }

    // Unload the current image...
    if (item->image && !item->changed && item != image_item_)
    {
      item->image->release();
      item->image = 0;
    }
  }

  export_footer(imagefile, footer);
  export_footer(thumbfile, footer);

 /*
  * Write a slideshow file...
  */

  snprintf(filename, sizeof(filename), "%s/slideshow.html", dstdir);
  if ((slidefile = fopen(filename, "w")) != NULL)
  {
    fputs("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\" "
          "\"http://www.w3.org/TR/html4/loose.dtd\">\n", slidefile);

    fputs("<HTML>\n", slidefile);
    fputs("<HEAD>\n", slidefile);
    fputs("<META http-equiv=\"Content-Type\" content=\"text/html; "
          "charset=iso-8859-1\">\n", slidefile);
    if (header && *header)
      export_string(slidefile, header, "TITLE");
    else
      export_string(slidefile, album_comment_, "TITLE");
    fputs("<SCRIPT TYPE=\"text/javascript\"><!--\n", slidefile);
    fputs("var imgnum=99999;\n", slidefile);
    fprintf(slidefile, "var imgdelay=%d;\n", delay * 1000);
    fputs("var imginterval=0;\n", slidefile);
    fputs("\n", slidefile);
    fputs("function next_image() {\n", slidefile);
    fputs("  var imgstr;\n", slidefile);
    fputs("\n", slidefile);
    fputs("  imgnum ++;\n", slidefile);
    fprintf(slidefile, "  if (imgnum > %d)\n", 99999 + browser_->count());
    fputs("    imgnum = 100000;\n", slidefile);
    fputs("\n", slidefile);
    fputs("  imgstr = imgnum.toString();\n", slidefile);
    fputs("\n", slidefile);
    fputs("  if (imginterval)\n", slidefile);
    fputs("    window.clearInterval(imginterval);\n", slidefile);
    fputs("  document.imgname.src = 'img' + imgstr.substring(1) + '.jpg';\n", slidefile);
    fputs("  imginterval = setInterval(\"next_image()\", imgdelay);\n", slidefile);
    fputs("}\n", slidefile);
    fputs("--></SCRIPT>\n", slidefile);
    fputs("</HEAD>\n", slidefile);
    fputs("<BODY BGCOLOR=\"#000000\" onLoad=\"next_image()\">\n", slidefile);
    fputs("<CENTER><TABLE CELLPADDING=\"0\" BORDER=\"0\" CELLSPACING=\"0\">\n", slidefile);
    fprintf(slidefile, "<TR><TD ALIGN=\"CENTER\" VALIGN=\"MIDDLE\" "
                       "WIDTH=\"%d\" HEIGHT=\"%d\"><IMG SRC=\"img00000.jpg\" "
		       "NAME=\"imgname\" onClick=\"next_image()\" "
		       "ALT=\"\"></TD>\n",
            imagesize, imagesize);
    fputs("</TR></TABLE></CENTER>\n", slidefile);
    fputs("</BODY>\n", slidefile);
    fputs("</HTML>\n", slidefile);
    fclose(slidefile);
  }

  if (strncmp(path, "http:", 5) == 0 ||
      strncmp(path, "https:", 5) == 0)
  {
    // Connect to the remote system...
    char	method[HTTP_MAX_URI],
		hostname[HTTP_MAX_URI],
		username[HTTP_MAX_URI],
		resource[HTTP_MAX_URI];
    int		port;

    HTTP::separate(path, method, sizeof(method), username, sizeof(username),
                   hostname, sizeof(hostname), &port,
		   resource, sizeof(resource));

    if (resource[strlen(resource) - 1] != '/')
      strlcat(resource, "/", sizeof(resource));

    export_progress_->label(_("Connecting to remote server..."));
    export_progress_->value(0);
    Fl::check();

    if (strcmp(method, "https") == 0)
      http = new HTTP(hostname, port, HTTP_ENCRYPT_ALWAYS);
    else
      http = new HTTP(hostname, port);

    // Apparently MKCOL doesn't create a directory, so for now
    // leave this code commented out until we figure things out.
#ifdef USE_MKCOL
    // Create the destination folder...
    export_progress_->label(_("Creating destination folder..."));
    export_progress_->value(0);
    Fl::check();

    do
    {
      status = export_dir(http, auth_user_field_->value(),
                          auth_pass_field_->value(), resource);

      if (status == HTTP_UNAUTHORIZED)
      {
        // Show the authentication window...
	auth_window_->show();
	while (auth_window_->shown())
	  Fl::wait();

        if (!auth_user_field_->value()[0])
	  break;
      }
    }
    while (status == HTTP_UNAUTHORIZED);

    if (status != HTTP_CREATED &&
        status != HTTP_OK &&
	status != HTTP_METHOD_NOT_ALLOWED)
    {
      fl_alert(_("Unable to create destination URL:\n\n%d - %s"), status,
               HTTP::status_string(status));
      delete http;
      export_group_->activate();
      export_window_->cursor(FL_CURSOR_DEFAULT);
      export_progress_->hide();
      return;
    }
#endif // USE_MKCOL

    for (i = 0; i < browser_->count(); i ++)
    {
      // Update progress...
      snprintf(label, sizeof(label), _("Uploading img%05d.jpg..."), i);
      export_progress_->label(label);
      export_progress_->value(20 * i / browser_->count());
      Fl::check();

      snprintf(filename, sizeof(filename), "%s/img%05d.jpg", dstdir, i);
      snprintf(url, sizeof(url), "%simg%05d.jpg", resource, i);

      do
      {
	status = export_file(http, auth_user_field_->value(),
                             auth_pass_field_->value(), filename, url);

	if (status == HTTP_UNAUTHORIZED)
	{
          // Show the authentication window...
	  auth_window_->show();
	  while (auth_window_->shown())
	    Fl::wait();

          if (!auth_user_field_->value()[0])
	    break;
	}
      }
      while (status == HTTP_UNAUTHORIZED);

      unlink(filename);

      if (status != HTTP_CREATED &&
          status != HTTP_OK &&
	  status != HTTP_METHOD_NOT_ALLOWED)
      {
        fl_alert(_("Unable to create destination URL:\n\n%d - %s"), status,
                 HTTP::status_string(status));
        delete http;
        export_group_->activate();
        export_window_->cursor(FL_CURSOR_DEFAULT);
	export_progress_->hide();
        return;
      }
    }

    for (i = 0; i < browser_->count(); i ++)
    {
      // Update progress...
      snprintf(label, sizeof(label), _("Uploading thm%05d.jpg..."), i);
      export_progress_->label(label);
      export_progress_->value(20 + 20 * i / browser_->count());
      Fl::check();

      snprintf(filename, sizeof(filename), "%s/thm%05d.jpg", dstdir, i);
      snprintf(url, sizeof(url), "%sthm%05d.jpg", resource, i);

      status = export_file(http, auth_user_field_->value(),
                           auth_pass_field_->value(), filename, url);

      unlink(filename);

      if (status != HTTP_CREATED &&
          status != HTTP_OK &&
	  status != HTTP_METHOD_NOT_ALLOWED)
      {
        fl_alert(_("Unable to create destination URL:\n\n%d - %s"), status,
                 HTTP::status_string(status));
        delete http;
        export_group_->activate();
        export_window_->cursor(FL_CURSOR_DEFAULT);
	export_progress_->hide();
        return;
      }
    }

    if (originals)
    {
      for (i = 0; i < browser_->count(); i ++)
      {
	// Update progress...
        item = browser_->value(i);

	snprintf(label, sizeof(label), _("Uploading org%05d%s..."), i,
	         fl_filename_ext(item->filename));
	export_progress_->label(label);
	export_progress_->value(40 + 20 * i / browser_->count());
	Fl::check();

	snprintf(filename, sizeof(filename), "%s/org%05d%s", dstdir, i,
	         fl_filename_ext(item->filename));
	snprintf(url, sizeof(url), "%sorg%05d%s", resource, i,
	         fl_filename_ext(item->filename));

	status = export_file(http, auth_user_field_->value(),
                             auth_pass_field_->value(), filename, url);

	unlink(filename);

	if (status != HTTP_CREATED &&
            status != HTTP_OK &&
	    status != HTTP_METHOD_NOT_ALLOWED)
	{
          fl_alert(_("Unable to create destination URL:\n\n%d - %s"), status,
                   HTTP::status_string(status));
          delete http;
          export_group_->activate();
          export_window_->cursor(FL_CURSOR_DEFAULT);
	  export_progress_->hide();
          return;
	}
      }
    }

    for (i = 0; i < thumbpages; i ++)
    {
      // Update progress...
      if (i == 0)
        strcpy(thumbname, "index.html");
      else
        snprintf(thumbname, sizeof(thumbname), "ind%05d.html", i);

      snprintf(label, sizeof(label), _("Uploading %s..."), thumbname);
      export_progress_->label(label);
      export_progress_->value(60 + 20 * i / thumbpages);
      Fl::check();

      snprintf(filename, sizeof(filename), "%s/%s", dstdir, thumbname);
      snprintf(url, sizeof(url), "%s%s", resource, thumbname);

      status = export_file(http, auth_user_field_->value(),
                           auth_pass_field_->value(), filename, url);

      unlink(filename);

      if (status != HTTP_CREATED &&
          status != HTTP_OK &&
	  status != HTTP_METHOD_NOT_ALLOWED)
      {
        fl_alert(_("Unable to create destination URL:\n\n%d - %s"), status,
                 HTTP::status_string(status));
        delete http;
        export_group_->activate();
        export_window_->cursor(FL_CURSOR_DEFAULT);
	export_progress_->hide();
        return;
      }
    }

    for (i = 0; i < imagepages; i ++)
    {
      // Update progress...
      snprintf(imagename, sizeof(imagename), "img%05d.html", i);

      snprintf(label, sizeof(label), _("Uploading %s..."), imagename);
      export_progress_->label(label);
      export_progress_->value(80 + 20 * i / imagepages);
      Fl::check();

      snprintf(filename, sizeof(filename), "%s/%s", dstdir, imagename);
      snprintf(url, sizeof(url), "%s%s", resource, imagename);

      status = export_file(http, auth_user_field_->value(),
                           auth_pass_field_->value(), filename, url);

      unlink(filename);

      if (status != HTTP_CREATED &&
          status != HTTP_OK &&
	  status != HTTP_METHOD_NOT_ALLOWED)
      {
        fl_alert(_("Unable to create destination URL:\n\n%d - %s"), status,
                 HTTP::status_string(status));
        delete http;
        export_group_->activate();
        export_window_->cursor(FL_CURSOR_DEFAULT);
	export_progress_->hide();
        return;
      }
    }

    export_progress_->label(_("Uploading slideshow.html..."));
    export_progress_->value(100);
    Fl::check();

    snprintf(filename, sizeof(filename), "%s/slideshow.html", dstdir);
    snprintf(url, sizeof(url), "%sslideshow.html", resource);

    status = export_file(http, auth_user_field_->value(),
                         auth_pass_field_->value(), filename, url);

    unlink(filename);

    if (status != HTTP_CREATED &&
        status != HTTP_OK &&
	status != HTTP_METHOD_NOT_ALLOWED)
    {
      fl_alert(_("Unable to create destination URL:\n\n%d - %s"), status,
               HTTP::status_string(status));
      delete http;
      export_group_->activate();
      export_window_->cursor(FL_CURSOR_DEFAULT);
      export_progress_->hide();
      return;
    }

    delete http;
    rmdir(dstdir);

    fl_message(_("Album exported successfully!"));
  }
  else if (strncmp(path, "ftp:", 4) == 0)
  {
    fl_alert(_("FTP upload currently not available, sorry!"));
  }

  // Hide the window...
  export_progress_->hide();
  export_group_->activate();
  export_window_->cursor(FL_CURSOR_DEFAULT);
  export_window_->hide();
}


//
// 'export_copy()' - Copy a file verbatim.
//

static int				// O - 0 on success, -1 on error
export_copy(const char *src,		// I - Source file
            const char *dst)		// I - Destination file
{
  FILE	*srcfp,				// Source file pointer
	*dstfp;				// Destination file pointer
  char	buffer[8192];			// Copy buffer
  int	bytes;				// Number of bytes read/written


  if ((srcfp = fopen(src, "rb")) == NULL)
    return (-1);

  if ((dstfp = fopen(dst, "wb")) == NULL)
  {
    fclose(srcfp);

    return (-1);
  }

  while ((bytes = fread(buffer, 1, sizeof(buffer), srcfp)) > 0)
    if ((int)fwrite(buffer, 1, bytes, dstfp) < bytes)
    {
      fclose(srcfp);
      fclose(dstfp);

      return (-1);
    }

  fclose(srcfp);
  fclose(dstfp);

  return (0);
}


#ifdef USE_MKCOL
//
// 'export_dir()' - Create the remote directory.
//

static HTTPStatus
export_dir(HTTP       *http,
           const char *username,
           const char *password,
	   const char *url)
{
  HTTPStatus	status;


  http->clear_fields();
  http->set_field(HTTP_FIELD_USER_AGENT, "flPhoto " FLPHOTO_VERSION);
  http->set_field(HTTP_FIELD_CONNECTION, "Keep-Alive");
  if (username[0])
  {
    char	userpass[1024],
		auth[1024];


    strcpy(auth, "Basic ");
    snprintf(userpass, sizeof(userpass), "%s:%s", username, password);
    HTTP::encode64(auth + 6, sizeof(auth) - 6, userpass);
    
    http->set_field(HTTP_FIELD_AUTHORIZATION, auth);
  }

  if (http->send_mkcol(url))
  {
    http->reconnect();

    if (http->send_mkcol(url))
      return (HTTP_ERROR);
  }

  while ((status = http->update()) == HTTP_CONTINUE);

  http->flush();

  if (status >= HTTP_BAD_REQUEST ||
      strcasecmp(http->get_field(HTTP_FIELD_CONNECTION), "close") == 0)
    http->reconnect();

  return (status);
}
#endif // USE_MKCOL


//
// 'export_file()' - Copy a local file to a remote file.
//

static HTTPStatus
export_file(HTTP       *http,
            const char *username,
            const char *password,
	    const char *filename,
	    const char *url)
{
  HTTPStatus	status;
  FILE		*fp;
  char		buffer[1024];
  int		bytes;


  if ((fp = fopen(filename, "rb")) == NULL)
    return (HTTP_ERROR);

  fseek(fp, 0, SEEK_END);
  bytes = ftell(fp);
  rewind(fp);

  http->clear_fields();
  http->set_field(HTTP_FIELD_USER_AGENT, "flPhoto " FLPHOTO_VERSION);
  http->set_field(HTTP_FIELD_CONNECTION, "Keep-Alive");
  sprintf(buffer, "%d", bytes);
  http->set_field(HTTP_FIELD_CONTENT_LENGTH, buffer);
  if (username[0])
  {
    char	userpass[1024],
		auth[1024];


    strcpy(auth, "Basic ");
    snprintf(userpass, sizeof(userpass), "%s:%s", username, password);
    HTTP::encode64(auth + 6, sizeof(auth) - 6, userpass);
    
    http->set_field(HTTP_FIELD_AUTHORIZATION, auth);
  }

  if (http->send_put(url))
  {
    http->reconnect();
    if (http->send_put(url))
    {
      fclose(fp);
      return (HTTP_ERROR);
    }
  }

  status = HTTP_CONTINUE;

  while ((bytes = fread(buffer, 1, sizeof(buffer), fp)) > 0)
  {
    if (http->check())
    {
      if ((status = http->update()) != HTTP_CONTINUE)
        break;
    }

    http->write(buffer, bytes);
  }

  fclose(fp);

  if (status == HTTP_CONTINUE)
  {
    while ((status = http->update()) == HTTP_CONTINUE);
  }

  http->flush();
  if (status >= HTTP_BAD_REQUEST ||
      strcasecmp(http->get_field(HTTP_FIELD_CONNECTION), "close") == 0)
    http->reconnect();

  return (status);
}


//
// 'export_footer()' - Write a HTML footer.
//

static void
export_footer(FILE       *fp,
              const char *footer)
{
  if (fp)
  {
    fputs("</TABLE></CENTER>\n", fp);
    if (footer && *footer)
    {
      fputs("<HR/>\n", fp);
      export_string(fp, footer, "P");
    }
    fputs("</BODY>\n", fp);
    fputs("</HTML>\n", fp);
    fclose(fp);
  }
}


//
// 'export_header()' - Write a HTML header.
//

static FILE *
export_header(const char *filename,
              const char *header,
              const char *comment)
{
  FILE	*fp;


  if ((fp = fopen(filename, "wb")) != NULL)
  {
    fputs("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\" "
          "\"http://www.w3.org/TR/html4/loose.dtd\">\n", fp);

    fputs("<HTML>\n", fp);
    fputs("<HEAD>\n", fp);
    fputs("<META http-equiv=\"Content-Type\" content=\"text/html; "
          "charset=iso-8859-1\">\n", fp);
    if (header && *header)
      export_string(fp, header, "TITLE");
    else
      export_string(fp, comment, "TITLE");
    fputs("</HEAD>\n", fp);
    fputs("<BODY>\n", fp);
    export_string(fp, header, "H1");
    export_string(fp, comment, "P");
    fputs("<CENTER><TABLE>\n", fp);
  }

  return (fp);
}


//
// 'export_jpeg()' - Write an image file in JPEG format.
//

static int					// O - 0 on failure, 1 on success
export_jpeg(const char      *filename,		// I - File to write to
            Fl_Shared_Image *img,		// I - Image to write
	    int             w,			// I - Width of image
	    int             h,			// I - Height of image
            int             quality)		// I - Image quality
{
  int				i;		// Looping var
  Fl_Shared_Image		*simg;		// Resized image
  uchar				*ptr;		// Pointer to image data
  FILE				*fp;		// File pointer
  struct jpeg_compress_struct	info;		// Compressor info
  struct jpeg_error_mgr		err;		// Error handler info


  // Create the output file...
  if ((fp = fopen(filename, "wb")) == NULL)
  {
    fl_alert(_("Unable to create JPEG image:\n\n%s"), strerror(errno));

    return (0);
  }

  // Resize the image...
  simg = (Fl_Shared_Image *)img->copy(w, h);

  // Setup the JPEG compression stuff...
  info.err = jpeg_std_error(&err);
  jpeg_create_compress(&info);
  jpeg_stdio_dest(&info, fp);

  info.image_width      = simg->w();
  info.image_height     = simg->h();
  info.input_components = simg->d();
  info.in_color_space   = simg->d() == 1 ? JCS_GRAYSCALE : JCS_RGB;

  jpeg_set_defaults(&info);
  jpeg_set_quality(&info, quality, 1);
  jpeg_simple_progression(&info);

  info.optimize_coding = 1;

  // Save the image...
  jpeg_start_compress(&info, 1);

  i = 0;
  while (info.next_scanline < info.image_height)
  {
    ptr = (uchar *)simg->data()[0] + info.next_scanline * simg->w() * simg->d();

    jpeg_write_scanlines(&info, &ptr, 1);
    i ++;
  }

  jpeg_finish_compress(&info);
  jpeg_destroy_compress(&info);

  fclose(fp);

  simg->release();

  return (1);
}


//
// 'export_string()' - Write a string to a HTML file.
//

static void
export_string(FILE       *fp,
              const char *s,
	      const char *el)
{
  if (!s || !*s)
    return;

  if (el)
    fprintf(fp, "<%s>", el);

  while (*s)
  {
    switch (*s)
    {
      case '&' :
          fputs("&amp;", fp);
	  break;
      case '<' :
          fputs("&lt;", fp);
	  break;
      case '\n' :
          fputs("<BR/>", fp);
	  break;
      default :
          putc(*s, fp);
	  break;
    }

    s ++;
  }

  if (el)
    fprintf(fp, "</%s>", el);
}


//
// End of "$Id: export.cxx,v 1.17 2003/09/05 12:47:07 easysw Exp $".
//
