//
// "$Id: print.cxx,v 1.30 2003/09/08 16:59:12 easysw Exp $"
//
// Print methods.
//
// Copyright 2002-2003 by Michael Sweet.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// Contents:
//
//   flphoto::print_album_cb() - Print an album.
//   flphoto::print_image_cb() - Print an image.
//   flphoto::print_cb()       - Generate the print file.
//   make_optimal_image()      - Make an optimally-sized copy of an image.
//   write_ascii85()           - Print binary data as a series of base-85 numbers.
//   write_calendar()          - Write a calendar.
//   write_end_page()          - Show the current page.
//   write_image()             - Write an image at the specified location, mode, and
//   write_matting()           - Write the matting around an image...
//   write_prolog()            - Write the PostScript prolog...
//   write_start_page()        - Start a page.
//   write_trailer()           - Write the PostScript trailer...
//

#include "flphoto.h"
#include "i18n.h"
#include <stdio.h>
#include <stdlib.h>
#include "flstring.h"
#include <unistd.h>
#include <errno.h>
#include <time.h>
#include "Fl_Print_Dialog.H"
#include <FL/Fl_File_Chooser.H>


//
// Local globals...
//

static Fl_Print_Dialog	*print_dialog = 0;


//
// Local functions...
//

static Fl_Shared_Image	*make_optimal_image(Fl_Shared_Image *img, int width,
			                    int length);
static void		write_ascii85(FILE *fp, uchar *data, int length);
static void		write_asciihex(FILE *fp, uchar *data, int length);
static void		write_calendar(FILE *fp, int firstday, int month,
			               int year, int left, int bottom,
				       int right, int top);
static void		write_end_page(FILE *fp);
static void		write_image(FILE *fp, Fl_Shared_Image *img,
			            int mode, int quality, int left, int bottom,
				    int right, int top, int autorotate);
static void		write_matting(FILE *fp, uchar *data, int type, int width,
			              int left, int bottom, int right, int top);
static void		write_prolog(FILE *fp, int mode, int left, int bottom,
				     int right, int top);
static void		write_start_page(FILE *fp, int page, int width,
			                 int length, int commands);
static void		write_trailer(FILE *fp, int pages);


//
// 'flphoto::print_album_cb()' - Print an album.
//

void
flphoto::print_album_cb()
{
  int	format;					// Print format


  if (!print_dialog)
  {
    print_dialog = new Fl_Print_Dialog();
    print_dialog->callback(print_cb, this);
  }

  flphoto::prefs.get("print_album_format", format, Fl_Print_Dialog::PRINT_INDEX);

  print_dialog->format(format);

  print_dialog->which(Fl_Print_Dialog::PRINT_ALL);

  print_dialog->matcolors(browser_);

  print_dialog->show();

  while (print_dialog->shown())
    Fl::wait();

  flphoto::prefs.set("print_album_format", print_dialog->format());
}


//
// 'flphoto::print_image_cb()' - Print an image.
//

void
flphoto::print_image_cb()
{
  int	i;					// Looping var
  int	format;					// Print format


  if (!print_dialog)
  {
    print_dialog = new Fl_Print_Dialog();
    print_dialog->callback(print_cb, this);
  }

  flphoto::prefs.get("print_image_format", format, Fl_Print_Dialog::PRINT_1UP);

  print_dialog->format(format);

  print_dialog->which(Fl_Print_Dialog::PRINT_CURRENT);

  for (i = 0; i < browser_->count(); i ++)
    if (browser_->selected(i) && browser_->value(i) != image_item_)
    {
      print_dialog->which(Fl_Print_Dialog::PRINT_SELECTED);
      break;
    }

  print_dialog->matcolors(browser_);

  print_dialog->show();

  while (print_dialog->shown())
    Fl::wait();

  flphoto::prefs.set("print_image_format", print_dialog->format());
}


//
// 'flphoto::print_cb()' - Generate the print file.
//

const char *
flphoto::print_cb(Fl_Print_Dialog *pd,
                  void            *d)
{
  flphoto		*album = (flphoto *)d;
#if !defined(WIN32) || defined(__CYGWIN__)
  int			fd;
#endif // !WIN32 || __CYGWIN__
  FILE			*fp;
  char			tmpdir[1024];
  int			page;
  int			left, bottom,
			right, top,
			width, length,
			pagew, pagel, pageo,
			loff, toff,
			ltemp, btemp,
			imgw, imgh, imgo,
			mattype;
  uchar			matrgb[3],
			matdata[64 * 64 * 3];
  float			matwidth,
			imgwidth,
			imgheight;
  int			imgcols,
			imgrows;
  int			i, x, y, copy, progval, progmax;
  int			day, month, year, orient;
  Fl_Shared_Image	*img;
  Fl_Image_Browser::ITEM *item;
  const char		*label;


  if (pd->printer())
  {
    // First create a temporary file for the print data...
    flphoto::prefs.getUserdataPath(tmpdir, sizeof(tmpdir));

#if defined(WIN32) && !defined(__CYGWIN__)
    // NOTE: this is just a temporary fix...
    snprintf(album->album_printname_, sizeof(album->album_printname_),
             "%s/flphoto%06d", tmpdir, GetCurrentProcessId);

    if ((fp = fopen(album->album_printname_, "wb")) == NULL)
    {
      unlink(album->album_printname_);
      return (0);
    }
#else
    snprintf(album->album_printname_, sizeof(album->album_printname_),
             "%s/flphotoXXXXXX", tmpdir);

    if ((fd = mkstemp(album->album_printname_)) < 0)
      return (0);

    if ((fp = fdopen(fd, "wb")) == NULL)
    {
      close(fd);
      unlink(album->album_printname_);
      return (0);
    }
#endif // WIN32 && !__CYGWIN__
  }
  else
  {
    const char	*f;


    if ((f = fl_file_chooser(_("Print To File?"),
                             _("PostScript Files (*.ps)"), 0, 1)) == NULL)
      return (0);

    if (!access(f, 0))
      if (!fl_ask(_("Print file %s already exists.\nOK to overwrite?"), f))
        return (0);

    strlcpy(album->album_printname_, f, sizeof(album->album_printname_));

    if ((fp = fopen(f, "wb")) == NULL)
    {
      fl_alert(_("Unable to create %s:\n\n%s"), f, strerror(errno));
      return (0);
    }
  }

  // Get the media dimensions...
  pd->imageable_area(left, bottom, right, top);
  pd->paper_dimension(width, length);

  // Figure out how many images are getting printed...
  switch (pd->which())
  {
    case Fl_Print_Dialog::PRINT_CURRENT :
        progmax = pd->copies();
        break;
    case Fl_Print_Dialog::PRINT_SELECTED :
        for (i = 0, progmax = 0; i < album->browser_->count(); i ++)
	  if (album->browser_->selected(i))
            progmax += pd->copies();

        if (progmax > 0)
          break;

	pd->which(Fl_Print_Dialog::PRINT_ALL);

    default :
        progmax = pd->copies() * album->browser_->count();
        break;
  }

  // Then write the standard PostScript prolog...
  write_prolog(fp, pd->mode(), left, bottom, right, top);
  page = 0;

  switch (pd->format())
  {
    case Fl_Print_Dialog::PRINT_INDEX :
	for (copy = 0, progval = 0; copy < pd->copies(); copy ++)
	{
          for (i = 0, x = -1, y = -1; i < album->browser_->count(); i ++)
	  {
	    if (pd->which() == Fl_Print_Dialog::PRINT_SELECTED)
	    {
	      for (; i < album->browser_->count(); i ++)
		if (album->browser_->selected(i))
	          break;

              if (i >= album->browser_->count())
		break;
            }

	    if (x < 0 || y < 0)
	    {
	      page ++;
	      write_start_page(fp, page, width, length, !pd->have_ppd());

	      fprintf(fp, "(Page %d) %d %d L\n",
	              page, (left + right) / 2, bottom);

	      x = left;
	      y = top;
	    }

            progval ++;
	    pd->progress_show(100 * progval / progmax, "Printing page %d...",
	                      page);

            item = album->browser_->value(i);
	    switch (pd->quality())
	    {
	      case 0 : // Draft
	      case 1 : // Normal
        	  img = item->thumbnail;
	          break;
              default : // Best
	          img = make_optimal_image(album->browser_->load_item(i),
		                           54, 54);
		  break;
	    }

	    write_image(fp, img, pd->mode(), pd->quality(), x, y - 54, x + 54, y, 0);

            if (img != item->image && img != item->thumbnail)
	      img->release();

            if ((label = strrchr(item->filename, '/')) != NULL)
	      label ++;
	    else
	      label = item->filename;

            fprintf(fp, "(%s) %d %d L\n", label, x + 27, y - 63);

	    if (item->image && item != album->image_item_ && !item->changed)
	    {
	      item->image->release();
	      item->image = 0;
	    }

	    x += 72;

	    if ((x + 54) > right)
	    {
	      x = left;
	      y -= 72;

	      if ((y - 63) < (bottom + 18))
	      {
		write_end_page(fp);
		x = y = -1;
	      }
	    }
	  }

	  if (x >= 0 || y >= 0)
	    write_end_page(fp);
	}
        break;

    case Fl_Print_Dialog::PRINT_1UP :
	for (copy = 0, progval = 0; copy < pd->copies(); copy ++)
	{
          for (i = 0; i < album->browser_->count(); i ++)
	  {
	    if (pd->which() == Fl_Print_Dialog::PRINT_SELECTED)
	    {
	      for (; i < album->browser_->count(); i ++)
		if (album->browser_->selected(i))
	          break;

              if (i >= album->browser_->count())
		break;
            }
	    else if (pd->which() == Fl_Print_Dialog::PRINT_CURRENT)
	    {
	      for (i = 0; i < album->browser_->count(); i ++)
		if (album->browser_->value(i) == album->image_item_)
		  break;

              if (i >= album->browser_->count())
		break;
            }

	    page ++;
	    progval ++;
	    pd->progress_show(100 * progval / progmax, "Printing page %d...",
	                      page);
	    write_start_page(fp, page, width, length, !pd->have_ppd());

            item = album->browser_->value(i);

	    switch (pd->quality())
	    {
	      case 0 : // Draft
	          img = make_optimal_image(album->browser_->load_item(i),
		                           (right - left) / 4,
					   (top - bottom) / 4);
	          break;
	      case 1 : // Normal
	          img = make_optimal_image(album->browser_->load_item(i),
		                           right - left, top - bottom);
	          break;
              default : // Best
                  img = album->browser_->load_item(i);
		  break;
	    }

	    write_image(fp, img, pd->mode(), pd->quality(), left, bottom,
	        	right, top, 1);

            if (img != item->image && img != item->thumbnail)
	      img->release();

	    if (item->image && item != album->image_item_ && !item->changed)
	    {
	      item->image->release();
	      item->image = 0;
	    }

            write_end_page(fp);

	    if (pd->which() == Fl_Print_Dialog::PRINT_CURRENT)
	      break;
	  }
	}
        break;

    case Fl_Print_Dialog::PRINT_2UP :
	for (copy = 0, y = -1, progval = 0; copy < pd->copies(); copy ++)
	{
          for (i = 0; i < album->browser_->count(); i ++)
	  {
	    if (pd->which() == Fl_Print_Dialog::PRINT_SELECTED)
	    {
	      for (; i < album->browser_->count(); i ++)
		if (album->browser_->selected(i))
	          break;

              if (i >= album->browser_->count())
		break;
            }
	    else if (pd->which() == Fl_Print_Dialog::PRINT_CURRENT)
	    {
	      for (i = 0; i < album->browser_->count(); i ++)
		if (album->browser_->value(i) == album->image_item_)
		  break;

              if (i >= album->browser_->count())
		break;
            }

            if (y < 0)
	    {
	      page ++;
	      write_start_page(fp, page, width, length, !pd->have_ppd());
	      y = top;
	    }

            progval ++;
	    pd->progress_show(100 * progval / progmax, "Printing page %d...",
	                      page);

            item = album->browser_->value(i);

	    switch (pd->quality())
	    {
	      case 0 : // Draft
	          img = make_optimal_image(album->browser_->load_item(i),
		                           (right - left) / 4,
					   (top - bottom) / 8);
	          break;
	      case 1 : // Normal
	          img = make_optimal_image(album->browser_->load_item(i),
		                           right - left, (top - bottom) / 2);
	          break;
              default : // Best
                  img = album->browser_->load_item(i);
		  break;
	    }

	    write_image(fp, img, pd->mode(), pd->quality(), left,
	                y - (top - bottom) / 2 + 9, right, y, 1);

            if (img != item->image && img != item->thumbnail)
	      img->release();

	    if (item->image && item != album->image_item_ && !item->changed)
	    {
	      item->image->release();
	      item->image = 0;
	    }

            y -= (top - bottom) / 2 + 9;
	    if (y < bottom)
	    {
              write_end_page(fp);
	      y = -1;
	    }

	    if (pd->which() == Fl_Print_Dialog::PRINT_CURRENT)
	      break;
	  }
	}

	if (y >= 0)
	  write_end_page(fp);
        break;

    case Fl_Print_Dialog::PRINT_4UP :
	for (copy = 0, x = -1, y = -1, progval = 0; copy < pd->copies(); copy ++)
	{
          for (i = 0; i < album->browser_->count(); i ++)
	  {
	    if (pd->which() == Fl_Print_Dialog::PRINT_SELECTED)
	    {
	      for (; i < album->browser_->count(); i ++)
		if (album->browser_->selected(i))
	          break;

              if (i >= album->browser_->count())
		break;
            }
	    else if (pd->which() == Fl_Print_Dialog::PRINT_CURRENT)
	    {
	      for (i = 0; i < album->browser_->count(); i ++)
		if (album->browser_->value(i) == album->image_item_)
		  break;

              if (i >= album->browser_->count())
		break;
            }

            if (x < 0 || y < 0)
	    {
	      page ++;
	      write_start_page(fp, page, width, length, !pd->have_ppd());
	      x = left;
	      y = top;
	    }

            progval ++;
	    pd->progress_show(100 * progval / progmax, "Printing page %d...",
	                      page);

            item = album->browser_->value(i);

	    switch (pd->quality())
	    {
	      case 0 : // Draft
	          img = make_optimal_image(album->browser_->load_item(i),
		                           (right - left) / 8,
					   (top - bottom) / 8);
	          break;
	      case 1 : // Normal
	          img = make_optimal_image(album->browser_->load_item(i),
		                           (right - left) / 2,
					   (top - bottom) / 2);
	          break;
              default : // Best
                  img = album->browser_->load_item(i);
		  break;
	    }

	    write_image(fp, img, pd->mode(), pd->quality(), x,
	                y - (top - bottom) / 2 + 9,
			x + (right - left) / 2 - 9, y, 1);

            if (img != item->image && img != item->thumbnail)
	      img->release();

	    if (item->image && item != album->image_item_ && !item->changed)
	    {
	      item->image->release();
	      item->image = 0;
	    }

            x += (right - left) / 2 + 9;
	    if (x >= right)
	    {
	      x = left;
              y -= (top - bottom) / 2 + 9;
	      if (y < bottom)
	      {
        	write_end_page(fp);
		x = y = -1;
	      }
	    }

	    if (pd->which() == Fl_Print_Dialog::PRINT_CURRENT)
	      break;
	  }
	}

	if (x >= 0 || y >= 0)
	  write_end_page(fp);
        break;

    case Fl_Print_Dialog::PRINT_PORTRAIT :
        progmax *= 8;

	for (copy = 0, progval = 0; copy < pd->copies(); copy ++)
	{
          for (i = 0; i < album->browser_->count(); i ++)
	  {
	    if (pd->which() == Fl_Print_Dialog::PRINT_SELECTED)
	    {
	      for (; i < album->browser_->count(); i ++)
		if (album->browser_->selected(i))
	          break;

              if (i >= album->browser_->count())
		break;
            }
	    else if (pd->which() == Fl_Print_Dialog::PRINT_CURRENT)
	    {
	      for (i = 0; i < album->browser_->count(); i ++)
		if (album->browser_->value(i) == album->image_item_)
		  break;

              if (i >= album->browser_->count())
		break;
            }

	    page ++;
            progval ++;
	    pd->progress_show(100 * progval / progmax, "Printing page %d...",
	                      page);
	    write_start_page(fp, page, width, length, !pd->have_ppd());

            item = album->browser_->value(i);

            int hh = (top - bottom) / 4 - 9;
            int hm = 2 * (top - bottom) / 3;
            int hs = top - bottom - hm - 9;
            int hp = (top - bottom) / 6 - 9;
            int ww = right - left - hh - 18;

            // Main portrait
	    switch (pd->quality())
	    {
	      case 0 : // Draft
	          img = make_optimal_image(album->browser_->load_item(i),
					   ww / 4, (top - bottom - hh - 9) / 4);
	          break;
	      case 1 : // Normal
	          img = make_optimal_image(album->browser_->load_item(i),
		                           ww, top - bottom - hh - 9);
	          break;
              default : // Best
                  img = album->browser_->load_item(i);
		  break;
	    }

            fputs("gsave\n", fp);
	    fprintf(fp, "%d %d %d %d rectclip\n", left, top - hm, ww, hm);
	    write_image(fp, img, pd->mode(), pd->quality(),
	                left, bottom + (hh + hs) / 2 + 9,
			left + ww, top + (hs - hh) / 2, 1);
            fputs("grestore\n", fp);

            progval ++;
	    pd->progress_show(100 * progval / progmax, "Printing page %d...",
	                      page);

            if (img != item->image && img != item->thumbnail)
	      img->release();

            // Smaller portrait underneath...
	    switch (pd->quality())
	    {
	      case 0 : // Draft
		  img = make_optimal_image(album->browser_->load_item(i),
		                           ww / 4, hs / 4 + 2);
	          break;
	      case 1 : // Normal
		  img = make_optimal_image(album->browser_->load_item(i),
		                           ww, hs + 8);
	          break;
              default : // Best
                  img = album->browser_->load_item(i);
		  break;
	    }

            fputs("gsave\n", fp);
	    fprintf(fp, "%d %d %d %d rectclip\n", left, bottom, ww, hs);
	    write_image(fp, img, pd->mode(), pd->quality(),
	                left, bottom - 4, left + ww, bottom + hs + 4, 1);
            fputs("grestore\n", fp);

            progval ++;
	    pd->progress_show(100 * progval / progmax, "Printing page %d...",
	                      page);

            if (img != item->image && img != item->thumbnail)
	      img->release();

	    switch (pd->quality())
	    {
	      case 0 : // Draft
		  img = make_optimal_image(album->browser_->load_item(i),
		                           hh / 4, hp / 4);
	          break;
	      case 1 : // Normal
		  img = make_optimal_image(album->browser_->load_item(i),
		                           hh, hp);
	          break;
              default : // Best
		  break;
	    }


            // Wallet-size portraits to the right...
            for (y = top; y > bottom; y -= hp + 11)
	    {
              progval ++;
	      pd->progress_show(100 * progval / progmax, "Printing page %d...",
	                	page);
	      write_image(fp, img, pd->mode(), pd->quality(),
		          right - hh, y - hp, right, y, 1);
            }

            if (img != item->image && img != item->thumbnail)
	      img->release();

	    if (item->image && item != album->image_item_ && !item->changed)
	    {
	      item->image->release();
	      item->image = 0;
	    }

            write_end_page(fp);

	    if (pd->which() == Fl_Print_Dialog::PRINT_CURRENT)
	      break;
	  }
	}
        break;

    case Fl_Print_Dialog::PRINT_CALENDAR :
	for (copy = 0, progval = 0; copy < pd->copies(); copy ++)
	{
	  pd->calendar(day, month, year, orient);

          for (i = 0; i < album->browser_->count(); i ++)
	  {
	    if (pd->which() == Fl_Print_Dialog::PRINT_SELECTED)
	    {
	      for (; i < album->browser_->count(); i ++)
		if (album->browser_->selected(i))
	          break;

              if (i >= album->browser_->count())
		break;
            }
	    else if (pd->which() == Fl_Print_Dialog::PRINT_CURRENT)
	    {
	      for (i = 0; i < album->browser_->count(); i ++)
		if (album->browser_->value(i) == album->image_item_)
		  break;

              if (i >= album->browser_->count())
		break;
            }

	    page ++;
	    progval ++;
	    pd->progress_show(100 * progval / progmax, "Printing page %d...",
	                      page);
	    write_start_page(fp, page, width, length, !pd->have_ppd());

            item = album->browser_->value(i);
            img  = album->browser_->load_item(i);

            if (orient == 1 ||
	        (orient < 0 &&
		 ((img->w() > img->h() && width > length) ||
	          (img->w() < img->h() && width < length))))
            {
	      // Show page in landscape orientation...
	      pagew = top - bottom;
	      pagel = right - left;
	      pageo = 1;

              imgh = pagel;
	      imgw = imgh * img->w() / img->h();

              if (imgw > (pagew / 2))
	      {
                imgw = pagew / 2;
	        imgh = imgw * img->h() / img->w();
	      }

              fprintf(fp, "%d 0 translate 90 rotate\n", width);
	      fprintf(fp, "%d %d translate\n", bottom, width - right);
	    }
	    else
	    {
	      pagew = right - left;
	      pagel = top - bottom;
	      pageo = 0;

              imgw = pagew;
	      imgh = imgw * img->h() / img->w();

              if (imgh > (pagel / 2))
	      {
                imgh = pagel / 2;
	        imgw = imgh * img->w() / img->h();
	      }

	      fprintf(fp, "%d %d translate\n", left, bottom);
	    }

	    switch (pd->quality())
	    {
	      case 0 : // Draft
	          img = make_optimal_image(album->browser_->load_item(i),
		                           imgw / 4, imgh / 4);
	          break;
	      case 1 : // Normal
	          img = make_optimal_image(album->browser_->load_item(i),
		                           imgw, imgh);
	          break;
              default : // Best
                  img = album->browser_->load_item(i);
		  break;
	    }

            if (pageo)
	    {
	      // Put image on the left side...
	      write_image(fp, img, pd->mode(), pd->quality(),
	                  0, 0, imgw, pagel, 0);
            }
	    else
	    {
	      // Put image on the top...
	      write_image(fp, img, pd->mode(), pd->quality(),
	                  0, pagel - imgh, pagew, pagel, 0);
            }

            if (img != item->image && img != item->thumbnail)
	      img->release();

	    if (item->image && item != album->image_item_ && !item->changed)
	    {
	      item->image->release();
	      item->image = 0;
	    }

            if (pageo)
	    {
	      // Put calendar on the right side...
              write_calendar(fp, day, month, year, imgw + 18, 0, pagew, pagel);
            }
	    else
	    {
	      // Put calendar on the bottom...
              write_calendar(fp, day, month, year, 0, 0, pagew, pagel - imgh);
            }

            write_end_page(fp);

	    if (pd->which() == Fl_Print_Dialog::PRINT_CURRENT)
	      break;

            month ++;
	    if (month > 11)
	    {
	      month = 0;
	      year ++;
	    }
	  }
	}
        break;


    case Fl_Print_Dialog::PRINT_MATTED :
        pd->matting(mattype, matwidth, matrgb, imgwidth, imgheight, imgcols,
	            imgrows);
        pd->matimage(matdata);

        imgw = (int)((imgwidth + matwidth) * imgcols + matwidth);
	imgh = (int)((imgheight + matwidth) * imgrows + matwidth);

        if ((imgw > imgh && width < length) ||
	    (imgw < imgh && width > length))
        {
	  // Show page in landscape orientation...
	  pagew = top - bottom;
	  pagel = right - left;
	  pageo = 1;
	}
	else
	{
	  pagew = right - left;
	  pagel = top - bottom;
	  pageo = 0;
	}

        loff = (int)(pagew - (imgcols + 1) * matwidth -
	             imgcols * imgwidth) / 2;
	toff = pagel - (int)(pagel - (imgrows + 1) * matwidth -
	                     imgrows * imgheight) / 2;

	for (copy = 0, x = -1, y = -1, progval = 0; copy < pd->copies(); copy ++)
	{
          for (i = 0; i < album->browser_->count(); i ++)
	  {
	    if (pd->which() == Fl_Print_Dialog::PRINT_SELECTED)
	    {
	      for (; i < album->browser_->count(); i ++)
		if (album->browser_->selected(i))
	          break;

              if (i >= album->browser_->count())
		break;
            }
	    else if (pd->which() == Fl_Print_Dialog::PRINT_CURRENT)
	    {
	      for (i = 0; i < album->browser_->count(); i ++)
		if (album->browser_->value(i) == album->image_item_)
		  break;

              if (i >= album->browser_->count())
		break;
            }

            if (x < 0 || y < 0)
	    {
	      page ++;
	      write_start_page(fp, page, width, length, !pd->have_ppd());
	      x = 0;
	      y = 0;

              if (pageo)
              {
		// Show page in landscape orientation...
        	fprintf(fp, "%d 0 translate 90 rotate\n", width);
		fprintf(fp, "%d %d translate\n", bottom, width - right);
	      }
	      else
		fprintf(fp, "%d %d translate\n", left, bottom);

	    }

            progval ++;
	    pd->progress_show(100 * progval / progmax, "Printing page %d...",
	                      page);

            item = album->browser_->value(i);

            // Figure out the maximum image size...
	    img  = album->browser_->load_item(i);

            if ((img->w() > img->h() && imgheight > imgwidth) ||
                (img->w() < img->h() && imgwidth > imgheight))
            {
	      // Rotate image size...
	      imgo = 1;
              imgw = (int)imgheight;
	      imgh = imgw * img->h() / img->w();
	      if (imgh < (int)imgwidth)
	      {
		imgh = (int)imgwidth;
		imgw = imgh * img->w() / img->h();
	      }
	    }
	    else
	    {
	      // Don't rotate the image...
	      imgo = 0;
              imgw = (int)imgwidth;
	      imgh = imgw * img->h() / img->w();
	      if (imgh < (int)imgheight)
	      {
		imgh = (int)imgheight;
		imgw = imgh * img->w() / img->h();
	      }
	    }

	    switch (pd->quality())
	    {
	      case 0 : // Draft
	          img = make_optimal_image(album->browser_->load_item(i),
		                           imgw / 2, imgh / 2);
	          break;
	      case 1 : // Normal
	          img = make_optimal_image(album->browser_->load_item(i),
		                           imgw, imgh);
	          break;
              default : // Best
                  img = album->browser_->load_item(i);
		  break;
	    }

            ltemp = (int)(loff + x * (imgwidth + matwidth) + matwidth);
	    btemp = (int)(toff - (y + 1) * (imgheight + matwidth));

            fprintf(fp, "gsave %d %d %d %d rectclip\n", ltemp, btemp,
	            (int)imgwidth, (int)imgheight);

            if (imgo)
	      write_image(fp, img, pd->mode(), pd->quality(),
	                  (int)(ltemp - (imgh - imgwidth) / 2),
	                  (int)(btemp - (imgw - imgheight) / 2),
	                  (int)(ltemp - (imgh - imgwidth) / 2) + imgh,
	                  (int)(btemp - (imgw - imgheight) / 2 + imgw),
			  1);
            else
	      write_image(fp, img, pd->mode(), pd->quality(),
	                  (int)(ltemp - (imgw - imgwidth) / 2),
	                  (int)(btemp - (imgh - imgheight) / 2),
	                  (int)(ltemp - (imgw - imgwidth) / 2) + imgw,
	                  (int)(btemp - (imgh - imgheight) / 2 + imgh),
			  1);

            fputs("grestore\n", fp);

            write_matting(fp, matdata, mattype, (int)matwidth,
	                  (int)(ltemp - matwidth),
	                  (int)(btemp - matwidth),
	                  (int)(ltemp + imgwidth + matwidth),
	                  (int)(btemp + imgheight + matwidth));

            if (img != item->image && img != item->thumbnail)
	      img->release();

	    if (item->image && item != album->image_item_ && !item->changed)
	    {
	      item->image->release();
	      item->image = 0;
	    }

            x ++;
	    if (x >= imgcols)
	    {
	      x = 0;
              y ++;
	      if (y >= imgrows)
	      {
        	write_end_page(fp);
		x = y = -1;
	      }
	    }

	    if (pd->which() == Fl_Print_Dialog::PRINT_CURRENT)
	      break;
	  }
	}

	if (x >= 0 || y >= 0)
	  write_end_page(fp);
        break;
  }

  pd->progress_hide();

  write_trailer(fp, page);
  fclose(fp);

  return (album->album_printname_);
}


//
// 'make_optimal_image()' - Make an optimally-sized copy of an image.
//

static Fl_Shared_Image	*
make_optimal_image(Fl_Shared_Image *img,
                   int             width,
		   int             length)
{
  int	i;
  int	neww, newh,
	minw, maxw,
	minh, maxh;


  // Figure out the size of the image in the bounding box...
  width  *= 2;
  length *= 2;

  if (img->w() > img->h() && length > width)
  {
    minh = width;
    minw = minh * img->w() / img->h();

    if (minw > length)
    {
      minw = length;
      minh = minw * img->h() / img->w();
    }
  }
  else
  {
    minw = width;
    minh = minw * img->h() / img->w();
    if (minh > length)
    {
      minh = length;
      minw = minh * img->w() / img->h();
    }
  }

  // Now figure out the optimal size of the bounding box...
  maxw = minw * 2;
  maxh = minh * 2;

#ifdef DEBUG
  printf("image = %dx%d, min = %dx%d, max = %dx%d\n", img->w(), img->h(),
         minw, minh, maxw, minh);
#endif // DEBUG

  for (i = 1; i < 20; i ++)
  {
    neww = img->w() / i;
    newh = img->h() / i;

    if (neww <= maxw && newh <= maxh)
      break;
  }

#ifdef DEBUG
  printf("new image = %dx%d\n", neww, newh);
#endif // DEBUG

  // Return the optimal image...
  return ((Fl_Shared_Image *)img->copy(neww, newh));
}


//
// 'write_ascii85()' - Print binary data as a series of base-85 numbers.
//

static void
write_ascii85(FILE  *fp,	// I - File to write to
              uchar *data,	// I - Data to print
	      int  length)	// I - Number of bytes to print
{
  unsigned	b;		// Binary data word
  unsigned char	c[5];		// ASCII85 encoded chars
  int		col;		// Current column


  col = 0;
  while (length > 3)
  {
    b = (((((data[0] << 8) | data[1]) << 8) | data[2]) << 8) | data[3];

    if (b == 0)
    {
      putc('z', fp);
      col ++;
    }
    else
    {
      c[4] = (b % 85) + '!';
      b /= 85;
      c[3] = (b % 85) + '!';
      b /= 85;
      c[2] = (b % 85) + '!';
      b /= 85;
      c[1] = (b % 85) + '!';
      b /= 85;
      c[0] = b + '!';

      fwrite(c, 5, 1, fp);
      col += 5;
    }

    data += 4;
    length -= 4;

    if (col >= 75)
    {
      putc('\n', fp);
      col = 0;
    }
  }

  if (length > 0)
  {
    memset(data + length, 0, 4 - length);
    b = (((((data[0] << 8) | data[1]) << 8) | data[2]) << 8) | data[3];

    c[4] = (b % 85) + '!';
    b /= 85;
    c[3] = (b % 85) + '!';
    b /= 85;
    c[2] = (b % 85) + '!';
    b /= 85;
    c[1] = (b % 85) + '!';
    b /= 85;
    c[0] = b + '!';

    fwrite(c, length + 1, 1, fp);
  }

  fputs("~>\n", fp);
}


//
// 'write_asciihex()' - Print binary data as a series of hex numbers.
//

static void
write_asciihex(FILE  *fp,	// I - File to write to
               uchar *data,	// I - Data to print
	       int  length)	// I - Number of bytes to print
{
  uchar			b;	// Current byte
  int			col;	// Current column
  static const char	*hex = "0123456789ABCDEF";
				// Hex digits...


  putc('<', fp);
  col = 0;
  while (length > 0)
  {
    b = *data >> 4;
    putc(hex[b], fp);
    putc(hex[*data & 15], fp);

    col += 2;
    length --;
    data ++;

    if (col >= 78)
    {
      putc('\n', fp);
      col = 0;
    }
  }

  fputs(">\n", fp);
}


//
// 'write_calendar()' - Write a calendar.
//

static void
write_calendar(FILE *fp,		// I - File to write
               int  firstday,		// I - Start day
               int  month,		// I - Month (0-11)
	       int  year,		// I - Year
               int  left,		// I - Left edge
	       int  bottom,		// I - Bottom edge
	       int  right,		// I - Right edge
	       int  top)		// I - Top edge
{
  int			x, y;
  int			width, height;
  int			wday,
			day, days,
			week, weeks;
  struct tm		start_date;
  time_t		start_time;
  struct tm		end_date;
  time_t		end_time;
  static const char	*months[] =
			{
			  _("January"),
			  _("February"),
			  _("March"),
			  _("April"),
			  _("May"),
			  _("June"),
			  _("July"),
			  _("August"),
			  _("September"),
			  _("October"),
			  _("November"),
			  _("December")
			};
  static const char	*wdays[] =
			{
			  _("Sunday"),
			  _("Monday"),
			  _("Tuesday"),
			  _("Wednesday"),
			  _("Thursday"),
			  _("Friday"),
			  _("Saturday")
			};


  memset(&start_date, 0, sizeof(start_date));
  start_date.tm_mon  = month;
  start_date.tm_mday = 1;
  start_date.tm_year = year - 1900;
  start_date.tm_hour = 12;
  start_time         = mktime(&start_date);

  memset(&end_date, 0, sizeof(end_date));

  if (month < 11)
  {
    end_date.tm_mon  = month + 1;
    end_date.tm_year = year - 1900;
  }
  else
    end_date.tm_year = year - 1900 + 1;

  end_date.tm_mday = 1;
  end_date.tm_hour = 12;
  end_time         = mktime(&end_date);

  days   = (end_time - start_time) / 86400;
  wday   = (start_date.tm_wday + firstday) % 7;
  weeks  = (wday + days + 6) / 7;
  width  = right - left;
  height = top - bottom - 45 - 18;

  fprintf(fp, "(%s %d) %d %d T\n", months[month], year,
          left + width / 2, top - 36);

  for (day = 0; day <= 7; day ++)
  {
    x = left + day * width / 7;
    fprintf(fp, "%d %d MO %d %d LI\n", x, bottom, x, bottom + height + 18);

    if (day < 7)
    {
      x = left + (2 * day + 1) * width / 14;
      fprintf(fp, "(%s) %d %d L\n", wdays[(14 + day - firstday) % 7],
              x, bottom + height + 5);
    }
  }

  for (week = 0; week <= weeks; week ++)
  {
    y = bottom + week * height / weeks;
    fprintf(fp, "%d %d MO %d %d LI\n", left, y, right, y);
  }

  y = bottom + height + 18;
  fprintf(fp, "%d %d MO %d %d LI\n", left, y, right, y);

  for (day = 1, week = 0; day <= days; day ++)
  {
    x = left + (wday + 1) * width / 7 - 4;
    y = bottom + height - week * height / weeks - 20;

    fprintf(fp, "(%d) %d %d R\n", day, x, y);

    wday ++;
    if (wday > 6)
    {
      wday = 0;
      week ++;
    }
  }
}


//
// 'write_end_page()' - Show the current page.
//

static void
write_end_page(FILE *fp)
{
  fputs("grestore\n", fp);
  fputs("showpage\n", fp);
  fputs("%%PageTrailer\n", fp);
}


//
// 'write_image()' - Write an image at the specified location, mode, and
//                   quality.
//

static void
write_image(FILE            *fp,
            Fl_Shared_Image *img,
            int             mode,
	    int             quality,
	    int             left,
	    int             bottom,
	    int             right,
	    int             top,
	    int             autorotate)
{
  float			x, y, w, h;
  int			rotate;
  Fl_RGB_Image		*gray;


  if (img->d() == 1)
    mode = Fl_Print_Dialog::MODE_GRAYSCALE;

  rotate = 0;
  if (autorotate &&
      ((img->w() > img->h() && (top - bottom) > (right - left)) ||
       (img->w() < img->h() && (right - left) > (top - bottom))))
    rotate = 1;

#ifdef DEBUG
  printf("write_image(fp=%p, img=%p, mode=%d, quality=%d, left=%d,\n"
         "            bottom=%d, right=%d, top=%d, autorotate=%d)\n",
         fp, img, mode, quality, left, bottom, right, top, autorotate);
#endif // DEBUG

  if (rotate)
  {
    w = right - left;
    h = w * img->w() / img->h();

    if (h > (top - bottom))
    {
      h = top - bottom;
      w = h * img->h() / img->w();
    }
  }
  else
  {
    w = right - left;
    h = w * img->h() / img->w();
    if (h > (top - bottom))
    {
      h = top - bottom;
      w = h * img->w() / img->h();
    }
  }

#ifdef DEBUG
  printf("    rotate=%d, w=%.1f, h=%.1f\n", rotate, w, h);
#endif // DEBUG

  x = left + (right - left - w) * 0.5;
  y = bottom + (top - bottom - h) * 0.5;

  fputs("gsave\n", fp);

  if (rotate)
  {
    fprintf(fp, "%.1f %.1f translate\n", x, y);
    fprintf(fp, "%.3f %.3f scale\n", w / img->h(), h / img->w());
  }
  else
  {
    fprintf(fp, "%.1f %.1f translate\n", x, y + h);
    fprintf(fp, "%.3f %.3f scale\n", w / img->w(), h / img->h());
  }

  if (mode == Fl_Print_Dialog::MODE_GRAYSCALE)
    fputs("/DeviceGray setcolorspace\n", fp);
  else
    fputs("/DeviceRGB setcolorspace\n", fp);

  fprintf(fp, "<<"
              "/ImageType 1"
	      "/Width %d"
	      "/Height %d"
	      "/BitsPerComponent 8",
	  img->w(), img->h());

  if (mode == Fl_Print_Dialog::MODE_GRAYSCALE)
    fputs("/Decode[0 1]", fp);
  else
    fputs("/Decode[0 1 0 1 0 1]", fp);

  fputs("/DataSource currentfile /ASCII85Decode filter", fp);

  fputs("/Interpolate true", fp);

  if (rotate)
    fputs("/ImageMatrix[0 1 1 0 0 0]>>image\n", fp);
  else
    fputs("/ImageMatrix[1 0 0 -1 0 1]>>image\n", fp);

  if (mode == Fl_Print_Dialog::MODE_GRAYSCALE && img->d() == 3)
  {
    // Convert image to grayscale...
    gray = new Fl_RGB_Image((uchar *)img->data()[0], img->w(), img->h(),
                            img->d());
    gray->desaturate();

    write_ascii85(fp, (uchar *)gray->data()[0], img->w() * img->h());

    delete gray;
  }
  else
    write_ascii85(fp, (uchar *)img->data()[0], img->w() * img->h() * img->d());

  fputs("grestore\n", fp);
}


//
// 'write_matting()' - Write the matting around an image...
//

static void
write_matting(FILE  *fp,			// I - File to write
              uchar *data,			// I - 64x64 matting pattern
	      int   type,			// I - Mat type
	      int   width,			// I - Mat width in points
              int   left,			// I - Left position in points
	      int   bottom,			// I - Bottom position in points
	      int   right,			// I - Right position in points
	      int   top)			// I - Top position in points
{
  int	x, y;					// Looping vars
  int	matsize,				// Mat size
	matscale;				// Mat scale
  uchar	zdata[128 * 128 * 3],			// Zoomed mat image
	*ptr,					// Pointer into mat image
	*zptr;					// Pointer into zoomed image


  if (width == 0 ||
      type == Fl_Print_Dialog::MAT_BLANK ||
      (data[0] == 255 && memcmp(data, data + 1, 64 * 64 * 3 - 1) == 0))
    return;					// Ignore blank or 0-width matting

  fputs("gsave\n", fp);
  fprintf(fp, "[ %d %d %d %d\n", left, bottom, width, top - bottom);
  fprintf(fp, "  %d %d %d %d\n", left, bottom, right - left, width);
  fprintf(fp, "  %d %d %d %d\n", right - width, bottom, width, top - bottom);
  fprintf(fp, "  %d %d %d %d ] rectclip\n", left, top - width, right - left, width);

  if (type >= Fl_Print_Dialog::MAT_STANDARD)
  {
    // Write a mat image...
    fputs("/matdata\n", fp);
    if (type == 2)
    {
      // Scale the mat image for velvet...
      matsize  = 128;
      matscale = 144;

      for (y = 64, ptr = data, zptr = zdata; y > 0; y --, zptr += 128 * 3)
      {
        for (x = 64; x > 0; x --, ptr += 3, zptr += 6)
	{
	  zptr[0] = ptr[0];
	  zptr[1] = ptr[1];
	  zptr[2] = ptr[2];

          if (y == 1)
	  {
	    zptr[128 * 3 + 0] = (ptr[0] + ptr[-63 * 64 * 3 + 0]) / 2;
	    zptr[128 * 3 + 1] = (ptr[1] + ptr[-63 * 64 * 3 + 1]) / 2;
	    zptr[128 * 3 + 2] = (ptr[2] + ptr[-63 * 64 * 3 + 2]) / 2;
	  }
	  else
	  {
	    zptr[128 * 3 + 0] = (ptr[0] + ptr[64 * 3 + 0]) / 2;
	    zptr[128 * 3 + 1] = (ptr[1] + ptr[64 * 3 + 1]) / 2;
	    zptr[128 * 3 + 2] = (ptr[2] + ptr[64 * 3 + 2]) / 2;
	  }

          if (x == 1)
	  {
	    zptr[3] = (ptr[0] + ptr[-63 * 3 + 0]) / 2;
	    zptr[4] = (ptr[1] + ptr[-63 * 3 + 1]) / 2;
	    zptr[5] = (ptr[2] + ptr[-63 * 3 + 2]) / 2;

            if (y == 1)
	    {
	      zptr[128 * 3 + 3] = (ptr[0] + ptr[-63 * 3 + 0] +
	                           ptr[-63 * 64 * 3 + 0] +
				   ptr[-63 * 64 * 3 - 63 * 3 + 0]) / 4;
	      zptr[128 * 3 + 4] = (ptr[1] + ptr[-63 * 3 + 1] +
	                           ptr[-63 * 64 * 3 + 1] +
				   ptr[-63 * 64 * 3 - 63 * 3 + 1]) / 4;
	      zptr[128 * 3 + 5] = (ptr[2] + ptr[-63 * 3 + 2] +
	                           ptr[-63 * 64 * 3 + 2] +
				   ptr[-63 * 64 * 3 - 63 * 3 + 2]) / 4;
	    }
	    else
	    {
	      zptr[128 * 3 + 3] = (ptr[0] + ptr[-63 * 3 + 0] +
	                           ptr[64 * 3 + 0] +
				   ptr[64 * 3 - 63 * 3 + 0]) / 4;
	      zptr[128 * 3 + 4] = (ptr[1] + ptr[-63 * 3 + 1] +
	                           ptr[64 * 3 + 1] +
				   ptr[64 * 3 - 63 * 3 + 1]) / 4;
	      zptr[128 * 3 + 5] = (ptr[2] + ptr[-63 * 3 + 2] +
	                           ptr[64 * 3 + 2] +
				   ptr[64 * 3 - 63 * 3 + 2]) / 4;
	    }
	  }
	  else
	  {
	    zptr[3] = (ptr[0] + ptr[3]) / 2;
	    zptr[4] = (ptr[1] + ptr[4]) / 2;
	    zptr[5] = (ptr[2] + ptr[5]) / 2;

            if (y == 1)
	    {
	      zptr[128 * 3 + 3] = (ptr[0] + ptr[3] +
	                           ptr[-63 * 64 * 3 + 0] +
				   ptr[-63 * 64 * 3 + 3]) / 4;
	      zptr[128 * 3 + 4] = (ptr[1] + ptr[4] +
	                           ptr[-63 * 64 * 3 + 1] +
				   ptr[-63 * 64 * 3 + 4]) / 4;
	      zptr[128 * 3 + 5] = (ptr[2] + ptr[5] +
	                           ptr[-63 * 64 * 3 + 2] +
				   ptr[-63 * 64 * 3 + 5]) / 4;
	    }
	    else
	    {
	      zptr[128 * 3 + 3] = (ptr[0] + ptr[3] +
	                           ptr[64 * 3 + 0] +
				   ptr[64 * 3 + 3]) / 4;
	      zptr[128 * 3 + 4] = (ptr[1] + ptr[4] +
	                           ptr[64 * 3 + 1] +
				   ptr[64 * 3 + 4]) / 4;
	      zptr[128 * 3 + 5] = (ptr[2] + ptr[5] +
	                           ptr[64 * 3 + 2] +
				   ptr[64 * 3 + 5]) / 4;
	    }
	  }
	}
      }

      write_asciihex(fp, zdata, 128 * 128 * 3);
    }
    else
    {
      // Use the standard 64x64 image for others...
      matsize  = 64;
      matscale = 36;

      write_asciihex(fp, data, 64 * 64 * 3);
    }

    fputs("def\n", fp);

    fputs("/DeviceRGB setcolorspace\n", fp);

    for (y = bottom - (bottom % matscale); y < top; y += matscale)
      for (x = left - (left % matscale); x < right; x += matscale)
      {
	fputs("gsave\n", fp);
	fprintf(fp, "%d %d translate\n", x, y + matscale);
	fprintf(fp, "%.3f dup scale\n", (float)matscale / (float)matsize);
	fprintf(fp, "<</ImageType 1/Width %d/Height %d/BitsPerComponent 8",
	        matsize, matsize);
	fputs("/Decode[0 1 0 1 0 1]/DataSource matdata", fp);
	fputs("/Interpolate true/ImageMatrix[1 0 0 -1 0 1]>>image\n", fp);
	fputs("grestore\n", fp);
      }
  }
  else
  {
    // Draw a solid color mat...
    fprintf(fp, "%.3f %.3f %.3f setrgbcolor\n",
            data[0] / 255.0, data[1] / 255.0, data[2] / 255.0);
    fprintf(fp, "%d %d %d %d rectfill\n", left, bottom, right - left,
            top - bottom);
  }

  fputs("grestore\n", fp);

  // Draw the matting bezel...
  left   += width;
  bottom += width;
  right  -= width;
  top    -= width;

  fputs("0.7 setgray\n", fp);
  fprintf(fp, "%d %d moveto %d %d lineto %d %d lineto %d %d lineto closepath fill\n",
          left, bottom, left, top, left + 2, top - 2, left + 2, bottom + 2);
  fputs("0.5 setgray\n", fp);
  fprintf(fp, "%d %d moveto %d %d lineto %d %d lineto %d %d lineto closepath fill\n",
          left, top, right, top, right - 2, top - 2, left + 2, top - 2);
  fputs("0.8 setgray\n", fp);
  fprintf(fp, "%d %d moveto %d %d lineto %d %d lineto %d %d lineto closepath fill\n",
          right, bottom, right, top, right - 2, top - 2, right - 2, bottom + 2);
  fputs("1.0 setgray\n", fp);
  fprintf(fp, "%d %d moveto %d %d lineto %d %d lineto %d %d lineto closepath fill\n",
          left, bottom, right, bottom, right - 2, bottom + 2, left + 2,
	  bottom + 2);
}


//
// 'write_prolog()' - Write the PostScript prolog...
//

static void
write_prolog(FILE *fp,
             int  mode,
	     int  left,
	     int  bottom,
	     int  right,
	     int  top)
{
  fputs("%!PS-Adobe-3.0\n", fp);
  fprintf(fp, "%%%%BoundingBox: %d %d %d %d\n", left, bottom, right, top);
  fputs("%%Creator: flPhoto " FLPHOTO_VERSION "\n", fp);
  fputs("%%LanguageLevel: 2\n", fp);
  fputs("%%DocumentData: Clean7Bit\n", fp);
  fputs("%%Pages: (atend)\n", fp);
  fputs("%%EndComments\n", fp);
  fputs("%%BeginProlog\n", fp);
  fputs("/BK { 0 setgray } bind def\n", fp);
  fputs("/LF /Helvetica findfont 9 scalefont def\n", fp);
  fputs("/L { moveto LF setfont BK dup stringwidth pop -0.5 mul 0 "
        "rmoveto show } bind def\n", fp);
  fputs("/RF /Helvetica findfont 18 scalefont def\n", fp);
  fputs("/R { moveto RF setfont BK dup stringwidth pop neg 0 "
        "rmoveto show } bind def\n", fp);
  fputs("/TF /Helvetica findfont 36 scalefont def\n", fp);
  fputs("/T { moveto TF setfont BK dup stringwidth pop -0.5 mul 0 "
        "rmoveto show } bind def\n", fp);
  fputs("/LI { lineto stroke } bind def\n", fp);
  fputs("/MO { moveto } bind def\n", fp);
  fputs("%%EndProlog\n", fp);
}


//
// 'write_start_page()' - Start a page.
//

static void
write_start_page(FILE *fp,			// I - File to write
                 int  page,			// I - Page to write
		 int  width,			// I - Page width in points
                 int  length,			// I - Page length in points
		 int  commands)			// I - Embed page commands?
{
  fprintf(fp, "%%%%Page: %d %d\n", page, page);

  if (commands)
  {
    fputs("%%BeginPageSetup\n", fp);
    fputs("[{\n", fp);

    if (width == 612 && length == 792)
      fputs("%%BeginFeature: *PageSize Letter\n", fp);
    else if (width == 595 && length == 842)
      fputs("%%BeginFeature: *PageSize A4\n", fp);
    else
      fprintf(fp, "%%%%BeginFeature: *PageSize w%dl%d\n", width, length);

    fprintf(fp, "<</PageSize[%d %d]/ImagingBBox null>>setpagedevice\n",
            width, length);

    fputs("%%EndFeature\n", fp);

    fputs("} stopped cleartomark\n", fp);
    fputs("%%EndPageSetup\n", fp);
  }

  fputs("gsave\n", fp);
}


//
// 'write_trailer()' - Write the PostScript trailer...
//

static void
write_trailer(FILE *fp,
              int  pages)
{
  fputs("%%Trailer\n", fp);
  fprintf(fp, "%%%%Pages: %d\n", pages);
  fputs("%%EOF\n", fp);
}


//
// End of "$Id: print.cxx,v 1.30 2003/09/08 16:59:12 easysw Exp $".
//
