//
// "$Id: Fl_CRW_Image.cxx,v 1.5 2003/01/03 04:41:15 easysw Exp $"
//
// Raw camera image routines for flphoto.
//
// Copyright 2002-2003 by Michael Sweet.
//
// Original code copyright 1997-2003 by Dave Coffin <dcoffin@shore.net>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// Contents:
//
//   Fl_CRW_Image::Fl_CRW_Image() - Load a PhotoCD image...
//   Fl_CRW_Image::Fl_CRW_Image() - Load a PhotoCD image...
//   Fl_CRW_Image::check()        - Try loading the named file.
//   Fl_CRW_Image::load_image()   - Load the image from the file.
//

//
// Include necessary header files...
//

#include <FL/Fl.H>
#include "Fl_CRW_Image.H"
#include <stdlib.h>
#include <math.h>
#include "flstring.h"


//
// 'Fl_CRW_Image::Fl_CRW_Image()' - Load a PhotoCD image...
//

Fl_CRW_Image::Fl_CRW_Image(const char *name)	// I - Filename to read from
  : Fl_RGB_Image(0,0,0) {
  FILE		*fp;				// File pointer


  // Open the image file...
  if ((fp = fopen(name, "rb")) == NULL)
    return;

  // Load the image...
  load_image(fp);

  // Close the image file...
  fclose(fp);
}


//
// 'Fl_CRW_Image::Fl_CRW_Image()' - Load a PhotoCD image...
//

Fl_CRW_Image::Fl_CRW_Image(FILE *fp)		// I - File stream to read from
  : Fl_RGB_Image(0,0,0) {
  // Load the image...
  load_image(fp);
}


//
// 'Fl_CRW_Image::check()' - Try loading the named file.
//

Fl_Image *					// O - Image or NULL
Fl_CRW_Image::check(const char *name,		// I - Name of file
		    uchar      *header,		// I - Header of file
		    int        headerlen)	// I - Number of header bytes
{
  if (memcmp(header, "II\032", 3) == 0)		// Raw camera file...
    return (new Fl_CRW_Image(name));
  else
    return (0);
}


//
// 'Fl_CRW_Image::load_image()' - Load the image from the file.
//

void
Fl_CRW_Image::load_image(FILE *fp)		// I - File to read
{
  int	i;
  int	y, x;
  int	dillon;
  int	smooth = 4;
  register unsigned c, val;
  uchar	*ptr;
  float	rgb[4], max, max2, expo, mul, scale;
  int	total, histogram[0x1000];


  gamma_val  = 0.8;
  bright     = 1.0;
  red_scale  = 1.0;
  blue_scale = 1.0;

  bitbuf     = 0;
  vbits      = 0;


  rewind(fp);

  if (open_and_id(fp))
    return;

  image = new ushort4[h() * w()];

  black = 0;

  read_crw(fp);

  if (black) {
#ifdef DEBUG
    printf("Subtracting thermal noise (%d)...\n", black);
#endif // DEBUG

    subtract_black();
  }
  dillon = 1;
  for (i=8; i < 32; i+=8)
    if ((filters >> i & 0xff) != (filters & 0xff))
      dillon = 0;
  if (dillon) {
#ifdef DEBUG
    puts("Dillon interpolation...");
#endif // DEBUG

    dillon_interpolate();
    trim = 2;
  } else {
#ifdef DEBUG
    puts("First interpolation...");
#endif // DEBUG

    first_interpolate();

    for (i=0; i < smooth; i++) {
#ifdef DEBUG
      puts("Second interpolation...");
#endif // DEBUG

      second_interpolate();
    }

    trim = 1;
  }

/*
   Build a histogram of magnitudes using 4096 bins of 64 values each.
 */
  memset (histogram, 0, sizeof histogram);
  for (y=trim; y < h()-trim; y++)
    for (x=trim; x < w()-trim; x++) {
      get_rgb (rgb, image[y*w()+x]);
      val = (int) sqrt(rgb[3]) >> 6;
      if (val > 0xfff) val=0xfff;
      histogram[val]++;
    }
/*
   Set the white point to the 99.66th percentile
 */
  for (val=0x1000, total=0; --val; )
    if ((total+=histogram[val]) > (int)(w()*h()*0.01)) break;
  max = val << 6;
  max2 = max * max;

  array = new uchar[(w() - 2 * trim) * (h() - 2 * trim) * 3];
  alloc_array = 1;

  expo = (gamma_val-1)/2;		/* Pull these out of the loop */
  mul = bright * 442 / max;

  for (y=trim, ptr = (uchar *)array; y < h()-trim; y++)
  {
    for (x=trim; x < w()-trim; x++)
    {
      get_rgb(rgb,image[y*w()+x]);
/* In some math libraries, pow(0,expo) doesn't return zero */
      scale = 0;
      if (rgb[3]) scale = mul * pow(rgb[3]/max2,expo);

      for (c=0; c < 3; c++)
      {
	val=(int)(rgb[c]*scale + 0.5);
	if (val > 255) val=255;
	*ptr++ = val;
      }
    }
  }

  w(w() - 2 * trim);
  h(h() - 2 * trim);

  delete[] image;
}


/*
   In order to inline this calculation, I make the risky
   assumption that all filter patterns can be described
   by a repeating pattern of eight rows and two columns

   Return values are either 0/1/2/3 = G/M/C/Y or 0/1/2 = R/G/B
 */
#define FC(row,col) \
	(filters >> ((((row) << 1 & 14) + ((col) & 1)) << 1) & 3)
/*
   PowerShot 600 uses 0xe1e4e1e4:

	  0 1 2 3 4 5
	0 G M G M G M
	1 C Y C Y C Y
	2 M G M G M G
	3 C Y C Y C Y

   PowerShot A5 uses 0x1e4e1e4e:

	  0 1 2 3 4 5
	0 C Y C Y C Y
	1 G M G M G M
	2 C Y C Y C Y
	3 M G M G M G

   PowerShot A50 uses 0x1b4e4b1e:

	  0 1 2 3 4 5
	0 C Y C Y C Y
	1 M G M G M G
	2 Y C Y C Y C
	3 G M G M G M
	4 C Y C Y C Y
	5 G M G M G M
	6 Y C Y C Y C
	7 M G M G M G

   PowerShot Pro70 uses 0x1e4b4e1b:

	  0 1 2 3 4 5
	0 Y C Y C Y C
	1 M G M G M G
	2 C Y C Y C Y
	3 G M G M G M
	4 Y C Y C Y C
	5 G M G M G M
	6 C Y C Y C Y
	7 M G M G M G

   PowerShots Pro90 and G1 use 0xb4b4b4b4:

	  0 1 2 3 4 5
	0 G M G M G M
	1 Y C Y C Y C

   Some RGB cameras use 0x94949494:

	  0 1 2 3 4 5
	0 R G R G R G
	1 G B G B G B
	2 R G R G R G
	3 G B G B G B

   The EOS-1D uses 0x61616161:

	  0 1 2 3 4 5
	0 G R G R G R
	1 B G B G B G
	2 G R G R G R
	3 B G B G B G

   The Nikon cameras use 0x16161616:

	  0 1 2 3 4 5
	0 B G B G B G
	1 G R G R G R
	2 B G B G B G
	3 G R G R G R
 */

void
Fl_CRW_Image::ps600_read_crw(FILE *ifp)
{
  uchar  data[1120], *dp;
  ushort pixel[896], *pix;
  int irow, orow, col;

/*
   Immediately after the 26-byte header come the data rows.  First
   the even rows 0..612, then the odd rows 1..611.  Each row is 896
   pixels, ten bits per pixel, packed into 1120 bytes (8960 bits).
 */
  for (irow=orow=0; irow < h(); irow++)
  {
    fread (data, 1120, 1, ifp);
    for (dp=data, pix=pixel; dp < data+1120; dp+=10, pix+=8)
    {
      pix[0]=(dp[0] << 2) + (dp[1] >> 6    );
      pix[1]=(dp[2] << 2) + (dp[1] >> 4 & 3);
      pix[2]=(dp[3] << 2) + (dp[1] >> 2 & 3);
      pix[3]=(dp[4] << 2) + (dp[1]      & 3);
      pix[4]=(dp[5] << 2) + (dp[9]      & 3);
      pix[5]=(dp[6] << 2) + (dp[9] >> 2 & 3);
      pix[6]=(dp[7] << 2) + (dp[9] >> 4 & 3);
      pix[7]=(dp[8] << 2) + (dp[9] >> 6    );
    }
/*
   Copy 854 pixels into the image[] array.  The other 42 pixels
   are black.  Left-shift by 4 for extra precision in upcoming
   calculations.
 */
    for (col=0; col < w(); col++)
      image[orow*w()+col][FC(orow,col)] = pixel[col] << 4;
    for (col=w(); col < 896; col++)
      black += pixel[col];

    if ((orow+=2) > h())	/* Once we've read all the even rows, */
      orow = 1;			/* read the odd rows. */
  }
  black = ((long long) black << 4) / ((896 - w()) * h());
}

void
Fl_CRW_Image::a5_read_crw(FILE *ifp)
{
  uchar  data[1240], *dp;
  ushort pixel[992], *pix;
  int row, col;

/*
   Each data row is 992 ten-bit pixels, packed into 1240 bytes.
 */
  for (row=0; row < h(); row++) {
    fread (data, 1240, 1, ifp);
    for (dp=data, pix=pixel; dp < data+1240; dp+=10, pix+=8)
    {
      pix[0]=(dp[1] << 2) + (dp[0] >> 6);
      pix[1]=(dp[0] << 4) + (dp[3] >> 4);
      pix[2]=(dp[3] << 6) + (dp[2] >> 2);
      pix[3]=(dp[2] << 8) + (dp[5]     );
      pix[4]=(dp[4] << 2) + (dp[7] >> 6);
      pix[5]=(dp[7] << 4) + (dp[6] >> 4);
      pix[6]=(dp[6] << 6) + (dp[9] >> 2);
      pix[7]=(dp[9] << 8) + (dp[8]     );
    }
/*
   Copy 960 pixels into the image[] array.  The other 32 pixels
   are black.  Left-shift by 4 for extra precision in upcoming
   calculations.
 */
    for (col=0; col < w(); col++)
      image[row*w()+col][FC(row,col)] = (pixel[col] & 0x3ff) << 4;
    for (col=w(); col < 992; col++)
      black += pixel[col] & 0x3ff;
  }
  black = ((long long) black << 4) / ((992 - w()) * h());
}

void
Fl_CRW_Image::a50_read_crw(FILE *ifp)
{
  uchar  data[1650], *dp;
  ushort pixel[1320], *pix;
  int row, col;

/*
  Each row is 1320 ten-bit pixels, packed into 1650 bytes.
 */
  for (row=0; row < h(); row++) {
    fread (data, 1650, 1, ifp);
    for (dp=data, pix=pixel; dp < data+1650; dp+=10, pix+=8)
    {
      pix[0]=(dp[1] << 2) + (dp[0] >> 6);
      pix[1]=(dp[0] << 4) + (dp[3] >> 4);
      pix[2]=(dp[3] << 6) + (dp[2] >> 2);
      pix[3]=(dp[2] << 8) + (dp[5]     );
      pix[4]=(dp[4] << 2) + (dp[7] >> 6);
      pix[5]=(dp[7] << 4) + (dp[6] >> 4);
      pix[6]=(dp[6] << 6) + (dp[9] >> 2);
      pix[7]=(dp[9] << 8) + (dp[8]     );
    }
/*
   Copy 1290 pixels into the image[] array.  The other 30 pixels
   are black.  Left-shift by 4 for extra precision in upcoming
   calculations.
 */
    for (col=0; col < w(); col++)
      image[row*w()+col][FC(row,col)] = (pixel[col] & 0x3ff) << 4;
    for (col=w(); col < 1320; col++)
      black += pixel[col] & 0x3ff;
  }
  black = ((long long) black << 4) / ((1320 - w()) * h());
}

void
Fl_CRW_Image::pro70_read_crw(FILE *ifp)
{
  uchar  data[1940], *dp;
  ushort pixel[1552], *pix;
  int row, col;

/*
  Each row is 1552 ten-bit pixels, packed into 1940 bytes.
 */
  for (row=0; row < h(); row++) {
    fread (data, 1940, 1, ifp);
    for (dp=data, pix=pixel; dp < data+1940; dp+=10, pix+=8)
    {
      pix[0]=(dp[1] << 2) + (dp[0] >> 6);	/* Same as PS A5 */
      pix[1]=(dp[0] << 4) + (dp[3] >> 4);
      pix[2]=(dp[3] << 6) + (dp[2] >> 2);
      pix[3]=(dp[2] << 8) + (dp[5]     );
      pix[4]=(dp[4] << 2) + (dp[7] >> 6);
      pix[5]=(dp[7] << 4) + (dp[6] >> 4);
      pix[6]=(dp[6] << 6) + (dp[9] >> 2);
      pix[7]=(dp[9] << 8) + (dp[8]     );
    }
/*
   Copy all pixels into the image[] array.  Left-shift by 4 for
   extra precision in upcoming calculations.  No black pixels?
 */
    for (col=0; col < w(); col++)
      image[row*w()+col][FC(row,col)] = (pixel[col] & 0x3ff) << 4;
  }
}

/*
   A rough description of Canon's compression algorithm:

+  Each pixel outputs a 10-bit sample, from 0 to 1023.
+  Split the data into blocks of 64 samples each.
+  Subtract from each sample the value of the sample two positions
   to the left, which has the same color filter.  From the two
   leftmost samples in each row, subtract 512.
+  For each nonzero sample, make a token consisting of two four-bit
   numbers.  The low nibble is the number of bits required to
   represent the sample, and the high nibble is the number of
   zero samples preceding this sample.
+  Output this token as a variable-length bitstring using
   one of three tablesets.  Follow it with a fixed-length
   bitstring containing the sample.

   The "first_decode" table is used for the first sample in each
   block, and the "second_decode" table is used for the others.
 */

/*
   Construct a decode tree according the specification in *source.
   The first 16 bytes specify how many codes should be 1-bit, 2-bit
   3-bit, etc.  Bytes after that are the leaf values.

   For example, if the source is

    { 0,1,4,2,3,1,2,0,0,0,0,0,0,0,0,0,
      0x04,0x03,0x05,0x06,0x02,0x07,0x01,0x08,0x09,0x00,0x0a,0x0b,0xff  },

   then the code is

	00		0x04
	010		0x03
	011		0x05
	100		0x06
	101		0x02
	1100		0x07
	1101		0x01
	11100		0x08
	11101		0x09
	11110		0x00
	111110		0x0a
	1111110		0x0b
	1111111		0xff
 */
void
Fl_CRW_Image::make_decoder(struct decode *dest, const uchar *source, int level)
{
  static struct decode *free;	/* Next unused node */
  static int leaf;			/* number of leaves already added */
  int i, next;

  if (level==0) {
    free = dest;
    leaf = 0;
  }
  free++;
/*
   At what level should the next leaf appear?
 */
  for (i=next=0; i <= leaf && next < 16; )
    i += source[next++];

  if (level < next) {		/* Are we there yet? */
    dest->branch[0] = free;
    make_decoder(free,source,level+1);
    dest->branch[1] = free;
    make_decoder(free,source,level+1);
  } else
    dest->leaf = source[16 + leaf++];
}

void
Fl_CRW_Image::init_tables(unsigned table)
{
  static const uchar first_tree[3][29] = {
    { 0,1,4,2,3,1,2,0,0,0,0,0,0,0,0,0,
      0x04,0x03,0x05,0x06,0x02,0x07,0x01,0x08,0x09,0x00,0x0a,0x0b,0xff  },

    { 0,2,2,3,1,1,1,1,2,0,0,0,0,0,0,0,
      0x03,0x02,0x04,0x01,0x05,0x00,0x06,0x07,0x09,0x08,0x0a,0x0b,0xff  },

    { 0,0,6,3,1,1,2,0,0,0,0,0,0,0,0,0,
      0x06,0x05,0x07,0x04,0x08,0x03,0x09,0x02,0x00,0x0a,0x01,0x0b,0xff  },
  };

  static const uchar second_tree[3][180] = {
    { 0,2,2,2,1,4,2,1,2,5,1,1,0,0,0,139,
      0x03,0x04,0x02,0x05,0x01,0x06,0x07,0x08,
      0x12,0x13,0x11,0x14,0x09,0x15,0x22,0x00,0x21,0x16,0x0a,0xf0,
      0x23,0x17,0x24,0x31,0x32,0x18,0x19,0x33,0x25,0x41,0x34,0x42,
      0x35,0x51,0x36,0x37,0x38,0x29,0x79,0x26,0x1a,0x39,0x56,0x57,
      0x28,0x27,0x52,0x55,0x58,0x43,0x76,0x59,0x77,0x54,0x61,0xf9,
      0x71,0x78,0x75,0x96,0x97,0x49,0xb7,0x53,0xd7,0x74,0xb6,0x98,
      0x47,0x48,0x95,0x69,0x99,0x91,0xfa,0xb8,0x68,0xb5,0xb9,0xd6,
      0xf7,0xd8,0x67,0x46,0x45,0x94,0x89,0xf8,0x81,0xd5,0xf6,0xb4,
      0x88,0xb1,0x2a,0x44,0x72,0xd9,0x87,0x66,0xd4,0xf5,0x3a,0xa7,
      0x73,0xa9,0xa8,0x86,0x62,0xc7,0x65,0xc8,0xc9,0xa1,0xf4,0xd1,
      0xe9,0x5a,0x92,0x85,0xa6,0xe7,0x93,0xe8,0xc1,0xc6,0x7a,0x64,
      0xe1,0x4a,0x6a,0xe6,0xb3,0xf1,0xd3,0xa5,0x8a,0xb2,0x9a,0xba,
      0x84,0xa4,0x63,0xe5,0xc5,0xf3,0xd2,0xc4,0x82,0xaa,0xda,0xe4,
      0xf2,0xca,0x83,0xa3,0xa2,0xc3,0xea,0xc2,0xe2,0xe3,0xff,0xff  },

    { 0,2,2,1,4,1,4,1,3,3,1,0,0,0,0,140,
      0x02,0x03,0x01,0x04,0x05,0x12,0x11,0x06,
      0x13,0x07,0x08,0x14,0x22,0x09,0x21,0x00,0x23,0x15,0x31,0x32,
      0x0a,0x16,0xf0,0x24,0x33,0x41,0x42,0x19,0x17,0x25,0x18,0x51,
      0x34,0x43,0x52,0x29,0x35,0x61,0x39,0x71,0x62,0x36,0x53,0x26,
      0x38,0x1a,0x37,0x81,0x27,0x91,0x79,0x55,0x45,0x28,0x72,0x59,
      0xa1,0xb1,0x44,0x69,0x54,0x58,0xd1,0xfa,0x57,0xe1,0xf1,0xb9,
      0x49,0x47,0x63,0x6a,0xf9,0x56,0x46,0xa8,0x2a,0x4a,0x78,0x99,
      0x3a,0x75,0x74,0x86,0x65,0xc1,0x76,0xb6,0x96,0xd6,0x89,0x85,
      0xc9,0xf5,0x95,0xb4,0xc7,0xf7,0x8a,0x97,0xb8,0x73,0xb7,0xd8,
      0xd9,0x87,0xa7,0x7a,0x48,0x82,0x84,0xea,0xf4,0xa6,0xc5,0x5a,
      0x94,0xa4,0xc6,0x92,0xc3,0x68,0xb5,0xc8,0xe4,0xe5,0xe6,0xe9,
      0xa2,0xa3,0xe3,0xc2,0x66,0x67,0x93,0xaa,0xd4,0xd5,0xe7,0xf8,
      0x88,0x9a,0xd7,0x77,0xc4,0x64,0xe2,0x98,0xa5,0xca,0xda,0xe8,
      0xf3,0xf6,0xa9,0xb2,0xb3,0xf2,0xd2,0x83,0xba,0xd3,0xff,0xff  },

    { 0,0,6,2,1,3,3,2,5,1,2,2,8,10,0,117,
      0x04,0x05,0x03,0x06,0x02,0x07,0x01,0x08,
      0x09,0x12,0x13,0x14,0x11,0x15,0x0a,0x16,0x17,0xf0,0x00,0x22,
      0x21,0x18,0x23,0x19,0x24,0x32,0x31,0x25,0x33,0x38,0x37,0x34,
      0x35,0x36,0x39,0x79,0x57,0x58,0x59,0x28,0x56,0x78,0x27,0x41,
      0x29,0x77,0x26,0x42,0x76,0x99,0x1a,0x55,0x98,0x97,0xf9,0x48,
      0x54,0x96,0x89,0x47,0xb7,0x49,0xfa,0x75,0x68,0xb6,0x67,0x69,
      0xb9,0xb8,0xd8,0x52,0xd7,0x88,0xb5,0x74,0x51,0x46,0xd9,0xf8,
      0x3a,0xd6,0x87,0x45,0x7a,0x95,0xd5,0xf6,0x86,0xb4,0xa9,0x94,
      0x53,0x2a,0xa8,0x43,0xf5,0xf7,0xd4,0x66,0xa7,0x5a,0x44,0x8a,
      0xc9,0xe8,0xc8,0xe7,0x9a,0x6a,0x73,0x4a,0x61,0xc7,0xf4,0xc6,
      0x65,0xe9,0x72,0xe6,0x71,0x91,0x93,0xa6,0xda,0x92,0x85,0x62,
      0xf3,0xc5,0xb2,0xa4,0x84,0xba,0x64,0xa5,0xb3,0xd2,0x81,0xe5,
      0xd3,0xaa,0xc4,0xca,0xf2,0xb1,0xe4,0xd1,0x83,0x63,0xea,0xc3,
      0xe2,0x82,0xf1,0xa3,0xc2,0xa1,0xc1,0xe3,0xa2,0xe1,0xff,0xff  }
  };

  if (table > 2) table = 2;
  memset( first_decode, 0, sizeof first_decode);
  memset(second_decode, 0, sizeof second_decode);
  make_decoder( first_decode,  first_tree[table], 0);
  make_decoder(second_decode, second_tree[table], 0);
}

/*
   getbits(ifp, -1) initializes the buffer
   getbits(ifp, n) where 0 <= n <= 25 returns an n-bit integer
 */
unsigned long
Fl_CRW_Image::getbits(FILE *ifp, int nbits)
{
  unsigned long	ret=0;
  unsigned char c;

  if (nbits == 0) return 0;
  if (nbits == -1)
    ret = bitbuf = vbits = 0;
  else {
    ret = bitbuf << (32 - vbits) >> (32 - nbits);
    vbits -= nbits;
  }
  while (vbits < 25) {
    c=fgetc(ifp);
    bitbuf = (bitbuf << 8) + c;
    if (c == 0xff && canon) fgetc(ifp); /* Canon puts an extra 0 after 0xff */
    vbits += 8;
  }
  return ret;
}

/*
   Decompress "count" blocks of 64 samples each.

   Note that the width passed to this function is slightly
   larger than the global width, because it includes some
   blank pixels that (*read_crw) will strip off.
 */
void
Fl_CRW_Image::decompress(FILE *ifp, ushort *outbuf, int count)
{
  struct decode *decode, *dindex;
  int i, leaf, len, sign, diff, diffbuf[64];
  static int carry, pixel, base[2];

  if (!outbuf) {			/* Initialize */
    carry = pixel = 0;
    fseek(ifp, count, SEEK_SET);
    getbits(ifp, -1);
    return;
  }
  while (count--) {
    memset(diffbuf,0,sizeof diffbuf);
    decode = first_decode;
    for (i=0; i < 64; i++ ) {

      for (dindex=decode; dindex->branch[0]; )
	dindex = dindex->branch[getbits(ifp, 1)];
      leaf = dindex->leaf;
      decode = second_decode;

      if (leaf == 0 && i) break;
      if (leaf == 0xff) continue;
      i  += leaf >> 4;
      len = leaf & 15;
      if (len == 0) continue;
      sign=(getbits(ifp, 1));	/* 1 is positive, 0 is negative */
      diff=getbits(ifp, len-1);
      if (sign)
	diff += 1 << (len-1);
      else
	diff += (-1 << len) + 1;
      if (i < 64) diffbuf[i] = diff;
    }
    diffbuf[0] += carry;
    carry = diffbuf[0];
    for (i=0; i < 64; i++ ) {
      if (pixel++ % raw_width == 0)
	base[0] = base[1] = 512;
      outbuf[i] = ( base[i & 1] += diffbuf[i] );
    }
    outbuf += 64;
  }
}

void
Fl_CRW_Image::pro90_read_crw(FILE *ifp)
{
  ushort pixel[1944*8];
  int row, r, col;

  decompress(ifp, 0,540);
/*
   Read eight rows at a time.
   Each row has 1896 image pixels and 48 black pixels.
 */
  for (row=0; row < h(); row += 8) {
    decompress(ifp, pixel,243);
    for (r=0; r < 8; r++) {
      for (col=0; col < w(); col++)
	image[(row+r)*w()+col][FC(row+r,col)] =
		(pixel[(r*1944)+col] & 0x3ff) << 4;
      for (col=w(); col < 1944; col++)
	black += pixel[(r*1944)+col] & 0x3ff;
    }
  }
  black = ((long long) black << 4) / ((1944 - w()) * h());
}

void
Fl_CRW_Image::g1_read_crw(FILE *ifp)
{
  ushort pixel[2144*2];
  int row, r, col;

  decompress(ifp, 0,540);
/*
   Read two rows at a time.
   The image has a black border, eight pixels wide on top,
   two on the bottom, four on the left, and 52 on the right.
 */
  for (row = -8; row < h()+2; row += 2) {
    decompress(ifp, pixel,67);
    for (r=0; r < 2; r++)
      for (col = -4; col < w()+52; col++)
	if ((row+r) < h() && col < w())
	  image[(row+r)*w()+col][FC(row+r,col)] =
		(pixel[(r*2144)+col+4] & 0x3ff) << 4;
	  else
	    black += pixel[(r*2144)+col+4] & 0x3ff;
  }
  black = ((long long) black << 4) / (10 * 2144 + 56 * h());
}

void
Fl_CRW_Image::g2_read_crw(FILE *ifp)
{
  ushort pixel[2376*8];
  int row, r, col;

  decompress(ifp, 0,540);
/*
   Read eight rows at a time.
   The image has a black border, six pixels wide on top,
   two on the bottom, 12 on the left, and 52 on the right.
 */
  for (row = -6; row < h()+2; row += 8) {
    decompress(ifp, pixel,297);
    for (r=0; r < 8; r++)
      for (col = -12; col < w()+52; col++)
	if ((row+r) < h() && col < w())
	  image[(row+r)*w()+col][FC(row+r,col)] =
		(pixel[(r*2376)+col+12] & 0x3ff) << 4;
	  else
	    black += pixel[(r*2376)+col+12] & 0x3ff;
  }
  black = ((long long) black << 4) / (8 * 2376 + 64 * h());
}

/*
   The "PowerShot" cameras provide 10 bits per pixel in one compressed
   chunk.  The "EOS" cameras provide 12 bits per pixel:  first the two
   low bits of every pixel, not compressed, followed by the top ten bits
   compressed as in the PowerShot cameras.
 */
void
Fl_CRW_Image::d30_read_crw(FILE *ifp)
{
  ushort *pixel, *prow;
  int i, row, r, col, save;
  int top=0, left=0, irow, icol;
  uchar c;

/* Set the width of the black borders */
  switch (raw_width) {
    case 2224:  top = 6;  left = 48;  break;	/* EOS D30 */
    case 3152:  top =12;  left = 64;  break;	/* EOS D60 */
  }
  pixel = new ushort[raw_width*4];
  if (!pixel) {
    perror("d30_read_crw() new failed");
    exit(1);
  }
  decompress(ifp, 0, 540 + raw_height*raw_width/4);
  for (row = 0; row < raw_height; row += 4) {
    decompress(ifp, pixel, raw_width/16);		/* Get four rows */
    save = ftell(ifp);				/* Don't lose our place */
    fseek(ifp, 26 + row*raw_width/4, SEEK_SET);/* Add the low bits */
    for (prow=pixel, i=0; i < raw_width; i++) {
      c = fgetc(ifp);
      for (r = 0; r < 8; r += 2)
	*prow++ = (*prow << 2) + ((c >> r) & 3);
    }
    fseek(ifp, save, SEEK_SET);
    for (r=0; r < 4; r++)
      for (col = 0; col < raw_width; col++) {
	irow = row+r-top;
	icol = col-left;
	if (irow >= h() || irow < 0) continue;
	if (icol < w() && icol >= 0)
	  image[irow*w()+icol][FC(irow,icol)] =
		(pixel[r*raw_width+col] & 0xfff) << 2;
	  else
	    black += pixel[r*raw_width+col] & 0xfff;
      }
  }
  delete[] pixel;
  black = ((long long) black << 2) / (left * h());
}

void
Fl_CRW_Image::eos1d_read_crw(FILE *ifp)
{
  // Need lossless JPEG code...
}


void
Fl_CRW_Image::nikon_d1x_read_crw(FILE *ifp)
{
  int waste=0, comp;
  static const uchar nikon_tree[] = {
    0,1,5,1,1,1,1,1,1,2,0,0,0,0,0,0,
    5,4,3,6,2,7,1,0,8,9,11,10,12
  };
  int vpred[4], hpred[2], csize, row, col, i, len;
  uchar test[256], skip16=0;
  ushort *curve;
  struct decode *dindex;
  register int diff;

  if (!strcmp(name,"NIKON D1X"))
    waste = 4;
/*
   Try to figure out if the image is compressed, based on
   my limited collection of NEF files.  For the D100, every
   16th byte of an uncompressed image is zero.
 */
  fseek(ifp, nef_data_offset+58, SEEK_SET);
  comp = fget2(ifp);
  fseek(ifp, nef_data_offset+142, SEEK_SET);
  fseek(ifp, fget4(ifp)+8, SEEK_SET);
  if (!strcmp(name,"NIKON D100")) {
    w(3034);
    fread (test, 1, 256, ifp);
    for (i=0; i < 16; i++)
      if (test[i*16+15]) goto compressed;
    fseek(ifp, -256, SEEK_CUR);
    w(3037);
    waste = 3;
    skip16 = 1;
  } else if (comp == 0x8799)
    goto compressed;

/* Read an uncompressed image */
  getbits(ifp, -1);
  for (row=0; row < h(); row++) {
    for (col=0; col < w()+waste; col++) {
      i = getbits(ifp, 12);
      if (col < w())
	image[row*w()+col][FC(row,col)] = i << 2;
      if (skip16 && (col % 10) == 9)
	getbits(ifp, 8);
    }
  }
  return;

/* Read an compressed image */
compressed:
  memset( first_decode, 0, sizeof first_decode);
  make_decoder( first_decode,  nikon_tree, 0);

  if (!strcmp(name,"NIKON D100"))
    fseek(ifp, 5974, SEEK_SET);
  else
    fseek(ifp, 3488, SEEK_SET);
  for (i=0; i < 4; i++)
    vpred[i] = fget2(ifp);
  csize = fget2(ifp);
  curve = new ushort[csize];
  if (!curve) {
    perror("nikon_d1x_read_crw() new failed");
    exit(1);
  }
  for (i=0; i < csize; i++)
    curve[i] = fget2(ifp);

  fseek(ifp, nef_data_offset+82, SEEK_SET);
  fseek(ifp, fget4(ifp), SEEK_SET);
  getbits(ifp, -1);

  for (row=0; row < h(); row++)
    for (col=0; col < w()+waste; col++) {

      for (dindex=first_decode; dindex->branch[0]; )
	dindex = dindex->branch[getbits(ifp, 1)];
      len = dindex->leaf;
      diff = getbits(ifp, len);
      if ((diff & (1 << (len-1))) == 0)
	diff -= (1 << len) - 1;
      if (col < 2) {
	i = 2*(row & 1) + col;
	vpred[i] += diff;
	hpred[col] = vpred[i];
      } else
	hpred[col & 1] += diff;
      if (col >= w()) continue;
      diff = hpred[col & 1];
      if (diff < 0) diff = 0;
      if (diff >= csize) diff = csize-1;
      image[row*w()+col][FC(row,col)] = curve[diff] << 2;
    }
  delete[] curve;
}

void
Fl_CRW_Image::nikon_e5700_read_crw(FILE *ifp)
{
  uchar  data[3864], *dp;
  ushort pixel[2576], *pix;
  int irow, row, col;

  fseek(ifp, 589226, SEEK_SET);
  for (irow=row=0; irow < h(); irow++)
  {
    fread (data, 3864, 1, ifp);
    for (dp=data, pix=pixel; dp < data+3864; dp+=3, pix+=2)
    {
      pix[0]=(dp[0] << 8) + (dp[1] & 0xf0) + (dp[1] >> 4);
      pix[1]=(dp[2] << 4) + (dp[1] & 0x0f);
    }
    for (col=0; col < w(); col++)
      image[row*w()+col][FC(row,col)] = (pixel[col] & 0xfff) << 2;

    if ((row+=2) >= h())	/* Once we've read all the even rows, */
      row = 1;			/* read the odd rows. */
  }
}

void
Fl_CRW_Image::olympus_read_crw(FILE *ifp)
{
  uchar *pixel;
  int row, col;

  pixel = new uchar[w() * 2];
  if (!pixel) {
    perror("olympus_read_crw() new failed");
    exit(1);
  }
  fseek(ifp, 0x4000, SEEK_SET);
  for (row=0; row < h(); row++) {
    fread (pixel, 2, w(), ifp);
    for (col=0; col < w(); col++)
      image[row*w()+col][FC(row,col)] =
          ((pixel[col * 2] << 8) | pixel[col * 2 + 1]) >> 2;
  }
  delete[] pixel;
}

void
Fl_CRW_Image::subtract_black()
{
  ushort *img;
  int size;

  img = image[0];
  size = w() * h() * 4;
  while (size--)
    if (*img > black)
      *img++ -= black;
    else
      *img++ = 0;
  rgb_max -= black;
}

/*
   RGB interpolation algorithm designed by Matt Dillon.
   GMCY cameras still use my algorithm.
   His explanation follows:

 * When this function is called, we only have one color for
 * each pixel.  This code searches the 5x5 neighborhood and
 * synthesizes the other colors based on interpolation.  The
 * algorithm works as follows.  Take one portion of the 
 * CCD filter pattern:
 *
 *	  0 1 2 3 4
 *	0 G R G R G
 *	1 B G B G B
 *	2 G R<G>R G
 *	3 B G B G B
 *	4 G R G R G
 *
 * Lets say we are on pixel (x,y) = (2,2), the bracketed green above, and
 * we are trying to synthesize the Blue and Red guns for that pixel based
 * on the surrounding pixels.
 *
 * We could average the surrounding blue pixels to synthesize a blue for
 * (2,2) but that will blur the image somewhat if we are on the boundary
 * of an edge.  Instead what we do is calculate a temporary green for each
 * blue pixel and then generate the blue for (2,2) by scaling it against
 * the differential between the temporary green and the green in (2,2).
 *
 * For example, the green for the blue pixel at (2,1) is calculated by
 * averaging the green at (2,0) and (2,2).  A scaling factor is generated
 * using this green and the green at (2,2), we multiply in the Blue pixel
 * at (2,1), and that is the Blue we store for (2,2).  It's actually somewhat
 * more complicated since there are other blue's around (2,2).  What we do
 * is execute this calculate for each blue and do a weighted average of the
 * result, and that final number is the Blue we store at (2,2).
 *
 * The scaling factor tends to blow up in low-light situations due to
 * the calculation of the scaling factor becoming skewed (think about the
 * difference between two pixel brightnesses of 10 and 15, and 4000 and 4005).
 * To compensate we introduce a constant in the scaling process, llfactor.
 * The higher the constant the closer the scaling factor gets to 1:1 (a
 * straight average of the surrounding blue's regardless of the scale 
 * generated using the green's).  This results in somewhat more blurring
 * but hides low-light dropouts.
 */
#define lowlight_val 0
void
Fl_CRW_Image::dillon_interpolate()
{
  int row, col, cc, c;
  int vb, vr, val;
  int dx, dy;
  int avg[4], sum[4];
  int base = 4096 + 64;
  int llfactor = 8 + lowlight_val * 16;
  int weight;
  int size;
  ushort4 *oimage;

  size = h() * w() * 8;
  oimage = new ushort4[size];
  if (!oimage) {
    perror("dillon_interpolate() new failed");
    exit(1);
  }
  memcpy (oimage, image, size);

  for (row=2; row < h()-2; row++) {
    for (col=2; col < w()-2; col++) {
      cc = FC(row,col);
      vb = oimage[row*w() + col][cc];
      avg[0] = avg[1] = avg[2] = avg[3] = 0;
      sum[0] = sum[1] = sum[2] = sum[3] = 0;
      for (dy = -1; dy <= 1; ++dy) {
	for (dx = -1; dx <= 1; ++dx) {
	  c = FC(row+dy, col+dx);
	  vr = oimage[(row+dy*2)*w() + (col+dx*2)][cc];
	  vr = (vb + vr) / 2;
	  /*
	   * RGB CCDs almost universally repeat the same
	   * color filter 2 pixels in any direction.
	   */
	  val = oimage[(row+dy)*w() + (col+dx)][c];
	  weight = base - abs(vr - vb);
	  val = (llfactor + vb) * val / (llfactor + vr);
	  weight = 100;
	  avg[c] += val * weight;
	  sum[c] += weight;
	}
      }
      for (c=0; c < colors; c++) {
	if (sum[c])
	  image[row*w()+col][c] = (avg[c] + (sum[c]/2)) / sum[c];
      }
    }
  }
  free (oimage);
}

/*
   When this function is called, we only have one color for
   each pixel.  Search the 3x3 neighborhood for pixels of
   other colors, and average them.  Diagonal neighbors get
   counted once, orthogonal neighbors twice.
 */
void
Fl_CRW_Image::first_interpolate()
{
  int row, col, cc, x, y, c, val;
  int avg[8];

  for (row=1; row < h()-1; row++)
    for (col=1; col < w()-1; col++) {
      cc = FC(row,col);
      memset (avg, 0, sizeof avg);
      for (y = row-1; y < row+2; y++)
	for (x = col-1; x < col+2; x++)
	  if ((c = FC(y,x)) != cc) {
	    val = image[y*w()+x][c];
	    avg[c] += val;
	    avg[c+4]++;
	    if (y==row || x==col) {	/* Orthogonal neighbor */
	      avg[c] += val;
	      avg[c+4]++;
	    }
	  }
      for (c=0; c < colors; c++)
	if (c != cc)
	  image[row*w()+col][c] = avg[c] / avg[c+4];
    }
}

/*
   We now have all color values for each pixel.  Smooth the
   color balance to avoid artifacts.  This function may be
   called more than once.
*/
void
Fl_CRW_Image::second_interpolate()
{
  ushort4 *last;
  ushort4 *current;
  void *tmp;
  int row, col, cc, x, y, c, val;
  int avg[8];

  last = new ushort4[w()];
  current = new ushort4[w()];
  if (!last || !current) {
    perror("second_interpolate() new failed");
    exit(1);
  }
  for (row=2; row < h()-2; row++) {
    for (col=2; col < w()-2; col++) {
      cc = FC(row,col);
      memset (avg, 0, sizeof avg);
      for (y = row-1; y < row+2; y++)
	for (x = col-1; x < col+2; x++)
	  if ((c = FC(y,x)) != cc && image[y*w()+x][cc]) {
	    val = ((unsigned long) image[y*w()+x][c] << 16) /
		image[y*w()+x][cc] * image[row*w()+col][cc] >> 16;
	    avg[c] += val;
	    avg[c+4]++;
	    if (y==row || x==col) {	/* Orthogonal neighbor */
	      avg[c] += val;
	      avg[c+4]++;
	    }
	  }
      current[col][cc] = image[row*w()+col][cc];
      for (c=0; c < colors; c++)
	if (c != cc)
	  current[col][c] = avg[c+4] ? avg[c] / avg[c+4] : 0;
    }
    if (row > 2)
      memcpy (image[(row-1)*w()+2], last+2, (w()-4)*sizeof *last);
    tmp = last;
    last = current;
    current = (ushort4 *)tmp;
  }
  memcpy (image[(row-1)*w()+2], last+2, (w()-4)*sizeof *last);
  delete[] last;
  delete[] current;
}

/*
   Get a 2-byte integer, making no assumptions about CPU byte order.
   Nor should we assume that the compiler evaluates left-to-right.
 */
ushort
Fl_CRW_Image::fget2(FILE *f)
{
  register uchar a, b;

  a = fgetc(f);
  b = fgetc(f);
  if (order == 0x4d4d)		/* "MM" means big-endian */
    return (a << 8) + b;
  else				/* "II" means little-endian */
    return a + (b << 8);
}

/*
   Same for a 4-byte integer.
 */
int
Fl_CRW_Image::fget4(FILE *f)
{
  register uchar a, b, c, d;

  a = fgetc(f);
  b = fgetc(f);
  c = fgetc(f);
  d = fgetc(f);
  if (order == 0x4d4d)
    return (a << 24) + (b << 16) + (c << 8) + d;
  else
    return a + (b << 8) + (c << 16) + (d << 24);
}

/*
   Parse a TIFF file looking for camera name and decompress offsets.
 */
void
Fl_CRW_Image::parse_tiff(FILE *ifp)
{
  int doff, entries, tag, type, len, val, save;

  fseek(ifp, 2, SEEK_SET);	/* open_and_id() already got byte order */
  val = fget2(ifp);		/* Should be 42 for standard TIFF */
  while ((doff = fget4(ifp))) {
    fseek(ifp, doff, SEEK_SET);
    entries = fget2(ifp);
    while (entries--) {
      tag  = fget2(ifp);
      type = fget2(ifp);
      len  = fget4(ifp);
      val  = fget4(ifp);
      save = ftell(ifp);
      switch (tag) {
	case 272:			/* Model tag */
	  fseek(ifp, val, SEEK_SET);
	  fread (name, 64, 1, ifp);
	  break;
	case 330:			/* SubIFD tag */
	  nef_data_offset = val;
      }
      fseek(ifp, save, SEEK_SET);
    }
  }
}

/*
   Parse the CIFF structure looking for two pieces of information:
   The camera name, and the decode table number.
 */
void
Fl_CRW_Image::parse(FILE *ifp, int offset, int length)
{
  int tboff, nrecs, i, type, len, roff, aoff, save;

  fseek(ifp, offset+length-4, SEEK_SET);
  tboff = fget4(ifp) + offset;
  fseek(ifp, tboff, SEEK_SET);
  nrecs = fget2(ifp);
  for (i = 0; i < nrecs; i++) {
    type = fget2(ifp);
    len  = fget4(ifp);
    roff = fget4(ifp);
    aoff = offset + roff;
    save = ftell(ifp);
    if (type == 0x080a) {		/* Get the camera name */
      fseek(ifp, aoff, SEEK_SET);
      while (fgetc(ifp));
      fread (name, 64, 1, ifp);
    }
    if (type == 0x1031) {		/* Get the raw width and height */
      fseek(ifp, aoff+2, SEEK_SET);
      raw_width  = fget2(ifp);
      raw_height = fget2(ifp);
    }
    if (type == 0x1835) {		/* Get the decoder table */
      fseek(ifp, aoff, SEEK_SET);
      init_tables (fget4(ifp));
    }
    if (type >> 8 == 0x28 || type >> 8 == 0x30)	/* Get sub-tables */
      parse(ifp, aoff, len);
    fseek(ifp, save, SEEK_SET);
  }
}


void
Fl_CRW_Image::read_crw(FILE *ifp)
{
  switch (type)
  {
    case CAMERA_A50 :
        a50_read_crw(ifp);
        break;
    case CAMERA_A5 :
        a5_read_crw(ifp);
        break;
    case CAMERA_D30 :
        d30_read_crw(ifp);
        break;
    case CAMERA_EOS1D :
        eos1d_read_crw(ifp);
        break;
    case CAMERA_G1 :
        g1_read_crw(ifp);
        break;
    case CAMERA_G2 :
        g2_read_crw(ifp);
        break;
    case CAMERA_NIKON_D1X :
        nikon_d1x_read_crw(ifp);
        break;
    case CAMERA_NIKON_E5700 :
        nikon_e5700_read_crw(ifp);
        break;
    case CAMERA_OLYMPUS :
        olympus_read_crw(ifp);
        break;
    case CAMERA_PRO70 :
        pro70_read_crw(ifp);
        break;
    case CAMERA_PRO90 :
        pro90_read_crw(ifp);
        break;
    case CAMERA_PS600 :
        ps600_read_crw(ifp);
        break;
  }
}


/*
   Open a CRW file, identify which camera created it, and set
   global variables accordingly.  Returns nonzero if an error occurs.
 */
int
Fl_CRW_Image::open_and_id(FILE *ifp)
{
  char head[8], *c;
  int hlen;

  rgb_mul[0] = 1.592;
  rgb_mul[1] = 1.0;
  rgb_mul[2] = 1.261;
  rgb_max = 0x4000;
  colors = 4;
  canon = 1;

  name[0] = 0;
  order = fget2(ifp);
  if (order == 0x4949 || order == 0x4d4d) {
    hlen = fget4(ifp);
    fread (head, 1, 8, ifp);
    if (!memcmp(head,"HEAPCCDR",8)) {
      fseek(ifp, 0, SEEK_END);
      parse(ifp, hlen, ftell(ifp) - hlen);
      fseek(ifp, hlen, SEEK_SET);
    } else
      parse_tiff(ifp);
  }
  c = name + strlen(name);	/* Remove trailing spaces */
  while (*--c == ' ') *c = 0;
  if (name[0] == 0) {
//    fprintf(stderr,"%s has an unknown format.\n",fname);
    return 1;
  }
  if (!strcmp(name,"Canon PowerShot 600")) {
    h(613);
    w(854);
    filters = 0xe1e4e1e4;
    type = CAMERA_PS600;
    rgb_mul[0] = 1.667;
    rgb_mul[2] = 1.667;
  } else if (!strcmp(name,"Canon PowerShot A5")) {
    h(776);
    w(960);
    filters = 0x1e4e1e4e;
    type = CAMERA_A5;
    rgb_mul[0] = 1.111;
    rgb_mul[2] = 0.978;
  } else if (!strcmp(name,"Canon PowerShot A50")) {
    h(968);
    w(1290);
    filters = 0x1b4e4b1e;
    type = CAMERA_A50;
    rgb_mul[0] = 1.316;
    rgb_mul[2] = 0.776;
  } else if (!strcmp(name,"Canon PowerShot Pro70")) {
    h(1024);
    w(1552);
    filters  = 0x1e4b4e1b;
    type = CAMERA_PRO70;
  } else if (!strcmp(name,"Canon PowerShot Pro90 IS")) {
    h(1416);
    w(1896);
    filters = 0xb4b4b4b4;
    type = CAMERA_PRO90;
  } else if (!strcmp(name,"Canon PowerShot G1")) {
    h(1550);
    w(2088);
    filters = 0xb4b4b4b4;
    type = CAMERA_G1;
    rgb_mul[0] = 1.469;
    rgb_mul[2] = 1.327;
  } else if (!strcmp(name,"Canon PowerShot S30")) {
    h(1550);
    w(2088);
    colors = 3;
    filters = 0x94949494;
    type = CAMERA_G1;
    rgb_mul[0] = 1.785;
    rgb_mul[2] = 1.266;
  } else if (!strcmp(name,"Canon PowerShot G2") ||
	     !strcmp(name,"Canon PowerShot S40")) {
    h(1720);
    w(2312);
    colors = 3;
    filters = 0x94949494;
    type = CAMERA_G2;
    rgb_mul[0] = 1.828;
    rgb_mul[2] = 1.326;
  } else if (!strcmp(name,"Canon EOS D30")) {
    h(1448);
    w(2176);
    colors = 3;
    filters = 0x94949494;
    type = CAMERA_D30;
  } else if (!strcmp(name,"Canon EOS D60")) {
    h(2056);
    w(3088);
    colors = 3;
    filters = 0x94949494;
    type = CAMERA_D30;
    rgb_mul[0] = 2.242;
    rgb_mul[2] = 1.245;
    rgb_max = 16000;
  } else if (!strcmp(name,"Canon EOS-1D")) {
#ifdef LJPEG_DECODE
    h(1662);
    w(2496);
    colors = 3;
    filters = 0x61616161;
    type = CAMERA_EOS1D;
    rgb_mul[0] = 1.976;
    rgb_mul[2] = 1.282;
#else
    fprintf(stderr,"crw.c was compiled without EOS-1D support.\n");
    return 1;
#endif
  } else if (!strcmp(name,"NIKON D1")) {
    h(1324);
    w(2012);
    colors = 3;
    canon = 0;
    filters = 0x16161616;
    type = CAMERA_NIKON_D1X;
    rgb_mul[0] = 0.838;
    rgb_mul[2] = 1.095;
  } else if (!strcmp(name,"NIKON D1H")) {
    h(1324);
    w(2012);
    colors = 3;
    canon = 0;
    filters = 0x16161616;
    type = CAMERA_NIKON_D1X;
    rgb_mul[0] = 1.347;
    rgb_mul[2] = 3.279;
  } else if (!strcmp(name,"NIKON D1X")) {
    h(1324);
    w(4024);
    colors = 3;
    canon = 0;
    filters = 0x16161616;
    type = CAMERA_NIKON_D1X;
    rgb_mul[0] = 1.910;
    rgb_mul[2] = 1.220;
  } else if (!strcmp(name,"NIKON D100")) {
    h(2024);
    w(3037);
    colors = 3;
    canon = 0;
    filters = 0x61616161;
    type = CAMERA_NIKON_D1X;
    rgb_mul[0] = 2.374;
    rgb_mul[2] = 1.677;
  } else if (!strcmp(name,"E5700")) {
    h(1924);
    w(2576);
    filters = 0xe1e1e1e1;
    type = CAMERA_NIKON_E5700;
    rgb_mul[0] = 2.126;
    rgb_mul[2] = 1.197;
  } else if (!strcmp(name,"E-10")) {
    h(1684);
    w(2256);
    colors = 3;
    filters = 0x94949494;
    type = CAMERA_OLYMPUS;
    rgb_mul[0] = 1.43;
    rgb_mul[2] = 1.77;
  } else if (!strncmp(name,"E-20",4)) {
    h(1924);
    w(2576);
    colors = 3;
    filters = 0x94949494;
    type = CAMERA_OLYMPUS;
    rgb_mul[0] = 1.43;
    rgb_mul[2] = 1.77;
  } else {
    fprintf(stderr,"Sorry, the %s is not yet supported.\n",name);
    return 1;
  }
  rgb_mul[0] *= red_scale;	/* Apply user-selected color balance */
  rgb_mul[2] *= blue_scale;
  if (colors == 4) make_coeff();
  return 0;
}

/*
   Given a matrix that converts RGB to GMCY, create a matrix to do
   the opposite.  Only square matrices can be inverted, so I create
   four 3x3 matrices by omitting a different GMCY color in each one.
   The final coeff[][] matrix is the sum of these four.
 */
void
Fl_CRW_Image::make_coeff()
{
  static const float gmcy[4][3] = {
/*    red  green  blue			   */
    { 0.11, 0.86, 0.08 },	/* green   */
    { 0.50, 0.29, 0.51 },	/* magenta */
    { 0.11, 0.92, 0.75 },	/* cyan    */
    { 0.81, 0.98, 0.08 }	/* yellow  */
  };
  double invert[3][6], num;
  int ignore, i, j, k, r, g;

  memset (coeff, 0, sizeof coeff);
  for (ignore=0; ignore < 4; ignore++) {
    for (j=0; j < 3; j++) {
      g = (j < ignore) ? j : j+1;
      for (r=0; r < 3; r++) {
	invert[j][r] = gmcy[g][r];	/* 3x3 matrix to invert */
	invert[j][r+3] = (r == j);	/* Identity matrix	*/
      }
    }
    for (j=0; j < 3; j++) {
      num = invert[j][j];		/* Normalize this row	*/
      for (i=0; i < 6; i++)
	invert[j][i] /= num;
      for (k=0; k < 3; k++) {		/* Subtract it from the other rows */
	if (k==j) continue;
	num = invert[k][j];
	for (i=0; i < 6; i++)
	  invert[k][i] -= invert[j][i] * num;
      }
    }
    for (j=0; j < 3; j++) {		/* Add the result to coeff[][] */
      g = (j < ignore) ? j : j+1;
      for (r=0; r < 3; r++)
	coeff[r][g] += invert[r][j+3];
    }
  }
  for (r=0; r < 3; r++)			/* Multiply coeff[][] by rgb_mul[] */
    for (g=0; g < 4; g++)
      coeff[r][g] *= rgb_mul[r];
}

void
Fl_CRW_Image::get_rgb(float rgb[4], ushort4 image)
{
  int r, g;

  memset (rgb, 0, 4 * sizeof (float));
  if (colors == 3)
    for (r=0; r < 3; r++) {		/* RGB from RGB */
      rgb[r] = image[r] * rgb_mul[r];
      if (rgb[r] > rgb_max)
	  rgb[r] = rgb_max;
      rgb[3] += rgb[r]*rgb[r];		/* Compute magnitude */
    }
  else
    for (r=0; r < 3; r++) {		/* RGB from GMCY */
      for (g=0; g < 4; g++)
	rgb[r] += image[g] * coeff[r][g];
      if (rgb[r] < 0) rgb[r] = 0;
      rgb[3] += rgb[r]*rgb[r];
    }
}


#if 0
/*
   Write the image to a 24-bit PPM file.
 */
void write_ppm(FILE *ofp)
{
  int y, x;
  register unsigned c, val;
  uchar (*ppm)[3];
  float rgb[4], max, max2, expo, mul, scale;
  int total, histogram[0x1000];

/*
   Build a histogram of magnitudes using 4096 bins of 64 values each.
 */
  memset (histogram, 0, sizeof histogram);
  for (y=trim; y < h()-trim; y++)
    for (x=trim; x < w()-trim; x++) {
      get_rgb (rgb, image[y*w()+x]);
      val = (int) sqrt(rgb[3]) >> 6;
      if (val > 0xfff) val=0xfff;
      histogram[val]++;
    }
/*
   Set the white point to the 99.66th percentile
 */
  for (val=0x1000, total=0; --val; )
    if ((total+=histogram[val]) > (int)(w()*h()*0.01)) break;
  max = val << 6;
  max2 = max * max;

  fprintf(ofp,"P6\n%d %d\n255\n",w()-trim*2,h()-trim*2);

  ppm = calloc(w()-trim*2,3);
  if (!ppm) {
    perror("ppm calloc failed");
    exit(1);
  }
  expo = (gamma_val-1)/2;		/* Pull these out of the loop */
  mul = bright * 442 / max;

  for (y=trim; y < h()-trim; y++)
  {
    for (x=trim; x < w()-trim; x++)
    {
      get_rgb(rgb,image[y*w()+x]);
/* In some math libraries, pow(0,expo) doesn't return zero */
      scale = 0;
      if (rgb[3]) scale = mul * pow(rgb[3]/max2,expo);

      for (c=0; c < 3; c++)
      {
	val=rgb[c]*scale;
	if (val > 255) val=255;
	ppm[x-trim][c]=val;
      }
    }
    fwrite (ppm, w()-trim*2, 3, ofp);
  }
  free(ppm);
}

int main(int argc, char **argv)
{
  char data[256];
  int i, arg, write_to_files=1, dillon=0, dillon_ok, smooth=1;
  void (*write_fun)(FILE *) = write_ppm;
  const char *write_ext = ".ppm";
  FILE *ofp;

  if (argc == 1)
  {
    fprintf(stderr,
    "\nCanon PowerShot Converter v3.06"
#ifdef LJPEG_DECODE
    " with EOS-1D support"
#endif
    "\nby Dave Coffin (dcoffin@shore.net)"
    "\n\nUsage:  %s [options] file1.crw file2.crw ...\n"
    "\nValid options:"
    "\n-c        Write to standard output"
    "\n-d        Use Dillon interpolation if possible"
    "\n-s <num>  Number of times to smooth colors (1 by default)"
    "\n-g <num>  Set gamma value (%5.3f by default, only for 24-bit output)"
    "\n-b <num>  Set brightness  (%5.3f by default)"
    "\n-r <num>  Set red  scaling (daylight = 1.0)"
    "\n-l <num>  Set blue scaling (daylight = 1.0)"
    "\n-2        Write 24-bit PPM (default)"
    "\n-3        Write 48-bit PSD (Adobe Photoshop)"
#ifndef NO_PNG
    "\n-4        Write 48-bit PNG"
#endif
    "\n\n",
      argv[0], gamma_val, bright);
    exit(1);
  }

/* Parse out the options */

  for (arg=1; argv[arg][0] == '-'; arg++)
    switch (argv[arg][1])
    {
      case 'c':
	write_to_files=0;  break;
      case 'd':
	dillon=1;  break;
      case 's':
	smooth=atoi(argv[++arg]);  break;
      case 'g':
	gamma_val=atof(argv[++arg]);  break;
      case 'b':
	bright=atof(argv[++arg]);  break;
      case 'r':
	red_scale=atof(argv[++arg]);  break;
      case 'l':
	blue_scale=atof(argv[++arg]);  break;
      case '2':
	write_fun = write_ppm;
	write_ext = ".ppm";
	break;
      case '3':
	write_fun = write_psd;
	write_ext = ".psd";
	break;
#ifndef NO_PNG
      case '4':
	write_fun = write_png;
	write_ext = ".png";
	break;
#endif
      default:
	fprintf(stderr,"Unknown option \"%s\"\n",argv[arg]);
	exit(1);
    }

/* Process the named files  */

  for ( ; arg < argc; arg++)
  {
    if (open_and_id(argv[arg])) {
      if (ifp) fclose(ifp);
      continue;
    }
    image = calloc (h() * w(), sizeof *image);
    if (!image) {
      perror("image calloc failed");
      exit(1);
    }
    black = 0;
    fprintf (stderr, "Loading %s image from %s...\n",name,argv[arg]);
    (*read_crw)();
    fclose(ifp);
    if (black) {
      fprintf (stderr, "Subtracting thermal noise (%d)...\n",black);
      subtract_black();
    }
    dillon_ok = dillon;
    for (i=8; i < 32; i+=8)
      if ((filters >> i & 0xff) != (filters & 0xff))
	dillon_ok = 0;
    if (dillon_ok) {
      fprintf (stderr, "Dillon interpolation...\n");
      dillon_interpolate();
      trim = 2;
    } else {
      if (dillon)
	fprintf (stderr, "Filter pattern is not Dillon-compatible.\n");
      fprintf (stderr, "First interpolation...\n");
      first_interpolate();
      for (i=0; i < smooth; i++) {
	fprintf (stderr, "Second interpolation...\n");
	second_interpolate();
      }
      trim = 1;
    }
    ofp = stdout;
    strcpy (data, "standard output");
    if (write_to_files) {
      exten(data, argv[arg], write_ext);
      ofp = fopen(data,"wb");
      if (!ofp) {
	perror(data);
	continue;
      }
    }
    fprintf (stderr, "Writing data to %s...\n",data);
    (*write_fun)(ofp);
    if (write_to_files)
      fclose(ofp);

    free(image);
  }
  return 0;
}
#endif // 0


//
// End of "$Id: Fl_CRW_Image.cxx,v 1.5 2003/01/03 04:41:15 easysw Exp $".
//
