//
// "$Id: camera.cxx,v 1.21 2003/09/09 15:11:55 easysw Exp $"
//
// Camera imput methods.
//
// Copyright 2002-2003 by Michael Sweet.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// Contents:
//
//   flphoto::camera_cb()          - Import images from a digital camera.
//   flphoto::camera_chooser_cb()  - Select a camera.
//   flphoto::camera_close_cb()    - Close the camera dialog.
//   flphoto::camera_delete_cb()   - Delete files on the camera.
//   flphoto::camera_download_cb() - Download files from the cmera.
//   get_files()                   - Recursively get files from the camera.
//   flphoto::camera_select_new()  - Select new images from the camera.
//   progress_start()              - Update the progress widget in the camera window.
//   progress_update()             - Update the progress widget in the camera window.
//

#include "flphoto.h"
#include "i18n.h"
#include "flstring.h"
#include <stdlib.h>
#if defined(WIN32) && !defined(__CYGWIN__)
#  include <direct.h>
#  include <io.h>
#  define fl_mkdir(p)	mkdir(p)
#else
#  include <unistd.h>
#  define fl_mkdir(p)	mkdir(p, 0777)
#endif // WIN32 && !__CYGWIN__
#include <errno.h>
#include "Fl_Print_Dialog.H"
#include <FL/fl_ask.H>
#include <FL/Fl_File_Chooser.H>

#ifdef HAVE_LIBGPHOTO2
#  include <gphoto2.h>
#endif // HAVE_LIBGPHOTO2


//
// Local functions...
//

#ifdef HAVE_LIBGPHOTO2
static void	get_files(Camera *camera, const char *folder,
		          CameraList *list, GPContext *context);
static unsigned	progress_start(GPContext *context, float target,
			       const char *format, va_list args,
        		       void *data);
static void	progress_update(GPContext *context, unsigned id, float current,
		                void *data);
#endif // HAVE_LIBGPHOTO2


//
// 'flphoto::camera_cb()' - Import images from a digital camera.
//

void
flphoto::camera_cb()
{
#ifdef HAVE_LIBGPHOTO2
  int			i, n;
  CameraList		*list;
  CameraAbilitiesList	*al;
  GPPortInfoList	*il;
  char			defmodel[255];
  const char		*model,
			*mptr;
  char			quoted[255],
			*qptr;
  char			temp[1024];


  // Get the default download directory...
  prefs.get("download_directory", temp, "", sizeof(temp));
  camera_download_field_->value(temp);

  // Get the default camera model...
  prefs.get("camera", defmodel, "", sizeof(defmodel));

  if (!context_)
  {
    context_ = gp_context_new();
    gp_context_set_progress_funcs((GPContext *)context_, progress_start,
                                  progress_update, 0, camera_progress_);
  }

  gp_abilities_list_new(&al);
  gp_abilities_list_load(al, (GPContext *)context_);
  gp_port_info_list_new(&il);
  gp_port_info_list_load(il);
  gp_list_new(&list);
  gp_abilities_list_detect(al, il, list, (GPContext *)context_);
  gp_abilities_list_free(al);
  gp_port_info_list_free(il);

  camera_chooser_->clear();
  camera_chooser_->add(_("No Camera Selected"));
  camera_chooser_->value(0);

  n = gp_list_count(list);

  for (i = 0; i < n; i ++)
  {
    gp_list_get_name(list, i, &model);

    for (qptr = quoted, mptr = model; *mptr;)
    {
      if (*mptr == '/' || *mptr == '\\')
        *qptr++ = '\\';

      *qptr++ = *mptr++;
    }

    *qptr = '\0';

    camera_chooser_->add(quoted);

    if (strcasecmp(defmodel, model) == 0)
      camera_chooser_->value(i + 1);
  }

  gp_list_free(list);

  camera_window_->hotspot(camera_window_);
  camera_window_->show();

  camera_chooser_cb();

  if (camera_chooser_->size() < 3)
    fl_message(_("Sorry, no cameras were found.\n"
                 "Is your camera connected to a USB port and turned on?"));
#else
  fl_message(_("Sorry, no camera support available."));
#endif // HAVE_LIBGPHOTO2
}


//
// 'flphoto::camera_chooser_cb()' - Select a camera.
//

void
flphoto::camera_chooser_cb()
{
  const char	*defmodel;


  camera_browser_->clear();

#ifdef HAVE_LIBGPHOTO2
  if (camera_)
  {
    gp_camera_unref((Camera *)camera_);
    camera_ = 0;
  }
#endif // HAVE_LIBGPHOTO2

  if (camera_chooser_->value() == 0)
    return;

  defmodel = camera_chooser_->text(camera_chooser_->value());
  prefs.set("camera", defmodel);

#ifdef HAVE_LIBGPHOTO2
  int			i, n;
  CameraList		*list;
  CameraAbilitiesList	*al;
  GPPortInfoList	*il;
  CameraAbilities	abilities;
  GPPortInfo		info;
  CameraFile		*src;
  const char		*srcname,
			*srcpath;
  FILE			*dst;
  char			dstname[1024];
  int			model,
			port;
//  const char		*portname;
  char			thumbdir[1024];
  const char		*data;
  unsigned long		size;


  // Open the camera...
  gp_camera_new((Camera **)&camera_);

  gp_abilities_list_new(&al);
  gp_abilities_list_load(al, (GPContext *)context_);
  gp_port_info_list_new(&il);
  gp_port_info_list_load(il);
  gp_list_new(&list);
  gp_abilities_list_detect(al, il, list, (GPContext *)context_);

  model = gp_abilities_list_lookup_model(al, defmodel);

  gp_abilities_list_get_abilities(al, model, &abilities);
  gp_camera_set_abilities((Camera *)camera_, abilities);

  //**** THIS NEEDS TO BE UPDATED TO SUPPORT NON-USB CAMERAS...
//  gp_list_get_value(list, 0, &portname);
  port = gp_port_info_list_lookup_path(il, "usb:" /*portname*/);
  gp_port_info_list_get_info(il, port, &info);
  gp_camera_set_port_info((Camera *)camera_, info);

  gp_abilities_list_free(al);
  gp_port_info_list_free(il);
  gp_list_free(list);

  // Create a temporary directory for thumbnails...
  prefs.getUserdataPath(thumbdir, sizeof(thumbdir));

  // Get a list of files on the camera...
  camera_progress_->value(0);
  camera_progress_->label(_("Getting Image List..."));
  camera_progress_->show();
  Fl::check();
  camera_group_->deactivate();

  gp_list_new(&list);
  get_files((Camera *)camera_, "/", list, (GPContext *)context_);

  n = gp_list_count(list);

  camera_window_->cursor(FL_CURSOR_WAIT);
  camera_progress_->maximum(n);

  for (i = 0; i < n; i ++)
  {
    // Get the source name and path...
    gp_list_get_name(list, i, &srcname);
    gp_list_get_value(list, i, &srcpath);

    // Create the destination file...
    snprintf(dstname, sizeof(dstname), "%s/%s", thumbdir, srcname);
    if (access(dstname, 0))
    {
      // Thumbnail not downloaded...
      if ((dst = fopen(dstname, "wb")) == NULL)
      {
	fl_alert(_("Unable to create temporary thumbnail image %s:\n\n%s"),
        	 dstname, strerror(errno));
	break;
      }

      // Update progress...
      camera_progress_->label(srcname);
      camera_progress_->value(i);
      Fl::check();

      // Open the source file on the camera and copy...
      gp_file_new(&src);
      gp_camera_file_get((Camera *)camera_, srcpath, srcname,
                	 GP_FILE_TYPE_PREVIEW, src, (GPContext *)context_);
      gp_file_get_data_and_size(src, &data, &size);
      fwrite(data, 1, size, dst);
      fclose(dst);
      gp_file_unref(src);
    }

    // Add the file to the browser...
    Fl::check();
    camera_browser_->add(dstname);
    camera_browser_->make_visible(camera_browser_->count() - 1);
  }

  camera_select_new();

  camera_browser_->redraw();
  camera_progress_->hide();
  camera_group_->activate();
  camera_window_->cursor(FL_CURSOR_DEFAULT);

  gp_list_free(list);
  gp_camera_exit((Camera *)camera_, (GPContext *)context_);
#endif // HAVE_LIBGPHOTO2
}


//
// 'flphoto::camera_close_cb()' - Close the camera dialog.
//

void
flphoto::camera_close_cb()
{
  if (!camera_group_->active())
  {
    Fl_Widget::default_callback(camera_close_button_, 0);
    return;
  }

#ifdef HAVE_LIBGPHOTO2
  if (camera_)
  {
    gp_camera_unref((Camera *)camera_);
    camera_ = 0;
  }
#endif // HAVE_LIBGPHOTO2

  camera_window_->hide();
}


//
// 'flphoto::camera_delete_cb()' - Delete files on the camera.
//

void
flphoto::camera_delete_cb()
{
#ifdef HAVE_LIBGPHOTO2
  int			i, j, n;	// Looping vars
  CameraList		*list;		// List of files
  const char		*srcname,	// Source name
			*srcpath;	// Source path
  char			*thumbname;	// Thunbnail name
  Fl_Widget		*button;	// Button pressed


  // Confirm the deletion...
  if (!fl_ask(_("Really delete the selected images?")))
    return;

  // Get a list of files on the camera...
  camera_close_button_->label(_("Cancel"));
  camera_close_button_->labelcolor(FL_WHITE);
  camera_close_button_->color(FL_RED);
  camera_close_button_->redraw();

  camera_progress_->value(0);
  camera_progress_->label(_("Getting Image List..."));
  camera_progress_->show();
  camera_group_->deactivate();
  Fl::check();

  gp_list_new(&list);
  get_files((Camera *)camera_, "/", list, (GPContext *)context_);

  n = gp_list_count(list);

  camera_window_->cursor(FL_CURSOR_WAIT);
  camera_progress_->maximum(n);

  while (Fl::readqueue());

  for (i = camera_browser_->count() - 1; i >= 0; i --)
    if (camera_browser_->selected(i))
    {
      camera_browser_->make_visible(i);

      thumbname = camera_browser_->value(i)->label;

      for (j = 0; j < n; j ++)
      {
	// Get the source name and path...
	gp_list_get_name(list, j, &srcname);
	gp_list_get_value(list, j, &srcpath);

        if (strcmp(srcname, thumbname) != 0)
	  continue;

	// Update progress...
        camera_progress_->label(srcname);
	Fl::check();

	while ((button = Fl::readqueue()) != NULL)
	  if (button == camera_close_button_)
	    break;

	if (button == camera_close_button_)
	{
	  i = 0;
	  break;
	}

	// Remove the file...
	gp_camera_file_delete((Camera *)camera_, srcpath, srcname,
                	      (GPContext *)context_);

	// Remove the file from the browser...
	camera_browser_->remove(i);
        break;
      }
    }

  camera_close_button_->label(_("Close"));
  camera_close_button_->labelcolor(FL_BLACK);
  camera_close_button_->color(FL_GRAY);
  camera_close_button_->redraw();
  camera_progress_->hide();
  camera_window_->cursor(FL_CURSOR_DEFAULT);
  camera_group_->activate();

  gp_list_free(list);
  gp_camera_exit((Camera *)camera_, (GPContext *)context_);

#endif // HAVE_LIBGPHOTO2
}


//
// 'flphoto::camera_download_cb()' - Download files from the camera.
//

void
flphoto::camera_download_cb()
{
#ifdef HAVE_LIBGPHOTO2
  int			i, j, n;	// Looping vars
  CameraList		*list;		// List of files
  CameraFile		*src;		// Source file
  const char		*srcname,	// Source filename
			*srcpath;	// Source path
  FILE			*dst;		// Destination file
  const char		*dstdir;	// Destination directory
  int			dstlen;		// Destination directory length
  char			dstname[1024],	// Destination filename
			*dstptr,	// Pointer into destination
			*thumbname;	// Thumbnail name
  const char		*data;		// File data
  unsigned long		size;		// Size of file data
  int			warning;	// 0 = no warning shown, 1 = warning shown
  Fl_Widget		*button;	// Button pressed


  // Get the destination directory...
  if ((dstdir = camera_download_field_->value()) == NULL)
    dstdir = "";

  if (!*dstdir)
  {
    if ((dstdir = fl_dir_chooser(_("Download Directory?"), NULL)) == NULL)
      return;

    camera_download_field_->value(dstdir);
    prefs.set("download_directory", dstdir);
  }

  if (access(dstdir, 0))
  {
    if (fl_ask(_("Directory %s does not exist.\n\nCreate directory?"), dstdir))
    {
      if (fl_mkdir(dstdir))
      {
	fl_alert(_("Unable to create directory %s:\n\n%s"), dstdir,
	         strerror(errno));
	return;
      }
    }
    else
      return;
  }

  dstlen = strlen(dstdir) + 1;

  // Get a list of files on the camera...
  camera_close_button_->label(_("Cancel"));
  camera_close_button_->labelcolor(FL_WHITE);
  camera_close_button_->color(FL_RED);
  camera_close_button_->redraw();
  camera_progress_->value(0);
  camera_progress_->label(_("Getting Image List..."));
  camera_progress_->show();
  camera_group_->deactivate();
  Fl::check();

  gp_list_new(&list);
  get_files((Camera *)camera_, "/", list, (GPContext *)context_);

  n       = gp_list_count(list);
  warning = 0;
  button  = 0;

  camera_window_->cursor(FL_CURSOR_WAIT);

  while (Fl::readqueue());

  for (i = 0; i < camera_browser_->count(); i ++)
    if (camera_browser_->selected(i))
    {
      camera_browser_->make_visible(i);

      thumbname = camera_browser_->value(i)->label;

      for (j = 0; j < n; j ++)
      {
	// Get the source name and path...
	gp_list_get_name(list, j, &srcname);
	gp_list_get_value(list, j, &srcpath);

        if (strcmp(srcname, thumbname) != 0)
	  continue;

	// Update progress...
        camera_progress_->label(srcname);
	Fl::check();

	while ((button = Fl::readqueue()) != NULL)
	  if (button == camera_close_button_)
	    break;

	if (button == camera_close_button_)
	  break;

	// Create the destination file...
	snprintf(dstname, sizeof(dstname), "%s/%s", dstdir, srcname);
	for (dstptr = dstname + dstlen; *dstptr; dstptr ++)
	  *dstptr = tolower(*dstptr);

	if ((dst = fopen(dstname, "wb")) == NULL)
	{
	  fl_alert(_("Unable to create image %s:\n\n%s"),
        	   dstname, strerror(errno));
	  return;
	}

	// Open the source file on the camera and copy...
	gp_file_new(&src);
	gp_camera_file_get((Camera *)camera_, srcpath, srcname,
                	   GP_FILE_TYPE_NORMAL, src, (GPContext *)context_);
	gp_file_get_data_and_size(src, &data, &size);
	fwrite(data, 1, size, dst);
	fclose(dst);
	gp_file_unref(src);

        // See if this is an image file or something else...
	if (fl_filename_match(srcname, "*.{bmp,crw,jpg,pcd,png,ppm}"))
	{
	  // Add the file to the main browser...
	  browser_->add(dstname);
	  browser_->make_visible(browser_->count() - 1);
	  update_stats();
	}
	else if (!warning)
	{
	  // This image isn't usable by flPhoto; let the user know...
	  fl_message(_("\"%s\" is a movie or sound file and will not be added to the current album."),
	             srcname);
          warning = 1;
	}
        break;
      }

      if (button == camera_close_button_)
	break;

      // Deselect this image now that it is downloaded...
      camera_browser_->value(i)->selected = 0;
      camera_browser_->redraw();
    }

  browser_->select(browser_->count() - 1);
  open_image_cb();

  camera_close_button_->label(_("Close"));
  camera_close_button_->labelcolor(FL_BLACK);
  camera_close_button_->color(FL_GRAY);
  camera_close_button_->redraw();
  camera_progress_->hide();
  camera_window_->cursor(FL_CURSOR_DEFAULT);
  camera_group_->activate();

  gp_list_free(list);
  gp_camera_exit((Camera *)camera_, (GPContext *)context_);
#endif // HAVE_LIBGPHOTO2
}


#ifdef HAVE_LIBGPHOTO2
//
// 'get_files()' - Recursively get files from the camera.
//

static void
get_files(Camera	*camera,
          const char	*folder,
	  CameraList	*list,
	  GPContext	*context)
{
  int		i, n;
  char		path[1024];
  CameraList	*flist;
  const char	*name;


#ifdef DEBUG
  printf("get_files(camera=%p, folder=\"%s\", list=%p, context=%p)\n",
         camera, folder, list, context);
#endif // DEBUG

  gp_list_new(&flist);

  // List the files in this folder...
  gp_camera_folder_list_files(camera, folder, flist, context);

  n = gp_list_count(flist);

#ifdef DEBUG
  printf("# files = %d\n", n);
#endif // DEBUG

  for (i = 0; i < n; i ++)
  {
    gp_list_get_name(flist, i, &name);

#ifdef DEBUG
    int t = gp_list_append(list, name, folder);

    puts(name);
    printf("    append returned %d, count = %d\n", t, gp_list_count(list));
#else
    gp_list_append(list, name, folder);
#endif // DEBUG
  }

  gp_list_reset(flist);

  // List the folders in this folder...
  gp_camera_folder_list_folders(camera, folder, flist, context);

  n = gp_list_count(flist);

#ifdef DEBUG
  printf("# folders = %d\n", n);
#endif // DEBUG

  for (i = 0; i < n; i ++)
  {
    gp_list_get_name(flist, i, &name);

    if (strcmp(folder, "/") == 0)
      snprintf(path, sizeof(path), "/%s", name);
    else
      snprintf(path, sizeof(path), "%s/%s", folder, name);

#ifdef DEBUG
    puts(path);
#endif // DEBUG

    get_files(camera, path, list, context);
  }

  gp_list_free(flist);
}
#endif // HAVE_LIBGPHOTO2


//
// 'flphoto::camera_select_new()' - Select new images from the camera.
//

void
flphoto::camera_select_new()
{
  int			i;
  const char		*dstdir;
  int			dstlen;
  char			dstname[1024],
			*dstptr;
  Fl_Image_Browser::ITEM *item;


  if ((dstdir = camera_download_field_->value()) == NULL)
    dstdir = ".";
  else if (!*dstdir)
    dstdir = ".";

  dstlen = strlen(dstdir) + 1;

  for (i = 0; i < camera_browser_->count(); i ++)
  {
    item = camera_browser_->value(i);

    snprintf(dstname, sizeof(dstname), "%s/%s", dstdir, item->label);
    for (dstptr = dstname + dstlen; *dstptr; dstptr ++)
      *dstptr = tolower(*dstptr);

    item->selected = access(dstname, 0) != 0;
  }

  camera_browser_->damage(FL_DAMAGE_SCROLL);
}


#ifdef HAVE_LIBGPHOTO2
//
// 'progress_start()' - Update the progress widget in the camera window.
//

static unsigned
progress_start(GPContext  *context,
               float      target,
	       const char *format,
	       va_list    args,
               void       *data)
{
  Fl_Progress	*p = (Fl_Progress *)data;


  p->value(0);
  p->maximum(target);
  Fl::check();
  return (0);
}


//
// 'progress_update()' - Update the progress widget in the camera window.
//

static void
progress_update(GPContext *context,
        	unsigned  id,
		float     current,
        	void      *data)
{
  Fl_Progress	*p = (Fl_Progress *)data;


  p->value(0);
  p->value(current);
  Fl::check();
}
#endif // HAVE_LIBGPHOTO2


//
// End of "$Id: camera.cxx,v 1.21 2003/09/09 15:11:55 easysw Exp $".
//
