//
// "$Id: slideshow.cxx,v 1.15 2003/09/05 00:44:21 easysw Exp $"
//
// Slideshow methods.
//
// Copyright 2002-2003 by Michael Sweet.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// Contents:
//
//   flphoto::slideshow_cb() - Show a slideshow.
//   flphoto::slidenext_cb() - Show the next image...
//

#include "flphoto.h"
#include "i18n.h"


//
// Local globals...
//

static flphoto		*current_album;
static int		current_image;
static int		*image_order;
static Fl_Double_Window	*window = 0;
static Fl_Image_Display	*display = 0;


//
// 'flphoto::slideshow_cb()' - Show a slideshow.
//

void
flphoto::slideshow_cb()
{
  int	i, j;					// Looping vars
  int	val;					// Preference value


  // See if we have any images in this album...
  if (!browser_->count())
  {
    fl_alert(_("No images to show in a slideshow!"));
    return;
  }

  // Save the current album...
  current_album = this;
  current_image = -1;

  // Get the saved slideshow preferences...
  prefs.get("slideshow_delay", val, 5);
  ssoptions_delay_value_->value(val);

  prefs.get("slideshow_first", val, 1);
  ssoptions_first_button_->value(val);

  prefs.get("slideshow_random", val, 0);
  ssoptions_random_button_->value(val);

  prefs.get("slideshow_repeat", val, 1);
  ssoptions_repeat_button_->value(val);

  // Show the preferences dialog and wait for the user to click "Continue"...
  ssoptions_window_->hotspot(ssoptions_window_);
  ssoptions_window_->show();

  while (ssoptions_window_->shown())
    Fl::wait();

  // Save the slideshow settings...
  prefs.set("slideshow_delay", (int)ssoptions_delay_value_->value());
  prefs.set("slideshow_first", ssoptions_first_button_->value());
  prefs.set("slideshow_random", ssoptions_random_button_->value());
  prefs.set("slideshow_repeat", ssoptions_repeat_button_->value());

  // Create the slideshow window as needed...
  if (!window)
  {
    window = new Fl_Double_Window(0, 0, Fl::w(), Fl::h());
    window->modal();

    display = new Fl_Image_Display(0, 0, Fl::w(), Fl::h());
    display->align((Fl_Align)(FL_ALIGN_BOTTOM | FL_ALIGN_INSIDE | FL_ALIGN_WRAP));
    display->labelcolor(FL_WHITE);
    display->labelsize(32);
    display->labelfont(FL_BOLD);
    display->labeltype(FL_SHADOW_LABEL);
    display->box(FL_FLAT_BOX);
    display->color(FL_BLACK);
    display->callback((Fl_Callback *)slidenext_cb);
  }

  // Create an array for the images...
  image_order = new int[browser_->count()];

  if (ssoptions_first_button_->value() || browser_->selected() < 0)
  {
    for (i = 0; i < browser_->count(); i ++)
      image_order[i] = i;
  }
  else
  {
    for (i = 0, j = browser_->selected(); i < browser_->count(); i ++)
    {
      image_order[i] = j;

      j ++;
      if (j >= browser_->count())
        j = 0;
    }
  }

  if (ssoptions_random_button_->value())
  {
    // Randomize the slideshow...
    for (i = 1; i < browser_->count(); i ++)
    {
      // Pick a slide to swap with...
      do
      {
        j = rand() % browser_->count();
      }
      while (j == 0 || j == i);

      // Swap...
      val            = image_order[i];
      image_order[i] = image_order[j];
      image_order[j] = val;
    }
  }

  // Clear the slideshow window and then show the next image...
  display->image(0);
  window->show();

  slidenext_cb();

  // Wait until the user closes the slideshow...
  while (window->shown())
    Fl::wait();

  // Remove any pending timeouts...
  Fl::remove_timeout((Fl_Timeout_Handler)slidenext_cb);

  // Show the current image in the window...
  open_image_cb(current_image);
}


//
// 'flphoto::slidenext_cb()' - Show the next image...
//

void
flphoto::slidenext_cb()
{
  Fl_Image_Browser::ITEM *item;			// Item in image browser...


  // Remove any prior slideshow timeouts...
  Fl::remove_timeout((Fl_Timeout_Handler)slidenext_cb);

  // If the window is no longer visible, then we're done!
  if (!window->shown())
    return;

  // Loop until we find the next image...
  for (;;)
  {
    // Unload the current image, if any...
    if (current_image >= 0)
    {
      item = current_album->browser_->value(image_order[current_image]);

      if (item->image && !item->changed && item != current_album->image_item_)
      {
        item->image->release();
	item->image = 0;
      }
    }

    // Advance to the next or previous image...
    if (Fl::event_button() == FL_RIGHT_MOUSE ||
        Fl::event_key() == FL_BackSpace)
      current_image --;
    else
      current_image ++;

    // Wrap to the beginning or end, as needed...
    if (current_image >= current_album->browser_->count())
    {
      current_image = 0;

      if (!current_album->ssoptions_repeat_button_->value())
      {
        // Don't go back to the first image...
        window->hide();
        return;
      }
    }
    else if (current_image < 0)
      current_image = current_album->browser_->count() - 1;

    // Load the current image...
    window->cursor(FL_CURSOR_WAIT);
    Fl::flush();

    display->value(current_album->browser_->load_item(image_order[current_image]));
    display->label(current_album->browser_->value(image_order[current_image])->comments);
    Fl::flush();

    // Pre-load next image...
    if ((current_image + 1) < current_album->browser_->count())
      current_album->browser_->load_item(image_order[current_image + 1]);
    else
      current_album->browser_->load_item(image_order[0]);

    window->cursor(FL_CURSOR_DEFAULT);

    break;
  }

  // Set a timer for the next image...
  if (current_album->ssoptions_delay_value_->value() > 0.0)
    Fl::add_timeout(current_album->ssoptions_delay_value_->value(),
                    (Fl_Timeout_Handler)slidenext_cb);
}


//
// End of "$Id: slideshow.cxx,v 1.15 2003/09/05 00:44:21 easysw Exp $".
//
