#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#ifdef FNCCHK_USE_BFD
#include <bfd.h>
#endif
#include "fnccheck.h"


/* changes from 1.1.4:
   now it is not "sub" time (time psend in child functions)
   but directly local time that is given by the stat file.
*/

/* names of sort types */
char *fncchk_sort_names[FNCCHK_SORT_MAX+2] = {"!", "Local time","Total time",
                                              "Call number","Function name",
					      "No sort", "!"};

/* the MAX macro */
#define HX_MAX(a,b)  (((a)>(b))?(a):(b))

/* string buffers */
#define HX_MAX_BUFFER 1024

#define EPS 0.000001  /* 1 usec */

/* the output file */
static char *HX_DROP_FILE=FNCCHK_DEFAULT_FILE;

/* not really nice. The max number of callers for each
   function that are registered. I use this to prevent
   reallocs during profile (to have a constant overcost). */
#define HX_MAX_CALLER 64

/* the kind of time used */
#define HX_TIME_EXT 0   /* absolute time */
#define HX_TIME_CPU 1   /* internal time (CPU) */
#define HX_TIME_SYS 2   /* CPU+SYSTEM times */
static int hx_time_type=HX_TIME_EXT;


/* temp file for 'nm' command output (is there a better/nicer way ?) */
#define HX_NM_FTEMP  "/tmp/.fnccheck_nm_temp_%d.txt"
#define HX_NM_FTEMP2 "/tmp/.fnccheck_nm2_temp_%d.txt"

/* table of stats for functions */
/*    structure of a function   */
typedef struct
{
  void *function;  /* the pointer to the function */
  char *file_line; /* the FILE:LINE info for the function */
  unsigned long int nb_calls; /* total # of calls */
  unsigned long int nb_sec;   /* total # of secs */
  unsigned long int nb_usec;  /* total # of usecs */
  /* local time spend in function */
  unsigned long int nb_locsec;   /* # of secs */
  unsigned long int nb_locusec;  /* # of usecs */
  /* min and max time for the function */
  unsigned long int min_sec;   /* # of secs */
  unsigned long int min_usec;       /* # of usecs */
  unsigned long int max_sec;   /* # of secs */
  unsigned long int max_usec;       /* # of usecs */
  /* computed values */
  double local;
  double total;
  double min, max;
  char *name;
  int no_symbol;
  void *called_by[HX_MAX_CALLER];
  void *rcalled_by[HX_MAX_CALLER];  /* real site call */
  char *ncalled_by[HX_MAX_CALLER];  /* datas about real calls: name */
  char *fcalled_by[HX_MAX_CALLER];  /* datas about real calls: file:line */
  unsigned int nb_of_callers;
  unsigned int my_index;
}HX_FDInfo;

HX_FDInfo *hx_dfinfo=NULL;
int hx_nb_dfinfo=0;
/* same, but used to temp moves when creating final list */
HX_FDInfo *hx_tmp_dfinfo=NULL;
int hx_nb_tmp_dfinfo=0;


/* structure of a node (for call-graph) */
typedef struct _HX_Node HX_Node;
struct _HX_Node
{
  HX_FDInfo *function;  /* corresponding function */
  HX_Node **nexts;      /* list of childs (ended by NULL) */
  HX_Node **prevs;      /* list of callers (ended by NULL) */
  int keep;             /* bool: if true, the function is keeped */
  int treated;          /* bool+: used to propagate "keep" */
  int tag;              /* tag for cycle detection */
};

/* mult for sort order */
int hx_mult_for_sort=-1;



/* create a new node for a given function */
HX_Node *hx_create_node(HX_FDInfo *function)
{
  HX_Node *tmp;

  if ((tmp = malloc(sizeof(HX_Node))) == NULL)
    {
    fprintf(stderr, "fncdump: Memory error.\n");
    return(NULL);
    }
  if ((tmp->nexts = malloc(sizeof(HX_Node*))) == NULL)
    {
    fprintf(stderr, "fncdump: Memory error.\n");
    free(tmp);
    return(NULL);
    }
  if ((tmp->prevs = malloc(sizeof(HX_Node*))) == NULL)
    {
    fprintf(stderr, "fncdump: Memory error.\n");
    free(tmp->nexts);
    free(tmp);
    return(NULL);
    }
  tmp->nexts[0] = NULL;
  tmp->prevs[0] = NULL;
  tmp->function = function;
  tmp->keep     = 1;
  tmp->treated  = 0;
  tmp->tag      = 0;
  return(tmp);
}

/* add a child to a node */
int hx_add_child(HX_Node *node, HX_Node *child)
{
  int nb=0;

  while(node->nexts[nb] != NULL)
    nb++;
  if ((node->nexts = realloc(node->nexts, (nb+2)*sizeof(HX_Node*))) == NULL)
    {
    fprintf(stderr, "fncdump: Memory error.\n");
    return(0);
    }
  node->nexts[nb]   = child;
  node->nexts[nb+1] = NULL;
  
  return(1);
}

/* add a child to a node */
int hx_add_prev(HX_Node *node, HX_Node *prev)
{
  int nb=0;

  while(node->prevs[nb] != NULL)
    nb++;
  if ((node->prevs = realloc(node->prevs, (nb+2)*sizeof(HX_Node*))) == NULL)
    {
    fprintf(stderr, "fncdump: Memory error.\n");
    return(0);
    }
  node->prevs[nb]   = prev;
  node->prevs[nb+1] = NULL;
  
  return(1);
}


/* insert the given name in the function list */
unsigned int hx_insert_name(char *name, void *fnc)
{
  unsigned int i;

  for(i=0; i<hx_nb_dfinfo; i++)
    {
    if (fnc == hx_dfinfo[i].function)
      {
      if ((hx_dfinfo[i].name == NULL)||(hx_dfinfo[i].name[0] == '<'))
        hx_dfinfo[i].name = strdup(name);
	if (hx_dfinfo[i].name == NULL)
	  {
	  fprintf(stderr, "fncdump: Memory error!\n");
	  }
	return(i+1);
      }
    }
  /* no match: the symbol not belong to tracked functions */
  return(0);
}

/* sort functions */
int hx_sort_by_name(const void*el1, const void*el2)
{
  HX_FDInfo *rel1, *rel2;

  rel1 = (HX_FDInfo*)el1;
  rel2 = (HX_FDInfo*)el2;
  
  return(hx_mult_for_sort*strcmp(rel1->name, rel2->name));
}
int hx_sort_by_calls(const void*el1, const void*el2)
{
  HX_FDInfo *rel1, *rel2;

  rel1 = (HX_FDInfo*)el1;
  rel2 = (HX_FDInfo*)el2;

  if (rel1->nb_calls < rel2->nb_calls)
    return(hx_mult_for_sort*(-1));
  if (rel1->nb_calls > rel2->nb_calls)
    return(hx_mult_for_sort*1);
  return(0);
}
int hx_sort_by_local(const void*el1, const void*el2)
{
  HX_FDInfo *rel1, *rel2;

  rel1 = (HX_FDInfo*)el1;
  rel2 = (HX_FDInfo*)el2;

  if (rel1->local < rel2->local)
    return(hx_mult_for_sort*(-1));
  if (rel1->local > rel2->local)
    return(hx_mult_for_sort*1);
  return(0);  /* may never occur on reals */
}
int hx_sort_by_total(const void*el1, const void*el2)
{
  HX_FDInfo *rel1, *rel2;

  rel1 = (HX_FDInfo*)el1;
  rel2 = (HX_FDInfo*)el2;

  if (rel1->total < rel2->total)
    return(hx_mult_for_sort*(-1));
  if (rel1->total > rel2->total)
    return(hx_mult_for_sort*1);
  return(0);  /* may never occur on reals */
}

char *hx_get_name_by_ptr(void *fnc, int nbinfo)
{
  int i;

  for(i=0; i<nbinfo; i++)
    {
    if (hx_dfinfo[i].function == fnc)
      return(hx_dfinfo[i].name);
    }
  return("<error>"); /* may not occur */
}

int hx_get_func_by_ptr(void *fnc, int nbinfo)
{
  int i;

  for(i=0; i<nbinfo; i++)
    {
    if (hx_dfinfo[i].function == fnc)
      return(i+1);
    }
  return(0); /* may not occur */
}

/* check if a given function is in the given list */
int hx_is_in_list(char *name, char **lst)
{
  int i;

  if (lst == NULL)
    return(0);  /* no list */
  i = 0;
  while(lst[i] != NULL)
    {
    if (strcmp(lst[i], name) == 0)
      return(1);
    i++;
    }
  return(0);
}


/* propagate given value (0/1) from the given node */ 
void hx_propagate_to_child(HX_Node*node, int val)
{
  int i;

  if (node->keep == val)
    return; /* still done */

  node->keep    = val;
  node->treated = 1;
  i = 0;
  while(node->nexts[i] != NULL)
    {
    hx_propagate_to_child(node->nexts[i], val);
    i++;
    }
}
/* retro-propagate given value (0/1) from the given node */ 
void hx_propagate_to_caller(HX_Node*node, int val)
{
  int i;

  if (node->keep == val)
    return; /* still done */

  node->keep = val;
  node->treated = 1;
  i = 0;
  while(node->prevs[i] != NULL)
    {
    hx_propagate_to_caller(node->prevs[i], val);
    i++;
    }
}

/* propagate given value (0/1) from the given node */ 
void hx_propagate_to_child_p(HX_Node*node)
{
  int i;

  if (node->treated == 1)
    return; /* still done */

  node->keep    = 1;
  node->treated = 1;
  i = 0;
  while(node->nexts[i] != NULL)
    {
    hx_propagate_to_child_p(node->nexts[i]);
    i++;
    }
}
/* retro-propagate given value (0/1) from the given node */ 
void hx_propagate_to_caller_p(HX_Node*node)
{
  int i;

  if (node->treated == 1)
    return; /* still done */

  node->keep    = 1;
  node->treated = 1;
  i = 0;
  while(node->prevs[i] != NULL)
    {
    hx_propagate_to_caller_p(node->prevs[i]);
    i++;
    }
}

/* return pointer on beginning of file basename */
char *hx_get_basename(char *name)
{
  int i;

  if (name == NULL)
    return(NULL);
  if (strlen(name) == 0)
    return(name);
  /* return the 1st '/' from the end */
  for(i=strlen(name)-1; i>=0; i--)
    {
    if (name[i] == '/')
      {
      if (i == strlen(name)-1)
        return(name); /* !?! */
      return(&(name[i+1]));
      }
    }
  return(name);
}

/* extract functions name from symbol addresses */
int hx_extract_names(int options, char *exec, int nbinfo)
{
  char buffer[HX_MAX_BUFFER];  /* temp file name */
  char buffer2[HX_MAX_BUFFER]; /* temp buffer */
  char buffer3[HX_MAX_BUFFER]; /* temp buffer */
  char cmd[HX_MAX_BUFFER];     /* command to be executed */
  char tname[HX_MAX_BUFFER], tmid[HX_MAX_BUFFER], tnumb[HX_MAX_BUFFER];
  char tt[HX_MAX_BUFFER];
  void *fnc;
  FILE *f=NULL;
  unsigned int i, j;
  int nocont, tempp;
#ifdef FNCCHK_USE_BFD
  char *ename, *rnm;
  bfd *core_bfd;  /* entree du BFD */
  int min_insn_size;
  int offset_to_code;
  asection *core_text_sect;
  int core_num_syms;
  asymbol **core_syms;
  const char *nm=NULL, *fnm=NULL;
  int l=0;
#endif

  /** read symbols for the exec **/
  /* old approach: nm on the exec to extract 'T' symbols (functions) */
  if (options & FNCCHK_USE_NM)
    {
    /* temp file name */
    sprintf(buffer, HX_NM_FTEMP, (unsigned int)getpid());
    /* command -> grab real function index */
    sprintf(cmd, "nm %s 2>/dev/null | grep \" T \" 2>/dev/null >%s", exec, buffer);
    /* execute the command */
    if (system(cmd) != 0)
      {
      fprintf(stderr, "fncdump: Error while extracting symbols.\n");
      fprintf(stderr, "fncdump: check if prog '%s' exists,\n", exec);
      fprintf(stderr, "fncdump:   or if 'nm' and 'grep' are installed.\n\n");
      fprintf(stderr, "fncdump: Generating stats without functions name.\n");
      }
    /* read the temp file content */
    if ((f = fopen(buffer, "r")) == NULL)
      {
      fprintf(stderr, "fncdump: Cant open temp file '%s'.\n", buffer);
      fprintf(stderr, "fncdump: Generating stats without functions name.\n");
      }
    else
      {
      /* parse for number / name */
      do
	{
	fscanf(f, "%1024s%1024s ", tnumb, tmid);
	fgets(tname, 1024, f);  /* this way we can read ALL the name */
	if (tname[strlen(tname)-1] == '\n')
	  tname[strlen(tname)-1] = '\0';
	sprintf(tt, "0x%s", tnumb);
	sscanf(tt, "%p", &fnc);
	tempp = hx_insert_name(tname, fnc);
	/* what is this ?!? I cant remember... */
	if (tempp > 0)
	  {
	  hx_dfinfo[tempp-1].file_line = NULL;
	  }
	}
      while(!feof(f)); /* got a duplicated element here... */
      fclose(f);
      /* callers info */
      for(i=0; i<nbinfo; i++)
        {
	for(j=0; j<hx_dfinfo[i].nb_of_callers; j++)
	  {
	  hx_dfinfo[i].ncalled_by[j] = NULL;
	  hx_dfinfo[i].fcalled_by[j] = NULL;
	  }
	}
      /* remove the temp file */
      sprintf(cmd, "rm -f %s 2>/dev/null", buffer);
      if (system(cmd) != 0)
	{
	fprintf(stderr, "fncdump: Cant remove temp file '%s'.\n", buffer);
	}
      }
    }
  /* addr2line with exec name and addr, gives the function
     name, and the file and line if compiled with debug options */
  else
#ifdef FNCCHK_USE_BFD
  if (options & FNCCHK_USE_ADDR2LINE)
#endif
    {
    nocont = 0;
    /* temp file name */
    sprintf(buffer,  HX_NM_FTEMP,  (unsigned int)getpid());
    sprintf(buffer2, HX_NM_FTEMP2, (unsigned int)getpid());
    /* we generate the addresses list */
    if ((f = fopen(buffer, "w")) == NULL)
      {
      fprintf(stderr, "fncdump: Cant open temp file '%s'.\n", buffer);
      fprintf(stderr, "fncdump: Generating stats without functions name.\n");
      nocont = 1;
      }

    if (!nocont)
      {
      /* requested symbols for treated functions */
      for(i=0; i<nbinfo; i++)
        {
	fprintf(f, "%p\n", hx_dfinfo[i].function);
	}
      /* adding real call sites to obtain calling infos */
      for(i=0; i<nbinfo; i++)
        {
	for(j=0; j<hx_dfinfo[i].nb_of_callers; j++)
	  {
	  fprintf(f, "%p\n", hx_dfinfo[i].rcalled_by[j]);
	  }
	}
      fclose(f);
      /* command -> addr2line gives fnc_name\nfile:line */
      if (options & FNCCHK_FULLNAME)
        sprintf(cmd, "cat %s 2>/dev/null | addr2line -C -f -e %s 2>/dev/null >%s", buffer, exec, buffer2);
      else
        sprintf(cmd, "cat %s 2>/dev/null | addr2line -C -f -s -e %s 2>/dev/null >%s", buffer, exec, buffer2);
      if (system(cmd) != 0)
	{
	fprintf(stderr, "fncdump: Error while extracting symbols.\n");
	fprintf(stderr, "fncdump: check if prog '%s' exists,\n", exec);
	fprintf(stderr, "fncdump:   or if 'cat' and 'addr2line' are installed.\n\n");
	fprintf(stderr, "fncdump: Generating stats without functions name.\n");
	nocont = 1;
	}
      if (!nocont)
	if ((f = fopen(buffer2, "r")) == NULL)
	  {
	  fprintf(stderr, "fncdump: Cant open temp file '%s'.\n", buffer2);
	  fprintf(stderr, "fncdump: Generating stats without functions name.\n");
	  nocont = 1;
	  }
      }
    if (!nocont)
      {
      for(i=0; i<nbinfo; i++)
        {
	fgets(buffer3, 1024, f);  /* function name */
	buffer3[1023] = '\0';     /* protect */
	if (buffer3[strlen(buffer3)-1] == '\n')
	  buffer3[strlen(buffer3)-1] = '\0'; /* we catch the \n */
	if (strcmp(buffer3, "??") == 0)  /* ?? -> not found */
	  {
	  sprintf(buffer3, "<%p>", hx_dfinfo[i].function);
	  }
	hx_insert_name(buffer3, hx_dfinfo[i].function);
	/* read the unused (at this time) file:line */
	fgets(buffer3, 1024, f);
	buffer3[1023] = '\0';     /* protect */
	if (buffer3[strlen(buffer3)-1] == '\n')
	  buffer3[strlen(buffer3)-1] = '\0'; /* we catch the \n */
	if ((hx_dfinfo[i].file_line = strdup(buffer3)) == NULL)
	  {
	  fprintf(f, "fncdump: Recoverable memory error.\n");
	  hx_dfinfo[i].file_line = "<merror>";
	  }
	}
      /* callers info */
      for(i=0; i<nbinfo; i++)
        {
	for(j=0; j<hx_dfinfo[i].nb_of_callers; j++)
	  {
	  fgets(buffer3, 1024, f);  /* function name */
	  buffer3[1023] = '\0';     /* protect */
	  if (buffer3[strlen(buffer3)-1] == '\n')
	    buffer3[strlen(buffer3)-1] = '\0'; /* we catch the \n */
	  hx_dfinfo[i].ncalled_by[j] = strdup(buffer3);
	  fgets(buffer3, 1024, f);  /* function name */
	  buffer3[1023] = '\0';     /* protect */
	  if (buffer3[strlen(buffer3)-1] == '\n')
	    buffer3[strlen(buffer3)-1] = '\0'; /* we catch the \n */
	  hx_dfinfo[i].fcalled_by[j] = strdup(buffer3);
	  }
	}
      fclose(f);
      /* remove the temp file */
      sprintf(cmd, "rm -f %s 2>/dev/null", buffer);
      if (system(cmd) != 0)
	{
	fprintf(stderr, "fncdump: Cant remove temp file '%s'.\n", buffer);
	}
      sprintf(cmd, "rm -f %s 2>/dev/null", buffer2);
      if (system(cmd) != 0)
	{
	fprintf(stderr, "fncdump: Cant remove temp file '%s'.\n", buffer2);
	}
      }
    }
#ifdef FNCCHK_USE_BFD
  else /* default behavior is USE_BFD */
    {
    ename = exec;
    core_bfd = bfd_openr(ename, 0);
    if (!core_bfd)
      {
      fprintf(stderr, "fncdump: Cant open %s\n", ename);
      return(0);
      }
    /* check format */
    if (!bfd_check_format(core_bfd, bfd_object))
      {
      fprintf(stderr, "fncdump: File %s is not an object.", ename);
      return(0);
      }
    /* get TEXT section */
    core_text_sect = bfd_get_section_by_name(core_bfd, ".text");
    if (!core_text_sect)
      {
      core_text_sect = bfd_get_section_by_name (core_bfd, "$CODE$");
      if (!core_text_sect)
	{
	fprintf(stderr, "fncdump: No TEXT section in object %s.\n", ename);
	return(0);
	}
      }
    /* read symbol table */
    core_num_syms = bfd_get_symtab_upper_bound(core_bfd);
    if (core_num_syms < 0)
      {
      fprintf(stderr, "fncdump: %s\n", bfd_errmsg(bfd_get_error()));
      return(0);
      }
    core_syms = (asymbol **) malloc(sizeof(asymbol*)*core_num_syms);
    if (core_syms == NULL)
      {
      fprintf(stderr, "fncdump: Memory error while allocating %d bytes.\n", (int)sizeof(asymbol*)*core_num_syms);
      fprintf(stderr, "fncdump: Fatal error!\n");
      exit(9);
      }
    core_num_syms = bfd_canonicalize_symtab(core_bfd, core_syms);
    if (core_num_syms < 0)
      {
      free(core_syms);
      fprintf(stderr, "fncdump: %s\n", bfd_errmsg(bfd_get_error()));
      return(0);
      }
    min_insn_size = 1;
    offset_to_code = 0;
    switch (bfd_get_arch(core_bfd))
      {
      case bfd_arch_vax:
      case bfd_arch_tahoe:
	offset_to_code = 2;
	break;
      case bfd_arch_alpha:
	min_insn_size = 4;
	break;
      default:
	break;
      }
    /* functions name/file/line */
    for(i=0; i<nbinfo; i++)
      {
      bfd_find_nearest_line(core_bfd, core_text_sect, core_syms,
			    (bfd_vma)hx_dfinfo[i].function - core_text_sect->vma,
			    &nm, &fnm, (unsigned int *)&l);
      hx_insert_name((char*)fnm, hx_dfinfo[i].function);
      if (options & FNCCHK_FULLNAME)
        rnm = (char*)nm;
      else
        rnm = hx_get_basename((char*)nm);
      sprintf(buffer3, "%s:%u", ((rnm!=NULL)?rnm:"??"), l);
      if ((hx_dfinfo[i].file_line = strdup(buffer3)) == NULL)
	{
	fprintf(f, "fncdump: Recoverable memory error.\n");
	hx_dfinfo[i].file_line = "<merror>";
	}
      }
    /* adding real call sites to obtain calling infos */
    for(i=0; i<nbinfo; i++)
      {
      for(j=0; j<hx_dfinfo[i].nb_of_callers; j++)
	{
	bfd_find_nearest_line(core_bfd, core_text_sect, core_syms,
			      (bfd_vma)hx_dfinfo[i].rcalled_by[j] - core_text_sect->vma,
			      &nm, &fnm, (unsigned int *)&l);
        hx_dfinfo[i].ncalled_by[j] = strdup((fnm!=NULL?fnm:"<??>"));
	if (options & FNCCHK_FULLNAME)
          rnm = (char*)nm;
	else
          rnm = hx_get_basename((char*)nm);
        sprintf(buffer3, "%s:%u", ((rnm!=NULL)?rnm:"??"), l);
	hx_dfinfo[i].fcalled_by[j] = strdup(buffer3);
	if (hx_dfinfo[i].ncalled_by[j] == NULL)
	  {
	  fprintf(f, "fncdump: Recoverable memory error.\n");
	  hx_dfinfo[i].ncalled_by[j] = "<merror>";
	  }
	if (hx_dfinfo[i].fcalled_by[j] == NULL)
	  {
	  fprintf(f, "fncdump: Recoverable memory error.\n");
	  hx_dfinfo[i].fcalled_by[j] = "<merror>";
	  }
	}
      }
    /* close and free */
    bfd_close(core_bfd);
    free(core_syms);
    }
#endif
  /* ok */
  return(1);
}


/* propagate tags to nodes */
void hx_tag_nodes(HX_Node *node, int tag)
{
  int i;

  /* stop if node still tagged */
  if (node->tag != 0)
    return;
  /* tag the node */
  node->tag = tag;
  /* tag each child */
  i = 0;
  while(node->nexts[i] != NULL)
    {
    hx_tag_nodes(node->nexts[i], tag+1);
    i++;
    }
}

void hx_display_recurs(HX_Node *node, unsigned int nc)
{
  printf("%d: Simple recursion:\n", nc);
  printf("  '%s' calls itself.\n", node->function->name);
  printf("  Total time for this function: %f\n", node->function->total);
  printf("\n");
}

void hx_display_cycle(HX_Node *end, HX_Node *start, unsigned int nc)
{
  printf("%d: Cycle:\n", nc);
  printf("  cycle from '%s' to '%s'\n", start->function->name,
                                        end->function->name);
  printf("  Total time for this cycle: %f\n", start->function->total);
  printf("\n");
}


/* compute list of cycles */
int hx_compute_cycles(HX_Node **nodes, int nbinfo)
{
  int i, j;
  unsigned int ncycle=1;  /* current cycle */

  /* first, we "tag" the nodes in calling order */
  /* this is done for each entering point, that
     is to say each node without callers  */
  for(i=0; i<nbinfo; i++)
    {
    if (nodes[i]->prevs[0] == NULL)
      {
      hx_tag_nodes(nodes[i], 1);
      }
    }
  /* if parts of the program are pure cycles (i.e.
     a recursive function compiled alone in a
     profiled module) they are not treated with
     the previous method. So we re-do this job on
     non-marked nodes to be sure to not forget nodes */
  for(i=0; i<nbinfo; i++)
    {
    if (nodes[i]->tag == 0)
      {
      hx_tag_nodes(nodes[i], 1);
      }
    }

  /* a cycle appears if a node's child got a tag
     that is greater or equal to its tag */
  for(i=0; i<nbinfo; i++)
    {
    j = 0;
    while(nodes[i]->nexts[j] != NULL)
      {
      /* simple recursion */
      if (nodes[i]->tag == nodes[i]->nexts[j]->tag)
        {
	hx_display_recurs(nodes[i], ncycle);
	ncycle++;
	}
      /* cycle */
      else
      if (nodes[i]->tag > nodes[i]->nexts[j]->tag)
        {
	hx_display_cycle(nodes[i], nodes[i]->nexts[j], ncycle);
	ncycle++;
	}
      j++;
      }
    }

  return(1);
}




/* these functions cumulate the profile data from two .out files
   into a single, summed profile file. Please be sure that the
   two files come from the SAME EXECUTABLE (same program, same
   compilation). Else, unpredictable results in output will
   occur (most of time: many unresolved function names and
   bad cumulation of times                                   */
typedef struct
{
  void *func;
  unsigned long int sec, usec;
  unsigned long int lsec, lusec;
  unsigned long int misec, miusec;
  unsigned long int masec, mausec;
  unsigned long int nbc;
  void *callers[HX_MAX_CALLER];
  void *rcallers[HX_MAX_CALLER];
  unsigned int nb_of_callers;
}HX_CumFunction;
void hx_register_caller(HX_CumFunction *who, void *from, void *rfrom)
{
  unsigned int i;

  if (who->nb_of_callers >= HX_MAX_CALLER)
    return; /* list is full */
  for(i=0; i<who->nb_of_callers; i++)
    {
    if (who->callers[i] == from)
      return;  /* still referenced */
    }
  who->callers[who->nb_of_callers] = from;
  who->rcallers[who->nb_of_callers] = rfrom;
  who->nb_of_callers++;
}
int hx_cumulate_profile(char *p1, char *p2, char *op)
{
  FILE *f1, *f2, *of;
  char buf1[HX_MAX_BUFFER], buf2[HX_MAX_BUFFER];
  char name1[HX_MAX_BUFFER], name2[HX_MAX_BUFFER];
  int i, j, k, curf, tm1, tm2, add;
  unsigned int n1, n2;
  HX_CumFunction *cumf1, *cumf2, *cumr;  /* list of functions */
  void *fnc;
  unsigned long int itemp2;
  unsigned int nr1, nr2, ssu1, ssu2, nf1, nf2;

  /* open the files */
  if ((f1 = fopen(p1, "r")) == NULL)
    {
    fprintf(stderr, "fncdump: cant open file '%s'.\n", p1);
    return(0);
    }
  if ((f2 = fopen(p2, "r")) == NULL)
    {
    fclose(f1);
    fprintf(stderr, "fncdump: cant open file '%s'.\n", p2);
    return(0);
    }
  /* check headers */
  fscanf(f1, "%1023s", buf1);
  fscanf(f2, "%1023s", buf2);
  if (strcmp(buf1, FNCCHK_HEADER) != 0)
    {
    fprintf(stderr, "fncdump: invalid header file in '%s'.\n", p1);
    fclose(f1);
    fclose(f2);
    return(0);
    }
  if (strcmp(buf2, FNCCHK_HEADER) != 0)
    {
    fprintf(stderr, "fncdump: invalid header file in '%s'.\n", p2);
    fclose(f1);
    fclose(f2);
    return(0);
    }
  /* number of elements */
  fscanf(f1, "%u", &n1);
  fscanf(f2, "%u", &n2);
  if ((n1 <= 0)||(n2 <= 0))
    {
    fprintf(stderr, "fncdump: invalid number of elements (%u||%u).\n", n1, n2);
    fclose(f1);
    fclose(f2);
    return(0);
    }
  /* alloc tables */
  if ((cumf1 = malloc(sizeof(HX_CumFunction)*n1)) == NULL)
    {
    fprintf(stderr, "fncdump: Memory error (%d bytes)!\n", sizeof(HX_CumFunction)*n1);
    fclose(f1);
    fclose(f2);
    return(0);
    }
  if ((cumf2 = malloc(sizeof(HX_CumFunction)*n2)) == NULL)
    {
    fprintf(stderr, "fncdump: Memory error (%d bytes)!\n", sizeof(HX_CumFunction)*n2);
    free(cumf1);
    fclose(f1);
    fclose(f2);
    return(0);
    }
  /* read elements */
  for(i=0; i<n1; i++)
    {
    fscanf(f1, "%p", &fnc);
    cumf1[i].func   = fnc;
    fscanf(f1, "%lu", &itemp2);
    cumf1[i].nbc    = (unsigned long int)itemp2;
    fscanf(f1, "%lu", &itemp2);
    cumf1[i].sec    = (unsigned long int)itemp2;
    fscanf(f1, "%lu", &itemp2);
    cumf1[i].usec   = (unsigned long int)itemp2;
    fscanf(f1, "%lu", &itemp2);
    cumf1[i].lsec   = (unsigned long int)itemp2;
    fscanf(f1, "%lu", &itemp2);
    cumf1[i].lusec  = (unsigned long int)itemp2;
    fscanf(f1, "%lu", &itemp2);
    cumf1[i].misec  = (unsigned long int)itemp2;
    fscanf(f1, "%lu", &itemp2);
    cumf1[i].miusec = (unsigned long int)itemp2;
    fscanf(f1, "%lu", &itemp2);
    cumf1[i].masec  = (unsigned long int)itemp2;
    fscanf(f1, "%lu", &itemp2);
    cumf1[i].mausec = (unsigned long int)itemp2;
    }
  fscanf(f1, "%u", &nr1);
  fscanf(f1, "%u", &ssu1);
  fscanf(f1, "%u", &nf1);
  for(i=0; i<n1; i++)
    {
    fscanf(f1, "%u", &(cumf1[i].nb_of_callers));
    for(j=0; j<cumf1[i].nb_of_callers; j++)
      {
      fscanf(f1, "%p", &(cumf1[i].callers[j]));
      fscanf(f1, "%p", &(cumf1[i].rcallers[j]));
      }
    }
  for(i=0; i<n2; i++)
    {
    fscanf(f2, "%p", &fnc);
    cumf2[i].func   = fnc;
    fscanf(f2, "%lu", &itemp2);
    cumf2[i].nbc    = (unsigned long int)itemp2;
    fscanf(f2, "%lu", &itemp2);
    cumf2[i].sec    = (unsigned long int)itemp2;
    fscanf(f2, "%lu", &itemp2);
    cumf2[i].usec   = (unsigned long int)itemp2;
    fscanf(f2, "%lu", &itemp2);
    cumf2[i].lsec   = (unsigned long int)itemp2;
    fscanf(f2, "%lu", &itemp2);
    cumf2[i].lusec  = (unsigned long int)itemp2;
    fscanf(f2, "%lu", &itemp2);
    cumf2[i].misec  = (unsigned long int)itemp2;
    fscanf(f2, "%lu", &itemp2);
    cumf2[i].miusec = (unsigned long int)itemp2;
    fscanf(f2, "%lu", &itemp2);
    cumf2[i].masec  = (unsigned long int)itemp2;
    fscanf(f2, "%lu", &itemp2);
    cumf2[i].mausec = (unsigned long int)itemp2;
    }
  fscanf(f2, "%u", &nr2);
  fscanf(f2, "%u", &ssu2);
  fscanf(f2, "%u", &nf2);
  for(i=0; i<n2; i++)
    {
    fscanf(f2, "%u", &(cumf2[i].nb_of_callers));
    for(j=0; j<cumf2[i].nb_of_callers; j++)
      {
      fscanf(f2, "%p", &(cumf2[i].callers[j]));
      fscanf(f2, "%p", &(cumf2[i].rcallers[j]));
      }
    }
  /* read the time modes */
  fscanf(f1, "%d", &tm1);
  fscanf(f2, "%d", &tm2);
  /* must be the same */
  if (tm1 != tm2)
    {
    fprintf(stderr,
      "fncdump: time mode in files %s and %s are not the same.\n", p1, p2);
    fclose(f1);
    fclose(f2);
    free(cumf1);
    free(cumf2);
    return(0);
    }
  /* read exec names (not implemented) */
  fscanf(f1, "%1023s", name1);
  fscanf(f2, "%1023s", name2);
  /* files readen */
  fclose(f1);
  fclose(f2);
  /* must be the same exec */
  if (strcmp(name1, name2) != 0)
    {
    fprintf(stderr,
      "fncdump: exec name in files %s and %s are not the same.\n", p1, p2);
    free(cumf1);
    free(cumf2);
    return(0);
    }
  
  /* now we duplicate the functions from #1 into a n1+n2 list
     (max of possible functions)  */
  if ((cumr = malloc(sizeof(HX_CumFunction)*(n1+n2))) == NULL)
    {
    fprintf(stderr, "fncdump: Memory error (%d bytes)!\n", sizeof(HX_CumFunction)*(n1+n2));
    free(cumf1);
    free(cumf2);
    return(0);
    }
  for(i=0; i<n1; i++)
    {
    cumr[i] = cumf1[i];
    }
  /* for each function from #2:  if not registered, duplicate
     it in the new list.  if registered, add its stats to the
     existing list */
  curf = n1;
  for(i=0; i<n2; i++)
    {
    /* search the function */
    add = 0;
    for(j=0; j<n1; j++)
      {
      if (cumr[j].func == cumf2[i].func)
        {
        add = 1;
        break;
        }
      }
    if (add)
      {/* add stats */
      /* # of call */
      cumr[j].nbc += cumf2[i].nbc;
      /* times */
      cumr[j].sec    += cumf2[i].sec;
      cumr[j].usec   += cumf2[i].usec;
      cumr[j].lsec   += cumf2[i].lsec;
      cumr[j].lusec  += cumf2[i].lusec;
      cumr[j].misec  += cumf2[i].misec;
      cumr[j].miusec += cumf2[i].miusec;
      cumr[j].masec  += cumf2[i].masec;
      cumr[j].mausec += cumf2[i].mausec;
      /* callers list */
      for(k=0; k<cumf2[i].nb_of_callers; k++)
        {
        hx_register_caller(&(cumr[j]), cumf2[i].callers[k], cumf2[i].rcallers[k]);
        }
      }
    else
      {/* duplicate element */
      cumr[curf] = cumf2[i];
      curf++;
      }
    }

  /* now we just output the new file */
  if (strcmp(op, "-") == 0)
    {
    of = stdout;
    }
  else
    {
    if ((of = fopen(op, "w")) == NULL)
      {
      fprintf(stderr, "fncdump: cant create output file '%s'.\n", op);
      free(cumf1);
      free(cumf2);
      free(cumr);
      }
    }
  fprintf(of, "%s\n", FNCCHK_HEADER);
  fprintf(of, "%d\n", curf);
  for(i=0; i<curf; i++)
    {
    fprintf(of, "%p %lu %lu %lu %lu %lu %lu %lu %lu %lu\n",
                  cumr[i].func, cumr[i].nbc,
                  cumr[i].sec, cumr[i].usec,
                  cumr[i].lsec, cumr[i].lusec,
                  cumr[i].misec, cumr[i].miusec,
                  cumr[i].masec, cumr[i].mausec);
    }
  fprintf(of, "%d\n", nr1+nr2); /* sum of reallocs */
  fprintf(of, "%u %u\n", HX_MAX(ssu1, ssu2), curf);
  for(i=0; i<curf; i++)
    {
    fprintf(of, "%u ", cumr[i].nb_of_callers);
    for(j=0; j<cumr[i].nb_of_callers; j++)
      {
      if (j+1 < cumr[i].nb_of_callers)
        fprintf(of, "%p %p ", cumr[i].callers[j], cumr[i].rcallers[j]);
      else
        fprintf(of, "%p %p", cumr[i].callers[j], cumr[i].rcallers[j]);
      }
    fprintf(of, "\n");
    }
  fprintf(of, "%d\n", tm1);  /* time mode */
  fprintf(of, "%s\n", name1); /* exec name */

  fclose(of);  

  /* leaving */
  free(cumf1);
  free(cumf2);
  free(cumr);
  return(1);  /* ok */
}


/* this function is in fnccheck lib */
void hx_compute_average();

/* do the output of file stat. exec in the exec name,
   fstat is the file stat name, sort_by is the kind of sort,
   options is ORed options, fonly is NULL or a comma separated
   list of functions name to use, fnot is NULL or a comma separated
   list of functions to reject */
int fnccheck_dump(char *exec, char *fstat, int sort_by, int options,
                  char *fonly_, char *fnot_)
{
  char buffer2[HX_MAX_BUFFER]; /* temp buffer */
  void *fnc;
  FILE *sf;
  unsigned int i, j, k, nbinfo, nb_reallocs;
  int prevd, nbe, lng;
  unsigned long int itemp2;
  unsigned int ssu, nf;
  int nb_hide, nb_spont, fc;
  double total;
  char deb[HX_MAX_BUFFER];
  HX_Node **nodes=NULL;
  char *fonly=NULL, *fnot=NULL;
  char **fo_lst=NULL, **fn_lst=NULL;
  int is_main;  /* true if total time is from main */

  /** special **/
  if (exec == NULL)
    {/* just estimate evarage time per call */
    /* extracted in 'fncaverage'. May not occur again */
    fprintf(stderr, "fncdump: called with 'exec==NULL'. This is no more suported.\n");
    fprintf(stderr, "fncdump: see 'fncaverage' instead.\n");
    return(1);
    }

  if (!(options & FNCCHK_CALLGRAPH))
    {
    printf("\n");
    printf("FunctionChecker V%s for gcc (by Y. Perret)\n", FNCCHK_FULL_VERSION);
    printf("Running profile data file version number %s\n\n", FNCCHK_HEADERVESRION);
    }

  /** read the stat file **/

  /* open the stat file */
  if ((sf = fopen(fstat==NULL?HX_DROP_FILE:fstat, "r")) == NULL)
    {
    fprintf(stderr, "fncdump: Cant open stat file '%s'.\n", fstat==NULL?HX_DROP_FILE:fstat);
    return(0);
    }
  /* header */
  fscanf(sf, "%1023s", buffer2);
  if (strcmp(buffer2, FNCCHK_HEADER) != 0)
    {
    fprintf(stderr, "fncdump: invalid header file in '%s'.\n", fstat==NULL?HX_DROP_FILE:fstat);
    fclose(sf);
    return(0);
    }
  /* # of elements */
  fscanf(sf, "%u", &nbinfo);
  if (nbinfo <= 0)
    {
    fprintf(stderr, "fncdump: Invalid number of elements (%u)!\n", nbinfo);
    fclose(sf);
    return(0);
    }
  hx_nb_dfinfo = nbinfo;
  if ((hx_dfinfo = malloc(sizeof(HX_FDInfo)*nbinfo)) == NULL)
    {
    fprintf(stderr, "fncdump: Memory error (%d bytes)!\n", sizeof(HX_FDInfo)*nbinfo);
    fclose(sf);
    return(0);
    }
  /* loop to read elements */
  for(i=0; i<nbinfo; i++)
    {
    hx_dfinfo[i].my_index   = i;
    hx_dfinfo[i].name       = NULL;
    hx_dfinfo[i].file_line  = NULL;
    hx_dfinfo[i].no_symbol  = 0;
    hx_dfinfo[i].total      = 0.;
    hx_dfinfo[i].local      = 0.;
    hx_dfinfo[i].min        = 0.;
    hx_dfinfo[i].max        = 0.;
    
    fscanf(sf, "%p", &fnc);
    hx_dfinfo[i].function   = fnc;
    fscanf(sf, "%lu", &itemp2);
    hx_dfinfo[i].nb_calls   = (unsigned long int)itemp2;
    fscanf(sf, "%lu", &itemp2);
    hx_dfinfo[i].nb_sec     = (unsigned long int)itemp2;
    fscanf(sf, "%lu", &itemp2);
    hx_dfinfo[i].nb_usec    = (unsigned long int)itemp2;
    fscanf(sf, "%lu", &itemp2);
    hx_dfinfo[i].nb_locsec  = (unsigned long int)itemp2;
    fscanf(sf, "%lu", &itemp2);
    hx_dfinfo[i].nb_locusec = (unsigned long int)itemp2;
    fscanf(sf, "%lu", &itemp2);
    hx_dfinfo[i].min_sec  = (unsigned long int)itemp2;
    fscanf(sf, "%lu", &itemp2);
    hx_dfinfo[i].min_usec = (unsigned long int)itemp2;
    fscanf(sf, "%lu", &itemp2);
    hx_dfinfo[i].max_sec  = (unsigned long int)itemp2;
    fscanf(sf, "%lu", &itemp2);
    hx_dfinfo[i].max_usec = (unsigned long int)itemp2;
    }
  fscanf(sf, "%u", &nb_reallocs);
  fscanf(sf, "%u", &ssu);
  fscanf(sf, "%u", &nf);
  for(i=0; i<nbinfo; i++)
    {
    fscanf(sf, "%u", &(hx_dfinfo[i].nb_of_callers));
    for(j=0; j<hx_dfinfo[i].nb_of_callers; j++)
      {
      fscanf(sf, "%p", &(hx_dfinfo[i].called_by[j]));
      fscanf(sf, "%p", &(hx_dfinfo[i].rcalled_by[j]));
      }
    }
  fscanf(sf, "%d", &hx_time_type);
  fclose(sf);


  /** read symbols for the exec **/
  if (!hx_extract_names(options, exec, nbinfo))
    {
    fprintf(stderr, "fncdump: Error while reading functions name.\n");
    }

  /** parse elements to compute needed values **/
  for(i=0; i<nbinfo; i++)
    {
    hx_dfinfo[i].local = (double)hx_dfinfo[i].nb_locsec+(double)hx_dfinfo[i].nb_locusec/1000000.;
    hx_dfinfo[i].total = (double)hx_dfinfo[i].nb_sec+(double)hx_dfinfo[i].nb_usec/1000000.;
    hx_dfinfo[i].min   = (double)hx_dfinfo[i].min_sec+(double)hx_dfinfo[i].min_usec/1000000.;
    hx_dfinfo[i].max   = (double)hx_dfinfo[i].max_sec+(double)hx_dfinfo[i].max_usec/1000000.;
    
    if (hx_dfinfo[i].name == NULL)
      { /* if not found, create a name */
      sprintf(deb, "<%p>", hx_dfinfo[i].function);
      hx_dfinfo[i].name = strdup(deb);
      hx_dfinfo[i].no_symbol = 1;
      }
    }


  /** now construct the call-graph of the profile **/
  if (fonly_ != NULL)
    fonly = strdup(fonly_);
  if (fnot_ != NULL)
    fnot  = strdup(fnot_);
  /* create the list of nodes */
  if ((nodes = malloc(sizeof(HX_Node*)*nbinfo)) == NULL)
    {
    fprintf(stderr, "fncdump: Memory error when creating call-graph.\n");
    fprintf(stderr, "fncdump: Fatal error! Abort!\n");
    return(0);
    }
  for(i=0; i<nbinfo; i++)
    {
    if ((nodes[i] = hx_create_node(&(hx_dfinfo[i]))) == NULL)
      {
      fprintf(stderr, "fncdump: Fatal error! Abort!\n");
      return(0);
      }
    }
  /* now add the links between nodes */
  for(i=0; i<nbinfo; i++)
    {
    for(j=0; j<hx_dfinfo[i].nb_of_callers; j++)
      {
      hx_add_child(nodes[hx_get_func_by_ptr(hx_dfinfo[i].called_by[j], nbinfo)-1], nodes[i]);
      hx_add_prev(nodes[i], nodes[hx_get_func_by_ptr(hx_dfinfo[i].called_by[j], nbinfo)-1]);
      }
    }
  /* search for total time: 'main' time if detected, else
     we compute the sum of each local time */
  total = 0.;
  is_main = 0;
  for(i=0; i<nbinfo; i++)
    {
    if (strcmp(hx_dfinfo[i].name, "main") == 0)
      {
      total = hx_dfinfo[i].total;
      is_main = 1;
      break;
      }
    }
  if (total == 0.)
    {
    /* the sum */
    total = 0.;
    for(i=0; i<nbinfo; i++)
      {
      total += hx_dfinfo[i].local;
      }
    /* add a small amount of time. It can cause max time not
       to be 100% but the alternative is a time > to 100%!.
       This is due to the fact the the sum of local times is
       a little bit small that total time (caused by the
       calling time to treatment functions) */
    total += total/100.;  /* add 1% */
    }
  if (total < EPS)
    {
    total = 0.;
    }

  /* display the cycles list (just displayed, not stored) */
  if (options & FNCCHK_CYCLES)
    {
    printf("Cycles during execution:\n\n");
    if (!hx_compute_cycles(nodes, nbinfo))
      {
      fprintf(stderr, "fncdump: Error while computing cycles.\n");
      }
    printf("\n");
    }

  /* extract functions name from lists */
  /* if both are used, 'fnot' is performed first,
     and then 'fonly' is added, in order to
     re-set functions which are shared by 'fnot's */ 
  nbe = 0;
  if (fonly != NULL)
    {
    for(i=0; i<strlen(fonly); i++)
      {
      if (fonly[i] == ',')
        nbe++;
      }
    if ((fo_lst = malloc(sizeof(char*)*(nbe+2))) == NULL)
      {
      fprintf(stderr, "fncdump: Memory error when treating functions.\n");
      fprintf(stderr, "fncdump: Fatal error! Abort!\n");
      return(0);
      }
    fo_lst[nbe+1] = NULL;
    fo_lst[0] = &(fonly[0]);
    j = 1;
    lng = strlen(fonly);
    for(i=1; i<lng; i++)
      {
      if (fonly[i] == ',')
        {
	fo_lst[j] = &(fonly[i+1]);
	fonly[i] = '\0';
        j++;
	}
      }
    }
  if (fnot != NULL)
    {
    for(i=0; i<strlen(fnot); i++)
      {
      if (fnot[i] == ',')
        nbe++;
      }
    if ((fn_lst = malloc(sizeof(char*)*(nbe+2))) == NULL)
      {
      fprintf(stderr, "fncdump: Memory error when treating functions.\n");
      fprintf(stderr, "fncdump: Fatal error! Abort!\n");
      return(0);
      }
    fn_lst[nbe+1] = NULL;
    fn_lst[0] = &(fnot[0]);
    j = 1;
    lng = strlen(fnot);
    for(i=1; i<lng; i++)
      {
      if (fnot[i] == ',')
        {
	fn_lst[j] = &(fnot[i+1]);
	fnot[i] = '\0';
        j++;
	}
      }
    }
  /* for each function, set it to "used" or "removed" */
  for(i=0; i<nbinfo; i++)
    {
    nodes[i]->treated = 0;
    if (fnot != NULL)
      {
      if (hx_is_in_list(nodes[i]->function->name, fn_lst))
        {
	nodes[i]->keep = 0;
	}
      else
        {
	nodes[i]->keep = 1;
	}
      }
    else
    if (fonly != NULL)
      {
      if (hx_is_in_list(nodes[i]->function->name, fo_lst))
        {
	nodes[i]->keep = 1;
	}
      else
        {
	nodes[i]->keep = 0;
	}
      }
    }
  /* propagate "used" or "removed" in call-graph if requested */
  if (options & FNCCHK_USE_CHILDS)
    {
    if (fnot != NULL)
      {
      for (i=0; i<nbinfo; i++)
        {
	if (nodes[i]->keep == 0)
	  {
	  nodes[i]->keep = 1;
          hx_propagate_to_child(nodes[i], 0);
	  }
	}
      }
    if ((fonly != NULL)&&(fnot != NULL))
      {/* just activate needed ones */
      for(i=0; i<nbinfo; i++)
        nodes[i]->treated = 0;
      for(i=0; i<nbinfo; i++)
        {
        if (hx_is_in_list(nodes[i]->function->name, fo_lst))
	  {
          hx_propagate_to_child_p(nodes[i]);
	  }
	}
      }
    else
    if (fonly != NULL)
      {
      for (i=0; i<nbinfo; i++)
        {
	if (nodes[i]->keep == 1)
	  {
	  nodes[i]->keep = 0;
          hx_propagate_to_child(nodes[i], 1);
	  }
	}
      }
    }

  /* propagate "used" or "removed" in call-graph if requested */
  if (options & FNCCHK_USE_CALLERS)
    {
    if (fnot != NULL)
      {
      for (i=0; i<nbinfo; i++)
        {
	if (nodes[i]->keep == 0)
	  {
	  nodes[i]->keep = 1;
          hx_propagate_to_caller(nodes[i], 0);
	  }
	}
      }
    if ((fonly != NULL)&&(fnot != NULL))
      {/* just activate needed ones */
      for(i=0; i<nbinfo; i++)
        nodes[i]->treated = 0;
      for(i=0; i<nbinfo; i++)
        {
        if (hx_is_in_list(nodes[i]->function->name, fo_lst))
	  {
          hx_propagate_to_caller_p(nodes[i]);
	  }
	}
      }
    else
    if (fonly != NULL)
      {
      for (i=0; i<nbinfo; i++)
        {
	if (nodes[i]->keep == 1)
	  {
	  nodes[i]->keep = 0;
          hx_propagate_to_caller(nodes[i], 1);
	  }
	}
      }
    }
  
  
  /* create the final list */
  hx_tmp_dfinfo = hx_dfinfo;
  hx_nb_tmp_dfinfo = hx_nb_dfinfo;
  hx_nb_dfinfo = 0;
  /* how many entities */
  for(i=0; i<nbinfo; i++)
    {
    if (nodes[i]->keep)
      hx_nb_dfinfo++;
    }
  if ((hx_dfinfo = malloc(sizeof(HX_FDInfo)*hx_nb_dfinfo)) == NULL)
    {
    fprintf(stderr, "fncdump: Memory error when creating new functions list.\n");
    fprintf(stderr, "fncdump: Fatal error! Abort!\n");
    }
  j = 0;
  for(i=0; i<nbinfo; i++)
    {
    if (nodes[i]->keep)
      {
      memcpy((void*)&(hx_dfinfo[j]), nodes[i]->function, sizeof(HX_FDInfo));
      j++;
      }
    }
  nbinfo = hx_nb_dfinfo;  /* new list completed */
  /* cleanup callers (remove those which are not in final list) */
  for(i=0; i<nbinfo; i++)
    {
    k = 0;
    for(j=0; j<hx_dfinfo[i].nb_of_callers; j++)
      {
      hx_dfinfo[i].called_by[k] = hx_dfinfo[i].called_by[j];
      if (hx_get_func_by_ptr(hx_dfinfo[i].called_by[j], nbinfo))
        {/* keep the caller */
	k++;
	}
      }
    hx_dfinfo[i].nb_of_callers = k;
    }


  /* if 'effective max time', recompute total time using this list */
  if (options & FNCCHK_MAXTIME_EFF)
    {
    /* search for total time: 'main' time if detected, else
       we compute the sum of each local time */
    total = 0.;
    is_main = 0;
    for(i=0; i<nbinfo; i++)
      {
      if (strcmp(hx_dfinfo[i].name, "main") == 0)
        {
	total = hx_dfinfo[i].total;
	is_main = 1;
	break;
	}
      }
    if (total == 0.)
      {
      /* the sum */
      total = 0.;
      for(i=0; i<nbinfo; i++)
	{
	total += hx_dfinfo[i].local;
	}
      /* add a small amount of time. It can cause max time not
	 to be 100% but the alternative is a time > to 100%!.
	 This is due to the fact the the sum of local times is
	 a little bit small that total time (caused by the
	 calling time to treatment functions) */
      total += total/100.;  /* add 1% */
      }
    if (total < EPS)
      {/* total=0 => no percentages performed */
      total = 0.;
      }
    }

  /** if requested, output call-graph and exit **/
  if (options & FNCCHK_CALLGRAPH)
    {
    printf("graph: {\n");
    printf(" orientation: left_to_right\n");
    /* display nodes */
    for(i=0; i<nbinfo; i++)
      {
      printf("  node: { title: \"%d\" label: \"%s\" borderwidth:0}\n",
                 i, hx_dfinfo[i].name);
      }
    /* display edges */
    for(i=0; i<nbinfo; i++)
      {
      j = 0;
      while(nodes[i]->nexts[j] != NULL)
        {
        printf("  edge: { sourcename: \"%d\" targetname: \"%d\" thickness: 1}\n",
                    i, nodes[i]->nexts[j]->function->my_index);
	j++;
	}
      }
    printf("    }\n");
    
    /** freeing all datas **/
    for(i=0; i<nbinfo; i++)
      {
      if (hx_dfinfo[i].name != NULL)
	free(hx_dfinfo[i].name);
      }
    free(hx_dfinfo);
    return(1);
    }
  
  /** sort the table **/
  if (sort_by < 0)
    {
    hx_mult_for_sort *= -1;
    sort_by = -sort_by;
    }
  switch(sort_by)
    {
    case FNCCHK_SORT_NO:  /* no sort */
      break;
    case FNCCHK_SORT_LOCAL:
      qsort((void*)hx_dfinfo, (size_t)nbinfo, sizeof(HX_FDInfo), hx_sort_by_local);
      break;
    default:   /* tri par defaut */
    case FNCCHK_SORT_TOTAL:
      qsort((void*)hx_dfinfo, (size_t)nbinfo, sizeof(HX_FDInfo), hx_sort_by_total);
      break;
    case FNCCHK_SORT_NAME:
      qsort((void*)hx_dfinfo, (size_t)nbinfo, sizeof(HX_FDInfo), hx_sort_by_name);
      break;
    case FNCCHK_SORT_CALLS:
      qsort((void*)hx_dfinfo, (size_t)nbinfo, sizeof(HX_FDInfo), hx_sort_by_calls);
      break;
    }

  nb_hide = 0;
  nb_spont = 0;
  for(i=0; i<nbinfo; i++)
    {
    if (((options & FNCCHK_NO_UNDEF) != 0)&&(hx_dfinfo[i].no_symbol))
      {
      nb_hide++;
      }
    else
      {
      if (((options & FNCCHK_NO_SPONTANEOUS) != 0)&&
          (hx_dfinfo[i].nb_of_callers == 0)&&
	  (strcmp(hx_dfinfo[i].name, "main") != 0))
        {
	nb_spont++;
	}
      }
    }

  /** generate formated output for the stats **/
  printf("General information:\n\n");
  printf("Profile for '%s'\n", exec);
  printf("Total execution time: %f", total);
  if (!is_main)
    printf(" (not main() time)");
  printf("\n");
  printf("Times computed using ");
  if (hx_time_type == HX_TIME_EXT)
    {
    printf("real clock time.\n");
    }
  else
  if (hx_time_type == HX_TIME_CPU)
    {
    printf("CPU process clock time.\n");
    }
  else
  if (hx_time_type == HX_TIME_SYS)
    {
    printf("CPU process plus SYSTEM clock time.\n");
    }
  else
    {
    printf("unknown clock time (invalid argument in stat. file!).\n");
    }
  printf("Number of realloc performed: %d\n", nb_reallocs);
  printf("Unresolved functions not shown    : %d\n", nb_hide);
  printf("Spontaneous functions not shown   : %d\n", nb_spont);
  printf("Hidden functions due to -not/-only: %d\n", hx_nb_tmp_dfinfo-nbinfo);
  printf("Final stack size used: %u\n", ssu);
  printf("Number of function(s): %u\n\n", nf);

  printf("Flat profile:\n\n");
  printf("|%17s|%17s|%10s|%11s|\n", "local      ", "total      ", "", "");
  printf("|%11s|%5s|%11s|%5s|%10s|%11s| name\n", "sec. ", "%  ", "sec. ", "%  ", "calls    ", "t. sec/call");
  printf("|-----------|-----|-----------|-----|----------|-----------|--------\n");
  for(i=0; i<nbinfo; i++)
    {
    if (!(((options & FNCCHK_NO_UNDEF) != 0)&&(hx_dfinfo[i].no_symbol)))
      {
      if (!(((options & FNCCHK_NO_SPONTANEOUS) != 0)&&
           (hx_dfinfo[i].nb_of_callers == 0)&&
	   (strcmp(hx_dfinfo[i].name, "main") != 0)))
        {
	printf("|%11f|%5.1f|%11f|%5.1f|%10u|%11f| %s\n",
	   hx_dfinfo[i].local, total==0.?0.:100.*hx_dfinfo[i].local/total,
	   hx_dfinfo[i].total, total==0.?0.:100.*hx_dfinfo[i].total/total,
	   (unsigned int)hx_dfinfo[i].nb_calls,
	   hx_dfinfo[i].total / (double)hx_dfinfo[i].nb_calls,
	   hx_dfinfo[i].name);
	}
      }
    }
  printf("\n");

  /* display MIN/MAX times */
  if (!(options & FNCCHK_NO_MINMAX))
    {
    printf("MIN/MAX total time list:\n\n");
    printf("|%4s|%13s|%13s|name\n", "indx", "MIN     ", "MAX     ");
    printf("|----|-------------|-------------|-----\n");
    for(i=0; i<nbinfo; i++)
      {
      if (!(((options & FNCCHK_NO_SPONTANEOUS) != 0)&&
           (hx_dfinfo[i].nb_of_callers == 0)&&
	   (strcmp(hx_dfinfo[i].name, "main") != 0)))
        {
	printf("|%4d|%13f|%13f| %s\n",
	       i, hx_dfinfo[i].min, hx_dfinfo[i].max,
	       hx_dfinfo[i].name);
        }
      }
    printf("\n");
    }

  /* display functions details (file:line) */
  if (options & FNCCHK_FUNC_MORE)
    {
    printf("Details for functions:\n\n");
    for(i=0; i<nbinfo; i++)
      {
      if (!(((options & FNCCHK_NO_SPONTANEOUS) != 0)&&
           (hx_dfinfo[i].nb_of_callers == 0)&&
	   (strcmp(hx_dfinfo[i].name, "main") != 0)))
        {
	printf("%s : %s\n", hx_dfinfo[i].name, ((hx_dfinfo[i].file_line!=NULL)?hx_dfinfo[i].file_line:"<n/a>"));
        }
      }
    printf("\n");
    }

  /* display who calls who */
  printf("Call-graph:\n\n");
  for(i=0; i<nbinfo; i++)
    {
    if (((options & FNCCHK_NO_UNDEF) != 0)&&(hx_dfinfo[i].no_symbol))
      {
      nb_hide++;
      }
    else
      {
      if (((options & FNCCHK_NO_SPONTANEOUS) != 0)&&
          (hx_dfinfo[i].nb_of_callers == 0)&&
	  (strcmp(hx_dfinfo[i].name, "main") != 0))
        {
	nb_spont++;
	}
      else
        {
	prevd = 0;
	if ((options & FNCCHK_CALLS_CALLED)||(!(options & FNCCHK_CALLS)))
	  {
	  prevd = 1;
	  printf("'%s' [%d] ", hx_dfinfo[i].name, i);
	  if (options & FNCCHK_CALLS_MORE)
	    printf("(%s) ", ((hx_dfinfo[i].file_line!=NULL)?hx_dfinfo[i].file_line:"<n/a>"));
	  if (hx_dfinfo[i].nb_of_callers > 0)
	    {
	    printf("called by:\n");
	    }
	  else
	    {
	    printf("spontaneously called.\n");
	    }
	  for(j=0; j<hx_dfinfo[i].nb_of_callers; j++)
	    {
	    if (options & FNCCHK_CALLS_MORE)
	      {
	      printf("   [%d](%s),\n",
	      hx_get_func_by_ptr(hx_dfinfo[i].called_by[j], nbinfo)-1,
	                              ((hx_dfinfo[i].fcalled_by[j]!=NULL)?hx_dfinfo[i].fcalled_by[j]:"<n/a>"));
	      }
	    else
 	      printf("   [%d],\n", hx_get_func_by_ptr(hx_dfinfo[i].called_by[j], nbinfo)-1);
	    }
	  printf("\n");
	  }
	if ((options & FNCCHK_CALLS_CALLED)||(options & FNCCHK_CALLS)) /* who this function calls */
	  {
	  printf("'%s' [%d] ", hx_dfinfo[i].name, i);
	  if (options & FNCCHK_CALLS_MORE)
	    printf("(%s) ", ((hx_dfinfo[i].file_line!=NULL)?hx_dfinfo[i].file_line:"<n/a>"));
	  printf("calls:");
	  fc = 1;
	  for(j=0; j<nbinfo; j++)
	    {
	    for(k=0; k<hx_dfinfo[j].nb_of_callers; k++)
	      {
	      if (hx_dfinfo[j].called_by[k] == hx_dfinfo[i].function)
		{
		if (fc == 1)
		  {
		  fc = 0;
		  printf("\n");
		  }
		if (options & FNCCHK_CALLS_MORE)
		  {
		  printf("   [%d](%s),\n", hx_get_func_by_ptr(hx_dfinfo[j].function, nbinfo)-1,
	                        	   ((hx_dfinfo[j].fcalled_by[k]!=NULL)?hx_dfinfo[j].fcalled_by[k]:"<n/a>"));
		  }
		else
		  {
		  printf("   [%d],\n", hx_get_func_by_ptr(hx_dfinfo[j].function, nbinfo)-1);
		  }
		}
	      }
	    }
	  if (fc == 1)
	    {
	    printf(" nobody.\n");
	    }
	  printf("\n");
	  }
	}
      }
    }

  /** freeing all datas **/
  for(i=0; i<nbinfo; i++)
    {
    if (hx_dfinfo[i].name != NULL)
      free(hx_dfinfo[i].name);
    }
  free(hx_dfinfo);
  if (nodes != NULL)
    free(nodes);

  /** all right, just leave **/
  return(1);
}


/*
display explanations about what 'fnccheck_dump' print.
also display details on stat-file format
*/
int hx_details(char *name)
{
  /* header */
  printf("\n");
  printf("FunctionChecker for gcc (by Hexasoft)\n\n");
  printf("\n");

  /* displayed datas */
  printf("Total execution time: ???  (a)\n");
  printf("Times computed using ???.  (b)\n");
  printf("|%20s|%20s|%12s|%13s|\n", "local      ", "total      ", "", "6 ");
  printf("|%13s|%6s|%13s|%6s|%12s|%13s| 7 name\n\n", "sec.1", "% 2", "sec.3", "% 4", "calls  5 ", "tot. sec/call");

  printf("a. total execution time\n");
  printf("b. kind of time used: real, CPU or SYSTEM\n");
  printf("1. local time spend in function. local time is total time (3) minus\n"
         "      time spend in known child-functions\n");
  printf("4. percentage of (1) regards to total execution time\n");
  printf("3. total number of seconds between enters and exits for this function\n");
  printf("4. percentage of (3) regards to total execution time\n");
  printf("5. number of function calls\n");
  printf("6. average total time per call for function\n");
  printf("7. name of the function (or symbol if not available)\n");
  printf("\n");

  printf("Unresolved function(s) not shown  : ??  (1)\n"
         "Spontaneous function(s) not shown : ??  (2)\n"
         "Hidden functions due to -not/-only: ??  (3)\n"
         "Final stack size used: ???              (4)\n"
         "Number of function(s): ???              (5)\n\n");

  printf("1. number of unresolved function hidden (see -no-unresolved)\n");
  printf("2. number of spontaneous function hidden (see -no-spontaneous)\n");
  printf("3. number of hidden functions due to -not/-only (see -not/-only)\n");
  printf("4. final stack size used during execution\n");
  printf("5. total number of functions stored during execution\n");
  printf("\n");

  printf("|%13s|%13s| name  3\n", "MIN  1  ", "MAX   2 ");
  printf("|-------------|-------------|-----\n\n");

  printf("1. minimum (total) time spend in function\n");
  printf("2. maximum (total) time spend in function\n");
  printf("3. name of the function\n");
  printf("\n");

  printf("'xxx' [n] called by:  (OR 'xxx' [n] called spontaneously.)\n"
         "[n1], [n2] (...)\n"
	 "(...)\n\n");

  printf("List of functions indexes ([n1], [n2]...) which call 'xxx'\n");
  printf("'called spontaneously' indicate that no one calls it (i.e. main)\n");
  printf("Note: * with -calls option, the list of functions CALLED BY 'xxx'\n"
         "          is displayed here.\n");
  printf("      * with +calls option, both this list and the CALLED BY list\n"
         "          are displayed.\n");
  printf("      * option -call-details add the file and line where calls\n"
         "          happend.\n");
  printf("      * option -func-details displays the list of functions, with\n"
         "          their correponding file and line.\n");
  printf("\n");

  /* file format */
  printf("Stat-file format (%s):\n", FNCCHK_FULL_VERSION);
  printf("\n"
	" HEADER\n"
	" # of elements\n"
	" f1_ptr  nb_calls nb_sec nb_usec nb_locsec nb_locusec\n"
	"        min_sec min_usec max_sec max_usec\n"
	" f2_ptr  nb_calls nb_sec nb_usec nb_locsec nb_locusec\n"
	"        min_sec min_usec max_sec max_usec\n"
	" ...\n"
	" # of realloc\n"
	" stack_size func_table_size\n"
	" nb_of_callers_f1 caller1_ptr rc1_ptr caller2_ptr rc1_ptr ...\n"
	" nb_of_callers_f2 caller1_ptr rc2_ptr caller2_ptr rc2_ptr ...\n"
	" ...\n"
	" time_mode\n"
	" exec_name\n");
  printf("HEADER    : identifier (including file-format version) of stat-files\n");
  printf("# el      : number of functions\n");
  printf("fi_ptr    : symbol of the function\n");
  printf("nb_calls  : number of calls for this function\n");
  printf("nb_sec    : total time in the function\n");
  printf("nb_usec   : corresponding usec\n");
  printf("_locsec   : local time in the function\n");
  printf("_locusec  : corresponding usec\n");
  printf("min_sec   : minimum time spend in the function\n");
  printf("_subusec  : corresponding usec\n");
  printf("min_sec   : maximum time spend in the function\n");
  printf("_subusec  : corresponding usec\n");
  printf("#realloc  : number of 'realloc' performed during execution\n");
  printf("stack_size: final stack size used during execution\n");
  printf("func_tbl_s: number of function stored during execution\n");
  printf("nbofcaller: number of callers for the corresp. function\n");
  printf("calleri_pt: symbol of ith caller for the function\n");
  printf("rci_ptr   : symbol of caller of the function\n");
  printf("time_mode : the FNCCHK_TIME type\n");
  printf("exec_name : corresp. exec name (not implemented)\n");
  
  printf("\n");
  return(1);
}



/* usage. called by main() */
void usage(char *name)
{
  int i;

  printf("fncdump by Hexasoft (Y.Perret, December 2000)\n");
  printf("Usage: %s exec [opts]\n", name);
  printf("   or: %s -avg\n", name);
  printf("Opts:  -sfile f        : use 'f' as stat file instead of 'fnccheck.out'\n");
  printf("       -sort n         : sort mode. See at the end for details\n");
  printf("       +sort n         : sort mode (reverse). See at the end for details\n");
  printf("       -no-spontaneous : dont print spontaneous called functions (dont apply to main)\n");
  printf("       -no-unresolved  : dont print unresolved symbols\n");
  printf("       -calls          : show 'called' functions instead of 'called by'\n");
  printf("       +calls          : show 'called' functions AND 'called by'\n");
  printf("       -no-minmax      : dont display MIN/MAX time for functions\n");
  printf("       -nm             : use 'nm' instead of 'libbfd'(1) to extract names\n");
  printf("       -addr2line      : use 'addr2line' instead of 'libbfd'(1) to extract names\n");
  printf("       -func-details   : add file/line for functions (not with '-nm')\n");
  printf("       -call-details   : add file/line for function calls (not with '-nm')\n");
  printf("       -fullname       : use full pathname for file/line info\n");
  printf("       -only <lst>     : just use functions that are in the comma separated list\n");
  printf("       -not <lst>      : dont use functions that are in the comma separated list\n");
  printf("       -propagate      : -only/-not is applied to childs\n");
  printf("       -rpropagate     : -only/-not is applied to callers\n");
  printf("       -real-maxtime   : use total execution time of displayed functions rather\n"
         "                         than total exec. time of all functions.\n");
  printf("       -call-graph     : output the call-graph in VCG format (nothing else displayed)\n");
  printf("       -cycles         : display detected cycles during execution\n");
  printf("    -sum-profile a b c : sum profile files 'a' and 'b' into 'c' ('-' means 'stdout')\n");
  printf("       --help          : this message\n");
  printf("       --version       : fnccheck/%s version\n", name);
  printf("       --misc          : author/contact/bugs\n");
  printf("       --details       : explanations about displayed informations\n");
  printf("(1): if 'fncdump' is compiled using 'libbfd' (standard behavior)\n");
  printf("     else (make fncdump_nobfd), -addr2line is the default approach\n");
  printf("Sort types:\n");
  for(i=FNCCHK_SORT_MIN; i<=FNCCHK_SORT_MAX; i++)
    printf("    %d:  sorted by '%s'\n", i, fncchk_sort_names[i]);
  printf("The -avg usage gives you the average time spend\n"
         "  in FncCheck treatments per each of your function.\n");
}

void misc(char *name)
{
  printf("FunctionCheck (%s) %s by Yannick Perret (Hexasoft), December 2000.\n",
              name, FNCCHK_FULL_VERSION);
  printf("For gcc V2.95.2 and higher.\n");
  printf("Author: Y. Perret (Hexasoft)\n");
  printf("EMail:  yperret@ligim.univ-lyon1.fr\n");
  printf("HTTP:   http://www710.univ-lyon1.fr/~yperret/profiler.html\n");
  printf("Send bugs report using email.\n");
}


/* options list */
typedef struct
{
  int id;
  char *opts[8];  /* max 7 alternative names   */
  int sup;        /* because last must be NULL */
}HX_Option;
typedef enum
{
  opt_ndef=0,
  opt_help,
  opt_version,
  opt_misc,
  opt_details,
  opt_average,
  opt_propagate,
  opt_rpropagate,
  opt_nm,
  opt_calls,
  opt_pcalls,
  opt_sort,
  opt_usort,
  opt_only,
  opt_not,
  opt_sfile,
  opt_call_details,
  opt_func_details,
  opt_no_minmax,
  opt_no_unresolved,
  opt_no_spontaneous,
  opt_addr2line,
  opt_fullname,
  opt_maxtimeeff,
  opt_callgraph,
  opt_cycles,
  opt_sumprof,
  opt_the_end,
}HXOL;
HX_Option hx_options[]=      {
    { opt_help, {"--help","-h","-help","--h",NULL}, 0},
    { opt_version, {"--version","-v","-version","--v",NULL}, 0},
    { opt_version, {"--misc",NULL}, 0},
    { opt_details, {"--details",NULL}, 0},
    { opt_average, {"-avg","-average",NULL}, 0},
    { opt_propagate, {"-propagate",NULL}, 0},
    { opt_rpropagate, {"-rpropagate",NULL}, 0},
    { opt_no_spontaneous, {"-no-spontaneous",NULL}, 0},
    { opt_no_unresolved, {"-no-unresolved",NULL}, 0},
    { opt_no_minmax, {"-no-minmax",NULL}, 0},
    { opt_nm, {"-nm",NULL}, 0},
    { opt_calls, {"-calls",NULL}, 0},
    { opt_pcalls, {"+calls",NULL}, 0},
    { opt_func_details, {"-function-details","-func-details","-func+",NULL}, 0},   /* -func+ is here for compatibility */
    { opt_call_details, {"-call-details", "-calls+",NULL}, 0}, /* -calls+ is here for compatibility */
    { opt_sfile, {"-sfile", "-stat-file",NULL}, 1},
    { opt_sort, {"-sort",NULL}, 1},
    { opt_usort, {"+sort",NULL}, 1},
    { opt_only, {"-only",NULL}, 1},
    { opt_not, {"-not",NULL}, 1},
    { opt_addr2line, {"-addr2line",NULL}, 0},
    { opt_fullname, {"-fullname",NULL}, 0},
    { opt_maxtimeeff, {"-real-maxtime",NULL}, 0},
    { opt_callgraph, {"-call-graph",NULL}, 0},
    { opt_cycles, {"-cycles",NULL}, 0},
    { opt_sumprof, {"-sum-profiles","-sum-profile",NULL}, 3},
    { opt_ndef, {NULL}, 0},
                            };

/* test if a given option is present */
int hx_test_option(HXOL opt, int i, int argc, char* argv[])
{
  int l, el;

  /* valid ? */
  el = 0;
  while(hx_options[el].id != opt_ndef)
    {
    if (hx_options[el].id == opt)
      break;
    el++;
    }
  if (hx_options[el].id == opt_ndef)
    return(0);   /* no match! */
  /* bounds */
  if (i+hx_options[el].sup >= argc)
    return(0);
  l = 0;
  while(hx_options[el].opts[l] != NULL)
    {
    if (strcmp(hx_options[el].opts[l], argv[i]) == 0)
      return(1);
    l++;
    }
  return(0);
}

/* just call the dump function on the given prog,
   with the alternate stat file "-file f" and the sort mode
   "-sort n", plus options and functions list */
int main(int argc, char *argv[])
{
  char *filename=NULL, *statname=NULL;
  int mode=0, options=FNCCHK_NONE;
  char *fonly=NULL, *fnot=NULL;
  int i;

  if (argc < 2)
    {
    usage(argv[0]);
    return(0);
    }


#define HX_TEST_OPTION(n) hx_test_option(n, i, argc, argv)
  /* parse options */
  for(i=1; i<argc; i++)
    {
    if ((argv[i][0] != '-')&&(argv[i][0] != '+'))
      {
      if ((filename = strdup(argv[i])) == NULL)
	{
	fprintf(stderr, "%s: Allocation error.\n", argv[0]);
	return(9);
	}
      }
    else
    if (HX_TEST_OPTION(opt_average))
      {/* special */
      if (!system("fncaverage"))
	{
	fprintf(stderr, "%s: Error while computing average time.\n", argv[0]);
	return(3);
	}
      return(0);
      }
    else
    if (HX_TEST_OPTION(opt_help))
      {
      usage(argv[0]);
      return(0);
      }
    else
    if (HX_TEST_OPTION(opt_misc))
      {
      misc(argv[0]);
      return(0);
      }
    else
    if (HX_TEST_OPTION(opt_details))
      {
      hx_details(argv[0]);
      return(0);
      }
    else
    if (HX_TEST_OPTION(opt_version))
      {
      printf("%s\n", FNCCHK_VERSION);
      return(0);
      }
    else
    if (HX_TEST_OPTION(opt_propagate))
      {
      options |= FNCCHK_USE_CHILDS;
      }
    else
    if (HX_TEST_OPTION(opt_rpropagate))
      {
      options |= FNCCHK_USE_CALLERS;
      }
    else
    if (HX_TEST_OPTION(opt_no_spontaneous))
      {
      options |= FNCCHK_NO_SPONTANEOUS;
      }
    else
    if (HX_TEST_OPTION(opt_maxtimeeff))
      {
      options |= FNCCHK_MAXTIME_EFF;
      }
    else
    if (HX_TEST_OPTION(opt_callgraph))
      {
      options |= FNCCHK_CALLGRAPH;
      }
    else
    if (HX_TEST_OPTION(opt_cycles))
      {
      options |= FNCCHK_CYCLES;
      }
    else
    if (HX_TEST_OPTION(opt_no_unresolved))
      {
      options |= FNCCHK_NO_UNDEF;
      }
    else
    if (HX_TEST_OPTION(opt_calls))
      {
      options |= FNCCHK_CALLS;
      }
    else
    if (HX_TEST_OPTION(opt_no_minmax))
      {
      options |= FNCCHK_NO_MINMAX;
      }
    else
    if (HX_TEST_OPTION(opt_func_details))
      {
      options |= FNCCHK_FUNC_MORE;
      }
    else
    if (HX_TEST_OPTION(opt_call_details))
      {
      options |= FNCCHK_CALLS_MORE;
      }
    else
    if (HX_TEST_OPTION(opt_pcalls))
      {
      options |= FNCCHK_CALLS_CALLED;
      }
    else
    if (HX_TEST_OPTION(opt_nm))
      {
      options |= FNCCHK_USE_NM;
      }
    else
    if (HX_TEST_OPTION(opt_addr2line))
      {
      options |= FNCCHK_USE_ADDR2LINE;
      }
    else
    if (HX_TEST_OPTION(opt_fullname))
      {
      options |= FNCCHK_FULLNAME;
      }
    else
    if (HX_TEST_OPTION(opt_sumprof))
      {
      if (!hx_cumulate_profile(argv[i+1], argv[i+2], argv[i+3]))
        {
        fprintf(stderr, "fncdump: error while adding profiles.\n");
        return(1);
        }
      return(0);
      }
    else
    if (HX_TEST_OPTION(opt_sfile))
      {
      if ((statname = strdup(argv[i+1])) == NULL)
        {
	fprintf(stderr, "%s: Allocation error.\n", argv[0]);
	return(9);
	}
      i++;
      }
    else
    if (HX_TEST_OPTION(opt_only))
      {
      fonly = argv[i+1];
      i++;
      }
    else
    if (HX_TEST_OPTION(opt_not))
      {
      fnot = argv[i+1];
      i++;
      }
    else
    if (HX_TEST_OPTION(opt_sort))
      {
      sscanf(argv[i+1], "%d", &mode);
      if (mode < FNCCHK_SORT_MIN)
        mode = FNCCHK_SORT_MIN;
      if (mode > FNCCHK_SORT_MAX)
        mode = FNCCHK_SORT_MAX;
      i++;
      }
    else
    if (HX_TEST_OPTION(opt_usort))
      {
      sscanf(argv[i+1], "%d", &mode);
      if (mode < FNCCHK_SORT_MIN)
        mode = FNCCHK_SORT_MIN;
      if (mode > FNCCHK_SORT_MAX)
        mode = FNCCHK_SORT_MAX;
      mode = -mode;
      i++;
      }
    else
      {
      fprintf(stderr, "%s: Option '%s' ignored.\n", argv[0], argv[i]);
      }
    }

  if (filename == NULL)
    {
    fprintf(stderr, "No program to analyse!\n");
    return(4);
    }

  if (!fnccheck_dump(filename, statname, mode, options, fonly, fnot))
    {
    fprintf(stderr, "%s: Error during dump proccess.\n", argv[0]);
    return(3);
    }

  free(filename);
  if (statname != NULL)
    free(statname);
  return(0);
}
#undef HX_TEST_OPTION
