/* fooseti
 * Copyright (c) 2001 Adam Gorski
 *
 * This file is part of fooseti.
 *
 * fooseti is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * fooseti is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with fooseti; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <signal.h>
#include <sys/stat.h>
#include <unistd.h>
#include "fileIO.h"
#include "error.h"

/* init() reads in the config file, sets up the path for the client and
   the files, and then proceeds to call call the update() function which
   reads in all the important values from those files */
  
void
init ()
{
  FILE* configfile;
  gchar buf[128];
  gchar* ptr;

  /* Read the config file */

  if(!(configfile = fopen(seti.configpath, "r"))) {
    show_error(NOCONFIG);
  } // if we can't open the config file

  else {
    /* read in the path */
      
    if (fscanf(configfile, "path=%s\n", buf) == EOF) {
      show_error(CONFRERR);
    }

    if (buf[strlen(buf) -1] == '/') {
      buf[strlen(buf) - 1] = '\0';
    } // if we have a trailing /

    /* Figure out the user, state, and pid file paths */
  
    if (seti.path == NULL) {
      if (g_strcasecmp(buf, ""))
        seti.path = g_strdup(buf);
    } // if the path is null

    /* Once a path is found, we figure out the path to each of the files
       that we're going to be reading from. If a path for that file already
       exists, we free the string and recreate it. This is necessary since
       the length of the new string is probably going to be different and
       we don't want to segfault */

    if (seti.path != NULL) {
      g_free(seti.path);
      seti.path = g_strdup(buf);

      if (seti.userpath == NULL)
        seti.userpath = g_strdup_printf("%s/user_info.sah", seti.path);

      else {
        g_free(seti.userpath);
        seti.userpath = g_strdup_printf("%s/user_info.sah", seti.path);
      } // else, if we are redefining the userpath

      if (seti.statepath == NULL)
        seti.statepath = g_strdup_printf("%s/state.sah", seti.path);

      else {
        g_free(seti.statepath);
        seti.statepath = g_strdup_printf("%s/state.sah", seti.path);
      } // else, if we are redefining the statepath

      if (seti.pidpath == NULL)
        seti.pidpath = g_strdup_printf("%s/pid.sah", seti.path);

      else {
        g_free(seti.pidpath);
        seti.pidpath = g_strdup_printf("%s/pid.sah", seti.path);
      } // else, if we are redefining the pidpath
    } // if the path isn't NULL

    /* read in the arguments to be passed to the client */

    fscanf(configfile, "%[^\n]%*[\n]", buf);
    ptr = buf + 5;

    if (data.args == NULL) {
      data.args = g_strdup(ptr);
    } // if the args are currently NULL

    else {
      g_free(data.args);
      data.args = g_strdup(ptr);
    } // else, if we are redefining the arguments

    /* read in client (binary) name */
      
    fscanf(configfile, "client=%s\n", buf);

    if (seti.client == NULL) {
      seti.client = g_strdup(buf);
    } // if the client is currently NULL
      
    else {
      g_free(seti.client);
      seti.client = g_strdup(buf);
    } // else, if we are redefining the client name

    /* User name */
      
    fscanf(configfile, "name=%d\n", &options.name);

    /* Packet number */
      
    fscanf(configfile, "packet=%d\n", &options.packet);

    /* Average time */
      
    fscanf(configfile, "atime=%d\n", &options.atime);

    /* Total CPU time */
      
    fscanf(configfile, "time=%d\n", &options.time);

    /* Stop on exit? */
      
    fscanf(configfile, "exit=%d", &options.exit);

    /* Client update time */
      
    fscanf(configfile, "update=%d", &seti.update);

    fclose(configfile);

    update();  
  } // else if config file is found 
} // init()

/* The update() function updates all the values that will be displayed from
   the various status files. It opens each file, skips to the line we want
   to read, reads it in, does any math that needs to be done, and proceeds
   to the next value in the file (or to the next file if we're done). */

void
update()
{ 
  FILE* userfile;
  FILE* statefile;
  gchar buf[128];
  gchar* ptr = buf;
  gint i, hours;
  gfloat time;

  if(!(userfile = fopen(seti.userpath, "r"))) {
    show_error(NOUSER);
  } // if we can't open the user file
    
  else {
    /* get the user name */
      
    if (fscanf(userfile, "%[^\n]%*[\n]", buf) == EOF) {
      show_error(FILERERR);
    } // if the user file is empty

    for (i = 1; i < 5; i++) {
      fscanf(userfile, "%[^\n]%*[\n]", buf);
    } // skip to the fifth line

    ptr = buf + 5;

    if (data.user == NULL) {  
      data.user = g_strdup(ptr);
    } // if this is the first run
    else {
      if (g_strcasecmp(data.user, ptr)) {
        g_free(data.user);
        data.user = g_strdup(ptr);
      } // if the name changes
    } // else, re-allocate the user name
      
    /* Get the number of packets completed */

    for (i = 1; i < 12; i++) {
      fscanf(userfile, "%[^\n]%*[\n]", buf);
    } // skip to the 16th line

    ptr = buf + 9;

    if (data.packet == NULL) {
      data.packet = g_strdup(ptr);
    } // if this is the first run
    else {
      if (g_strcasecmp(data.packet, ptr)) {
        g_free(data.packet);
        data.packet = g_strdup(ptr);
      } // if the packet changes
    } // else, re-allocate the packet number

    /* I know there is probably a better way of doing this, I
       just haven't come up with it yet.. heh */
         
    /* read total time */

    fscanf(userfile, "%[^\n]%*[\n]", buf);
    ptr = buf + 10;
    time = atof(ptr);

    /* convert time from seconds to the larger unit. I'm not sure
       if all the math in this is correct since I'm the only one who
       tested it, but I think it should all be right */

    if (time < 31536000) {
      hours = (int) time / 3600;

      if (data.time == NULL) {
        data.time = g_strdup_printf("%d hours, %d minutes", hours,
                                    (int)((time / 3600 - hours) * 60));
      } // if this is the first run
      else {
        g_free(data.time);
        data.time = g_strdup_printf("%d hours, %d minutes", hours,
                                    (int)((time / 3600 - hours) * 60));
      } // else, if the time changes
    } // if we haven't reached a year yet

    else {
      if (data.time == NULL) {
        data.time = g_strdup_printf("%.3f years", time / 31536000);
      } // if this is the first run
      else {
        g_free(data.time);
        data.time = g_strdup_printf("%.3f years", time / 31536000);
      } // else, if the time changes
    } // else, if we're at or above 1 year

    hours = (int) time / (atoi(data.packet) * 3600);

    if (data.atime == NULL) {
      data.atime = g_strdup_printf("%d hours, %d minutes", hours, 
                    (int)((time / (atof(data.packet) * 3600) - hours) * 60));
    } // if this is the first run
    else {
      g_free(data.atime);
      data.atime = g_strdup_printf("%d hours, %d minutes", hours,
                      (int)((time / (atof(data.packet) * 3600) - hours) * 60));
    } // else, if the atime changes

    fclose(userfile);
  } // else if we can open the user file

  /* Read the state file */

  if(!(statefile = fopen(seti.statepath, "r"))) {
    show_error(NOSTATE);
  } // if we can't open the state file

  else {  
    if (fscanf(statefile, "%[^\n]%*[\n]", buf) == EOF) {
      show_error(FILERERR);
    } // if the state file is empty

    for (i = 1; i < 5; i++) {
      fscanf(statefile, "%[^\n]%*[\n]", buf);
    } // skip to the fifth line

    ptr = buf + 5;
    data.packetStatus = atof(ptr);

    fclose(statefile);
  } // if the file has something in it
} // update()

/* The checkStatus() function checks the status of the client by getting
   the (hopefully) current PID and calling kill() on it. It returns true
   if the client is running, and false if it's not */

gint
checkStatus()
{
  FILE* pidfile;
  gchar pidstring[10];
  
  if(!(pidfile = fopen(seti.pidpath, "r"))) {
    return 0;
  } // if we can't open the file

  else {  
    fscanf(pidfile, "%s", pidstring);

    fclose(pidfile);
    
    seti.pid = atoi(pidstring);
  
    return !kill(seti.pid, 0);
  } // else if we open the pid file
} // checkStatus()

/* stopClient() stops the client by calling kill SIGTERM on the client's
   PID. The function then sleeps to allow for the client to actually stop
   and exit since otherwise checkStatus() would return true even though
   the client was on it's way to stop */

void
stopClient()
{
  kill(seti.pid, SIGTERM);
  g_print("Client Stopped\n");
  update();
  usleep(RWSPEED);
} // stop();

/* runClient() starts the client by going into the client's directory and
   executing the simple commands (along with any arguments). This works for
   me, but the downfall is that the binary name is assumed to be the default
   one. I might have to work on this. */

void
runClient()
{
  struct stat statBuf;
  char* command;
  char* client;

  if (seti.path == NULL) {
    show_error(NULLPATH);
  } // if the path is null
  else {
    client = g_strdup_printf("%s/%s", seti.path, seti.client);

    if(!stat(client, &statBuf)) {
      command = g_strdup_printf("cd %s ; ./%s %s &", seti.path, seti.client, data.args); 
      system(command);
      g_free(command);
      update();
      usleep(RWSPEED);
    } // if the command is found

    else {
      show_error(NOCLIENT);
    } // if the client binary is not found

    g_free(client);
  } // else if path is not null
} // startClient()
