%{

#include <iostream.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#ifdef _WIN32
#   include <strstrea.h>
#else
#   ifdef __SUNPRO_CC
#      include <strstream.h>
#   else
#      include <strstream>
#   endif
#endif

// Include malloc.h to satisfy reference to "alloca" in bison-generated
// code on win32 platforms

#ifdef _WIN32
#   include <malloc.h>
#endif

#include "dcomplex.h"

#include "Fractal.h"

#include "DoubleNodeStack.h"
#include "BinaryDoubleNode.h"
#include "UnaryDoubleNode.h"
#include "VariableDoubleNode.h"
#include "ConstantDoubleNode.h"
#include "AttributeDoubleNode.h"
#include "OneArgFunctionDoubleNode.h"
#include "TwoArgFunctionDoubleNode.h"
#include "ThreeArgFunctionDoubleNode.h"
#include "MapColorFunctionDoubleNode.h"
#include "TernaryDoubleNode.h"
#include "TextIntersectNode.h"
#include "MapColorFunctionDoubleNodeStack.h"
#include "AssignmentDoubleNode.h"
#include "GetEntryDoubleNode.h"

#include "ComplexNodeStack.h"
#include "BinaryComplexNode.h"
#include "UnaryComplexNode.h"
#include "NumericComplexNode.h"
#include "VariableComplexNode.h"
#include "OneArgFunctionComplexNode.h"
#include "TwoArgFunctionComplexNode.h"
#include "MixedThreeArgFunctionComplexNode.h"
#include "TernaryComplexNode.h"
#include "AssignmentComplexNode.h"

#include "BooleanNodeStack.h"
#include "BooleanNode.h"
#include "ConditionalBooleanNode.h"
#include "RelationalBooleanNode.h"
#include "TrueBooleanNode.h"
#include "FalseBooleanNode.h"
#include "NotBooleanNode.h"

#include "ColorNodeStack.h"
#include "ColorStack.h"
#include "ColorsColorNode.h"
#include "Counter.h"
#include "DefineColorNode.h"
#include "GradientColorNode.h"
#include "RepeatColorNode.h"
#include "SingleColorColorNode.h"
#include "ColorTable.h"

#include "ConditionalStatement.h"
#include "CompoundStatement.h"
#include "CompoundStatementStack.h"
#include "ComplexAssignmentStatement.h"
#include "DoubleAssignmentStatement.h"
#include "RGBSetColorStatement.h"
#include "HSVSetColorStatement.h"
#include "IndexedSetColorStatement.h"
#include "ForStatement.h"
#include "NullStatement.h"
#include "PrintStatement.h"
#include "BreakStatement.h"
#include "ContinueStatement.h"
#include "ComplexNodePrintArg.h"
#include "DoubleNodePrintArg.h"
#include "LiteralPrintArg.h"
#include "FlagStack.h"
#include "SwitchStatement.h"
#include "SwitchStatementStack.h"

#include "InsideBooleanNode.h"
#include "RegionNodeStack.h"
#include "CircleRegionNode.h"
#include "EllipseRegionNode.h"
#include "RectRegionNode.h"
#include "PolyRegionNode.h"
#include "SPolyRegionNode.h"
#include "CrossRegionNode.h"
#include "NotRegionNode.h"
#include "AndRegionNode.h"
#include "OrRegionNode.h"
#include "XorRegionNode.h"

#include "FractalLexer.h"
#include "fractal_parse.h"

#define YYERROR_VERBOSE (1)
#define YYDEBUG         (0)
#define YYPARSE_PARAM   voidPointer
#define ENV(x)          ((FractalParse *)voidPointer)->x

// A little trickiness to allow an additional variable to be passed
// to fractal_yyerror when it is called - this avoids the need for
// static (or global) data to record fractal_yyerror's error messages

#define fractal_yyerror(message) handleError(message, voidPointer)

extern "C" {
   int yylex(...);
}

// Local class representing parsing environment

struct FractalParse
{
   Fractal *fractalPtr;

   ComplexNodeStack                  complexNodeStack;
   BooleanNodeStack                  booleanNodeStack;
   DoubleNodeStack                   doubleNodeStack;
   RegionNodeStack                   regionNodeStack;
   ColorStack                        colorStack;
   ColorNodeStack                    colorNodeStack;
   CompoundStatementStack            compoundStack;
   MapColorFunctionDoubleNodeStack   mapStack;
   const DoubleAssignmentStatement  *currentDoubleAssign;
   const ComplexAssignmentStatement *currentComplexAssign;
   Counter                           theCounter;
   Counter                           conditionalCounter;
   Counter                           pointCounter;
   PrintStatement                   *printStatement;
   strstream                         errStream;
   FlagStack                         breakContinueAllowed;
   SwitchStatementStack              switchStack;

   int                    insideGradient;
   int                    mappingSectionStartOffset;
   int                    mappingSectionEndOffset;
   int                    errorLineNumber;

   FractalParse() :
      fractalPtr(0),
      currentDoubleAssign(0),
      currentComplexAssign(0),
      printStatement(0),
      insideGradient(0),
      mappingSectionStartOffset(0),
      mappingSectionEndOffset(0),
      errorLineNumber(0)
   {
      // Nothing to do
   }
};

/*****************************************************************************
 * Local function prototypes
 *****************************************************************************/

static int handleError(char *s, void *voidPointer);

static Fractal *fractal_createFromText(
   const char *text,
   FractalParse &env,
   char **errorMessagePointer,
   int *errorLineNumberPointer
);

static void copyFromStrStream(strstream &theStream, char **dest);

static int handleComplexOperatorAssign(
   void *voidPointer,
   char *id,
   char *token,
   BinaryComplexNode::Operator op,
   int isPostAssign = FALSE
);

static void doubleToComplex(void *voidPointer, int reverseOrder = 0);

static int handleDoubleOperatorAssign(
   void *voidPointer,
   char *id,
   char *token,
   BinaryDoubleNode::Operator op,
   int isPostAssign = FALSE
);

%}

%pure_parser

%union
{
   double dvalue;
   int ivalue;
   char *identifier;
   char *complexAttribute;
   char *literal;
}

%token <dvalue>       REAL
%token <ivalue>       INTEGER
%token <identifier>   IDENTIFIER
%token <literal>      LITERAL

%token FRACTAL
%token MAPPING
%token POINTTO
%token FORMULA
%token WHILE
%token FOR
%token COLORS
%token DEFINE
%token REPEAT
%token GRADIENT
%token COMPLEX
%token SET_COLOR
%token SET_COLOR_HSV

%token EQUALS
%token PLUS_EQUALS
%token MINUS_EQUALS
%token TIMES_EQUALS
%token DIVIDE_EQUALS
%token DOT

%token TOKEN_TRUE
%token TOKEN_FALSE
%token TOKEN_NULL

%token PLUS_PLUS
%token MINUS_MINUS

%token PRINT
%token PRINTLN
%token MAP_COLOR
%token BREAK
%token CONTINUE

%token R_CIRCLE
%token R_ELLIPSE
%token R_RECT
%token R_POLY
%token R_SPOLY
%token R_CROSS
%token R_NOT
%token R_AND
%token R_OR
%token R_XOR
%token INSIDE
%token TEXT_INTERSECT
%token SWITCH
%token CASE
%token DEFAULT

%token <complexAttribute> ATTRIBUTE_REAL
%token <complexAttribute> ATTRIBUTE_IMAGINARY
%token <complexAttribute> ATTRIBUTE_DEGREES
%token <complexAttribute> ATTRIBUTE_OLD_DEGREES
%token <complexAttribute> ATTRIBUTE_RADIANS
%token <complexAttribute> ATTRIBUTE_OLD_RADIANS
%token <complexAttribute> ATTRIBUTE_MAGNITUDE
%token <complexAttribute> ATTRIBUTE_SUM_OF_SQUARES

%token LT
%token GT
%token LE
%token GE
%token NE
%token EQ
%token OR
%token AND

%token IF
%token ELSE
%token ELSEIF

%right DOT
%right PLUS_EQUALS MINUS_EQUALS TIMES_EQUALS DIVIDE_EQUALS
%right EQUALS
%right '?' ':'
%left OR
%left AND
%left LT GT LE GE NE EQ
%left '-' '+'
%left '*' '/'
%left '%'
%right '^'
%nonassoc '!'
%nonassoc UMINUS

%type <dvalue>           number
%type <dvalue>           numeric_expression
%type <complexAttribute> complex_attribute

%%

begin: FRACTAL
       {
          // Create our fractal object

          ENV(fractalPtr) = new Fractal();
       }
       '{'
          sections
       '}'
   ;

sections: mapping colors formula | mapping formula
   ;

mapping: MAPPING
          {
            ENV(mappingSectionStartOffset) =
            FractalLexer::getInstance()->getFileOffset();
          }
        '{'
           '('
              numeric_expression ','
              numeric_expression ','
              numeric_expression ','
              numeric_expression
           ')'
           POINTTO
           '('
              numeric_expression ',' numeric_expression
           ')'
        '}'
         {
            ENV(mappingSectionEndOffset) =
            FractalLexer::getInstance()->getFileOffset();

            // Set our complex coordinates and our dimension values

            ENV(fractalPtr)->setComplexCoordinates(
               $5, $7, $9, $11
            );

            // Ensure both the height and width are > 0 (after rounding
            // them to the nearest integer)

            {
               int width  = (int)($15 < 0.0 ? $15 - 0.5 : $15 + 0.5);
               int height = (int)($17 < 0.0 ? $17 - 0.5 : $17 + 0.5);

               if(width <= 0 || height <= 0)
               {
                  strstream temp;

                  temp << "Width and height must be > 0: ["
                     << $15 << ", " << $17 << "]" << '\0';

                  handleError(temp.str(), voidPointer);
                  return(-1);
               }

               ENV(fractalPtr)->setDimensions(width, height);
            }
         }
   ;

colors: COLORS
         '{'
            { ENV(theCounter).beginCount(); }
            entries
         '}'
         {
            const ColorsColorNode *colorsPtr = new ColorsColorNode(
               ENV(colorNodeStack), ENV(theCounter).endCount()
            );

            colorsPtr->process(ENV(fractalPtr)->getColorTable());

            // Sun's compiler warns that calling delete on a const object
            // isn't kosher

            delete (ColorsColorNode *)colorsPtr;
         }
   ;

entries: entry
   |     entries entry
   ;

entry: DEFINE '(' INTEGER ')'
      '{'
         { ENV(theCounter).beginCount(); }
         color_entries
      '}'
      {
         ENV(colorNodeStack).push(
            new DefineColorNode(
               $3, ENV(colorNodeStack), ENV(theCounter).endCount()
            )
         );

         ENV(theCounter).increment();
      }
   |   DEFINE
      '{'
         { ENV(theCounter).beginCount(); }
         color_entries
      '}'
      {
         ENV(colorNodeStack).push(
            new DefineColorNode(
               ENV(colorNodeStack), ENV(theCounter).endCount()
            )
         );

         ENV(theCounter).increment();
      }
   ;

color_entries: color_entry
   |           color_entries color_entry
   ;

color_entry: REPEAT '(' INTEGER ')' 
            '{'
               { ENV(theCounter).beginCount(); }
               color_entries
            '}'
            {
               ENV(colorNodeStack).push(
                  new RepeatColorNode(
                     $3, ENV(colorNodeStack), ENV(theCounter).endCount()
                  )
               );

               ENV(theCounter).increment();
            }
   |         GRADIENT '(' INTEGER ')'
            '{'
               { ENV(insideGradient) = 1; ENV(theCounter).beginCount(); }
               gradient
               { ENV(insideGradient) = 0; }
            '}'
            {
               strstream temp;

               GradientColorNode *nodePtr =
                  GradientColorNode::create(
                     $3, ENV(colorStack), ENV(theCounter).endCount(), temp
                  );

               if(nodePtr)
               {
                  ENV(colorNodeStack).push(nodePtr);
                  ENV(theCounter).increment();
               }
               else
               {
                  handleError(temp.str(), voidPointer);
                  return(-1);
               }
            }
   |         single_color_entry
   ;

gradient: single_color_entry single_color_entries
   ;

single_color_entries: single_color_entry
   |                  single_color_entries single_color_entry
   ;

single_color_entry: '[' INTEGER INTEGER INTEGER ']'
                     {
                        if(ENV(insideGradient))
                        {
                           int *values = new int[3];
                           values[0]   = $2;
                           values[1]   = $3;
                           values[2]   = $4;

                           ENV(colorStack).push(values);
                        }
                        else
                        {
                           ENV(colorNodeStack).push(
                              new SingleColorColorNode(
                                 $2, $3, $4
                              )
                           );
                        }

                        ENV(theCounter).increment();
                     }
   ;

zero_or_more_statements: /* empty */
   |                     statements;

statement_body: '{' zero_or_more_statements '}'
   ;

formula: FORMULA
         '{'
            {
               ENV(compoundStack).push(new CompoundStatement());
               ENV(breakContinueAllowed).push(FALSE);
            }

         zero_or_more_statements

            {
               const CompoundStatement *ptr = ENV(compoundStack).pop();
               ENV(fractalPtr)->setBeforeStatement(ptr);
            }

         WHILE
         '('
            boolean_expression
            {
               ENV(fractalPtr)->setCondition(ENV(booleanNodeStack).pop());
            }
         ')'
            {
               ENV(compoundStack).push(new CompoundStatement());
               ENV(breakContinueAllowed).push(TRUE);
            }
          statement_body
            {
               const CompoundStatement *ptr = ENV(compoundStack).pop();
               ENV(fractalPtr)->setStatement(ptr);
               ENV(breakContinueAllowed).pop();
               ENV(compoundStack).push(new CompoundStatement());
            }

         zero_or_more_statements
            {
               const CompoundStatement *ptr = ENV(compoundStack).pop();
               ENV(fractalPtr)->setAfterStatement(ptr);
               ENV(breakContinueAllowed).pop();
            }
         '}'
   ;

statements: statement
   |        statements statement
   ;

simple_statement: complex_assignment_statement
                  {
                     const Statement *s        = ENV(currentComplexAssign);
                     ENV(currentComplexAssign) = NULL;

                     ENV(compoundStack).top()->add(s);
                  }
   |              double_assignment_statement
                  {
                     const Statement *s       = ENV(currentDoubleAssign);
                     ENV(currentDoubleAssign) = NULL;

                     ENV(compoundStack).top()->add(s);
                  }
   |              set_color_statement
   |              null_statement
   |              print_statement
   |              break_statement
   |              continue_statement
   ;

statement: simple_statement ';'
   |       conditional_statement
   |       for_statement
   |       switch_statement
   ;

complex_assignment_statement: IDENTIFIER EQUALS complex_expression
   {
      dcomplex *ptr =
         ENV(fractalPtr)->getComplexVariables().assign($1);

      ComplexAssignmentStatement *statement =
         new ComplexAssignmentStatement(
            ptr, $1, ENV(complexNodeStack).pop()
      );

      ENV(currentComplexAssign) = statement;

      free($1);
   }
   | IDENTIFIER EQUALS double_expression
   {
      doubleToComplex(voidPointer);

      dcomplex *ptr =
         ENV(fractalPtr)->getComplexVariables().assign($1);

      ComplexAssignmentStatement *statement =
         new ComplexAssignmentStatement(
            ptr, $1, ENV(complexNodeStack).top()
      );

      ENV(currentComplexAssign) = statement;

      free($1);
   }
   |
   IDENTIFIER PLUS_EQUALS complex_expression
   {
      int status = handleComplexOperatorAssign(
         voidPointer, $1, "+=", BinaryComplexNode::_PLUS_
      );

      if(status == -1)
         return(-1);
   }
   |
   IDENTIFIER PLUS_EQUALS double_expression
   {
      doubleToComplex(voidPointer);

      int status = handleComplexOperatorAssign(
         voidPointer, $1, "+=", BinaryComplexNode::_PLUS_
      );

      if(status == -1)
         return(-1);
   }
   |
   IDENTIFIER MINUS_EQUALS complex_expression
   {
      int status = handleComplexOperatorAssign(
         voidPointer, $1, "-=", BinaryComplexNode::_MINUS_
      );

      if(status == -1)
         return(-1);
   }
   |
   IDENTIFIER MINUS_EQUALS double_expression
   {
      doubleToComplex(voidPointer);

      int status = handleComplexOperatorAssign(
         voidPointer, $1, "-=", BinaryComplexNode::_MINUS_
      );

      if(status == -1)
         return(-1);
   }
   |
   IDENTIFIER TIMES_EQUALS complex_expression
   {
      int status = handleComplexOperatorAssign(
         voidPointer, $1, "*=", BinaryComplexNode::_MULTIPLY_
      );

      if(status == -1)
         return(-1);
   }
   |
   IDENTIFIER TIMES_EQUALS double_expression
   {
      doubleToComplex(voidPointer);

      int status = handleComplexOperatorAssign(
         voidPointer, $1, "*=", BinaryComplexNode::_MULTIPLY_
      );

      if(status == -1)
         return(-1);
   }
   |
   IDENTIFIER DIVIDE_EQUALS complex_expression
   {
      int status = handleComplexOperatorAssign(
         voidPointer, $1, "/=", BinaryComplexNode::_DIVIDE_
      );

      if(status == -1)
         return(-1);
   }
   |
   IDENTIFIER DIVIDE_EQUALS double_expression
   {
      doubleToComplex(voidPointer);

      int status = handleComplexOperatorAssign(
         voidPointer, $1, "/=", BinaryComplexNode::_DIVIDE_
      );

      if(status == -1)
         return(-1);
   }
   |
   IDENTIFIER PLUS_PLUS
   {
      ENV(complexNodeStack).push(
         new NumericComplexNode(new ConstantDoubleNode(1.0))
      );

      int status = handleComplexOperatorAssign(
         voidPointer, $1, "++", BinaryComplexNode::_PLUS_, TRUE
      );

      if(status == -1)
         return(-1);
   }
   |
   PLUS_PLUS IDENTIFIER
   {
      ENV(complexNodeStack).push(
         new NumericComplexNode(new ConstantDoubleNode(1.0))
      );

      int status = handleComplexOperatorAssign(
         voidPointer, $2, "++", BinaryComplexNode::_PLUS_
      );

      if(status == -1)
         return(-1);
   }
   |
   IDENTIFIER MINUS_MINUS
   {
      ENV(complexNodeStack).push(
         new NumericComplexNode(new ConstantDoubleNode(1.0))
      );

      int status = handleComplexOperatorAssign(
         voidPointer, $1, "--", BinaryComplexNode::_MINUS_, TRUE
      );

      if(status == -1)
         return(-1);
   }
   |
   MINUS_MINUS IDENTIFIER
   {
      ENV(complexNodeStack).push(
         new NumericComplexNode(new ConstantDoubleNode(1.0))
      );

      int status = handleComplexOperatorAssign(
         voidPointer, $2, "--", BinaryComplexNode::_MINUS_
      );

      if(status == -1)
         return(-1);
   }
   ;

double_assignment_statement: '$' IDENTIFIER EQUALS double_expression
   {
      double *ptr = ENV(fractalPtr)->getDoubleVariables().assign($2);
      const DoubleNode *node = ENV(doubleNodeStack).top();

      DoubleAssignmentStatement *statement =
         new DoubleAssignmentStatement(ptr, $2, node);

      ENV(currentDoubleAssign) = statement;

      free($2);
   }
  |
   '$' IDENTIFIER PLUS_EQUALS double_expression
   {
      int status = handleDoubleOperatorAssign(
         voidPointer, $2, "+=", BinaryDoubleNode::_PLUS_
      );

      if(status == -1)
         return(-1);
   }
   |
   '$' IDENTIFIER MINUS_EQUALS double_expression
   {
      int status = handleDoubleOperatorAssign(
         voidPointer, $2, "-=", BinaryDoubleNode::_MINUS_
      );

      if(status == -1)
         return(-1);
   }
   |
   '$' IDENTIFIER TIMES_EQUALS double_expression
   {
      int status = handleDoubleOperatorAssign(
         voidPointer, $2, "*=", BinaryDoubleNode::_MULTIPLY_
      );

      if(status == -1)
         return(-1);
   }
   |
   '$' IDENTIFIER DIVIDE_EQUALS double_expression
   {
      int status = handleDoubleOperatorAssign(
         voidPointer, $2, "/=", BinaryDoubleNode::_DIVIDE_
      );

      if(status == -1)
         return(-1);
   }
   |
   '$' IDENTIFIER PLUS_PLUS
   {
      ENV(doubleNodeStack).push(new ConstantDoubleNode(1.0));

      int status = handleDoubleOperatorAssign(
         voidPointer, $2, "++", BinaryDoubleNode::_PLUS_, TRUE
      );

      if(status == -1)
         return(-1);
   }
   |
   PLUS_PLUS '$' IDENTIFIER
   {
      ENV(doubleNodeStack).push(new ConstantDoubleNode(1.0));

      int status = handleDoubleOperatorAssign(
         voidPointer, $3, "++", BinaryDoubleNode::_PLUS_
      );

      if(status == -1)
         return(-1);
   }
   |
   '$' IDENTIFIER MINUS_MINUS
   {
      ENV(doubleNodeStack).push(new ConstantDoubleNode(1.0));

      int status = handleDoubleOperatorAssign(
         voidPointer, $2, "--", BinaryDoubleNode::_MINUS_, TRUE
      );

      if(status == -1)
         return(-1);
   }
   |
   MINUS_MINUS '$' IDENTIFIER
   {
      ENV(doubleNodeStack).push(new ConstantDoubleNode(1.0));

      int status = handleDoubleOperatorAssign(
         voidPointer, $3, "--", BinaryDoubleNode::_MINUS_
      );

      if(status == -1)
         return(-1);
   }
   ;

set_color_statement: indexed_set_color_statement
   |                 rgb_set_color_statement
   |                 hsv_set_color_statement
   ;

indexed_set_color_statement: SET_COLOR '(' double_expression ')'
   {
      const DoubleNode *index = ENV(doubleNodeStack).pop();
      Statement *s = new IndexedSetColorStatement(index, *ENV(fractalPtr));

      ENV(compoundStack).top()->add(s);
   }
   ;

rgb_set_color_statement: SET_COLOR
                            '('
                               double_expression ','
                               double_expression ','
                               double_expression
                            ')'
   {
      const DoubleNode *blue   = ENV(doubleNodeStack).pop();
      const DoubleNode *green  = ENV(doubleNodeStack).pop();
      const DoubleNode *red    = ENV(doubleNodeStack).pop();

      Statement *s = new RGBSetColorStatement(
         red, green, blue, *ENV(fractalPtr)
      );

      ENV(compoundStack).top()->add(s);
   }
   ;

hsv_set_color_statement: SET_COLOR_HSV
                            '('
                               double_expression ','
                               double_expression ','
                               double_expression
                            ')'
   {
      const DoubleNode *v = ENV(doubleNodeStack).pop();
      const DoubleNode *s = ENV(doubleNodeStack).pop();
      const DoubleNode *h = ENV(doubleNodeStack).pop();

      ENV(compoundStack).top()->add(
         new HSVSetColorStatement(h, s, v, *ENV(fractalPtr))
      );
   }
   ;

null_statement: TOKEN_NULL
   {
      ENV(compoundStack).top()->add(new NullStatement());
   }
   ;

break_statement: BREAK
   {
      if(ENV(breakContinueAllowed).top())
      {
         ENV(compoundStack).top()->add(new BreakStatement());
      }
      else
      {
         handleError(
            "Illegal use of: 'break' statement", voidPointer
         );

         return(-1);
      }
   }
   ;

continue_statement: CONTINUE
   {
      if(ENV(breakContinueAllowed).top())
      {
         ENV(compoundStack).top()->add(new ContinueStatement());
      }
      else
      {
         handleError(
            "Illegal use of: 'continue' statement", voidPointer
         );

         return(-1);
      }
   }
   ;

map_color:  MAP_COLOR '('
            {
               ENV(mapStack).push(new MapColorFunctionDoubleNode());
            }
            map_color_arg     ':'
            map_color_args    ':'
            map_color_args ')'
            {
               // Make sure we got at least 4 arguments and that the
               // total number of arguments is even

               const MapColorFunctionDoubleNode *node = ENV(mapStack).pop();

               int size = node->size();

               if(size < 4 || size % 2 != 0)
               {
                  strstream temp;

                  temp << "map_color: invalid number of " <<
                          "arguments (" << size << ")\n"  <<
                          "expected an even number of arguments >= 4" << '\0';

                  handleError(temp.str(), voidPointer);
                  return(-1);
               }

               // Ok, all's cool - push our node onto doubleNodeStack

               ENV(doubleNodeStack).push(node);
            }
   ;

map_color_args: map_color_args ',' map_color_arg
   |            map_color_arg
   ;

map_color_arg: double_expression
               {
                  ENV(mapStack).top()->addArgument(
                     ENV(doubleNodeStack).pop()
                  );
               }
   ;

print_function: PRINT
               {
                  ENV(printStatement) = new PrintStatement();
               }
|               PRINTLN
               {
                  ENV(printStatement) = new PrintStatement(1);
               }
   ;

print_statement:  print_function '('
                  print_statement_args ')'
                  {
                     const Statement *statement = ENV(printStatement);
                     ENV(compoundStack).top()->add(statement);
                     ENV(printStatement) = NULL;
                  }
   ;

print_statement_args:  print_dest ',' print_args
   |                   print_args
   ;

print_args: print_args DOT print_arg
   |        print_arg
   ;

print_arg: double_expression
   {
      const DoubleNode *node = ENV(doubleNodeStack).pop();

      ENV(printStatement)->addPrintArg(
         new DoubleNodePrintArg(node)
      );
   }
   | complex_expression
   {
      const ComplexNode *node = ENV(complexNodeStack).pop();

      ENV(printStatement)->addPrintArg(
         new ComplexNodePrintArg(node)
      );
   }
   | LITERAL
   {
      ENV(printStatement)->addPrintArg(new LiteralPrintArg($1));
      free($1);
   }
   ;

print_dest: LITERAL
            {
               ENV(printStatement)->setDestination($1);
               free($1);
            }
   ;

if_statement: IF '(' boolean_expression ')'
               {
                  ENV(compoundStack).push(new CompoundStatement());

                  ENV(conditionalCounter).beginCount();
                  ENV(conditionalCounter).increment();
               }
               statement_body
   ;

else_statement: ELSE
                  {
                     ENV(compoundStack).push(new CompoundStatement());
                  }
                statement_body
   ;

elseif_statement: ELSEIF '(' boolean_expression ')'
                  {
                     ENV(compoundStack).push(
                        new CompoundStatement()
                     );

                     ENV(conditionalCounter).increment();
                  }
                  statement_body
   ;

start_conditional_statement: if_statement
   |                         if_statement elseif_statements
   ;

elseif_statements: elseif_statements elseif_statement
   |               elseif_statement
   ;

conditional_statement: start_conditional_statement
                        {
                           int size = ENV(conditionalCounter).endCount();

                           const Statement **statements =
                              new const Statement *[size];

                           const BooleanNode **conditions =
                              new const BooleanNode *[size];

                           for(int i = 0, j = size - 1; i < size; i ++, j --)
                           {
                              statements[j] = ENV(compoundStack).pop();
                              conditions[j] = ENV(booleanNodeStack).pop();
                           }

                           ConditionalStatement *s =
                              new ConditionalStatement(
                                 statements, conditions, size
                              );

                           ENV(compoundStack).top()->add(s);
                       }
|                      start_conditional_statement else_statement
                       {
                           int size = ENV(conditionalCounter).endCount();

                           // The first statement on our stack will be
                           // our else statement - grab it now

                           const Statement *elseStatement =
                              ENV(compoundStack).pop();

                           const Statement **statements =
                              new const Statement *[size];

                           const BooleanNode **conditions =
                              new const BooleanNode *[size];

                           for(int i = 0, j = size - 1; i < size; i ++, j --)
                           {
                              statements[j] = ENV(compoundStack).pop();
                              conditions[j] = ENV(booleanNodeStack).pop();
                           }

                           ConditionalStatement *s =
                              new ConditionalStatement(
                                 statements, conditions, size, elseStatement
                              );

                           ENV(compoundStack).top()->add(s);
                       }
   ;

for_statement_list: /* empty */
   |                simple_statement
   |                for_statement_list ',' simple_statement
   ;

for_statement: FOR '('
               {
                  ENV(compoundStack).push(new CompoundStatement());
               }
               for_statement_list ';'
               boolean_expression ';'
               {
                  ENV(compoundStack).push(new CompoundStatement());
               }
               for_statement_list
               ')'
               {
                  ENV(compoundStack).push(new CompoundStatement());
                  ENV(breakContinueAllowed).push(TRUE);
               }
               statement_body
               {
                  const CompoundStatement *bodyStatement =
                     ENV(compoundStack).pop();

                  const CompoundStatement *incrementStatement =
                     ENV(compoundStack).pop();

                  const CompoundStatement *initializeStatement =
                     ENV(compoundStack).pop();

                  const BooleanNode *condition =
                     ENV(booleanNodeStack).pop();

                  Statement *s = new ForStatement(
                     initializeStatement,
                     condition,
                     incrementStatement,
                     bodyStatement
                  );

                  ENV(compoundStack).top()->add(s);
                  ENV(breakContinueAllowed).pop();
               }
   ;

switch_statement: SWITCH '(' double_expression ')'
                  {
                     SwitchStatement *sw = new SwitchStatement(
                        ENV(doubleNodeStack).pop()
                     );

                     ENV(switchStack).push(sw);
                     ENV(breakContinueAllowed).push(FALSE);
                  }
                  '{'
                      switch_body
                  '}'
                  {
                     const SwitchStatement *sw = ENV(switchStack).pop();
                     ENV(compoundStack).top()->add(sw);
                     ENV(breakContinueAllowed).pop();
                  }
   ;

switch_body: /* empty */
   |         case_statements
   |         case_statements default_statement
   |         default_statement
   ;

case_statements: case_statement
   |             case_statements case_statement
   ;

case_statement:
               {
                  ENV(switchStack).top()->caseBegin();
               }
               case_entries
               {
                  ENV(compoundStack).push(new CompoundStatement());
               }
               case_body
               {
                  ENV(switchStack).top()->caseAddStatements(
                     ENV(compoundStack).pop()
                  );
               }
   ;

case_entries: case_entries case_entry
   |          case_entry
   ;

case_entry: CASE double_expression ':'
            {
               ENV(switchStack).top()->caseAddNode(
                  ENV(doubleNodeStack).pop()
               );
            }
   ;

default_statement:
               {
                  ENV(switchStack).top()->caseBegin();
               }
               default_entry
               {
                  ENV(compoundStack).push(new CompoundStatement());
               }
               case_body
               {
                  ENV(switchStack).top()->caseAddStatements(
                     ENV(compoundStack).pop()
                  );
               }
   ;

default_entry: DEFAULT ':'
   ;

case_body: '{' zero_or_more_statements '}'
   |       '{' zero_or_more_statements '}' BREAK ';'
            {
               ENV(switchStack).top()->caseSetBreak();
            }
   ;

boolean_expression: TOKEN_TRUE
                     {
                        ENV(booleanNodeStack).push(new TrueBooleanNode());
                     }
|
                    TOKEN_FALSE
                     {
                        ENV(booleanNodeStack).push(new FalseBooleanNode());
                     }
|
                    boolean_expression AND boolean_expression
                     {
                        const BooleanNode *right = ENV(booleanNodeStack).pop();
                        const BooleanNode *left  = ENV(booleanNodeStack).pop();

                        ENV(booleanNodeStack).push(
                           new ConditionalBooleanNode(
                              left,
                              ConditionalBooleanNode::_AND_,
                              right
                           )
                        );
                     }
   |                boolean_expression OR  boolean_expression
                     {
                        const BooleanNode *right = ENV(booleanNodeStack).pop();
                        const BooleanNode *left  = ENV(booleanNodeStack).pop();

                        ENV(booleanNodeStack).push(
                           new ConditionalBooleanNode(
                              left,
                              ConditionalBooleanNode::_OR_,
                              right
                           )
                        );
                     }
   |                '(' boolean_expression ')'
                     {
                        /* -- Nothing to do -- */
                     }
   |                double_expression LT double_expression
                     {
                        const DoubleNode *right = ENV(doubleNodeStack).pop();
                        const DoubleNode *left  = ENV(doubleNodeStack).pop();

                        ENV(booleanNodeStack).push(
                           new RelationalBooleanNode(
                              left,
                              RelationalBooleanNode::_LT_,
                              right
                           )
                        );
                     }
   |                double_expression LE double_expression
                     {
                        const DoubleNode *right = ENV(doubleNodeStack).pop();
                        const DoubleNode *left  = ENV(doubleNodeStack).pop();

                        ENV(booleanNodeStack).push(
                           new RelationalBooleanNode(
                              left,
                              RelationalBooleanNode::_LE_,
                              right
                           )
                        );
                     }
   |                double_expression GT double_expression
                     {
                        const DoubleNode *right = ENV(doubleNodeStack).pop();
                        const DoubleNode *left  = ENV(doubleNodeStack).pop();

                        ENV(booleanNodeStack).push(
                           new RelationalBooleanNode(
                              left,
                              RelationalBooleanNode::_GT_,
                              right
                           )
                        );
                     }
   |                double_expression GE double_expression
                     {
                        const DoubleNode *right = ENV(doubleNodeStack).pop();
                        const DoubleNode *left  = ENV(doubleNodeStack).pop();

                        ENV(booleanNodeStack).push(
                           new RelationalBooleanNode(
                              left,
                              RelationalBooleanNode::_GE_,
                              right
                           )
                        );
                     }
   |                double_expression EQ double_expression
                     {
                        const DoubleNode *right = ENV(doubleNodeStack).pop();
                        const DoubleNode *left  = ENV(doubleNodeStack).pop();

                        ENV(booleanNodeStack).push(
                           new RelationalBooleanNode(
                              left,
                              RelationalBooleanNode::_EQ_,
                              right
                           )
                        );
                     }
   |                double_expression NE double_expression
                     {
                        const DoubleNode *right = ENV(doubleNodeStack).pop();
                        const DoubleNode *left  = ENV(doubleNodeStack).pop();

                        ENV(booleanNodeStack).push(
                           new RelationalBooleanNode(
                              left,
                              RelationalBooleanNode::_NE_,
                              right
                           )
                        );
                     }
   |                inside_boolean_expression
   |                '!' boolean_expression
                    {
                        ENV(booleanNodeStack).push(
                           new NotBooleanNode(ENV(booleanNodeStack).pop())
                        );
                    }
   ;

double_ternary_expression:  boolean_expression '?'
                            double_expression ':' double_expression
   |                        '{' double_ternary_expression '}'
   ;

double_expression: double_expression '+' double_expression
            {
               const DoubleNode *right = ENV(doubleNodeStack).pop();
               const DoubleNode *left  = ENV(doubleNodeStack).pop();

               ENV(doubleNodeStack).push(
                  new BinaryDoubleNode(
                     left,
                     BinaryDoubleNode::_PLUS_,
                     right
                  )
               );
            }
   |        double_expression '-' double_expression
            {
               const DoubleNode *right = ENV(doubleNodeStack).pop();
               const DoubleNode *left  = ENV(doubleNodeStack).pop();

               ENV(doubleNodeStack).push(
                  new BinaryDoubleNode(
                     left,
                     BinaryDoubleNode::_MINUS_,
                     right
                  )
               );
            }
   |        double_expression '*' double_expression
            {
               const DoubleNode *right = ENV(doubleNodeStack).pop();
               const DoubleNode *left  = ENV(doubleNodeStack).pop();

               ENV(doubleNodeStack).push(
                  new BinaryDoubleNode(
                     left,
                     BinaryDoubleNode::_MULTIPLY_,
                     right
                  )
               );
            }
   |        double_expression '/' double_expression
            {
               const DoubleNode *right = ENV(doubleNodeStack).pop();
               const DoubleNode *left  = ENV(doubleNodeStack).pop();

               ENV(doubleNodeStack).push(
                  new BinaryDoubleNode(
                     left,
                     BinaryDoubleNode::_DIVIDE_,
                     right
                  )
               );
            }
   |        double_expression '%' double_expression
            {
               // We just map this to 'mod', so acquire that function
               // pointer

               const char *mod = TwoArgFunctionDoubleNode::MOD;

               TwoArgFunctionDoublePointer fPtr =
                  TwoArgFunctionDoubleNode::lookup(mod);

               const DoubleNode *second = ENV(doubleNodeStack).pop();
               const DoubleNode *first  = ENV(doubleNodeStack).pop();

               ENV(doubleNodeStack).push(
                  new TwoArgFunctionDoubleNode(
                     mod, fPtr, first, second
                  )
               );
            }
   |        double_expression '^' double_expression
            {
               // We just map this to 'pow', so acquire that function
               // pointer

               const char *pow = TwoArgFunctionDoubleNode::POW;

               TwoArgFunctionDoublePointer fPtr =
                  TwoArgFunctionDoubleNode::lookup(pow);

               const DoubleNode *second = ENV(doubleNodeStack).pop();
               const DoubleNode *first  = ENV(doubleNodeStack).pop();

               ENV(doubleNodeStack).push(
                  new TwoArgFunctionDoubleNode(
                     pow, fPtr, first, second
                  )
               );
            }
   |        '-' double_expression %prec UMINUS
            {
               ENV(doubleNodeStack).push(
                  new UnaryDoubleNode(
                     ENV(doubleNodeStack).pop()
                  )
               );
            }
   |        '(' double_expression ')'
            {
               /* -- Nothing to do -- */
            }
   |        number
            {
               ENV(doubleNodeStack).push(
                  new ConstantDoubleNode($1)
               );
            }
   |        '$' IDENTIFIER
            {
               // Ensure this variable has been defined

               const double *ptr =
                  ENV(fractalPtr)->getDoubleVariables().lookup($2);

               if(ptr == NULL)
               {
                  strstream temp;

                  temp << "Undefined double variable: [$" << $2 << "]" << '\0';

                  handleError(temp.str(), voidPointer);

                  free($2);
                  return(-1);
               }

               ENV(doubleNodeStack).push(
                  new VariableDoubleNode($2, ptr)
               );

               free($2);
            }
   |        complex_attribute '(' complex_expression ')'
            {
               AttributeDoubleNode *ptr =
                  AttributeDoubleNode::create(
                     $1, ENV(complexNodeStack).pop()
                  );

               if(!ptr)
               {
                  /**********************************************
                   * This really shouldn't be possible unless
                   * the set of tokens (ATTRIBUTE_REAL,
                   * ATTRIBUTE_IMAGINARY, etc.) and the set of
                   * attribute functions in the source file
                   * AttributeDoubleNode.cpp have gotten out
                   * of sync
                   **********************************************/

                  strstream temp;

                  temp << "No matching attribute: " << $1 <<
                     "(complex_expression)" << '\0';

                  handleError(temp.str(), voidPointer);

                  free($1);
                  return(-1);
               }

               free($1);

               ENV(doubleNodeStack).push(ptr);
            }
   |        map_color
            {
               /* -- Nothing to do -- */
            }
   |        IDENTIFIER '(' double_expression ')'
            {
               /* -- Attempt to look up the function requested -- */

               int error = FALSE;

               OneArgFunctionDoublePointer fPtr =
                  OneArgFunctionDoubleNode::lookup($1);

               if(!fPtr)
               {
                  // This could also be a GetEntryDoubleNode so
                  // try that as well

                  GetEntryDoubleNode::WhichEntry entry =
                     GetEntryDoubleNode::lookup($1);

                  if(entry == GetEntryDoubleNode::INVALID)
                  {
                     strstream temp;

                     temp << "No matching function: " << $1 <<
                        "(double)" << '\0';

                     handleError(temp.str(), voidPointer);
                     error = TRUE;
                  }
                  else
                  {
                     ENV(doubleNodeStack).push(
                        new GetEntryDoubleNode(
                           $1,
                           entry,
                           ENV(doubleNodeStack).pop(),
                           *ENV(fractalPtr)
                        )
                     );
                  }
               }
               else
               {
                  ENV(doubleNodeStack).push(
                     new OneArgFunctionDoubleNode(
                        $1, fPtr, ENV(doubleNodeStack).pop()
                     )
                  );
               }

               free($1);

               if(error)
                  return(-1);
            }
   |        IDENTIFIER '(' double_expression ',' double_expression ')'
            {
               /* -- Attempt to look up the function requested -- */

               TwoArgFunctionDoublePointer fPtr =
                  TwoArgFunctionDoubleNode::lookup($1);

               if(!fPtr)
               {
                  strstream temp;

                  temp << "No matching function: " << $1 <<
                     "(double, double)" << '\0';
                
                  handleError(temp.str(), voidPointer);

                  free($1);
                  return(-1);
               }

               const DoubleNode *second = ENV(doubleNodeStack).pop();
               const DoubleNode *first  = ENV(doubleNodeStack).pop();

               ENV(doubleNodeStack).push(
                  new TwoArgFunctionDoubleNode(
                     $1, fPtr, first, second
                  )
               );

               free($1);
            }
   |        IDENTIFIER '('
               double_expression ',' double_expression ',' double_expression
            ')'
            {
               /* -- Attempt to look up the function requested -- */

               ThreeArgFunctionDoublePointer fPtr =
                  ThreeArgFunctionDoubleNode::lookup($1);

               if(!fPtr)
               {
                  strstream temp;

                  temp << "No matching function: " << $1 <<
                     "(double, double, double)" << '\0';
                
                  handleError(temp.str(), voidPointer);

                  free($1);
                  return(-1);
               }

               const DoubleNode *third  = ENV(doubleNodeStack).pop();
               const DoubleNode *second = ENV(doubleNodeStack).pop();
               const DoubleNode *first  = ENV(doubleNodeStack).pop();

               ENV(doubleNodeStack).push(
                  new ThreeArgFunctionDoubleNode(
                     $1, fPtr, first, second, third
                  )
               );

               free($1);
            }
   |        TEXT_INTERSECT '('
               LITERAL            ','
               complex_expression ','
               double_expression  ','
               double_expression  ','
               complex_expression
            ')'
            {
               const ComplexNode *point  = ENV(complexNodeStack).pop();
               const DoubleNode  *thick  = ENV(doubleNodeStack).pop();
               const DoubleNode  *height = ENV(doubleNodeStack).pop();
               const ComplexNode *origin = ENV(complexNodeStack).pop();

               ENV(doubleNodeStack).push(
                  new TextIntersectNode(
                     $3, origin, height, thick, point
                  )
               );

               free($3);
            }
   |        double_ternary_expression
            {
               const BooleanNode *test   = ENV(booleanNodeStack).pop();
               const DoubleNode  *second = ENV(doubleNodeStack).pop();
               const DoubleNode  *first  = ENV(doubleNodeStack).pop();

               ENV(doubleNodeStack).push(
                  new TernaryDoubleNode(test, first, second)
               );
            }
   |        double_assignment_statement
            {
               const DoubleAssignmentStatement *s =
                  ENV(currentDoubleAssign);

               ENV(currentDoubleAssign) = NULL;

               ENV(doubleNodeStack).push(new AssignmentDoubleNode(s));
            }
   ;

complex_attribute: ATTRIBUTE_REAL
   |               ATTRIBUTE_IMAGINARY
   |               ATTRIBUTE_DEGREES
   |               ATTRIBUTE_OLD_DEGREES
   |               ATTRIBUTE_RADIANS
   |               ATTRIBUTE_OLD_RADIANS
   |               ATTRIBUTE_SUM_OF_SQUARES
   |               ATTRIBUTE_MAGNITUDE
   ;

mixed_add_expression: complex_expression '+' double_expression
                     { doubleToComplex(voidPointer);    }
   |                  double_expression  '+' complex_expression
                     { doubleToComplex(voidPointer, 1); }
   ;

complex_add_expression: complex_expression '+' complex_expression
   |                    mixed_add_expression
   ;

mixed_subtract_expression: complex_expression '-' double_expression
                           { doubleToComplex(voidPointer);    }
   |                       double_expression  '-' complex_expression
                           { doubleToComplex(voidPointer, 1); }
   ;

complex_subtract_expression: complex_expression '-' complex_expression
   |                         mixed_subtract_expression
   ;

mixed_multiply_expression: complex_expression '*' double_expression
                           { doubleToComplex(voidPointer);    }
   |                       double_expression  '*' complex_expression
                           { doubleToComplex(voidPointer, 1); }
   ;

complex_multiply_expression: complex_expression '*' complex_expression
   |                         mixed_multiply_expression
   ;

mixed_divide_expression: complex_expression '/' double_expression
                           { doubleToComplex(voidPointer);    }
   |                     double_expression  '/' complex_expression
                           { doubleToComplex(voidPointer, 1); }
   ;

complex_divide_expression: complex_expression '/' complex_expression
   |                       mixed_divide_expression
   ;

mixed_power_expression: complex_expression '^' double_expression
                           { doubleToComplex(voidPointer);    }
   |                    double_expression '^' complex_expression
                           { doubleToComplex(voidPointer, 1); }
   ;

complex_power_expression: complex_expression '^' complex_expression
   |                      mixed_power_expression
   ;

mixed_two_arg_expression: complex_expression ',' double_expression
                           { doubleToComplex(voidPointer);    }
   |                      double_expression  ',' complex_expression
                           { doubleToComplex(voidPointer, 1); }
   ;

complex_two_arg_expression: complex_expression ',' complex_expression
   |                        mixed_two_arg_expression
   ;

mixed_ternary_arguments:  complex_expression ':' double_expression
                           { doubleToComplex(voidPointer);    }
   |                      double_expression  ':' complex_expression
                           { doubleToComplex(voidPointer, 1); }
   ;

complex_ternary_arguments: complex_expression ':' complex_expression
   |                        mixed_ternary_arguments
   ;

complex_ternary_expression: boolean_expression '?' complex_ternary_arguments
   |                        '{' complex_ternary_expression '}'
   ;

complex_expression: complex_add_expression
            {
               const ComplexNode *right = ENV(complexNodeStack).pop();
               const ComplexNode *left  = ENV(complexNodeStack).pop();

               ENV(complexNodeStack).push(
                  new BinaryComplexNode(
                     left,
                     BinaryComplexNode::_PLUS_,
                     right
                  )
               );
            }
   |        complex_subtract_expression
            {
               const ComplexNode *right = ENV(complexNodeStack).pop();
               const ComplexNode *left  = ENV(complexNodeStack).pop();

               ENV(complexNodeStack).push(
                  new BinaryComplexNode(
                     left,
                     BinaryComplexNode::_MINUS_,
                     right
                  )
               );
            }
   |        complex_multiply_expression
            {
               const ComplexNode *right = ENV(complexNodeStack).pop();
               const ComplexNode *left  = ENV(complexNodeStack).pop();

               ENV(complexNodeStack).push(
                  new BinaryComplexNode(
                     left,
                     BinaryComplexNode::_MULTIPLY_,
                     right
                  )
               );
            }
   |        complex_divide_expression
            {
               const ComplexNode *right = ENV(complexNodeStack).pop();
               const ComplexNode *left  = ENV(complexNodeStack).pop();

               ENV(complexNodeStack).push(
                  new BinaryComplexNode(
                     left,
                     BinaryComplexNode::_DIVIDE_,
                     right
                  )
               );
            }
   |        complex_power_expression
            {
               // We just map this to 'pow', so acquire that function
               // pointer

               const char *pow = TwoArgFunctionDoubleNode::POW;

               TwoArgFunctionComplexPointer fPtr =
                  TwoArgFunctionComplexNode::lookup(pow);

               const ComplexNode *second = ENV(complexNodeStack).pop();
               const ComplexNode *first  = ENV(complexNodeStack).pop();

               ENV(complexNodeStack).push(
                  new TwoArgFunctionComplexNode(
                     pow, fPtr, first, second
                  )
               );
            }
   |        '-' complex_expression %prec UMINUS
            {
               ENV(complexNodeStack).push(
                  new UnaryComplexNode(
                     ENV(complexNodeStack).pop()
                  )
               );
            }
   |        '(' complex_expression ')'
            {
               /* -- Nothing to do -- */
            }
   |        COMPLEX '(' double_expression ')'
            {
               // Strictly speaking, we don't actually need COMPLEX
               // at all anymore now that we've revamped the automatic
               // double -> complex conversions but there's no harm
               // in supporting it for compatibility reasons

               doubleToComplex(voidPointer);
            }
   |        '[' double_expression ',' double_expression ']'
            {
               const DoubleNode *imaginaryPart = ENV(doubleNodeStack).pop();
               const DoubleNode *realPart      = ENV(doubleNodeStack).pop();

               ENV(complexNodeStack).push(
                  new NumericComplexNode(realPart, imaginaryPart)
               );
            }
   |        IDENTIFIER
            {
               /* -- Ensure this variable has been defined -- */

               dcomplex *ptr =
                  ENV(fractalPtr)->getComplexVariables().lookup($1);

               if(ptr == NULL)
               {
                  strstream temp;

                  temp << "Undefined variable: [" << $1 << "]" << '\0';

                  handleError(temp.str(), voidPointer);

                  free($1);
                  return(-1);
               }

               ENV(complexNodeStack).push(
                  new VariableComplexNode($1, ptr)
               );

               free($1);
            }
   |        IDENTIFIER '(' complex_expression ')'
            {
               /* -- Attempt to look up the function requested -- */

               OneArgFunctionComplexPointer fPtr =
                  OneArgFunctionComplexNode::lookup($1);

               if(!fPtr)
               {
                  strstream temp;

                  temp << "No matching function: " << $1
                     << "(complex)" << '\0';

                  handleError(temp.str(), voidPointer);

                  free($1);
                  return(-1);
               }

               ENV(complexNodeStack).push(
                  new OneArgFunctionComplexNode(
                     $1, fPtr, ENV(complexNodeStack).pop()
                  )
               );

               free($1);
            }
   |        IDENTIFIER '(' complex_two_arg_expression ')'
            {
               /* -- Attempt to look up the function requested -- */

               TwoArgFunctionComplexPointer fPtr =
                  TwoArgFunctionComplexNode::lookup($1);

               if(!fPtr)
               {
                  strstream temp;

                  temp << "No matching function: " << $1
                       << "(complex, complex)" << '\0';

                  handleError(temp.str(), voidPointer);
                  
                  free($1);
                  return(-1);
               }

               const ComplexNode *second = ENV(complexNodeStack).pop();
               const ComplexNode *first  = ENV(complexNodeStack).pop();

               ENV(complexNodeStack).push(
                  new TwoArgFunctionComplexNode(
                     $1, fPtr, first, second
                  )
               );

               free($1);
            }
   |        IDENTIFIER '('
               complex_two_arg_expression ',' double_expression
            ')'
            {
               /* -- Attempt to look up the function requested -- */

               MixedThreeArgFunctionComplexPointer fPtr =
                  MixedThreeArgFunctionComplexNode::lookup($1);

               if(!fPtr)
               {
                  strstream temp;

                  temp << "No matching function: " << $1
                       << "(complex, complex, double)" << '\0';

                  handleError(temp.str(), voidPointer);
                  
                  free($1);
                  return(-1);
               }

               const ComplexNode *second = ENV(complexNodeStack).pop();
               const ComplexNode *first  = ENV(complexNodeStack).pop();
               const DoubleNode *third   = ENV(doubleNodeStack).pop();

               ENV(complexNodeStack).push(
                  new MixedThreeArgFunctionComplexNode(
                     $1, fPtr, first, second, third
                  )
               );

               free($1);
            }
   |        complex_ternary_expression
            {
               const BooleanNode *test    = ENV(booleanNodeStack).pop();
               const ComplexNode  *second = ENV(complexNodeStack).pop();
               const ComplexNode  *first  = ENV(complexNodeStack).pop();

               ENV(complexNodeStack).push(
                  new TernaryComplexNode(test, first, second)
               );
            }
   |        complex_assignment_statement
            {
               const ComplexAssignmentStatement *s =
                  ENV(currentComplexAssign);

               ENV(currentComplexAssign) = NULL;

               ENV(complexNodeStack).push(new AssignmentComplexNode(s));
            }
   ;

number: REAL
         {
            $$ = $1;
         }
   |    INTEGER
         {
            $$ = (double)$1;
         }
   ;

numeric_expression: numeric_expression '+' numeric_expression
                     {
                        $$ = $1 + $3;
                     }
   |                numeric_expression '-' numeric_expression
                     {
                        $$ = $1 - $3;
                     }
   |                numeric_expression '*' numeric_expression
                     {
                        $$ = $1 * $3;
                     }
   |                numeric_expression '/' numeric_expression
                     {
                        $$ = $1 / $3;
                     }
   |                '-' numeric_expression %prec UMINUS
                     {
                        $$ = -$2;
                     }
   |                 '(' numeric_expression ')'
                     {
                        $$ = $2;
                     }
   |                 number
                     {
                        $$ = $1;
                     }
   ;

more_r_poly_points: more_r_poly_points ',' complex_expression
               {
                  ENV(pointCounter).increment();
               }
   |         complex_expression
               {
                  ENV(pointCounter).increment();
               }
   ;


region: R_CIRCLE '(' complex_expression ',' double_expression ')'
         {
            const ComplexNode *center = ENV(complexNodeStack).pop();
            const DoubleNode  *radius = ENV(doubleNodeStack).pop();

            ENV(regionNodeStack).push(new CircleRegionNode(center, radius));
         }
|       R_ELLIPSE '(' complex_expression ','
           double_expression ',' double_expression ')'
         {
            const ComplexNode *center  = ENV(complexNodeStack).pop();
            const DoubleNode  *yradius = ENV(doubleNodeStack).pop();
            const DoubleNode  *xradius = ENV(doubleNodeStack).pop();

            ENV(regionNodeStack).push(
               new EllipseRegionNode(center, xradius, yradius)
            );
         }
   |    R_RECT '(' complex_expression ',' complex_expression ')'
         {
            const ComplexNode *second = ENV(complexNodeStack).pop();
            const ComplexNode *first  = ENV(complexNodeStack).pop();

            ENV(regionNodeStack).push(new RectRegionNode(first, second));
         }
   |    R_POLY '(' complex_expression ',' complex_expression ','
         {
            ENV(pointCounter).beginCount();
            ENV(pointCounter).increment();
            ENV(pointCounter).increment();
         }
         more_r_poly_points
        ')'
         {
            int size = ENV(pointCounter).endCount();

            const ComplexNode **points =
               new const ComplexNode *[size];

            for(int i = 0, j = size - 1; i < size; i ++, j --)
               points[j] = ENV(complexNodeStack).pop();

            ENV(regionNodeStack).push(
               new PolyRegionNode(points, size)
            );
         }
   |    R_SPOLY '(' complex_expression ',' double_expression ','
                    double_expression  ',' double_expression ')'
         {
            const ComplexNode *center = ENV(complexNodeStack).pop();
            const DoubleNode  *angle  = ENV(doubleNodeStack).pop();
            const DoubleNode  *radius = ENV(doubleNodeStack).pop();
            const DoubleNode  *nSides = ENV(doubleNodeStack).pop();

            ENV(regionNodeStack).push(
               new SPolyRegionNode(center, nSides, radius, angle)
            );
         }
   |    R_CROSS '(' complex_expression ',' complex_expression ','
                    double_expression  ',' double_expression ')'
                 
         {
            const ComplexNode *p2     = ENV(complexNodeStack).pop();
            const ComplexNode *p1     = ENV(complexNodeStack).pop();
            const DoubleNode  *height = ENV(doubleNodeStack).pop();
            const DoubleNode  *width  = ENV(doubleNodeStack).pop();

            ENV(regionNodeStack).push(
               new CrossRegionNode(p1, p2, width, height)
            );
         }
   |    R_NOT '(' region ')'
         {
            const RegionNode *node = ENV(regionNodeStack).pop();

            ENV(regionNodeStack).push(new NotRegionNode(node));
         }
   |    R_AND '(' region ',' region ')'
         {
            const RegionNode *second = ENV(regionNodeStack).pop();
            const RegionNode *first  = ENV(regionNodeStack).pop();

            ENV(regionNodeStack).push(new AndRegionNode(first, second));
         }
   |    R_OR  '(' region ',' region ')'
   {
            const RegionNode *second = ENV(regionNodeStack).pop();
            const RegionNode *first  = ENV(regionNodeStack).pop();

            ENV(regionNodeStack).push(new OrRegionNode(first, second));
   }
   |    R_XOR '(' region ',' region ')'
   {
            const RegionNode *second = ENV(regionNodeStack).pop();
            const RegionNode *first  = ENV(regionNodeStack).pop();

            ENV(regionNodeStack).push(new XorRegionNode(first, second));
   }
   ;

inside_boolean_expression: INSIDE '(' complex_expression ',' region ')'
   {
      const ComplexNode *point = ENV(complexNodeStack).pop();
      const RegionNode *region = ENV(regionNodeStack).pop();

      ENV(booleanNodeStack).push(new InsideBooleanNode(point, region));
   }
   ;

%%

/*****************************************************************************
 * Local functions
 *****************************************************************************/

static int handleError(char *s, void *voidPointer)
{
   char currentLine[FractalLexer::MAX_LINE_LENGTH + 1];
   FractalLexer *env = FractalLexer::getInstance();

   // Record the line number where the error took place

   ENV(errorLineNumber) = env->getLineNumber();

   env->getCurrentLine(currentLine);

   ENV(errStream) << "line " << ENV(errorLineNumber) << ": " << s << "\n";
   ENV(errStream) << currentLine << "\n";

   int lineOffset = env->getLineOffset();

   for(int i = 0; i < lineOffset; i ++)
   {
      ENV(errStream) << ' ';
   }

   ENV(errStream) << '^' << '\0';

   return(0);
}

static Fractal *fractal_createFromText(
   const char *text,
   FractalParse &env,
   char **errorMessagePointer,
   int *errorLineNumberPointer
)
{
#if YYDEBUG != 0
   yydebug = 1;
#endif

   *errorMessagePointer    = 0;
   *errorLineNumberPointer = 0;

   FractalLexer::getInstance()->begin(text);

   int success = (yyparse(&env) == 0);

   FractalLexer::getInstance()->end();

   if(success)
   {
      // Uncomment the following for debugging output
      // cout << *env.fractalPtr << "\n";

      return(env.fractalPtr);
   }

   // Delete fractalPtr as it is very likely in some intermediate state

   delete env.fractalPtr;

   // Ensure any error messages make their way to the client

   copyFromStrStream(env.errStream, errorMessagePointer);

   // Also grab the line number of the error

   *errorLineNumberPointer = env.errorLineNumber;

   // Return null indicating an unsuccessful parse

   return(0);
}

static void copyFromStrStream(strstream &theStream, char **dest)
{
   char *ptr = theStream.str();
   *dest = new char[strlen(ptr) + 1];
   strcpy(*dest, ptr);
}

static int handleComplexOperatorAssign(
   void *voidPointer,
   char *id,
   char *token,
   BinaryComplexNode::Operator op,
   int isPostAssign
)
{
   dcomplex *ptr =
      ENV(fractalPtr)->getComplexVariables().lookup(id);

   if(ptr == NULL)
   {
      strstream temp;

      temp << "Undefined variable: [" << id << "] "
           << "used with " << token << '\0';

      handleError(temp.str(), voidPointer);

      free(id);
      return(-1);
   }

   const ComplexNode *rhs           = ENV(complexNodeStack).pop();
   const VariableComplexNode *value = new VariableComplexNode(id, ptr);

   const ComplexNode *node = new BinaryComplexNode(value, op, rhs);

   ComplexAssignmentStatement *statement =
      new ComplexAssignmentStatement(ptr, id, node, isPostAssign);

   ENV(currentComplexAssign) = statement;

   return(0);
}

static void doubleToComplex(void *voidPointer, int reverseOrder)
{
   ENV(complexNodeStack).push(
      new NumericComplexNode(ENV(doubleNodeStack).pop())
   );

   if(reverseOrder)
   {
      const ComplexNode *first  = ENV(complexNodeStack).pop();
      const ComplexNode *second = ENV(complexNodeStack).pop();

      ENV(complexNodeStack).push(first);
      ENV(complexNodeStack).push(second);
   }
}

static int handleDoubleOperatorAssign(
   void *voidPointer,
   char *id,
   char *token,
   BinaryDoubleNode::Operator op,
   int isPostAssign
)
{
   double *ptr = ENV(fractalPtr)->getDoubleVariables().lookup(id);

   if(ptr == NULL)
   {
      strstream temp;

      temp << "Undefined variable: [" << id << "] "
           << "used with " << token << '\0';

      handleError(temp.str(), voidPointer);

      free(id);
      return(-1);
   }

   const DoubleNode *rhs           = ENV(doubleNodeStack).pop();
   const VariableDoubleNode *value = new VariableDoubleNode(id, ptr);

   const DoubleNode *node = new BinaryDoubleNode(value, op, rhs);

   DoubleAssignmentStatement *statement =
      new DoubleAssignmentStatement(ptr, id, node, isPostAssign);

   ENV(currentDoubleAssign) = statement;

   free(id);

   return(0);
}

/*****************************************************************************
 * Globally exported functions
 *****************************************************************************/

char *fractal_slurpFile(const char *fileName, char **errorMessagePointer)
{
   strstream errStream;
   FILE *f = 0;

   if((f = fopen(fileName, "rb")) == 0)
   {
      errStream << "Unable to open input file: [" << fileName << "]" << '\0';
      copyFromStrStream(errStream, errorMessagePointer);

      return(0);
   }

   fseek(f, 0, 2);
   int size = ftell(f);
   fseek(f, 0, 0);

   char *text = new char[size + 1];
   int count  = fread(text, size, 1, f);
   text[size] = '\0';
   fclose(f);

   if(count != 1)
   {
      errStream << "Error reading: [" << size <<
         "] bytes from file: [" << fileName << "]" << '\0';

      copyFromStrStream(errStream, errorMessagePointer);

      delete [] text;
      return(0);
   }

   return(text);
}

Fractal *fractal_createFromText(
   const char *text,
   char **errorMessagePointer,
   int *errorLineNumberPointer
)
{
   FractalParse env;

   return(
      fractal_createFromText(
         text,
         env,
         errorMessagePointer,
         errorLineNumberPointer
      )
   );
}

int fractal_createZoomBuffer(
   const char *buffer,
   char **zoomBufferPtr,
   double x0, double y0, double x1, double y1,
   char **errorMessagePointer,
   int *errorLineNumberPointer,
   void *clientData,
   void (*mapFunction)(
      FractalDefinition *parent,
      FractalDefinition *child,
      double x0, double y0, double x1, double y1,
      void *clientData
   )
)
{
   FractalParse env;

   int returnValue = 0;

   Fractal *fractalPtr = fractal_createFromText(
      buffer,
      env,
      errorMessagePointer,
      errorLineNumberPointer
   );

   if(fractalPtr)
   {
      // Create and fill out our parent FractalDefinition structure

      FractalDefinition parent;

      parent.xStart      = fractalPtr->getXStart();
      parent.yStart      = fractalPtr->getYStart();
      parent.xEnd        = fractalPtr->getXEnd();
      parent.yEnd        = fractalPtr->getYEnd();
      parent.pixelWidth  = fractalPtr->getWidth();
      parent.pixelHeight = fractalPtr->getHeight();

      // 'mapFunction' will fill out child

      FractalDefinition child;

      child.xStart      = 0.0;
      child.yStart      = 0.0;
      child.xEnd        = 0.0;
      child.yEnd        = 0.0;
      child.pixelWidth  = 0.0;
      child.pixelHeight = 0.0;

      mapFunction(
         &parent,
         &child,
         x0, y0, x1, y1,
         clientData
      );

      char newMapping[1024];

      sprintf(
         newMapping,
         " {\n"
         "      (%.20f,\n"
         "       %.20f,\n"
         "       %.20f,\n"
         "       %.20f) => (%d, %d)\n"
         "   }",
         child.xStart, child.yStart,
         child.xEnd,   child.yEnd,
         (int)(child.pixelWidth  + 0.5),
         (int)(child.pixelHeight + 0.5)
      );

      int leadingChars  = env.mappingSectionStartOffset;
      int trailingChars = strlen(buffer) - env.mappingSectionEndOffset;

      // Here we seemingly allocate our buffer one byte too small but
      // in reality this is correct as we're going to leave off the
      // last byte of our trailing chars as otherwise we end up with
      // a spurious '\n' and our text grows longer by one (blank) line
      // each time we alter our mapping section via this function

      *zoomBufferPtr =
         new char[leadingChars + trailingChars + strlen(newMapping)];

      // Note the use of 'trailingChars - 1' as per the above comment ...

      sprintf(
         *zoomBufferPtr,
         "%.*s%s%.*s",
         leadingChars, buffer,
         newMapping,
         trailingChars - 1,
         buffer + env.mappingSectionEndOffset
      );

      delete fractalPtr;

      returnValue = 1;
   }

   return(returnValue);
}
