// C/C++ Headers

#include <stdio.h>
#include <string.h>
#include <ctype.h>

// Local Headers

#include "kvenvironment.h"

/*****************************************************************************
 * Local function prototypes
 *****************************************************************************/

static char *openErrorMessage(char *name);
static char *trim(char *buffer);
static int parseLine(
   char *buffer, char **key, char **value, char **errMsgPtr
);
static int ignoreThisLine(char *line);

/*****************************************************************************
 * Definition of class KVEnvironment
 *****************************************************************************/

KVEnvironment *KVEnvironment::create()
{
   return(new KVEnvironment());
}

KVEnvironment *KVEnvironment::create(
   char *fileName,
   Option errorOption,
   char **errMsgPtr
)
{
   KVEnvironment *environment = NULL;
   FILE *f                    = NULL;

   if((f = fopen(fileName, "r")) == NULL)
   {
      if(errorOption == ErrorIfOpenFails)
         *errMsgPtr = openErrorMessage(fileName);

      else
         environment = new KVEnvironment();
   }
   else
   {
      environment = new KVEnvironment();

      if(!environment->read(f, errMsgPtr))
      {
         delete environment;
         environment = NULL;
      }

      fclose(f);
   }

   return(environment);
}

KVEnvironment::~KVEnvironment()
{
   destroy();
}

Environment *KVEnvironment::clone(char **errMsgPtr)
{
   KVEnvironment *env = new KVEnvironment();

   for(int i = 0; i < mCount; i ++)
   {
      if(!env->add(mKeys[i], mValues[i], errMsgPtr))
      {
         delete env;
         return(NULL);
      }
   }

   return(env);
}

int KVEnvironment::keyDefined(char *key)
{
   return(findIndex(key) != -1);
}

char *KVEnvironment::getValue(char *key, char **errMsgPtr)
{
   int index   = findIndex(key);
   char *value = NULL;

   if(index != -1)
   {
      value = new char[strlen(mValues[index]) + 1];

      sprintf(value, "%s", mValues[index]);

      return(value);        
   }
   else
   {
      *errMsgPtr = keyNotDefinedMessage(key);
   }

   return(value);

}

int KVEnvironment::setValue(char *key, char *value, char **errMsgPtr)
{
   int index       = findIndex(key);
   int returnValue = FALSE;

   if(index != -1)
   {
      delete [] mValues[index];

      mValues[index] = new char[strlen(value) + 1];

      sprintf(mValues[index], "%s", value);

      returnValue = TRUE;
   }
   else
   {
      *errMsgPtr = keyNotDefinedMessage(key);
   }

   return(returnValue);

}

int KVEnvironment::add(char *key, char *value, char **errMsgPtr)
{
   int index       = findIndex(key);
   int returnValue = FALSE;

   if(index == -1)
   {
      addEntry(
         strcpy(new char[strlen(key)   + 1], key),
         strcpy(new char[strlen(value) + 1], value)
      );

      returnValue = TRUE;
   }
   else
   {
      *errMsgPtr = keyAlreadyDefinedMessage(key);
   }

   return(returnValue);

}

int KVEnvironment::remove(char *key, char **errMsgPtr)
{
   int index       = findIndex(key);
   int returnValue = FALSE;

   if(index == -1)
   {
      removeEntry(index);
      returnValue = FALSE;
   }
   else
   {
      *errMsgPtr = keyNotDefinedMessage(key);
   }

   return(returnValue);
}

int KVEnvironment::print(FILE *out, char **)
{
   // Make things look nice, figure out the longest key ...

   int i       = 0;
   int longest = 0;

   for(i = 0; i < mCount; i ++)
   {
      int length = strlen(mKeys[i]);

      if(length > longest)
         longest = length;
   }

   for(i = 0; i < mCount; i ++)
   {
      fprintf(out, "%s", mKeys[i]);

      int length = strlen(mKeys[i]);

      for(int j = length; j < longest; j ++)
         fprintf(out, " ");

      fprintf(out, " = %s\n", mValues[i]);
   }

   return(TRUE);
}

KVEnvironment::KVEnvironment()
   : mCount(0), mKeys(0), mValues(0)
{
}

int KVEnvironment::read(FILE *f, char **errMsgPtr)
{
   char buffer[255] = { '\0' };
   int c            = 0;
   int returnValue  = TRUE;

   while((c = fgetc(f)) != EOF)
   {
      ungetc(c, f);

      fgets(buffer, sizeof(buffer), f);

      if(!ignoreThisLine(buffer))
      {
         char *key   = NULL;
         char *value = NULL;

         if(!parseLine(buffer, &key, &value, errMsgPtr))
         {
            returnValue = FALSE;
            break;
         }

         addEntry(key, value);
      }
   }

   return(returnValue);
}

void KVEnvironment::destroy()
{
   for(int i = 0; i < mCount; i ++)
   {
      delete [] mKeys[i];
      delete [] mValues[i];
   }

   delete [] mKeys;
   delete [] mValues;

   mCount  = 0;
   mKeys   = NULL;
   mValues = NULL;
}

int KVEnvironment::findIndex(char *key)
{
   for(int i = 0; i < mCount; i ++)
      if(strcmp(key, mKeys[i]) == 0)
         return(i);

   return(-1);
}

void KVEnvironment::addEntry(char *key, char *value)
{
   int size         = mCount + 1;
   char **newKeys   = new char *[size];
   char **newValues = new char *[size];
   int i            = 0;

   for(i = 0; i < mCount; i ++)
   {
      newKeys[i]   = mKeys[i];
      newValues[i] = mValues[i];
   }

   newKeys[mCount]   = key;
   newValues[mCount] = value;

   delete [] mKeys;
   delete [] mValues;

   mKeys   = newKeys;
   mValues = newValues;

   ++mCount;
}

void KVEnvironment::removeEntry(int index)
{
   if(mCount == 1)
   {
      destroy();
   }
   else
   {
      int size         = mCount - 1;
      char **newKeys   = new char *[size];
      char **newValues = new char *[size];

      for(int i = 0, j = 0; i < mCount; i ++)
      {
         if(i != index)
         {
            newKeys[j]   = mKeys[i];
            newValues[j] = mValues[i];
            j++;
         }
      }

      delete [] mKeys[index];
      delete [] mValues[index];

      delete [] mKeys;
      delete [] mValues;

      mKeys   = newKeys;
      mValues = newValues;

      --mCount;
   }
}

/*****************************************************************************
 * Local Functions
 *****************************************************************************/

static char *openErrorMessage(char *name)
{
   static char *m1 = "Error reading file: [";
   static char *m2 = "]";

   int length   = strlen(m1) + strlen(name) + strlen(m2);

   char *buffer = new char[length];

   sprintf(buffer, "%s%s%s", m1, name, m2);

   return(buffer);
}

static char *trim(char *buffer)
{
   int start = -1;
   int end   = -1;

   for(int i = 0; buffer[i]; i ++)
   {
      if(!isspace(buffer[i]))
      {
         if(start == -1)
            start = i;

         end = i;
      }
   }

   if(start == -1)
      return(NULL);

   int nChars   = (end - start) + 1;
   char *result = new char[nChars + 1];

   sprintf(result, "%.*s", nChars, buffer + start);

   return(result);
}

static int parseLine(
   char *buffer, char **key, char **value, char **errMsgPtr
)
{
   char *theKey   = NULL;
   char *theValue = NULL;
   char *ptr      = NULL;

   if((ptr = strchr(buffer, '=')) != NULL)
   {
      *ptr = '\0';

      theKey   = trim(buffer);
      theValue = trim(ptr + 1);

      *ptr = '=';

      if(theKey == NULL || theValue == NULL)
      {
         delete [] theKey;
         delete [] theValue;

         theKey   = NULL;
         theValue = NULL;
      }
   }

   if(theKey != NULL)
   {
      *key   = theKey;
      *value = theValue;

      return(TRUE);
   }

   static char *m1 = "Illegal line: [";
   static char *m2 = "]";

   int length = strlen(m1) + strlen(buffer) + strlen(m2) + 1;

   *errMsgPtr = new char[length];

   sprintf(*errMsgPtr, "%s%s%s", m1, buffer, m2);

   return(FALSE);
}

static int ignoreThisLine(char *s)
{
   for(int i = 0; s[i]; i ++)
      if(!isspace(s[i]))
         return(s[i] == Environment::getCommentChar());

   return(TRUE);
}

