#!/uns/bin/python

import os, time, select, string, tempfile, sys, traceback, types
import pygale, gale_pack, sign, gale_env, engine, userinfo
import openssl.evp
from gale_const import *

GALE_DIR = gale_env.get('GALE_DIR', os.path.join(userinfo.home_dir,
	'.gale'))
DEBUG = 0
PRIVATE_MAGIC = 'h\023\000\001'
PRIVATE_MAGIC2 = 'h\023\000\003'
PRIVATE_MAGIC3 = 'GALE'
SUBVERSION = '\000\002'

KEY_RESPONSE_CAT = '_gale.key.'
OLD_KEY_RESPONSE_CAT = '@%s/auth/key/%s/'
KEY_REQUEST_CAT = '_gale.query.'
OLD_KEY_REQUEST_CAT = '@%s/auth/query/%s/'

NORMAL = 0
TRUSTED = 1

# Timeout for AKD requests (in milliseconds)
AKD_TIMEOUT = 30000
# Time between AKD requests for same key (in seconds)
AKD_INTERVAL = 60

# ----------------------------------------------------------------------
# Collector abstract data type
class NullValueClass: pass
NullValue = NullValueClass()

class Collector:
	def __init__(self, inputs):
		"""Inputs is the list of items for which outputs need to be
		collected"""
		self.__inputs = inputs
		self.__values = {}
		for item in inputs:
			self.__values[item] = NullValue
	
	def setoutput(self, input, output):
		"""Set this output for this input"""
		if DEBUG > 1:
			print 'COLL: set[%s] = %s' % (input, output)
		self.__values[input] = output
	
	def done(self):
		"""Return 1 iff the collector has collected outputs for all its
		inputs"""
		for item in self.__values.keys():
			if self.__values[item] == NullValue:
				return 0
		if DEBUG > 1:
			print 'COLL: done'
		return 1
	
	def getvalues(self):
		"""Return a list of outputs corresponding to my list of inputs"""
		return map(lambda x, d=self.__values: d[x], self.__inputs)

# Request a set of keys.  Use the first key to return.
class AnyCollector:
	"Done as soon as any input receives an output"
	def __init__(self, keynames):
		"""Inputs is the list of items for which outputs need to be
		collected"""
		self.__inputs = keynames
		self.__values = {}
		self.__done = 0	# after done once, ignore remaining outputs
		for item in keynames:
			self.__values[item] = NullValue
	
	def setoutput(self, keyname, keyobj):
		"""Set this output for this input"""
		if self.__done: return
		if DEBUG > 1:
			print 'ANYCOLL: set[%s] = %s' % (keyname, keyobj)
		self.__values[keyname] = keyobj
	
	def done(self):
		"""Return 1 iff the collector has collected at least one
		output"""
		if self.__done:
			# If already done, don't fire again
			return 0
		for item in self.__values.keys():
			if self.__values[item] is not None and\
				self.__values[item] != NullValue:
				self.__done = 1
				if DEBUG > 1:
					print 'ANYCOLL: done (with something)'
				return 1
		values = self.__values.values()
		if NullValue not in values:
			# Means we got a response from everything, even if each one
			# was None
			if DEBUG > 1:
				print 'ANYCOLL: done (with None)'
			return 1
		if DEBUG > 1:
			print 'ANYCOLL: not done... still waiting'
		return 0
	
	def getvalue(self):
		"""Return the first output"""
		values = self.__values.values()
		goodvalues = filter(lambda x: x is not None and x != NullValue,
			values)
		if not goodvalues:
			goodvalues = filter(lambda x: x != NullValue, values)
		return goodvalues[0]

# ----------------------------------------------------------------------
class KeyCacheClass:
	def __init__(self):
		self.__keys = {}		# in-memory key cache
		self.__last_akd = {}	# Time we did last AKD request for key
		self.__keychain = {}	# List of callbacks waiting for a key
	def set(self, name, val):
		self.__keys[name] = val
	def get(self, name):
		return self.__keys[name]
	def __len__(self):
		return len(self.__keys)
	def has_key(self, name):
		return self.__keys.has_key(name)

	def get_key(self, name):
		# TODO: check for newer on-disk version
		if self.__keys.has_key(name):
			return self.__keys[name]
		else:
			return None

	def add_to_memory_cache(self, keyobj):
		self.__keys[keyobj.name()] = keyobj

	# AKD
	def find_pubkey_akd(self, keyobj, callback):
		name = keyobj.name()
		if DEBUG: print 'Finding public key via AKD:', name
		if '@' in name:
			i = string.find(name, '@')
			domain = name[i+1:]
			user = name[:i]
		else:
			# Hmm, we can't request a domain key without the user
			# Invalid key name
			keyobj.setpublic(None)
			callback(keyobj)
			return
		if ':' in name:
			keyobj.setpublic(None)
			pygale.call_error_handler(
				'Ignoring AKD request for %s with colon' % name)
			callback(keyobj)
			return

		# First see when was the last time we did AKD for this key
		if self.__last_akd.has_key(name):
			last_checked = self.__last_akd[name]
			now = time.time()
			if now - last_checked < AKD_INTERVAL:
				# Chain requests
				if DEBUG:
					print 'not doing AKD for %s; done too recently' %\
						name
				pygale.call_update_handler('Chaining to existing key '+
					'request for %s' % name)
				self.__keychain[name] = self.__keychain.get(name, []) +\
					[callback]
				return
		else:
			self.__last_akd[name] = time.time()
	
		self.__keychain[name] = self.__keychain.get(name, []) +\
			[callback]
		pygale.call_update_handler('Requesting key %s' % name)
		client = pygale.GaleClient()
		client.connect(lambda h, s=self, d=domain, u=user, n=name, k=keyobj,
			c=client: s.find_pubkey_akd2(h, d, u, n, c, k))

	def find_pubkey_akd2(self, hostname, domain, user, name, client,
		keyobj):
		if hostname is None:
			pygale.call_error_handler('Unable to connect to gale ' +
				'server for AKD')
			keyobj.setpublic(None)
			self.cleanup_akd(keyobj)
			return

		flipped_user = string.split(user, '.')
		flipped_user.reverse()
		flipped_user = string.join(flipped_user, '.')
		locs = ['%s%s@%s' % (KEY_RESPONSE_CAT, user, domain),
			OLD_KEY_RESPONSE_CAT % (domain, flipped_user)]
		cb = lambda b, g, s=self, u=user, d=domain, n=name, c=client, \
			k=keyobj: s.find_pubkey_akd3(b, g, u, d, n, c, k)
		client.sub_to(locs, cb)
	
	def find_pubkey_akd3(self, bad_locs, good_locs, user, domain, name,
		client, keyobj):

		if bad_locs:
			pygale.call_error_handler('Error subscribing to AKD ' +
				'location %s' % string.join(bad_locs, ', '))
			keyobj.setpublic(None)
			self.cleanup_akd(keyobj)
			return
		p = pygale.Puff()

		# for backwards compatibility with old key serving clients
		# dangermouse
		p.set_text('question.key', name)
		p.set_text('question/key', sign.flip_local_key_part(name))

		flipped_user = string.split(user, '.')
		flipped_user.reverse()
		flipped_user = string.join(flipped_user, '.')

		auth_locs = ['%s%s@%s' % (KEY_REQUEST_CAT, user, domain)]
		auth_locs.append(OLD_KEY_REQUEST_CAT % (domain, flipped_user))
		p.set_loc(string.join(auth_locs, ' '))

		timeout_callback = lambda k=keyobj, s=self, l=client:\
			s.handle_AKD_timeout(k, l)
		# add_timeout returns a handle to use to cancel the timeout
		timeout_handle = engine.engine.add_timeout(AKD_TIMEOUT,
			timeout_callback)
		client.set_puff_callback(lambda p, s=self, k=keyobj,
			l=client, t=timeout_handle, d=domain:
			s.handle_key_puff(p, k, l, t, d))
		client.transmit_puff(p)
	
	def cleanup_akd(self, keyobj):
		name = keyobj.name()
		if keyobj.public():
			for keyname in [name] + keyobj.get_links_to_me():
				self.__keys[keyname] = keyobj
		if DEBUG:
			print 'Cleaning up after', name
			print 'Links to me:', keyobj.get_links_to_me()

		for name in [keyobj.name()] + keyobj.get_links_to_me():
			if self.__keychain.has_key(name):
				# make copy of callbacks
				cbs = self.__keychain[name][:]
				del self.__keychain[name]
				for cb in cbs:
					if DEBUG: print 'Cleanup CB:', cb
					cb(keyobj)
			if self.__last_akd.has_key(name):
				del self.__last_akd[name]

	def handle_AKD_timeout(self, key, client):
		if DEBUG: print 'AKD request timeout, key', key.name()
		pygale.call_update_handler('key request for %s timed out' %
			key.name())
		client.del_puff_callback()
		key.setdata(None)
		self.cleanup_akd(key)

	def handle_key_puff(self, puff, keyobj, client, timeout_handle,
		domain):
		if DEBUG: print 'Key response puff returned for', keyobj.name()
		if DEBUG: print 'Key response puff loc:', `puff.get_loc()`
		if puff.get_loc():
			# New-style AKD category
			if '_gale.key.' + keyobj.name() == puff.get_loc():
				if DEBUG: print '... and it is for the right key'
			else:
				if DEBUG: print '... and it is for the wrong key; ignoring'
				return
		
		keyobj.setpublic(None)
		if puff.get_text('answer/key/error'):
			if DEBUG: print '... but it contains an error:',\
				puff.get_text_first('answer/key/error')
			signer = puff.get_signer(None)
			if signer is None or signer != domain:
				if DEBUG: print 'Received unverified AKD response'
				pygale.call_update_handler(
					'Received unverified AKD response from ' + `signer`)
				pygale.call_update_handler(
					'Unverified response: ' + puff.get_text_first(
						'answer/key/error'))
				return
			client.del_puff_callback()
			engine.engine.del_timeout(timeout_handle)
			pygale.call_update_handler(
				puff.get_text_first('answer/key/error', 'AKD error'))
			self.cleanup_akd(keyobj)
			return
		if puff.get_binary('answer/key'):
			if DEBUG: print '... and it contains answer/key'
			key = puff.get_binary_first('answer/key')
			keyobj.setdata(key)
			sign.import_pubkey(keyobj, lambda k, to=timeout_handle,
				c=client, s=self: s.finish_key_puff(k, c, to), 0)
		elif puff.get_binary('answer.key'):
			if DEBUG: print '... and it contains answer.key'
			key = puff.get_binary_first('answer.key')
			keyobj.setdata(key)
			sign.import_pubkey(keyobj, lambda k, to=timeout_handle,
				c=client, s=self: s.finish_key_puff(k, c, to), 0)
		else:
			if DEBUG: print '... but it does not contain a key; waiting'
	
	def finish_key_puff(self, keyobj, client, timeout_handle):
		if DEBUG:
			print 'Finished key puff for', keyobj.name()
#		if not keyobj.public() and not keyobj.members():
		if not keyobj.verified():
			# Got a bad public key; don't bother cleaning up just yet
			if DEBUG: print 'Got a bad public key; waiting for another'
			return
		client.del_puff_callback()
		engine.engine.del_timeout(timeout_handle)
		self.cleanup_akd(keyobj)

# ----------------------------------------------------------------------

# ----------------------------------------------------------------------
# Private keys

def import_v3_privkey(key):
	fraglist = gale_pack.group_to_FragList(key, DAN_IS_STOOPID=1)
	privkey = openssl.evp.PKEY()
	privkey.assign_RSA(openssl.rsa.RSA())
	rsa = privkey.pkey.rsa
	rsa.n = openssl.bn.bin2bn(fraglist.get_binary_first('rsa.modulus'))
	rsa.e = openssl.bn.bin2bn(fraglist.get_binary_first('rsa.exponent'))
	rsa.d = openssl.bn.bin2bn(fraglist.get_binary_first('rsa.private.exponent'))
	prime_data = fraglist.get_binary_first('rsa.private.prime')
	rsa.p = openssl.bn.bin2bn(prime_data[:GALE_RSA_PRIME_LEN])
	rsa.q = openssl.bn.bin2bn(prime_data[GALE_RSA_PRIME_LEN:])
	prime_data = fraglist.get_binary_first('rsa.private.prime.exponent')
	rsa.dmp1 = openssl.bn.bin2bn(prime_data[:GALE_RSA_PRIME_LEN])
	rsa.dmq1 = openssl.bn.bin2bn(prime_data[GALE_RSA_PRIME_LEN:])
	rsa.iqmp = openssl.bn.bin2bn(fraglist.get_binary_first('rsa.private.coefficient'))
	return privkey


def import_v1or2_privkey(key):
	privkey = openssl.evp.PKEY()
	privkey.assign_RSA(openssl.rsa.RSA())
	(bits, key) = gale_pack.pop_int(key)
	if bits > GALE_RSA_MODULUS_BITS:
		pygale.call_error_handler('wrong number of bits in private key')
		return None
	(modulus, key) = gale_pack.pop_rle(key, GALE_RSA_MODULUS_LEN)
	privkey.pkey.rsa.n = openssl.bn.bin2bn(modulus)
	(pubexp, key) = gale_pack.pop_rle(key, GALE_RSA_MODULUS_LEN)
	privkey.pkey.rsa.e = openssl.bn.bin2bn(pubexp)
	(exp, key) = gale_pack.pop_rle(key, GALE_RSA_MODULUS_LEN)
	privkey.pkey.rsa.d = openssl.bn.bin2bn(exp)
	(prime, key) = gale_pack.pop_rle(key, GALE_RSA_PRIME_LEN*2)
	privkey.pkey.rsa.p = openssl.bn.bin2bn(prime[:GALE_RSA_PRIME_LEN])
	privkey.pkey.rsa.q = openssl.bn.bin2bn(prime[GALE_RSA_PRIME_LEN])
	(primeexp, key) = gale_pack.pop_rle(key, GALE_RSA_PRIME_LEN*2)
	privkey.pkey.dmp1 = openssl.bn.bin2bn(primeexp[:GALE_RSA_PRIME_LEN])
	privkey.pkey.dmq1 = openssl.bn.bin2bn(prime[GALE_RSA_PRIME_LEN:])
	(coef, key) = gale_pack.pop_rle(key, GALE_RSA_PRIME_LEN)
	privkey.pkey.iqmp = openssl.bn.bin2bn(coef)
	return privkey

def import_privkey(key):
	# _ga_import_priv
	(magic, key) = gale_pack.pop_data(key, len(PRIVATE_MAGIC))
	if PRIVATE_MAGIC == magic:
		version = 1
		(keyname, key) = gale_pack.pop_nulltermstr(key)
	elif PRIVATE_MAGIC2 == magic:
		version = 2
		(keyname, key) = gale_pack.pop_lenstr(key, chars=1)
	elif PRIVATE_MAGIC3 == magic:
		(subversion, key) = gale_pack.pop_data(key, len(SUBVERSION))
		if SUBVERSION == subversion:
			version = 3
		else:
			pygale.call_error_handler('Unsupported version 3 key format')
			return None
		(keyname, key) = gale_pack.pop_lenstr(key, chars=1)
	else:
		pygale.call_error_handler('invalid private key format')
		return None

	if version == 3:
		privkey = import_v3_privkey(key)
	else:
		privkey = import_v1or2_privkey(key)

	if privkey is None:
		return None
	
	if not KeyCache.has_key(keyname):
		keyobj = sign.Key(keyname)
		keyobj.setprivate(privkey)
		add_to_memory_cache(keyobj)
	else:
		keyobj = KeyCache.get(keyname)
		keyobj.setprivate(privkey)

	return keyobj

def have_a_privkey(recipients):
	"""Return 1 iff I have at least one of the private keys in this
	list of ids"""
	for recp in recipients:
		keyobj = get_key(recp)
		if keyobj and keyobj.private():
			return 1
		else:
			keyobj = find_privkey_fromdisk(recp)
			if keyobj and keyobj.private():
				return 1
	return 0
# ----------------------------------------------------------------------

# Cache a public key if one can be found in the key path
# Return the name of the key if found, None otherwise
def find_pubkey_fromdisk(keyobj, callback):
	name = keyobj.name()
	revname = sign.flip_local_key_part(name)
	if DEBUG: print 'finding pubkey from disk cache: %s' % (name)
	# First look for key with trailing .gpub
	for (prefix, suffix, trust) in GPUBKEYPATH:
		gpubpath = os.path.join(prefix, suffix, name + '.gpub')
		if DEBUG >= 2: print 'authcache: checking', gpubpath
		if os.path.exists(gpubpath):
			if DEBUG: print 'found key at', gpubpath
			f = open(gpubpath, 'rb')
			keydata = f.read()
			f.close()
			keyobj.setdata(keydata)
			sign.import_pubkey(keyobj,
				lambda k, cb=callback: cache_key_cb(k, cb), trust)
			return
	# Then look for key without trailing .gpub (old-style)
	for (prefix, suffix, trust) in KEYPATH:
		revpath = os.path.join(prefix, suffix, revname)
		if DEBUG >= 2: print 'authcache: checking', revpath
		if os.path.exists(revpath):
			if DEBUG: print 'found key at', revpath
			f = open(revpath, 'rb')
			keydata = f.read()
			f.close()
			keyobj.setdata(keydata)
			sign.import_pubkey(keyobj,
				lambda k, cb=callback: cache_key_cb(k, cb), trust)
			return
	# Otherwise, we didn't find it; return None
	keyobj.setpublic(None)
	callback(keyobj)
	return

def cache_key_cb(keyobj, callback):
	# Save it to in-memory cache
	if keyobj.data():
		if DEBUG: print 'caching key to memory:', keyobj
		KeyCache.add_to_memory_cache(keyobj)
	callback(keyobj)

# Find a private key on disk
# Return the key object, or None if not found
def find_privkey_fromdisk(name):
	# look for gpri first
	for (prefix, suffix) in PRIVKEYPATH:
		path = os.path.join(prefix, suffix, name + '.gpri')
		if os.path.exists(path):
			f = open(path, 'rb')
			key = f.read()
			f.close()
			return import_privkey(key)

	# then look for old-style key name
	revname = sign.flip_local_key_part(name)
	for (prefix, suffix) in PRIVKEYPATH:
		path = os.path.join(prefix, suffix, revname)
		if os.path.exists(path):
			f = open(path, 'rb')
			key = f.read()
			f.close()
			return import_privkey(key)

	return None

def save_pubkey_todisk(keyobj):
	"""Save public key bits to disk; also save copy in in-memory
	cache"""
	if keyobj.data() is None:
		if DEBUG: print 'Got empty public key to save to disk!'
		return
	# First store it in the in-memory cache
	KeyCache.add_to_memory_cache(keyobj)
	if DEBUG: print 'save_pubkey_todisk(%s)' % keyobj.name()
	# Save it to a file
	tempfile.tempdir = CACHE_PATH
	tmpfile = tempfile.mktemp()
	# Add '.gpub' on to the name of the key
	gpubname = keyobj.name() + '.gpub'
	keypath = os.path.join(CACHE_PATH, gpubname)
	# Don't bother overwriting the old (non-.gpub) key
	if os.path.exists(keypath):
		pygale.call_update_handler('overwriting old '+
			'public key %s with a new one' % keyobj.name())
	try:
		f = open(tmpfile, 'wb')
		f.write(keyobj.data())
		f.close()
		# Fix for win32: rename won't overwrite an existing file
		if sys.platform == 'win32':
			if os.path.exists(keypath):
				os.unlink(keypath)
		os.rename(tmpfile, keypath)
	except IOError, e:
		pygale.call_update_handler(
			'Error saving key to cache: ' + str(e))

def find_pubkey_list(keyobjs, callback, do_akd=1):
	if DEBUG:
		print 'Finding all pubkeys in list:', keyobjs
	c = Collector(map(lambda x: x.name(), keyobjs))
	for key in keyobjs:
		if DEBUG > 1:
			print 'Finding pubkey', key.name()
		find_pubkey(key, lambda key, c=c, cb=callback:
			_collector_callback(key, c, cb), do_akd=do_akd)

def _collector_callback(key, collector, callback):
	if DEBUG > 1: print 'collector_callback_all, key.name:', key.name()
	collector.setoutput(key.name(), key)
	if collector.done():
		if DEBUG > 1:
			print 'collector_callback_all, done!'
		# Call back with the response
		callback(collector.getvalues())

def find_pubkey_any(keyobjs, callback, do_akd=1):
	"Find any of the keys on this list"
	if DEBUG:
		print 'Finding any pubkey in list:', keyobjs
	c = AnyCollector(map(lambda x: x.name(), keyobjs))
	for key in keyobjs:
		# TODO: this may break for redirected keys
		find_pubkey(key, lambda key, c=c, cb=callback:
			_collector_callback_any(key, c, cb), do_akd=do_akd)

def _collector_callback_any(key, collector, callback):
	if DEBUG > 1: print 'collector_callback_any, key.name:', key.name()
#	if key.public() is None and not key.members():
	if not key.verified():
		if DEBUG > 1:
			print 'collector_callback_any: key %s is not verified' %\
				key.name()
		collector.setoutput(key.name(), None)
	else:
		# Got a real key with stuff in it
		collector.setoutput(key.name(), key)
	if collector.done():
		retval = collector.getvalue()
		if DEBUG > 1:
			print 'collector_callback_any, done! calling cb with',\
				retval, callback
		# Call back with the first response
		callback(retval)

def find_pubkey(keyobj, callback, do_akd=1):
#		if DEBUG: print 'Find_pubkey: %s (%s)' % (`keyobj`, keyobj.name())
	if DEBUG: print 'Find_pubkey (%s), cb:' % keyobj.name(), callback
	keyobj.setpublic(None)
	if KeyCache.has_key(keyobj.name()) and\
		KeyCache.get(keyobj.name()).public():
		if DEBUG: print 'Returning key from in-memory cache'
		cached_key = KeyCache.get(keyobj.name())
		callback(cached_key)
		return

	# Go to the disk cache
	find_pubkey_fromdisk(keyobj, lambda k, cb=callback,
		d=do_akd: _find_pubkey2(k, cb, do_akd=d))

def _find_pubkey2(keyobj, callback, do_akd=1):
	if keyobj.public() is not None:
		# Put it in the in-memory cache
		KeyCache.add_to_memory_cache(keyobj)
		callback(keyobj)
		return

	# Oops, we gotta do AKD
	if do_akd:
		find_pubkey_akd(keyobj, lambda k, cb=callback:
			_find_pubkey3(k, cb))
	else:
		# Can't find it without AKD, oh well
		callback(keyobj)

def _find_pubkey3(key, callback):
	if key.verified():
		# Put it in the in-memory cache
		KeyCache.add_to_memory_cache(key)
		callback(key)
		return
	if DEBUG: print 'In _find_pubkey3, calling callback'
	callback(key)

# ----------------------------------------------------------------------
def find_pubkey_akd(*args):
	return apply(KeyCache.find_pubkey_akd, args)
def get_key(name):
	return KeyCache.get_key(name)
def add_to_memory_cache(*args):
	return apply(KeyCache.add_to_memory_cache, args)

def init():
	global KeyCache
	if DEBUG: print 'Initializing keycache'
	KeyCache = KeyCacheClass()

	if not gale_env.has_key('GALE_SYS_DIR'):
		print 'Warning: GALE_SYS_DIR not set; using ""'
	GALE_SYS_DIR = gale_env.get('GALE_SYS_DIR', '')
	global GPUBKEYPATH, KEYPATH, CACHE_PATH, PRIVKEYPATH
	GPUBKEYPATH = [
		(GALE_DIR, os.path.join('auth','trusted'), TRUSTED),
#		(GALE_DIR, os.path.join('auth','private'), NORMAL),
		(GALE_DIR, os.path.join('auth','local'), NORMAL),
		(GALE_DIR, os.path.join('auth','cache'), NORMAL),
		(GALE_SYS_DIR, os.path.join('auth', 'trusted'), TRUSTED),
		(GALE_SYS_DIR, os.path.join('auth', 'local'), NORMAL),
		(GALE_SYS_DIR, os.path.join('auth', 'cache'), NORMAL)]
	KEYPATH = [
		(GALE_DIR, os.path.join('auth','trusted'), TRUSTED),
		(GALE_DIR, os.path.join('auth','local'), NORMAL),
		(GALE_DIR, os.path.join('auth','cache'), NORMAL),
		(GALE_SYS_DIR, os.path.join('auth', 'trusted'), TRUSTED),
		(GALE_SYS_DIR, os.path.join('auth', 'local'), NORMAL),
		(GALE_SYS_DIR, os.path.join('auth', 'cache'), NORMAL)]

	CACHE_PATH = os.path.join(GALE_SYS_DIR, 'auth', 'cache')

	PRIVKEYPATH = [
		(GALE_DIR, os.path.join('auth','private')),
		(GALE_SYS_DIR, os.path.join('auth', 'private'))]


def shutdown():
	global KeyCache
	del KeyCache
