/*
 * $Id: fioservice.cpp,v 1.1 2004/04/01 08:52:24 daichi Exp $
 *
 * Copyright 2003- ONGS Inc. All rights reserved.
 * 
 * author: Masanori OZAWA (ozawa@ongs.co.jp)
 * version: $Revision: 1.1 $
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 *   1. Redistributions of source code must retain the above copyright
 *      notice, this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY ONGS INC ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL ONGS INC OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the ONGS Inc.
 * 
 */

#include "../include/fusenshi.h"
#include "fioservice.h"
#include "fusenshiloader.h"
#include "fusenshisaver.h"
#include "filemanager.h"

#include <signal.h>

#define PRINT_TMP_FILE "print.tmp"

/**
 * ͣΥ󥹥
 */
FioService* FioService::m_pInstance = NULL;

/**
 * 䵻
 */
PropertyData FioService::m_cProperty;

/**
 * 󥹥ȥ饯
 */
FioService::FioService(int inFD, int outFD)
{
    m_nIn = inFD;
    m_nOut = outFD;
    m_bStop = FALSE;
    m_bRunning = FALSE;
    m_pCommand = new CommandControl();
    
    if (!m_pCommand) {
        throw "Out of Memory Error!";
    }
    
    m_pMutex = g_mutex_new();
    
    if (!m_pMutex) {
        throw "Can't create mutex!";
    }
}

/**
 * ǥȥ饯
 */
FioService::~FioService()
{
    if (m_bRunning) {
        stop();
    }
    
    if (m_pCommand) {
        delete m_pCommand;
    }
    
    if (m_pMutex) {
        g_mutex_free(m_pMutex);
    }
}

/**
 * ̿򳫻Ϥޤ
 */
void FioService::run()
{
    // ¹Ծ֤γǧ
    int bRunning = FALSE;
    
    g_mutex_lock(m_pMutex);
    bRunning = m_bRunning;
    if (!m_bRunning) {
        m_bRunning = TRUE;
    }
    m_bStop = FALSE;
    g_mutex_unlock(m_pMutex);
    
    if (bRunning) {
        throw "FioService is already running.";
    }
    
    // ¹Գ
    int bStop = FALSE;
    LP_COMMAND lpCommand = NULL;
    
    guint nFileID = 0;
    GString* pStrData = NULL;
    FusenshiServer *pServer = NULL;
    
    try {
        while (!bStop) {
            // ɥץ׵å
            lpCommand = m_pCommand->receiveCommand(m_nIn);
            
            if (lpCommand) {
                actionCommand(lpCommand);
                m_pCommand->destroyCommand(lpCommand);
            }

            // ͥåȥ׵å
            if (pServer = getServer(), pServer) {
                // å
                pStrData = pServer->checkAndReceive();

                if (pStrData) {
                    // 䵻ե¸
                    nFileID = FusenshiSaver::saveNewFusenshiData(pStrData);

                    if (nFileID) {
                        // ɽĤ
                        m_pCommand->addCommand
                            (VERIFY_RECEIVE_DATA, nFileID, pStrData);
                    }
                    else {
                        g_string_free(pStrData, TRUE);
                    }
                }
            }
            else {
                usleep(100 * 1000 /* 100ms */);
            }
            
            // ɥץؤޥɤ̵ͭǧ
            lpCommand = m_pCommand->getCommand();
            
            if (lpCommand) {
                m_pCommand->sendCommand(m_nOut, lpCommand);
                m_pCommand->destroyCommand(lpCommand);
            }
            
            // λ׵å
            g_mutex_lock(m_pMutex);
            bStop = m_bStop;
            g_mutex_unlock(m_pMutex);
        }
    }
    catch (char* pErrorMsg) {
        fprintf(stderr, "%s: FioService: %s\n", APP_NAME, pErrorMsg);
    }
    catch (const char* pErrorMsg) {
        fprintf(stderr, "%s: FioService: %s\n", APP_NAME, pErrorMsg);
    }
    catch (...) {
        fprintf(stderr, "%s: FioService: unknown error.\n", APP_NAME);
    }
    
    // λ
    g_mutex_lock(m_pMutex);
    m_bRunning = FALSE;
    m_bStop = FALSE;
    g_mutex_unlock(m_pMutex);
}

/**
 * 䵻եɤ߹ߡɥץΤޤ
 * 
 * @param pFileName 䵻ե̾
 */
void FioService::open(const char* pFileName)
{
    GString* pData = FusenshiLoader::loadFusenshiFile(pFileName);
    
    if (pData) {
        guint nFileID = FileManager::getInstance().getIndex(pFileName);
        
        if (nFileID) {
            // Хåե˥ޥɤȥǡɲ
            m_pCommand->addCommand
                (CREATE_NEW_FUSENSHI_WINDOW, nFileID, (gpointer)pData);
        }
        else {
            // ե륤ǥåϿ˼
            g_string_free(pData, TRUE);
        }
    }
}

/**
 * ̿λޤ
 */
void FioService::stop()
{
    g_mutex_lock(m_pMutex);
    
    if (m_bRunning) {
        m_bStop = TRUE;
    }

    g_mutex_unlock(m_pMutex);
}

/**
 * ̿δλԤޤ
 */
void FioService::waitFor()
{
    int bEnable = TRUE;
    
    while (bEnable) {
        g_mutex_lock(m_pMutex);
        bEnable = m_bRunning;
        g_mutex_unlock(m_pMutex);
        
        if (bEnable) {
            usleep(250 * 1000 /* 250ms*/);
        }
    }
}

/**
 * 󥹥󥹤˴ץꥱνλ
 * Ԥޤ
 */
void FioService::destroy()
{
    FioService* instance = getInstance();
    
    if (instance) {
        instance->stop();
        instance->waitFor();
        
        delete instance;
        m_pInstance = NULL;
    }
}

/**
 * ۺѤFioService饹Υ󥹥󥹤ޤ
 * 
 * @return ͣΥ󥹥
 */
FioService* FioService::getInstance()
{
    if (!m_pInstance) {
        throw "FioService has no Instance.";
    }
    
    return m_pInstance;
}

/**
 * ɬפ˱FioService饹Υ󥹥󥹤ۤ
 * ͣΥ󥹥󥹤ޤ
 * 
 * @param inFD ɥץȤΥѥ
 * @param outFD ɥץȤΥѥ
 * @return ͣΥ󥹥
 */
FioService* FioService::getInstance(int inFD, int outFD)
{
    if (!m_pInstance) {
        m_pInstance = new FioService(inFD, outFD);
        
        if (!m_pInstance) {
            throw "Out of Memory Error!";
        }
    }
    
    return m_pInstance;
}

/**
 * 䵻ޤ
 * 
 * @return 䵻
 */
PropertyData FioService::getProperty()
{
    static bool bLoad = false;

    if (!bLoad) {
        try {
            FioService::m_cProperty = PropertyData::loadProperty();
        }
        catch (...) {
            fprintf(stderr, "%s: Can't load property file.", APP_NAME);
        }

        bLoad = true;
    }

    return FioService::m_cProperty;
}

/**
 * 䵻ꤷľݴɤޤ
 * 
 * @param property 䵻
 */
void FioService::setProperty(const PropertyData& property)
{
    if (!PropertyData::saveProperty(property)) {
        fprintf(stderr, "%s: Can't save property file.", APP_NAME);
    }
    else {
        // ѹݻ
        FioService::m_cProperty = property;
        
        // ѹȿ
        FioService::updateProperty();

        // ѹΤ
        FioService *pInstance = FioService::getInstance();
        pInstance->m_pCommand->addCommand(PROPERTY_CHANGED, 0, NULL);
    }
}

/**
 * ץѥƥѹɬפʽ¹Ԥޤ
 */
void FioService::updateProperty()
{
    PropertyData property = FioService::getProperty();
    
#ifndef WITHOUT_NETWORK
    // ͥåȥ̵ͭĴ
    if (property.isEnableNetwork()) {
        /**
         * no operation.
         * FioService::run()᥽åǼưŪͭˤʤ뤿ᡢ
         * ⤷ޤ
         */
    }
    else {
        FusenshiServer::destroy();
    }
#endif
}

/**
 * ޥɤޤ
 * 
 * @param lpCommand ޥ
 */
void FioService::actionCommand(LP_COMMAND lpCommand)
{
    if (!lpCommand) {
        throw "Null Pointer Exception.";
    }
    
    FileManager& fm = FileManager::getInstance();
    
    const char* pFileName = NULL;
    GString* lpStrData = NULL;
    
    switch (lpCommand->nCommandID) {
    case SAVE_FUSENSHI:
        pFileName = fm.getFileName(lpCommand->nFileID);
        lpStrData = (GString*)lpCommand->pData;
        if (pFileName && lpStrData) {
            FusenshiSaver::saveFusenshiData(pFileName, lpStrData);
        }
        break;
    case CREATE_NEW_FUSENSHI:
        pFileName = fm.createFile();
        if (pFileName) {
            this->open(pFileName);
            free((void*)pFileName);
        }
        break;
    case DELETE_FUSENSHI:
        fm.removeFile(lpCommand->nFileID);
        break;
    case PRINT_FUSENSHI:
        lpStrData = (GString*)lpCommand->pData;
        if (lpStrData) {
            printFusenshi(lpCommand->nFileID, lpStrData->str);
        }
        break;
    case PROPERTY_CHANGED:
        FioService::m_cProperty = PropertyData::loadProperty();
        FioService::updateProperty();
        break;
    case EXIT_APP:
        stop();
        break;
    default:
        break;
    }
}

/**
 * ͥåȥǽͭʾ䵻極ӥ楯饹
 * ޤ
 * 
 * @return 䵻極ӥ楯饹ޤϡNULLᤷޤ
 */
FusenshiServer* FioService::getServer()
{
#ifndef WITHOUT_NETWORK
    PropertyData property = FioService::getProperty();

    if (property.isEnableNetwork()) {
        return FusenshiServer::getInstance();
    }
#endif

    return NULL;
}

/**
 * 䵻ޤ
 * 
 * @param nFileID ե̤륤ǥå
 * @param pCommand ޥ
 */
void FioService::printFusenshi(guint nFileID, const char* pCommand)
{
    Glib::ustring command = pCommand;
    Glib::ustring tmpFileName = getAppDir(APP_NAME);
    Glib::ustring errMsg;
    std::string text;
    FILE* fp = NULL;
    
    // 䵻ե
    try {
        FileManager& fm = FileManager::getInstance();
        const char* pFileName = fm.getFileName(nFileID);
        FusenshiData *pData =
            FusenshiLoader::loadFusenshiData(pFileName);
        text = Glib::locale_from_utf8(pData->getText());
        delete pData;
    }
    catch (char* pMessage) {
        fprintf(stderr, "%s: %s\n", APP_NAME, pMessage);
        return;
    }
    catch (const char* pMessage) {
        fprintf(stderr, "%s: %s\n", APP_NAME, pMessage);
        return;
    }
    catch (...) {
        fprintf(stderr, "%s: Can't read fusenshi file.\n", APP_NAME);
        return;
    }
    
    // ե˥ǡ¸
    tmpFileName += "/" PRINT_TMP_FILE;
    
    if (fp = fopen(tmpFileName.c_str(), "wb"), !fp) {
        errMsg = _("Can't create print file.");
    }
    else {
        if (text.size() != fwrite(text.c_str(), 1, text.size(), fp)) {
            errMsg = _("Can't write print file.");
        }
        fclose(fp);
        
        if (0 >= errMsg.length()) {
            // եִ̾
            Glib::ustring::size_type pos = command.find(PRINT_FILE_KEY);
            if (0 <= pos) {
                command.replace(pos, strlen(PRINT_FILE_KEY), tmpFileName);
            }
            
            // μ¹
            system(command.c_str());
        }

        // ե
        remove(tmpFileName.c_str());
    }
    
    if (0 < errMsg.length()) {
        fprintf(stderr, "%s: %s\n", APP_NAME, errMsg.c_str());
    }
}
