#include "Arena.h"

#include "Game.h"
#include "File.h"
#include "Exception.h"
#include "System.h"
#include "XmlUtils.h"

#include "Area.h"
#include "Graphics.h"
#include "Camera.h"
#include "Physics.h"

#include "SpawnArea.h"
#include "ConquestArea.h"
#include "RegenerationArea.h"
#include "TeleportationArea.h"

Arena::Arena(){
	this->game = NULL;

//	this->arenaFilePath = "<uninitialized>";
	this->checksum = 0;

	this->name = "<untitled>";
//	this->versionString = "<uninitialized>";

	this->info.description = "<no description available>";

	this->graphicalProperties.sceneTypeMask = Ogre::ST_GENERIC;
	this->graphicalProperties.worldGeometrySource = "";
	this->graphicalProperties.ambientEffect.clear();
	this->graphicalProperties.ambientLight = FColor::BLACK;
	this->graphicalProperties.backgroundColor = FColor::BLACK;
	this->graphicalProperties.shadow.technique = Ogre::SHADOWTYPE_NONE;
	this->graphicalProperties.shadow.color = FColor::BLACK;
	this->graphicalProperties.shadow.farDistance = 100.0f;
	this->graphicalProperties.shadow.textureSize = 512;
	this->graphicalProperties.fog.mode = Ogre::FOG_NONE;
	this->graphicalProperties.fog.color = FColor::WHITE;
	this->graphicalProperties.fog.expDensity = 0.001f;
	this->graphicalProperties.fog.linearStart = 0.0f;
	this->graphicalProperties.fog.linearEnd = 1.0f;
	this->graphicalProperties.skyBox.material = "";
	this->graphicalProperties.skyBox.distance = 1000.0f;
	this->graphicalProperties.skyBox.drawFirst = true;
	this->graphicalProperties.skyBox.orientation = FQuaternion::IDENTITY;

	this->physicalProperties.gravity = FVector3::ZERO;
	this->physicalProperties.wind = FVector3::ZERO;
	this->physicalProperties.collisionSpace.quadTreeLevels = 0;
	this->physicalProperties.collisionSpace.min = FVector3(-1.0f, -1.0f, -1.0f);
	this->physicalProperties.collisionSpace.max = FVector3( 1.0f,  1.0f,  1.0f);

	this->objects.objectDescriptorsByName.clear();
	this->areas.areasByName.clear();
}


Arena::~Arena(){
	// delete areas
	for( areasByName_t::iterator iter = this->areas.areasByName.begin(); iter != this->areas.areasByName.end(); ++iter ){
		delete iter->second;
	}
}



Arena* Arena::clone() const {
	Arena* ret = new Arena();

	*ret = *this;

	ret->areas.areasByName.clear();
	// copy areas
	for( areasByName_t::const_iterator c_iter = this->areas.areasByName.begin(); c_iter != this->areas.areasByName.end(); ++c_iter ){
		Area* a = c_iter->second->clone();
		ret->areas.areasByName.insert( areasByName_t::value_type( a->getName(), a ) );
	}

	return ret;
}



void Arena::attach(Game* game){
	if( this->game != NULL ){
		throw Exception("Arena is already attached to a game object.", "Arena::attach()");
	}



	// do graphics
	Graphics* graphics = game->getSystem()->getGraphics();


	graphics->setSceneManagerByType( this->graphicalProperties.sceneTypeMask );
	Ogre::SceneManager* sm = graphics->getSceneManager();

//	sm->showBoundingBoxes( true );

	// set shadow params
	graphics->setShadowTechnique( this->graphicalProperties.shadow.technique );
	sm->setShadowColour( this->graphicalProperties.shadow.color.toOgreColourValue() );
	sm->setShadowFarDistance( this->graphicalProperties.shadow.farDistance );
	sm->setShadowTextureSize( this->graphicalProperties.shadow.textureSize );

	// FIXME: do this somewhere else...
	//vp->setBackgroundColour( this->graphicalProperties.backgroundColor.toOgreColourValue() );


	// set up ambient stuff
	sm->setAmbientLight( this->graphicalProperties.ambientLight.toOgreColourValue() );
	if( !this->graphicalProperties.ambientEffect.empty() ){
		game->spawnEffect( this->graphicalProperties.ambientEffect, game->getCamera() );
	}

	if( !this->graphicalProperties.skyBox.material.empty() ){
		sm->setSkyBox( true, this->graphicalProperties.skyBox.material, this->graphicalProperties.skyBox.distance, this->graphicalProperties.skyBox.drawFirst, this->graphicalProperties.skyBox.orientation.toOgreQuaternion() );
	}

	// set fog
	sm->setFog(this->graphicalProperties.fog.mode, this->graphicalProperties.fog.color.toOgreColourValue(),
		this->graphicalProperties.fog.expDensity, this->graphicalProperties.fog.linearStart, this->graphicalProperties.fog.linearEnd);

	// set world geometry
	if( !this->graphicalProperties.worldGeometrySource.empty() ){
		sm->setWorldGeometry( game->getSystem()->getDataDir() + this->graphicalProperties.worldGeometrySource );
	}



	// do physics
	Physics* physics = game->getPhysics();
	physics->createSpaces(this->physicalProperties.collisionSpace.quadTreeLevels, this->physicalProperties.collisionSpace.min, this->physicalProperties.collisionSpace.max);
	physics->setGravity( this->physicalProperties.gravity );
	physics->setCollideAgainstTerrain( !this->graphicalProperties.worldGeometrySource.empty() );

	// spawn geometry and effects
//	Ogre::SceneNode* rootSceneNode = sm->getRootSceneNode();
	for( objectDescriptorsByName_t::iterator o_iter = this->objects.objectDescriptorsByName.begin(); o_iter != this->objects.objectDescriptorsByName.end(); ++o_iter ){
//		System::log("Arena: Attaching object  '%s'...", o_iter->first.c_str());
		if( o_iter->second.type == GameObject::GAME_OBJECT_TYPE_GEOMETRY ){
			game->spawnGeometry( o_iter->second.templateName, o_iter->second.position, o_iter->second.orientation );
		}else if( o_iter->second.type == GameObject::GAME_OBJECT_TYPE_EFFECT ){
			game->spawnEffect( o_iter->second.templateName, o_iter->second.position, o_iter->second.orientation );
		}else{
		}
	}

	// attach areas
	for( areasByName_t::iterator a_iter = this->areas.areasByName.begin(); a_iter != this->areas.areasByName.end(); ++a_iter ){
		a_iter->second->attach( game );
	}

	this->game = game;

}


void Arena::detach(){
	if( this->game == NULL ){
		throw Exception("Arena is not attached to a game object.", "Arena::detach()");
	}

	// detach areas
	for( areasByName_t::iterator a_iter = this->areas.areasByName.begin(); a_iter != this->areas.areasByName.end(); ++a_iter ){
		if( a_iter->second->isAttached() ){
			a_iter->second->detach();
		}
	}

	// THINKABOUTME: unspawn objects?

	// TODO: detatch ambient effect! (is done in Game::cleanup()!)

	// clearscene is done in Game::cleanup()
	Graphics* graphics = game->getSystem()->getGraphics();
	Ogre::SceneManager* sm = graphics->getSceneManager();
	sm->setSkyBox( false, this->graphicalProperties.skyBox.material );

	this->game = NULL;
}

bool Arena::isAttached() const {
	return (this->game != NULL);
}



void Arena::loadFromFile(const string& arenaFilePath){


	if( !File::exists( arenaFilePath ) ){
		throw Exception("File '" + arenaFilePath + "' does not exist.", "Arena::loadFromFile()");
	}

	TiXmlDocument doc( arenaFilePath.c_str() );
	if( !doc.LoadFile() ){
		throw Exception("Couldn't load file '" + arenaFilePath + "'. (Maybe not a correct xml file?)", "Arena::loadFromFile()");
	}
//	doc.Print();

	TiXmlNode* rootElement = doc.RootElement();
	TiXmlElement* element = rootElement->ToElement();
	string rootElementName = element->Value();
	if( rootElementName != "arena" ){
		throw Exception("File '" + arenaFilePath + "' does not have an <arena> root node.", "Arena::loadFromFile()");
	}


	this->fromXmlElement( element );
}



void Arena::fromXmlElement(TiXmlElement* xmlElement){

	TiXmlElement* element = xmlElement;

	// check version
	const char* tmp = element->Attribute( "formatVersion" );
	if( tmp == NULL ){
		throw Exception("<arena> node has no version information.", "Arena::fromXmlElement()");
	}else{
		string version = tmp;

		if( version != System::getInstance()->getVersionString() ){
			System::warn("(in Arena::fromXmlElement()): <arena> node has version %s. Trying to parse anyway.", version.c_str() );
		}
	}

	// check name
	tmp = element->Attribute( "name" );
	if( tmp == NULL ){
		throw Exception("<arena> node has no name information.", "Arena::fromXmlElement()");
	}else{
		this->name = string( tmp );
	}
	if( this->name.length() > 32 ){
		System::error( "(in Arena::fromXmlElement()): Arena name '%s' is longer than 32 characters.", this->name.c_str() );
		throw Exception( "XmlElement is not a valid arena description.", "Arena::fromXmlElement()" );
	}


	// parse info
	element = xmlElement->FirstChildElement("info");
	if( element ){
		this->parseInfo(element);
	}else{
		throw Exception("<arena> node has no <info> child node.", "Arena::fromXmlElement()");
	}

	// parse graphics
	element = xmlElement->FirstChildElement("graphicalProperties");
	if( element ){
		this->parseGraphicalProperties(element);
	}else{
		throw Exception("<arena> node has no <graphicalProperties> child node.", "Arena::fromXmlElement()");
	}

	// parse physics
	element = xmlElement->FirstChildElement("physicalProperties");
	if( element ){
		this->parsePhysicalProperties(element);
	}else{
		throw Exception("<arena> node has no <physicalProperties> child node.", "Arena::fromXmlElement()");
	}

	// parse objects
	element = xmlElement->FirstChildElement("objects");
	if( element ){
		this->parseObjects(element);
	}else{
		throw Exception("<arena> node has no <objects> child node.", "Arena::fromXmlElement()");
	}

	// parse areas
	element = xmlElement->FirstChildElement("areas");
	if( element ){
		this->parseAreas(element);
	}else{
		throw Exception("<arena> node has no <areas> child node.", "Arena::fromXmlElement()");
	}

}

void Arena::parseInfo(TiXmlElement* infoElement){

	TiXmlElement* elt = infoElement->FirstChildElement("description");
	if( elt != NULL ){
		this->parseDescription(elt);
	}

}

void Arena::parseDescription(TiXmlElement* descriptionElement){

	TiXmlNode* n = descriptionElement->FirstChild();
	if( n != NULL && n->ToText() != NULL ){
		this->info.description = string( n->ToText()->Value() );
	}
}


void Arena::parseGraphicalProperties(TiXmlElement* graphicalPropertiesElement){

	string tmp("generic");
	XmlUtils::readStringProperty( graphicalPropertiesElement, "sceneType", tmp );
	if( tmp == string("generic") ){
		this->graphicalProperties.sceneTypeMask = Ogre::ST_GENERIC;
	}else if( tmp == string("exterior_close") ){
		this->graphicalProperties.sceneTypeMask = Ogre::ST_EXTERIOR_CLOSE;
	}else{
		System::warn( "(in Arena::parseGraphicalProperties()): Unknown scene type '%s'. Using 'generic'.", tmp.c_str() );
	}

	XmlUtils::readStringProperty( graphicalPropertiesElement, "worldGeometrySource", this->graphicalProperties.worldGeometrySource );
	XmlUtils::readFColorProperty( graphicalPropertiesElement, "backgroundColor", this->graphicalProperties.backgroundColor );
	XmlUtils::readFColorProperty( graphicalPropertiesElement, "ambientLight", this->graphicalProperties.ambientLight );
	XmlUtils::readStringProperty( graphicalPropertiesElement, "ambientEffect", this->graphicalProperties.ambientEffect );

//	TiXmlElement* elt = NULL;
	for( TiXmlElement* elt = graphicalPropertiesElement->FirstChildElement(); elt != NULL; elt = elt->NextSiblingElement() ){
		string value = string( elt->Value() );

		if( value == string("fog") ){
			this->parseFog(elt);
		}else if( value == string("shadow") ){
			this->parseShadow(elt);
		}else if( value == string("skyBox") ){
			this->parseSkyBox(elt);
		}else if( value == string("skyPlane") ){
			this->parseSkyPlane(elt);
		}else if( value == string("skyDome") ){
			this->parseSkyDome(elt);
		}else if( value == string("property") ){
			// ignore

		}else{
			System::warn("(in Arena::parseGraphics()): Unknown element: '%s' (row: %i). Ignoring.", value.c_str(), elt->Row() );
		}

	}

}

void Arena::parseShadow(TiXmlElement* shadowElement){
	// check technique
	const char* tmp = shadowElement->Attribute( "technique" );
	if( tmp == NULL ){
		throw Exception("<shadow> node has no technique information.", "Arena::parseShadow()");
	}else{
		string shadowTechnique = string( tmp );

		if( shadowTechnique == string("stencil_additive") ){
			this->graphicalProperties.shadow.technique = Ogre::SHADOWTYPE_STENCIL_ADDITIVE;
		}else if( shadowTechnique == string("stencil_modulative") ){
			this->graphicalProperties.shadow.technique = Ogre::SHADOWTYPE_STENCIL_MODULATIVE;
		}else if( shadowTechnique == string("texture_modulative") ){
			this->graphicalProperties.shadow.technique = Ogre::SHADOWTYPE_TEXTURE_MODULATIVE;
		}else if( shadowTechnique == string("none") ){
			this->graphicalProperties.shadow.technique = Ogre::SHADOWTYPE_NONE;

		}else{
			throw Exception("<shadow> node references unsupported shadowTechnique (" + shadowTechnique + ").", "Arena::parseShadow()");
		}
	}


	this->graphicalProperties.shadow.farDistance = shadowElement->Attribute("farDistance") ? atof( shadowElement->Attribute("farDistance") ) : this->graphicalProperties.shadow.farDistance;
	this->graphicalProperties.shadow.textureSize = shadowElement->Attribute("textureSize") ? atoi( shadowElement->Attribute("textureSize") ) : this->graphicalProperties.shadow.textureSize;

	// color
	TiXmlElement* colorElement = shadowElement->FirstChildElement("color");
	if( colorElement != NULL ){
		this->graphicalProperties.shadow.color = XmlUtils::readFColor(colorElement);
	}

}

void Arena::parseFog(TiXmlElement* fogElement){
	// check mode
	const char* tmp = fogElement->Attribute( "mode" );
	if( tmp == NULL ){
		throw Exception("<fog> node has no mode information.", "Arena::parseFog()");
	}else{
		string fogMode = string( tmp );

		if( fogMode == string("none") ){
			this->graphicalProperties.fog.mode = Ogre::FOG_NONE;
		}else if( fogMode == string("exp") ){
			this->graphicalProperties.fog.mode = Ogre::FOG_EXP;
		}else if( fogMode == string("exp2") ){
			this->graphicalProperties.fog.mode = Ogre::FOG_EXP2;
		}else if( fogMode == string("linear") ){
			this->graphicalProperties.fog.mode = Ogre::FOG_LINEAR;

		}else{
			throw Exception("<fog> node references unsupported mode.", "Arena::parseFog()");
		}
	}


	this->graphicalProperties.fog.expDensity = fogElement->Attribute("expDensity") ? atof( fogElement->Attribute("expDensity") ) : this->graphicalProperties.fog.expDensity;
	this->graphicalProperties.fog.linearStart = fogElement->Attribute("linearStart") ? atof( fogElement->Attribute("linearStart") ) : this->graphicalProperties.fog.linearStart;
	this->graphicalProperties.fog.linearEnd = fogElement->Attribute("linearEnd") ? atof( fogElement->Attribute("linearEnd") ) : this->graphicalProperties.fog.linearEnd;

	// color
	TiXmlElement* colorElement = fogElement->FirstChildElement("color");
	if( colorElement != NULL ){
		this->graphicalProperties.fog.color = XmlUtils::readFColor(colorElement);
	}
}
void Arena::parseSkyBox(TiXmlElement* skyBoxElement){
	// material
	this->graphicalProperties.skyBox.material = skyBoxElement->Attribute("material") ? skyBoxElement->Attribute("material") : this->graphicalProperties.skyBox.material;

	// distance
	this->graphicalProperties.skyBox.distance = skyBoxElement->Attribute("distance") ? atof( skyBoxElement->Attribute("distance") ) : this->graphicalProperties.skyBox.distance;

	// drawFirst
	this->graphicalProperties.skyBox.drawFirst = skyBoxElement->Attribute("drawFirst") ? string( skyBoxElement->Attribute("drawFirst") ) == string("true") : this->graphicalProperties.skyBox.drawFirst;

	// orientation
	TiXmlElement* orientationElement = skyBoxElement->FirstChildElement("orientation");
//	graphics.skyBox.orientation = FQuaternion::IDENTITY;
	if( orientationElement != NULL ){
		this->graphicalProperties.skyBox.orientation = XmlUtils::readFQuaternion(orientationElement);
	}
}
void Arena::parseSkyDome(TiXmlElement* skyDomeElement){
}
void Arena::parseSkyPlane(TiXmlElement* skyPlaneElement){
}



void Arena::parsePhysicalProperties(TiXmlElement* physicalPropertiesElement){

	XmlUtils::readFVector3Property( physicalPropertiesElement, "gravity", this->physicalProperties.gravity );
	XmlUtils::readFVector3Property( physicalPropertiesElement, "wind", this->physicalProperties.wind );

	for( TiXmlElement* elt = physicalPropertiesElement->FirstChildElement(); elt != NULL; elt = elt->NextSiblingElement() ){
		string value = string( elt->Value() );

		if( value == string("collisionSpace") ){
			this->parseCollisionSpace(elt);
		}else if( value == string("property") ){
			// ignore
		}else{
			System::warn("(in Arena::parsePhysicalProperties()): Unknown element: '%s' (row: %i). Ignoring.", value.c_str(), elt->Row() );
		}
	}
}

void Arena::parseCollisionSpace(TiXmlElement* collisionSpaceElement){
	this->physicalProperties.collisionSpace.quadTreeLevels = collisionSpaceElement->Attribute("quadTreeLevels") ? atoi(collisionSpaceElement->Attribute("quadTreeLevels")) : this->physicalProperties.collisionSpace.quadTreeLevels;

	for( TiXmlElement* elt = collisionSpaceElement->FirstChildElement(); elt != NULL; elt = elt->NextSiblingElement() ){
		string value = string( elt->Value() );

		if( value == string("min") ){
			this->physicalProperties.collisionSpace.min = XmlUtils::readFVector3(elt);
		}else if( value == string("max") ){
			this->physicalProperties.collisionSpace.max = XmlUtils::readFVector3(elt);
		}else{
			System::warn("(in Arena::parseCollisionSpace()): Unknown element: '%s' (row: %i). Ignoring.", value.c_str(), elt->Row() );
		}
	}
}





void Arena::parseObjects(TiXmlElement* objectsElement){

	for( TiXmlElement* elt = objectsElement->FirstChildElement(); elt != NULL; elt = elt->NextSiblingElement() ){
		string value = string( elt->Value() );

		try{

			objectDescriptor_s d;

			if( value == "geometry" ){
				d.type = GameObject::GAME_OBJECT_TYPE_GEOMETRY;
			}else if( value == "effect" ){
				d.type = GameObject::GAME_OBJECT_TYPE_EFFECT;
			}else{
				throw Exception("Unknown object type '" + value + "'.", "Arena::parseObjects()");
			}

			const char* tmp = elt->Attribute("name");
			if( tmp != NULL ){
				d.name = string( tmp );
			}else{
				throw Exception("Object has no name.", "Arena::parseObjects()");
			}

			tmp = elt->Attribute("template");
			if( tmp != NULL ){
				d.templateName= string( tmp );
			}else{
				throw Exception("Object '" + d.name + "' has no 'template' attribute.", "Arena::parseObjects()");
			}


			d.position = FVector3::ZERO;
			if( elt->FirstChildElement("position") != NULL ){
				d.position = XmlUtils::readFVector3( elt->FirstChildElement("position") );
			}
			d.orientation = FQuaternion::IDENTITY;
			if( elt->FirstChildElement("orientation") != NULL ){
				d.orientation = XmlUtils::readFQuaternion( elt->FirstChildElement("orientation") );
			}

			objectDescriptorsByName_t::const_iterator c_iter = this->objects.objectDescriptorsByName.find( d.name );
			if( c_iter != this->objects.objectDescriptorsByName.end() ){ // another object with that name is already registered -> throw exception
				throw Exception("An object with name '" + d.name + "' is already registered.", "Arena::parseObjects()");
			}
			this->objects.objectDescriptorsByName.insert( objectDescriptorsByName_t::value_type(d.name, d) );

		}catch( Exception& e ){
			System::warn("(Arena::parseObjects()): An exception occured while parsing object at row %i. Ignoring object.", elt->Row() );
			System::log("Arena: The exception was: %s", e.getDescription().c_str());
		}

	}

}


void Arena::parseAreas(TiXmlElement* areasElement){
	for( TiXmlElement* elt = areasElement->FirstChildElement(); elt != NULL; elt = elt->NextSiblingElement() ){
		string value = string( elt->Value() );

		try{

			Area* a = NULL;

			if(value == "spawnArea"){
				a = new SpawnArea();
			}else if(value == "conquestArea"){
				a = new ConquestArea();
			}else if(value == "regenerationArea"){
				a = new RegenerationArea();
			}else if(value == "upgradeArea"){

			}else if(value == "teleportationArea"){
				a = new TeleportationArea();
			}else{
				System::warn("(Arena::parseAreas()): Unknown are type '%s' at row %i. Ignoring area.", value.c_str(), elt->Row() );
			}

			if( a != NULL ){
				a->fromXmlElement( elt );
				a->calculateChecksum();

				areasByName_t::const_iterator c_iter = this->areas.areasByName.find( a->getName() );
				if( c_iter != this->areas.areasByName.end() ){
					throw Exception("An area with name '" + a->getName() + "' is already registered.", "Arena::parseAreas()");
				}
				this->areas.areasByName.insert( areasByName_t::value_type( a->getName(), a ) );
			}

		}catch( Exception& e ){
			System::warn("(Arena::parseAreas()): An exception occured while parsing area at row %i. Ignoring area.", elt->Row() );
			System::log("Arena: The exception was: %s", e.getDescription().c_str());
		}
	}
}



void Arena::collectAreas( list<Area*>& areas ) const {
	for( areasByName_t::const_iterator a_iter = this->areas.areasByName.begin(); a_iter != this->areas.areasByName.end(); ++a_iter ){
		areas.push_back( a_iter->second );
	}
}


const string& Arena::getName() const {
	return this->name;
}
const string& Arena::getDescription() const {
	return this->info.description;
}

Game* Arena::getGame() const {
	return this->game;
}

void Arena::calculateChecksum(){
	this->checksum = 0;
}


