#include "ArenaFactory.h"

#include "System.h"
#include "Exception.h"
#include "Arena.h"

ArenaFactory::ArenaFactory(Game* game){
	this->game = game;

	this->arenasByName.clear();
}

ArenaFactory::~ArenaFactory(){
	// delete arenas
	for( arenasByName_t::iterator iter = arenasByName.begin(); iter != arenasByName.end(); iter++ ){
		delete iter->second;
	}

}

void ArenaFactory::addArena(const string& arenaFilePath){

	try{
//		System::log( "ArenaFactory: Parsing arena '%s'...", arenaFilePath.c_str() );

		Arena* a = new Arena();
		a->loadFromFile(arenaFilePath);

		arenasByName_t::const_iterator c_iter = arenasByName.find( a->getName() );
		if( c_iter != arenasByName.end() ){ // another arena with that name is already registered -> throw exception
			throw Exception("An arena with name '" + a->getName() +"' is already registered.", "ArenaFactory::addArena()");
		}

		// calculate checksum
		a->calculateChecksum();

		// insert the new arena into the list
		arenasByName. insert( arenasByName_t::value_type(a->getName(), a) );
		System::log("ArenaFactory: Added arena '%s' to list.", a->getName().c_str());
	}catch( Exception& e ){
		System::warn("(ArenaFactory::addArena()): An exception occured while parsing arena file '%s'. Ignoring arena.", arenaFilePath.c_str() );
		System::log("ArenaFactory: The exception was: %s", e.getDescription().c_str());
	}
//	printf("\n\n##JO!\n");

}

void ArenaFactory::removeArena(const string& name){
	arenasByName_t::iterator iter = arenasByName.find( name );
	if( iter == arenasByName.end() ){
		throw Exception("An arena with name '" + name +"' is not registered.", "ArenaFactory::removeArena()");
	}
	this->arenasByName.erase(iter);

}

Arena* ArenaFactory::createArena(const string& name) const {

	arenasByName_t::const_iterator c_iter = arenasByName.find( name );
	if( c_iter == arenasByName.end() ){
		throw Exception("An arena with name '" + name +"' is not registered.", "ArenaFactory::createArena()");
	}

	Arena* ret = c_iter->second->clone();

	return ret;
}

void ArenaFactory::destroyArena(Arena* arena) const {
	delete arena;
}


vector<string> ArenaFactory::getArenaNames() const {
	vector<string> ret;

	for( arenasByName_t::const_iterator c_iter = this->arenasByName.begin(); c_iter != this->arenasByName.end(); c_iter++ ){
		ret.push_back( c_iter->first );
	}

	return ret;
}

const string& ArenaFactory::getArenaDescription( const string& name ) const {
	arenasByName_t::const_iterator c_iter = arenasByName.find( name );
	if( c_iter == arenasByName.end() ){
		throw Exception("An arena with name '" + name +"' is not registered.", "ArenaFactory::getArenaDescription()");
	}

    return c_iter->second->getDescription();
}
