#include "CVar.h"

#include <limits.h>
#include <float.h>
#include <stdio.h>


CVar::CVar(const string& name){
	this->name = name;
	this->infoString = "";
	this->changeString = "";
	this->flags = FLAG_NONE;
	this->instantUpdate = false;
}

CVar::~CVar(){
}

string CVar::getTypeString() const {
	switch( this->type ){
	case TYPE_INT: return "integer";
	case TYPE_BOOL: return "boolean";
	case TYPE_STRING: return "string";
	case TYPE_REAL: return "real";
	case TYPE_REAL_VECTOR: return "real vector";
	default: return "unknown";
	};
}

string CVar::getFlagsString() const {
	return string("jo");
}

const string& CVar::getName() const {
	return this->name;
}
const string& CVar::getInfoString() const {
	return this->infoString;
}
void CVar::setChangeString(const string& newChangeString){
	this->changeString = newChangeString;
}
const string& CVar::getChangeString() const {
	return this->changeString;
}

void CVar::setFlags(unsigned int newFlags){
	this->flags = newFlags;
}
unsigned int CVar::getFlags() const {
	return this->flags;
}





CVarBool::CVarBool(const string& name, bool value): CVar(name){

	this->type = TYPE_BOOL;
	this->referencedVariable = NULL;
	this->value = value;
	this->minValue = 0;
	this->maxValue = 1;

}


CVarBool::CVarBool(const string& name, bool* referencedVariable, bool instantUpdate): CVar(name){
	this->type = TYPE_BOOL;
	this->referencedVariable = referencedVariable;
	this->value = *referencedVariable;
	this->instantUpdate = instantUpdate;
	this->minValue = 0;
	this->maxValue = 1;
}

CVarBool::~CVarBool(){
}

bool CVarBool::isValueStringValid(const string& valueString) const{
	const char* cStr = valueString.c_str();
	int tmp;
	int check = sscanf(cStr, "%i", &tmp);

	return !( check == 0  || check == EOF || tmp < (int)minValue || tmp > (int)maxValue );
}

void CVarBool::setValueString(const string& valueString){
	const char* cStr = valueString.c_str();
	int tmp;
	int check = sscanf(cStr, "%i", &tmp);

	if( check == 0  || check == EOF || tmp < (int)minValue || tmp > (int)maxValue )
		return;

	value = (tmp != 0);		// set local var

	if( instantUpdate ) 	// and the real variable if instantUpdate is true
		updateReferencedVariable();

}

string CVarBool::getValueString() const {
	char buff[256];
	sprintf(buff, "%i", value);

	return string(buff);
}

string CVarBool::getReferencedVariableValueString() const {
	if( referencedVariable == NULL ){
		return this->getValueString();
	}else{
		char buff[256];
		sprintf(buff, "%i", *referencedVariable);

		return string(buff);
	}

}

void CVarBool::toggle(){
	value = !value;
	if( instantUpdate )
		updateReferencedVariable();

}

void CVarBool::updateReferencedVariable(){
	if( referencedVariable != NULL)
		(*referencedVariable) = value;
}

void CVarBool::setValueRange(bool minValue, bool maxValue){
	this->minValue = minValue;
	this->maxValue = maxValue;
}

string CVarBool::getValueRangeString() const{
	char buff[256];
	sprintf(buff, "%i - %i", minValue, maxValue);

	return string(buff);
}

void CVarBool::setValue(bool newValue){
	this->value = newValue;

	if( instantUpdate )
		updateReferencedVariable();
}

bool CVarBool::getValue() const {
	return this->value;
}





CVarInt::CVarInt(const string& name, int value): CVar(name){

	this->type = TYPE_INT;
	this->referencedVariable = NULL;
	this->value = value;
	this->minValue = INT_MIN;
	this->maxValue = INT_MAX;

}


CVarInt::CVarInt(const string& name, int* referencedVariable, bool instantUpdate): CVar(name){
	this->type = TYPE_INT;
	this->referencedVariable = referencedVariable;
	this->value = *referencedVariable;
	this->instantUpdate = instantUpdate;
	this->minValue = INT_MIN;
	this->maxValue = INT_MAX;
}

CVarInt::~CVarInt(){
}

bool CVarInt::isValueStringValid(const string& valueString) const{
	const char* cStr = valueString.c_str();
	int tmp;
	int check = sscanf(cStr, "%i", &tmp);

	return !( check == 0  || check == EOF || tmp < minValue || tmp > maxValue );
}

void CVarInt::setValueString(const string& valueString){
	const char* cStr = valueString.c_str();
	int tmp;
	int check = sscanf(cStr, "%i", &tmp);

	if( check == 0  || check == EOF || tmp < minValue || tmp > maxValue )
		return;

	value = tmp;		// set local var

	if( instantUpdate ) 	// and the real variable if instantUpdate is true
		updateReferencedVariable();

}

string CVarInt::getValueString() const {
	char buff[256];
	sprintf(buff, "%i", value);

	return string(buff);
}

string CVarInt::getReferencedVariableValueString() const {
	if( referencedVariable == NULL ){
		return this->getValueString();
	}else{
		char buff[256];
		sprintf(buff, "%i", *referencedVariable);

		return string(buff);
	}

}

void CVarInt::toggle(){
	value = value + 1;
	if( value > maxValue){
		value = minValue;
	}

	if( instantUpdate )
		updateReferencedVariable();

}

void CVarInt::updateReferencedVariable(){
	if( referencedVariable != NULL)
		(*referencedVariable) = value;
}

void CVarInt::setValueRange(int minValue, int maxValue){
	this->minValue = minValue;
	this->maxValue = maxValue;
}

string CVarInt::getValueRangeString() const{
	char buff[256];
	sprintf(buff, "%i - %i", minValue, maxValue);

	return string(buff);
}

void CVarInt::setValue(int newValue){
	this->value = newValue;

	if( instantUpdate )
		updateReferencedVariable();
}
int CVarInt::getValue() const {
	return this->value;
}






CVarReal::CVarReal(const string& name, float value): CVar(name){

	this->type = TYPE_REAL;
	this->referencedVariable = NULL;
	this->value = value;
	this->minValue = FLT_MIN;
	this->maxValue = FLT_MAX;

}


CVarReal::CVarReal(const string& name, float* referencedVariable, bool instantUpdate): CVar(name){
	this->type = TYPE_REAL;
	this->referencedVariable = referencedVariable;
	this->value = *referencedVariable;
	this->instantUpdate = instantUpdate;
	this->minValue = FLT_MIN;
	this->maxValue = FLT_MAX;
}
CVarReal::~CVarReal(){
}


bool CVarReal::isValueStringValid(const string& valueString) const{
	const char* cStr = valueString.c_str();
	float tmp;
	int check = sscanf(cStr, "%f", &tmp);

	return !( check == 0  || check == EOF || tmp < minValue || tmp > maxValue );
}

void CVarReal::setValueString(const string& valueString){
	const char* cStr = valueString.c_str();
	float tmp;
	int check = sscanf(cStr, "%f", &tmp);

	if( check == 0  || check == EOF || tmp < minValue || tmp > maxValue )
		return;

	value = tmp;		// set local var

	if( instantUpdate ) 	// and the real variable if instantUpdate is true
		updateReferencedVariable();

}

string CVarReal::getValueString() const {
	char buff[256];
	sprintf(buff, "%f", value);

	return string(buff);
}

string CVarReal::getReferencedVariableValueString() const {
	if( referencedVariable == NULL ){
		return this->getValueString();
	}else{
		char buff[256];
		sprintf(buff, "%f", *referencedVariable);

		return string(buff);
	}

}

void CVarReal::toggle(){
	value = value + 1.0f;
	if( value > maxValue){
		value = minValue;
	}

	if( instantUpdate )
		updateReferencedVariable();

}

void CVarReal::updateReferencedVariable(){
	if( referencedVariable != NULL)
		(*referencedVariable) = value;
}

void CVarReal::setValueRange(float minValue, float maxValue){
	this->minValue = minValue;
	this->maxValue = maxValue;
}

string CVarReal::getValueRangeString() const {
	char buff[256];
	sprintf(buff, "%f - %f", minValue, maxValue);

	return string(buff);
}

void CVarReal::setValue(float newValue){
	this->value = newValue;

	if( instantUpdate )
		updateReferencedVariable();
}

float CVarReal::getValue() const {
	return this->value;
}








CVarString::CVarString(const string& name, const string& value): CVar(name){

	this->type = TYPE_STRING;
	this->referencedVariable = NULL;
	this->value = value;
	this->minLength = 0;
	this->maxLength = INT_MAX;

}


CVarString::CVarString(const string& name, string* referencedVariable, bool instantUpdate): CVar(name){
	this->type = TYPE_REAL;
	this->referencedVariable = referencedVariable;
	this->value = *referencedVariable;
	this->instantUpdate = instantUpdate;
	this->minLength = 0;
	this->maxLength = INT_MAX;
}
CVarString::~CVarString(){
}


bool CVarString::isValueStringValid(const string& valueString) const{
	return ( valueString.length() >= (unsigned int)minLength && valueString.length() <= (unsigned int)maxLength);
}

void CVarString::setValueString(const string& valueString){
	if( !this->isValueStringValid(valueString) )
		return;

	value = valueString;		// set local var

	if( instantUpdate ) 	// and the real variable if instantUpdate is true
		updateReferencedVariable();

}

string CVarString::getValueString() const {
	return this->value;
}

string CVarString::getReferencedVariableValueString() const {
	if( referencedVariable == NULL ){
		return this->value;
	}else{
		return (*referencedVariable);

	}

}

void CVarString::toggle(){
}

void CVarString::updateReferencedVariable(){
	if( referencedVariable != NULL)
		(*referencedVariable) = value;
}

void CVarString::setLengthRange(int minLength, int maxLength){
	this->minLength = minLength;
	this->maxLength = maxLength;
}

string CVarString::getValueRangeString() const {	// this is actually getLengthRangeString()
	char buff[256];
	sprintf(buff, "%i - %i", minLength, maxLength);

	return string(buff);
}

void CVarString::setValue(const string& newValue){
	this->value = newValue;

	if( instantUpdate )
		updateReferencedVariable();
}

const string& CVarString::getValue() const {
	return this->value;
}

