#include "ConquestArea.h"

#include "System.h"
#include "Exception.h"
#include "XmlUtils.h"
#include "Game.h"
#include "Vehicle.h"
#include "Graphics.h"
#include "Gui.h"
#include "Hud.h"
#include "Network.h"

const unsigned long ConquestArea::CAPTURE_INTERVAL = 1000;

ConquestArea::ConquestArea(): Area( Area::AREA_TYPE_CONQUEST ) {
	this->teamFlags = 0;

	this->capturePercentage = 0.0f;
	this->captureState = CAPTURE_STATE_NEUTRAL;

	this->particleSystem = NULL;
}

ConquestArea::~ConquestArea(){
}


Area* ConquestArea::clone() const {
	ConquestArea* ret = new ConquestArea();

	*ret = *this;

	return ret;
}




void ConquestArea::fromXmlElement( TiXmlElement* xmlElement ){
	// name
	const char* tmp = xmlElement->Attribute("name");
	if( tmp != NULL ){
		this->name = string( tmp );
	}else{
		throw Exception( "ConquestArea element has no name attribute.", "ConquestArea::fromXmlElement()" );
	}
	if( this->name.length() > 32 ){
		System::error( "(in ConquestArea::fromXmlElement()): ConquestArea name '%s' is longer than 32 characters.", this->name.c_str() );
		throw Exception( "XmlElement is not a valid conquest area description.", "ConquestArea::fromXmlElement()" );
	}

	// teamFlags
	tmp = xmlElement->Attribute("teamFlags");
	if( tmp != NULL ){
		string flags = string( tmp );
		if( flags.find("s") != string::npos ){
			this->teamFlags |= Game::TEAM_FLAG_SPECTATORS;
		}
		if( flags.find("p") != string::npos ){
			this->teamFlags |= Game::TEAM_FLAG_PLAYERS;
		}
		if( flags.find("r") != string::npos ){
			this->teamFlags |= Game::TEAM_FLAG_RED;
		}
		if( flags.find("b") != string::npos ){
			this->teamFlags |= Game::TEAM_FLAG_BLUE;
		}
	}else{
		throw Exception( "ConquestArea element has no teamFlags attribute.", "ConquestArea::fromXmlElement()" );
	}



	for( TiXmlElement* elt = xmlElement->FirstChildElement(); elt != NULL; elt = elt->NextSiblingElement() ){
		string value = string( elt->Value() );
		
		if( value == "edgeLengths" ){
			this->edgeLengths = XmlUtils::readFVector3( elt );

		}else if( value == "position" ){
			this->position = XmlUtils::readFVector3( elt );
		}else if( value == "orientation" ){
			this->orientation = XmlUtils::readFQuaternion( elt );


		}else{
			System::warn("(in ConquestArea::fromXmlElement()): Unknown child element: '%s' (row: %i). Ignoring.", value.c_str(), elt->Row() );
		}
	}


}

TiXmlElement ConquestArea::toXmlElement() const {
	return TiXmlElement("dummy");
}

void ConquestArea::calculateChecksum(){
	this->checksum = 0;
}



void ConquestArea::attach( Game* game ){

	if( this->game != NULL ){
		throw Exception("ConquestArea is already attached to a game.", "ConquestArea::attach()");
	}

	Ogre::SceneManager* sm = game->getSystem()->getGraphics()->getSceneManager();
	Ogre::SceneNode* rootNode = sm->getRootSceneNode();

	this->sceneNode = rootNode->createChildSceneNode( this->name );
	this->sceneNode->setPosition( this->position.toOgreVector3() );
	this->sceneNode->setOrientation( this->orientation.toOgreQuaternion() );
//	this->ogreSceneNode->setScale( this->edgeLengths.toOgreVector3() );
/*
	Ogre::ParticleSystemManager& psm = Ogre::ParticleSystemManager::getSingleton();
	this->particleSystem = Ogre::ParticleSystemManager::getSingleton().createSystem( this->name + " particle system", "are/regeneration_area_sparkle" );
	Ogre::ParticleEmitter* e = this->particleSystem->getEmitter(0);
	e->setParameter("position", Ogre::StringConverter::toString(Ogre::Vector3(0.0, -this->edgeLengths.y*0.5f, 0.0)));
	e->setParameter("width", Ogre::StringConverter::toString(Ogre::Real(this->edgeLengths.x)));
	e->setParameter("height", Ogre::StringConverter::toString(Ogre::Real(this->edgeLengths.z)));
	e->setParameter("depth", Ogre::StringConverter::toString(Ogre::Real(1.0)));
	e->setParameter("time_to_live", Ogre::StringConverter::toString(Ogre::Real(this->edgeLengths.y*0.1)));
	e->setParameter("emission_rate", Ogre::StringConverter::toString(Ogre::Real(this->edgeLengths.x * this->edgeLengths.y * 0.02)));
	this->sceneNode->attachObject( this->particleSystem );
*/
	this->game = game;
}

void ConquestArea::detach(){
	if( this->game == NULL ){
		throw Exception("RegenerationArea is not attached to a game.", "RegenerationArea::detach()");
	}
/*
	Ogre::ParticleSystemManager& psm = Ogre::ParticleSystemManager::getSingleton();
	psm.destroySystem( this->particleSystem );
*/	
	Ogre::SceneManager* sm = this->game->getSystem()->getGraphics()->getSceneManager();
	this->sceneNode->getParentSceneNode()->removeAndDestroyChild( this->sceneNode->getName() );
	this->sceneNode = NULL;

	this->game = NULL;
}

bool ConquestArea::canAffectVehicle( Vehicle* vehicle ) const {
//	unsigned long currentMillis = this->game->getSystem()->getTimer()->getMilliseconds();

	if( this->game->getMode() != Game::MODE_CONQUEST ){
		return false;
	}

	FVector3 doubleDiff = (vehicle->position - this->position) * 2.0f;
	return ( fabs(doubleDiff.x) < this->edgeLengths.x 
			&& fabs(doubleDiff.y) < this->edgeLengths.y 
			&& fabs(doubleDiff.z) < this->edgeLengths.z );
}

void ConquestArea::affectVehicle( Vehicle* vehicle ){
	float deltaT = this->game->getSystem()->getTimer()->getDeltaT();
	Client* c = vehicle->getClient();
	int team = c->info.team;


	// set center of screen message
	if( fabs( this->capturePercentage ) < 100.0f
		&& c == this->game->getSystem()->getNetwork()->getClient()
		){
		this->game->getSystem()->getGui()->getHud()->setCenterOfScreenMessage(
			"Capturing " + this->name + "...", 1000 );
	}


	// capture
	if( team == Game::TEAM_RED ){
		this->capturePercentage -= deltaT * 5.0f;
		this->capturePercentage = this->capturePercentage >= -100.0f ? this->capturePercentage : -100.0f;

		if( this->game->isArenaOwnedBy( Game::TEAM_RED ) ){
			this->game->getSystem()->getGui()->getHud()->setCenterOfScreenMessage(
				"Red team owns the arena!" );
		}

	}else if( team == Game::TEAM_BLUE ){
		this->capturePercentage += deltaT * 5.0f;
		this->capturePercentage = this->capturePercentage <= 100.0f ? this->capturePercentage : 100.0f;

		if( this->game->isArenaOwnedBy( Game::TEAM_BLUE ) ){
			this->game->getSystem()->getGui()->getHud()->setCenterOfScreenMessage(
				"Blue team owns the arena!" );
		}

	}else{
	}

}

float ConquestArea::getCapturePercentage() const {
	return this->capturePercentage;
}

ConquestArea::captureStates_e ConquestArea::getCaptureState() const {
	return this->captureState;
}
