#include "ConsoleCCmds.h"

#include "System.h"
#include "File.h"
#include "Console.h"
#include "CVar.h"

CCmdRegisterCVar::CCmdRegisterCVar(): CCmd("registercvar"){
	this->usageString = "registercvar [cvar] <initial value>";
	this->infoString = "registers a new cvar";
}
CCmdRegisterCVar::~CCmdRegisterCVar(){
}

void CCmdRegisterCVar::exec( const vector<string>& arguments ){
	Console& console = System::getInstance()->getConsoleRef();

	if( arguments.size() == 1 ){
		if( console.getCVar( arguments[0] ) != NULL ){
			System::log("registercvar: A cvar with name '%s' is already registered.", arguments[0].c_str() );
			return;
		}
		CVarString* cVar = new CVarString( arguments[0], "<empty>" );
		cVar->setFlags( CVar::FLAG_USER_DEFINED );
		console.registerCVar(cVar);
		System::log("registercvar: cvar '%s' registered and set to '%s'.", cVar->getName().c_str(), cVar->getValueString().c_str() );
	}else if( arguments.size() == 2 ){
		if( console.getCVar( arguments[0] ) != NULL ){
			System::log("registercvar: A cvar with name '%s' is already registered.", arguments[0].c_str() );
			return;
		}
		CVarString* cVar = new CVarString( arguments[0], arguments[1] );
		cVar->setFlags( CVar::FLAG_USER_DEFINED );
		console.registerCVar(cVar);
		System::log("registercvar: cvar '%s' registered and set to '%s'.", cVar->getName().c_str(), cVar->getValueString().c_str() );
	}else{
		System::log("registercvar: usage: %s", usageString.c_str());
		return;
	}
}

CCmdUnregisterCVar::CCmdUnregisterCVar(): CCmd("unregistercvar"){
	usageString = "unregistercvar [variable]";
	infoString = "unregisters a variable";
}
CCmdUnregisterCVar::~CCmdUnregisterCVar(){
}

void CCmdUnregisterCVar::exec( const vector<string>& arguments ){
	Console& console = System::getInstance()->getConsoleRef();

	if( arguments.size() == 1 ){
		CVar* cVar = console.getCVar( arguments[0] );
		if( cVar == NULL ){
			System::log("unregistercvar: No cvar with name '%s' registered.", arguments[0].c_str());
			return;
		}else{
			if( (cVar->getFlags() & CVar::FLAG_USER_DEFINED) == CVar::FLAG_USER_DEFINED ){
				console.unregisterCVar(cVar);
				System::log("unregistercvar: cvar '%s' unregistered.", cVar->getName().c_str() );
				return;
			}else{
				System::log("unregistercvar: '%s' is not an user-defined cvar.", cVar->getName().c_str());
				return;
			}
		}
	}else{
		System::log("unregistercvar: usage: %s", usageString.c_str());
		return;
	}
}


CCmdSet::CCmdSet():CCmd("set"){
	usageString = "set [variable] [value]";
	infoString = "sets a variable to a given value";
}
CCmdSet::~CCmdSet(){
}

void CCmdSet::exec( const vector<string>& arguments ){
	Console& console = System::getInstance()->getConsoleRef();

	if( arguments.size() == 2 ){
		CVar* cVar = console.getCVar(arguments[0]);
		if(cVar == NULL){
			System::log("set: No cvar with name '%s' found.", arguments[0].c_str());
			return;
		}else{
			if( (cVar->getFlags() & CVar::FLAG_READ_ONLY) != CVar::FLAG_READ_ONLY ){
				if( cVar->isValueStringValid(arguments[1]) ){
					cVar->setValueString( arguments[1] );
					System::log("set: '%s' set to '%s' %s", cVar->getName().c_str(), cVar->getValueString().c_str(), cVar->getChangeString().c_str() );
					return;
				}else{
					System::log("set: '%s' is not a valid value for '%s'", arguments[1].c_str(), cVar->getName().c_str() );
					return;
				}
			}else{
				System::log("set: '%s' is read-only.", cVar->getName().c_str() );
				return;
			}
		}
	}else{
		System::log("set: usage: %s", usageString.c_str() );
		return;
	}
}


CCmdVal::CCmdVal():CCmd("val"){
	usageString = "val [variable]";
	infoString = "displays the current value of a variable";
}
CCmdVal::~CCmdVal(){
}
void CCmdVal::exec( const vector<string>& arguments ){
	Console& console = System::getInstance()->getConsoleRef();

	if( arguments.size() == 1 ){
		CVar* cVar = console.getCVar( arguments[0] );
		if( cVar != NULL ){
			System::log("val: value of '%s': '%s' (internal value: '%s')"
				, cVar->getName().c_str(), cVar->getValueString().c_str(), cVar->getReferencedVariableValueString().c_str() );
			return;
		}else{
			System::log("val: No cvar with name '%s' found.", arguments[0].c_str());
			return;
		}
	}else{
		System::log("val: usage: %s", usageString.c_str() );
		return;
	}
}


CCmdHelp::CCmdHelp():CCmd("help"){
	usageString = "help";
	infoString = "displays a short help message on how to use the console";
}
CCmdHelp::~CCmdHelp(){
}
void CCmdHelp::exec( const vector<string>& arguments ){
	System::log("This is the FWP command console.");
	System::log("Type 'info [command/variable]' to get information on a special command/variable.");
	System::log("The commands 'ccmdlist' and 'cvarlist' list all console commands/variables.");
	return;
}

CCmdInfo::CCmdInfo():CCmd("info"){
	usageString = "info [variable/command]";
	infoString = "displays some information about a variable or a command";
}
CCmdInfo::~CCmdInfo(){
}
void CCmdInfo::exec( const vector<string>& arguments ){
	Console& console = System::getInstance()->getConsoleRef();

	if( arguments.size() == 1 ){
		CCmd* cCmd = console.getCCmd(arguments[0]);
		if( cCmd != NULL ){
			System::log("info: %s: %s", cCmd->getName().c_str(), cCmd->getInfoString().c_str() );
			System::log("info: usage: %s",cCmd->getUsageString().c_str() );
			System::log("info: flags: %s",cCmd->getFlagsString().c_str() );
			return;
		}else{
			CVar* cVar = console.getCVar(arguments[0]);
			if(cVar!=NULL){
				System::log("info: %s: %s", cVar->getName().c_str(), cVar->getInfoString().c_str() );
				System::log("info: type: %s | value: %s (internal value: %s) | valRange: %s"
					, cVar->getTypeString().c_str(), cVar->getValueString().c_str(), cVar->getReferencedVariableValueString().c_str(), cVar->getValueRangeString().c_str() );
				System::log("info: flags: %s", cVar->getFlagsString().c_str() );
					return;
			}else{
				System::log("info: No cvar/ccmd with name '%s' found.", arguments[0].c_str() );
				return;
			}
		}
	}else{
		System::log("info: usage: %s", usageString.c_str() );
		return;
	}
}



CCmdToggle::CCmdToggle():CCmd("toggle"){
	usageString = "toggle [variable]";
	infoString = "cycles through all valid values for a variable";
}
CCmdToggle::~CCmdToggle(){
}
void CCmdToggle::exec( const vector<string>& arguments ){
	Console& console = System::getInstance()->getConsoleRef();

	if( arguments.size() == 1 ){
		CVar* cVar = console.getCVar( arguments[0] );
		if(cVar == NULL){
			System::log("toggle: No cvar with name '%s' found.",arguments[0].c_str());
			return;
		}else{
			if( (cVar->getFlags() & CVar::FLAG_READ_ONLY) != CVar::FLAG_READ_ONLY ){
				cVar->toggle();
				System::log("toggle: '%s' set to '%s' %s", cVar->getName().c_str(), cVar->getValueString().c_str(), cVar->getChangeString().c_str() );
				return;
			}else{
				System::log("toggle: '%s' is read-only.", cVar->getName().c_str() );
				return;
			}
		}
	}else{
		System::log("toggle: usage: %s", usageString.c_str());
		return;
	}
}


CCmdExec::CCmdExec():CCmd("exec"){
	usageString = "exec [cvarname/filename]";
	infoString = "executes the commands stored in a cvar or in a file";
}
CCmdExec::~CCmdExec(){
}
void CCmdExec::exec( const vector<string>& arguments ){
	Console& console = System::getInstance()->getConsoleRef();

	if( arguments.size() == 1 ){
		CVar* cVar = console.getCVar( arguments[0] );
		if( cVar != NULL ){
			System::log("exec: executing cvar '%s'", cVar->getName().c_str() );
			console.parse( cVar->getValueString() );
			System::log("exec: finished execution of cvar '%s'", cVar->getName().c_str() );
			return;
		}else{	// search for a file
			File f(arguments[0], ios::in);
			if( f.isOpen() ){
				System::log("exec: executing file '%s'", f.getPath().c_str() );
				string line;
				while( f.readLine( line, "#" ) ){
					console.parse( line );
				}	//while
				f.close();
				System::log("exec: finished execution of file '%s'", f.getPath().c_str() );
				return;
			}else{
				System::log("no cvar or file with name '%s' found.", arguments[0].c_str() );
				return;
			}
		}// if cVar!=NULL
	}else{
		System::log("exec: usage: %s", usageString.c_str());
		return;
	}
}

CCmdCVarList::CCmdCVarList():CCmd("cvarlist"){
	usageString = "cvarlist <pattern>";
	infoString = "lists all console-variables";
}
CCmdCVarList::~CCmdCVarList(){
}
void CCmdCVarList::exec( const vector<string>& arguments ){
//	Console& console = System::getInstance().getConsole();

	if( arguments.size() == 0 ){
		System::log("cvarlist: Listing not yet implemented.");
		return;
	}else if( arguments.size() == 1 ){// pattern search
		System::log("cvarlist: Pattern search not yet implemented.");
		return;
	}else{
		System::log("cvarlist: usage: %s", usageString.c_str() );
		return;
	}
}


CCmdCCmdList::CCmdCCmdList():CCmd("ccmdlist"){
	usageString = "ccmdlist <pattern>";
	infoString = "lists all console-commands";
}
CCmdCCmdList::~CCmdCCmdList(){
}
void CCmdCCmdList::exec( const vector<string>& arguments ){
//	Console& console = System::getInstance().getConsole();

	if( arguments.size() == 0 ){
		System::log("ccmdlist: Listing not yet implemented.");
		return;
	}else if( arguments.size() == 1 ){// pattern search
		System::log("ccmdlist: Pattern search not yet implemented.");
		return;
	}else{
		System::log("ccmdlist: usage: %s", usageString.c_str() );
		return;
	}
}

CCmdEcho::CCmdEcho():CCmd("echo"){
	usageString = "echo [string]";
	infoString = "prints a string to the console window";
}
CCmdEcho::~CCmdEcho(){
}
void CCmdEcho::exec( const vector<string>& arguments ){
//	Console& console = System::getInstance().getConsole();

	if( arguments.size() == 1 ){
		System::log( arguments[0].c_str() );
		return;
	}else{
		System::log("echo: usage: %s", usageString.c_str() );
		return;
	}
}















