#include "Drone.h"

#include "System.h"
#include "Weapon.h"
#include "Game.h"
#include "Effect.h"

Drone::Drone(): Vehicle(Vehicle::VEHICLE_TYPE_DRONE) {
//	this->propulsionUnitPitch = 0.0f;
}

Drone::~Drone(){
}

Vehicle* Drone::clone() const {
	Drone* ret = new Drone();
	*ret = *this;

	return ret;
}

void Drone::reset(){
	this->health = this->physicalProperties.healthCapacity;
	this->energy = this->physicalProperties.energyCapacity;
}

void Drone::takeOverClientState( const Client::state_s& clientState ){
	this->health = clientState.health;
	this->energy = clientState.energy;

	this->position = clientState.position;
	this->orientation = clientState.orientation;
	this->velocity = clientState.velocity;
	this->angularVelocity = clientState.angularVelocity;

//	this->mainUnitYaw = *( (float*)(&clientState.vehicleSpecificBytes[0]) );
//	this->mainUnitPitch = *( (float*)(&clientState.vehicleSpecificBytes[4]) );
	this->boostersActive = clientState.vehicleSpecificBytes[5] != 0;
}

void Drone::fillInClientState( Client::state_s& clientState ) const {
	clientState.health = (int)this->health;
	clientState.energy = (int)this->energy;

	clientState.position = this->position;
	clientState.orientation = this->orientation;
	clientState.velocity = this->velocity;
	clientState.angularVelocity = this->angularVelocity;

//	memcpy( &clientState.vehicleSpecificBytes[0], &this->mainUnitYaw, 4 );
//	memcpy( &clientState.vehicleSpecificBytes[4], &this->mainUnitPitch, 4 );
	clientState.vehicleSpecificBytes[5] = this->boostersActive ? 1 : 0;
}


void Drone::processInput(Input::actionMap_t& actionMap, Input::mouseState_t& mouseState){

	FVector3 linearInput = FVector3::ZERO;
	FVector3 angularInput = FVector3::ZERO;
	this->boostersActive = false;
	this->recoveryActive = false;

	// keys
	if( actionMap[Input::AC_TURN_LEFT].triggered ){
			angularInput.y += 1.0f;
			angularInput.z += 0.5f;
	}
	if( actionMap[Input::AC_TURN_RIGHT].triggered ){
			angularInput.y -= 1.0f;
			angularInput.z -= 0.5f;
	}
	if( actionMap[Input::AC_TURN_UP].triggered ){
		angularInput.x -= 1.0f;
	}
	if( actionMap[Input::AC_TURN_DOWN].triggered ){
		angularInput.x += 1.0f;
	}

	if( actionMap[Input::AC_MOVE_LEFT].triggered ){
		linearInput -= FVector3::UNIT_X;
	}
	if( actionMap[Input::AC_MOVE_RIGHT].triggered ){
		linearInput += FVector3::UNIT_X;
	}
	if( actionMap[Input::AC_MOVE_UP].triggered ){
		linearInput += FVector3::UNIT_Y;
	}
	if( actionMap[Input::AC_MOVE_DOWN].triggered ){
		linearInput -= FVector3::UNIT_Y;
	}
	if( actionMap[Input::AC_MOVE_FORWARD].triggered ){
		linearInput -= FVector3::UNIT_Z;
	}
	if( actionMap[Input::AC_MOVE_BACKWARD].triggered ){
		linearInput += FVector3::UNIT_Z;
	}
	if( actionMap[Input::AC_ROLL_LEFT].triggered ){
		angularInput.z += 1.0f;
	}
	if( actionMap[Input::AC_ROLL_RIGHT].triggered ){
		angularInput.z -= 1.0f;
	}

	if( actionMap[Input::AC_FIRE1].triggered ){
		if( this->weapons[0] != NULL && this->weapons[0]->isReady() ){
			this->weapons[0]->fire();
		}
	}
	if( actionMap[Input::AC_FIRE2].triggered ){
		if( this->weapons[1] != NULL && this->weapons[1]->isReady() ){
			this->weapons[1]->fire();
		}
	}
	if( actionMap[Input::AC_FIRE3].triggered ){
		if( this->weapons[2] != NULL && this->weapons[2]->isReady() ){
			this->weapons[2]->fire();
		}
	}
	if( actionMap[Input::AC_FIRE4].triggered ){
		if( this->weapons[3] != NULL && this->weapons[3]->isReady() ){
			this->weapons[3]->fire();
		}
	}

	if( actionMap[Input::AC_ACTIVATE_BOOSTERS].triggered && this->energy > 1.0f ){
		this->boostersActive = true;
	}

	if( actionMap[Input::AC_RECOVER].triggered ){
		this->recoveryActive = true;
	}

	// mouse
	angularInput.y -= mouseState.xRel * 20.0f * mouseState.sensitivity;
	angularInput.z -= mouseState.xRel * 10.0f * mouseState.sensitivity;

	angularInput.x += mouseState.yRel * 20.0f * mouseState.sensitivity;



	linearInput.normalize();
	this->propulsiveForce = linearInput * this->physicalProperties.propulsiveForce;


	angularInput.normalize();
	this->propulsiveTorque = angularInput * this->physicalProperties.propulsiveTorque;
}

void Drone::prepareForSimulationStep(){
	float deltaT = this->game->getSystem()->getTimer()->getDeltaT();

	dVector3 tmpv3;
	dVector4 tmpv4;

	dBodySetPosition( this->body, this->position.x, this->position.y, this->position.z );
	this->orientation.toOdeVector4(tmpv4);
	dBodySetQuaternion( this->body, tmpv4 );
	dBodySetLinearVel( this->body, this->velocity.x, this->velocity.y, this->velocity.z );
	dBodySetAngularVel( this->body, this->angularVelocity.x, this->angularVelocity.y, this->angularVelocity.z );



	if( this->recoveryActive ){
		const FVector3& ng = this->game->getPhysics()->getNormalizedGravity();
		float mod = 1.0f + this->up.dotProduct(ng);

		if( this->direction.dotProduct(ng) < 0.0f ){
			this->propulsiveTorque.x = -mod * this->physicalProperties.propulsiveTorque;
		}else{
			this->propulsiveTorque.x = +mod * this->physicalProperties.propulsiveTorque;
		}

		if( this->right.dotProduct(ng) < 0.0f ){
			this->propulsiveTorque.z = -mod * this->physicalProperties.propulsiveTorque;
		}else{
			this->propulsiveTorque.z = +mod * this->physicalProperties.propulsiveTorque;
		}
	}

	// propulsive force
	dBodyAddRelForce( this->body,
		this->propulsiveForce.x, this->propulsiveForce.y, this->propulsiveForce.z );


	// boosters
	if( this->energy < 0.1f ){	// check energy first...
		this->boostersActive = false;
	}
	if( this->boostersActive ){
		this->energy -= deltaT * 100.0f;
		this->energy = this->energy < 0.0f ? 0.0f : this->energy;

		FVector3 f = this->propulsiveForce;
		f.normalize();
		f *= this->physicalProperties.boosterForce;
		dBodyAddRelForce( this->body, f.x, f.y, f.z );
	}else{
		this->energy += deltaT * 100.0f;
		this->energy = this->energy > this->physicalProperties.energyCapacity ? this->physicalProperties.energyCapacity : this->energy;

		// stabilizer force
		dBodyVectorFromWorld( this->body,
			this->velocity.x, this->velocity.y, velocity.z, tmpv3);
		FVector3 relV(tmpv3);
		float sLinear = relV.normalize() / this->physicalProperties.maxLinearVelocity * this->physicalProperties.propulsiveForce;
		FVector3 sf = -relV * sLinear;
		dBodyAddRelForce( this->body, sf.x, sf.y, sf.z );
	}

	// prop torque
	dBodyAddRelTorque( this->body,
		this->propulsiveTorque.x, this->propulsiveTorque.y, this->propulsiveTorque.z );

	dBodyVectorFromWorld( this->body,
		this->angularVelocity.x, this->angularVelocity.y, this->angularVelocity.z, tmpv3);
	FVector3 relAV(tmpv3);
	float sAngular = relAV.normalize() / this->physicalProperties.maxAngularVelocity * this->physicalProperties.propulsiveTorque;
	FVector3 st = -relAV * sAngular;
	dBodyAddRelTorque( this->body, st.x, st.y, st.z );
}

void Drone::updateAfterSimulationStep(){
//	printf("1 %i\n", this->body);
	const dReal* pos = dBodyGetPosition( this->body );
	const dReal* ori = dBodyGetQuaternion( this->body );
	const dReal* vel = dBodyGetLinearVel( this->body );
	const dReal* avel = dBodyGetAngularVel( this->body );

	this->position = FVector3(pos);
	this->orientation = FQuaternion(ori);
	this->velocity = FVector3(vel);
	this->angularVelocity = FVector3(avel);

	this->mainUnitSceneNode->setPosition( this->position.toOgreVector3() );
	this->mainUnitSceneNode->setOrientation( this->orientation.toOgreQuaternion() );
	Ogre::Quaternion q = this->mainUnitSceneNode->getOrientation();
	this->right = FVector3(q.xAxis().val);
	this->up = FVector3(q.yAxis().val);
	this->direction = -FVector3(q.zAxis().val);
	this->angles.x = q.getYaw().valueRadians();
	this->angles.y = q.getPitch().valueRadians();
	this->angles.z = q.getRoll().valueRadians();

	this->propulsionUnitSceneNode->setPosition( this->position.toOgreVector3() );
	this->propulsionUnitSceneNode->setOrientation( this->orientation.toOgreQuaternion() );
		
	if( this->physicalProperties.propulsiveForce != 0.0f ){
		float mod = this->velocity.dotProduct(this->direction) / this->physicalProperties.maxLinearVelocity;
		mod = mod < 1.0f ? mod : 1.0f;
		mod = mod > -1.0f ? mod : -1.0f;
		float propulsionUnitPitch = -mod * 0.5f*FLOAT_PI;

		this->propulsionUnitSceneNode->rotate( Ogre::Vector3::UNIT_X,
			Ogre::Radian(propulsionUnitPitch) );
	}
//	printf("2: %f\n", this->position.y);

	float deltaT = this->game->getSystem()->getTimer()->getDeltaT();

	if( this->boosterEffect != NULL ){
		if( this->boostersActive ){
			if( !this->boosterEffect->isAttached() ){
				this->boosterEffect->attach( this->game, this->propulsionUnitSceneNode );
			}
			this->boosterEffect->update( deltaT );
		}else{
			if( this->boosterEffect->isAttached() ){
				this->boosterEffect->detach();
			}
		}
	}

}
