#include "Effect.h"

#include "System.h"
#include "Exception.h"
#include "XmlUtils.h"
#include "StringConverter.h"
#include "File.h"
#include "Game.h"
#include "Graphics.h"
#include "Sound.h"

Effect::Effect(): GameObject( GameObject::GAME_OBJECT_TYPE_EFFECT ) {
	this->instanceId = 0;
	this->instanceName = "<uninitialized>";

	this->name = "<uninitialized>";
	this->duration = 0.0f;
	this->livesForever = false;
	this->gameObject = NULL;

	this->sceneNode = NULL;
	this->particleSystems.clear();
	this->animationStates.clear();
	this->lights.clear();

	this->graphicalProperties.meshes.clear();
	this->graphicalProperties.particleSystems.clear();
	this->graphicalProperties.lights.clear();

	this->acousticalProperties.samples.clear();
}

Effect::~Effect(){
}

Effect* Effect::clone() const {
	Effect* ret = new Effect();

	(*ret) = (*this);

	// TODO: duplicate pointered things

	return ret;
}

void Effect::loadFromFile( const string& effectFilePath ){

	if( !File::exists( effectFilePath ) ){
		throw Exception("File '" + effectFilePath + "' does not exist.", "Effect::loadFromFile()");
	}

	TiXmlDocument doc( effectFilePath.c_str() );
	if( !doc.LoadFile() ){
		throw Exception("Couldn't load file '" + effectFilePath + "'. (Maybe not a correct xml file?)", "Effect::loadFromFile()");
	}
//	doc.Print();

	TiXmlNode* rootElement = doc.RootElement();
	TiXmlElement* element = rootElement->ToElement();
	string rootElementName = element->Value();
	if( rootElementName != "effect" ){
		throw Exception("File '" + effectFilePath + "' does not have an <effect> root node.", "Effect::loadFromFile()");
	}


	this->fromXmlElement( element );

}

void Effect::fromXmlElement( TiXmlElement* xmlElement ){

	TiXmlElement* element = xmlElement;
/*
	// check version
	const char* tmp = element->Attribute( "formatVersion" );
	if( tmp == NULL ){
		throw Exception("<effect> node has no version information.", "Effect::fromXmlElement()");
	}else{
		string version = tmp;

		if( version != System::getInstance()->getVersionString() ){
			System::warn("(in Effect::fromXmlElement()): <effect> node has version %s. Trying to parse anyway.", version.c_str() );
		}
//		this->versionString = version;
	}
*/
	// check name
	const char* tmp = element->Attribute( "name" );
	if( tmp == NULL ){
		throw Exception("<effect> node has no name information.", "Effect::fromXmlElement()");
	}else{
		this->name = string( tmp );
	}
	if( this->name.length() > 32 ){
		System::error( "(in Effect::fromXmlElement()): Effect name '%s' is longer than 32 characters.", this->name.c_str() );
		throw Exception( "XmlElement is not a valid effect description.", "Effect::fromXmlElement()" );
	}

	// duration
	tmp = element->Attribute( "duration" );
	if( tmp == NULL ){
		throw Exception("<effect> node has no duration information.", "Effect::fromXmlElement()");
	}else{
		this->duration = (float)atof( tmp );
		this->livesForever = (this->duration < 0.0f);
	}


	for( TiXmlElement* elt = xmlElement->FirstChildElement(); elt != NULL; elt = elt->NextSiblingElement() ){
		string value = string( elt->Value() );

		if(value == "graphicalProperties"){
			this->parseGraphicalProperties(elt);
		}else if(value == "acousticalProperties"){
			this->parseAcousticalProperties(elt);

		}else{
			System::warn("(in Effect::fromXmlElement()): Unknown child element: '%s' (row: %i). Ignoring.", value.c_str(), elt->Row() );
		}
	}

}

TiXmlElement Effect::toXmlElement() const {
	return TiXmlElement("dummy");
}



void Effect::attach(Game* game, GameObject* gameObject){ // exclusive scene node
	if( this->game != NULL ){
		throw Exception("Effect is already attached to a game.", "Effect::attach()");
	}

	this->instanceId = game->getNextId();
//	printf("id: %d\n", id);
	char tmp[512];
	sprintf(tmp, "effect '%s' (instance id: %d)", this->name.c_str(), this->instanceId);
	this->instanceName = string(tmp);

	this->gameObject = gameObject;

	Ogre::SceneManager* sm = game->getSystem()->getGraphics()->getSceneManager();

	this->sceneNode = sm->getRootSceneNode()->createChildSceneNode( this->instanceName + "/SceneNode" );
	this->sceneNode->setPosition( this->position.toOgreVector3() );
	this->sceneNode->setOrientation( this->orientation.toOgreQuaternion() );

	this->attachStuff( game );

	// TODO: sound

	this->game = game;

}

void Effect::attach(Game* game, Ogre::SceneNode* sceneNode){ // shared scene node
	if( this->game != NULL ){
		throw Exception("Effect is already attached to a game.", "Effect::attach()");
	}

	this->instanceId = game->getNextId();
//	printf("id: %d\n", id);
	char tmp[512];
	sprintf(tmp, "effect '%s' (instance id: %d)", this->name.c_str(), this->instanceId);
	this->instanceName = string(tmp);

	this->gameObject = NULL;
	this->sceneNode = sceneNode;


	this->attachStuff( game );

	this->game = game;
}



void Effect::attachStuff( Game* game ){
	Ogre::SceneManager* sm = game->getSystem()->getGraphics()->getSceneManager();

	// meshes
	for( list<mesh_s>::const_iterator mesh_c_iter = this->graphicalProperties.meshes.begin(); mesh_c_iter != this->graphicalProperties.meshes.end(); mesh_c_iter++ ){
		Ogre::Entity *ent = sm->createEntity( this->instanceName + "/Mesh '" + mesh_c_iter->fileName + "'", mesh_c_iter->fileName );
//		ent->setCastShadows(this->graphicalProperties.castShadows);
		this->sceneNode->attachObject(ent);

		if( mesh_c_iter->animated ){
			Ogre::AnimationState* as = ent->getAnimationState("animation");
			as->setEnabled(true);
			this->animationStates.push_back( as );
		}
	}

	// particle systems
//	Ogre::ParticleSystemManager& psm = Ogre::ParticleSystemManager::getSingleton();
	for( list<particleSystem_s>::const_iterator ps_c_iter = this->graphicalProperties.particleSystems.begin(); ps_c_iter != this->graphicalProperties.particleSystems.end(); ps_c_iter++ ){
		Ogre::ParticleSystem* ps = sm->createParticleSystem( this->instanceName + "/ParticleSystem '" + ps_c_iter->templateName + "'", ps_c_iter->templateName );
		this->sceneNode->attachObject(ps);

		this->particleSystems.push_back(ps);
	}

	// lights
	for( list<light_s>::const_iterator l_c_iter = this->graphicalProperties.lights.begin(); l_c_iter != this->graphicalProperties.lights.end(); ++l_c_iter ){
		Ogre::Light* l = sm->createLight( this->instanceName + "/Light '" + StringConverter::toString( this->lights.size() ) + "'" );

		l->setVisible(true);
		l->setType(l_c_iter->lightType);
		l->setCastShadows(l_c_iter->castShadows);
		l->setDiffuseColour( l_c_iter->diffuseColor.toOgreColourValue() );
		l->setSpecularColour( l_c_iter->specularColor.toOgreColourValue() );
		l->setAttenuation(l_c_iter->attenuationRange, l_c_iter->attenuationConstant, l_c_iter->attenuationLinear, l_c_iter->attenuationQuadratic);
		l->setDirection( l_c_iter->direction.toOgreVector3() );
		if( l_c_iter->lightType == Ogre::Light::LT_SPOTLIGHT ){
			l->setSpotlightRange(Ogre::Radian(l_c_iter->spotlightInnerAngle), Ogre::Radian(l_c_iter->spotlightOuterAngle), l_c_iter->spotlightFalloff);
		}

		if( l_c_iter->attachToSceneNode ){
			this->sceneNode->attachObject( l );
		}else{
			l->setPosition( this->position.toOgreVector3() );
		}

		this->lights.push_back( l );
	}

	// samples
	Sound* sound = game->getSystem()->getSound();
	if( sound->isEnabled() ){
		for( list<sample_s>::const_iterator s_c_iter = this->acousticalProperties.samples.begin(); s_c_iter != this->acousticalProperties.samples.end(); ++s_c_iter ){
			Sound::sample_t* s = sound->getSample( s_c_iter->fileName );
			sound->playSample( this->position, s, s_c_iter->numLoops );
		}
	}
}


void Effect::detach(){
	if( this->game == NULL ){
		throw Exception("Effect is not attached to a game.", "Effect::detach()");
	}

	Ogre::SceneManager* sm = this->game->getSystem()->getGraphics()->getSceneManager();
	// destroy particle systems
//	Ogre::ParticleSystemManager& psm = Ogre::ParticleSystemManager::getSingleton();
	for( list<Ogre::ParticleSystem*>::iterator ps_iter = this->particleSystems.begin(); ps_iter != this->particleSystems.end(); ++ps_iter ){
		sm->destroyParticleSystem( *ps_iter );
	}
	this->particleSystems.clear();

	// destroy light
	for( list<Ogre::Light*>::iterator l_iter = this->lights.begin(); l_iter != this->lights.end(); ++l_iter ){
		sm->destroyLight( *l_iter );
	}
	this->lights.clear();

	// THINKABOUTME: destroy meshes as well?

	// destroy scene node
	if( this->gameObject != NULL ){ // exclusive scene node
		this->sceneNode->getParentSceneNode()->removeAndDestroyChild( this->sceneNode->getName() );
	}
	this->sceneNode = NULL;

	this->gameObject = NULL;
	this->game = NULL;
}


void Effect::update(float deltaT){
	if( !this->livesForever ){
		this->duration -= deltaT;
	}

	if( this->gameObject != NULL ){
		this->position = this->gameObject->position;
		this->orientation = this->gameObject->orientation;
		this->sceneNode->setPosition( this->position.toOgreVector3() );
		this->sceneNode->setOrientation( this->orientation.toOgreQuaternion() );
	}else{
		this->position = this->sceneNode->getPosition();
		this->orientation = this->sceneNode->getOrientation();
	}


	for( list<Ogre::AnimationState*>::iterator iter = this->animationStates.begin(); iter != this->animationStates.end(); iter++ ){
		(*iter)->addTime( deltaT );
	}

	// TODO: set sound to pos
}




void Effect::parseGraphicalProperties( TiXmlElement* graphicalPropertiesElement ){
//	XmlUtils::readStringProperty(graphicalPropertiesElement, "mesh", this->graphicalProperties.mesh);
//	XmlUtils::readStringProperty(graphicalPropertiesElement, "particleSystem", this->graphicalProperties.particleSystem);
	for( TiXmlElement* elt = graphicalPropertiesElement->FirstChildElement(); elt != NULL; elt = elt->NextSiblingElement() ){
		string value = string( elt->Value() );

		if( value == string("mesh") ){
			this->parseMesh(elt);
		}else if( value == string("particleSystem") ){
			this->parseParticleSystem(elt);
		}else if( value == string("light") ){
			this->parseLight(elt);

		}else{
			System::warn("(in Effect::parseGraphicalProperties()): Unknown child element: '%s' (row: %i). Ignoring.", value.c_str(), elt->Row() );
		}
	}

}


void Effect::parseAcousticalProperties( TiXmlElement* acousticalPropertiesElement ){
	for( TiXmlElement* elt = acousticalPropertiesElement->FirstChildElement(); elt != NULL; elt = elt->NextSiblingElement() ){
		string value = string( elt->Value() );

		if( value == string("sample") ){
			this->parseSample(elt);

		}else{
			System::warn("(in Effect::parseAcousticalProperties()): Unknown child element: '%s' (row: %i). Ignoring.", value.c_str(), elt->Row() );
		}
	}

}

void Effect::parseSample( TiXmlElement* sampleElement ){
	sample_s s;

	const char* tmp = sampleElement->Attribute("file");
	if( tmp != NULL ){
		s.fileName = string( tmp );
	}else{
		throw Exception("<sample> node has no file attribute", "Effect::parseSample()");
	}

	tmp = sampleElement->Attribute("numLoops");
	if( tmp != NULL ){
		s.numLoops = atoi( tmp );
	}else{
		s.numLoops = 0;
	}

	this->acousticalProperties.samples.push_back( s );

}

void Effect::parseMesh(TiXmlElement* meshElement ){
	mesh_s m;

	const char* tmp = meshElement->Attribute("file");
	if( tmp != NULL ){
		m.fileName = string(tmp);
	}else{
		throw Exception("<mesh> node has no file attribute", "Effect::parseMesh()");
	}

	tmp = meshElement->Attribute("animated");
	if( tmp != NULL ){
		m.animated = string(tmp) == string("true") ? true : false;
	}else{
		m.animated = false;
	}

	this->graphicalProperties.meshes.push_back( m );
}

void Effect::parseParticleSystem( TiXmlElement* particleSystemElement ){
	particleSystem_s ps;

	const char* tmp = particleSystemElement->Attribute("template");
	if( tmp != NULL ){
		ps.templateName = string(tmp);
	}else{
		throw Exception("<particleSystem> node has no template attribute", "Effect::parseParticleSystem()");
	}

	this->graphicalProperties.particleSystems.push_back( ps );

}

void Effect::parseLight( TiXmlElement* lightElement ){
	light_s light;

	light.diffuseColor = FColor::WHITE;
	light.specularColor = FColor::WHITE;
	light.direction = FVector3::UNIT_Z;

	// type
	string type = lightElement->Attribute("type") ? lightElement->Attribute("type") : "point";
	if( type == "point" ){
		light.lightType = Ogre::Light::LT_POINT;
	}else if( type == "directional" ){
		light.lightType = Ogre::Light::LT_DIRECTIONAL;
	}else if( type == "spotlight" ){
		light.lightType = Ogre::Light::LT_SPOTLIGHT;
	}

	// shadows
	light.castShadows = true;
	if( lightElement->Attribute("castShadows") != NULL && string( lightElement->Attribute("castShadows") ) == string("false") ){
		light.castShadows = false;
	}

	// attach
	light.attachToSceneNode = true;
	if( lightElement->Attribute("attachToSceneNode") != NULL && string( lightElement->Attribute("attachToSceneNode") ) == string("false") ){
		light.attachToSceneNode = false;
	}


	for( TiXmlElement* elt = lightElement->FirstChildElement(); elt != NULL; elt = elt->NextSiblingElement() ){
		string value = string( elt->Value() );

		if(value == "diffuseColor"){
			light.diffuseColor = XmlUtils::readFColor( elt );
		}else if(value == "specularColor"){
			light.diffuseColor = XmlUtils::readFColor( elt );

		}else if(value == "attenuation"){
			light.attenuationRange = elt->Attribute("range") ? atof(elt->Attribute("range")) : 1000.0;
			light.attenuationConstant = elt->Attribute("constant") ? atof(elt->Attribute("constant")) : 1.0;
			light.attenuationLinear = elt->Attribute("linear") ? atof(elt->Attribute("linear")) : 0.0;
			light.attenuationQuadratic = elt->Attribute("quadratic") ? atof(elt->Attribute("quadratic")) : 0.0;
		}else if(value == "spotlight"){
			light.spotlightInnerAngle = elt->Attribute("innerAngle") ? atof(elt->Attribute("innerAngle")) : 1.0;
			light.spotlightOuterAngle = elt->Attribute("outerAngle") ? atof(elt->Attribute("outerAngle")) : 1.0;
			light.spotlightFalloff = elt->Attribute("falloff") ? atof(elt->Attribute("falloff")) : 0.0;

		}else if(value == "direction"){
			light.direction = XmlUtils::readFVector3( elt );

//		}else if(value == "position"){
//			light.position = XmlUtils::readFVector3( elt );
//		}else if(value == "orientation"){
//			light.orientation = XmlUtils::readFQuaternion( elt );

		}else{
			System::warn("(in Effect::parseLight()): Unknown child element: '%s' (row: %i). Ignoring.", value.c_str(), elt->Row() );
		}
	}

	this->graphicalProperties.lights.push_back( light );
}

bool Effect::shouldBeDestroyed() const {
	return ( !this->livesForever && this->duration < 0.0f );
}
const string& Effect::getName() const {
	return this->name;
}
float Effect::getDuration() const {
	return this->duration;
}
GameObject* Effect::getGameObject() const {
	return this->gameObject;
}
