#ifndef __FVectors_h__
#define __FVectors_h__

#include <ostream>
#include <math.h>
#include <float.h>

#include <Ogre.h>
//#include <OgreNoMemoryMacros.h>
#include <ode/ode.h>
//#include <OgreMemoryMacros.h>


#ifndef FLOAT_PI
#define FLOAT_PI 3.141592654f
#endif

#ifndef FLOAT_EPSILON
#define FLOAT_EPSILON FLT_EPSILON
#endif

class FVector2{
public:
	union {
		struct {
			float x, y;
		};
		float components[2];
	};

//	static const float EPSILON;
	static const FVector2 ZERO;
	static const FVector2 UNIT_X;
	static const FVector2 UNIT_Y;

	FVector2();
	FVector2( float x, float y);
	FVector2( const float components[2] );
//	FVector2( const float* const components );
	FVector2( const FVector2& other );

	float operator [] ( size_t i ) const;
	float& operator [] ( size_t i );
	FVector2& operator = ( const FVector2& other );
	bool operator == ( const FVector2& other ) const;
	bool operator != ( const FVector2& other ) const;

	FVector2 operator + ( const FVector2& other ) const;
	FVector2 operator - ( const FVector2& other ) const;
	FVector2 operator * ( float scalar ) const;
	FVector2 operator * ( const FVector2& other) const;
	FVector2 operator / ( float scalar ) const;
	FVector2 operator / ( const FVector2& other) const;
	FVector2 operator - () const;

	FVector2& operator += ( const FVector2& other );
	FVector2& operator -= ( const FVector2& other );
	FVector2& operator *= ( float scalar );
	FVector2& operator *= ( const FVector2& other );
	FVector2& operator /= ( float scalar );
	FVector2& operator /= ( const FVector2& other );
	
	float length() const;
	float squaredLength () const;
	float normalize();
	float dotProduct( const FVector2& other ) const;

	friend FVector2 operator * ( float scalar, const FVector2& vector );
	friend std::ostream& operator << ( std::ostream& os, const FVector2& vector );
};



class FVector3{
public:
	union {
		struct {
			float x, y, z;
		};
		float components[3];
	};

//	static const float EPSILON;
	static const FVector3 ZERO;
	static const FVector3 UNIT_X;
	static const FVector3 UNIT_Y;
	static const FVector3 UNIT_Z;

	FVector3();
	FVector3( float x, float y, float z );
	FVector3( const float components[3] );
//	FVector3( const float* const components );

	FVector3( double x, double y, double z );
	FVector3( const double components[3] );
//	FVector3( const double* const components );

	FVector3( const FVector3& other );
	FVector3( const Ogre::Vector3& ogreVector3 );

	float operator [] ( size_t i ) const;
	float& operator [] ( size_t i );
	FVector3& operator = ( const FVector3& other );
	bool operator == ( const FVector3& other ) const;
	bool operator != ( const FVector3& other ) const;

	FVector3 operator + ( const FVector3& other ) const;
	FVector3 operator - ( const FVector3& other ) const;
	FVector3 operator * ( float scalar ) const;
	FVector3 operator * ( const FVector3& other) const;
	FVector3 operator / ( float scalar ) const;
	FVector3 operator / ( const FVector3& other) const;
	FVector3 operator - () const;

	FVector3& operator += ( const FVector3& other );
	FVector3& operator -= ( const FVector3& other );
	FVector3& operator *= ( float scalar );
	FVector3& operator *= ( const FVector3& other );
	FVector3& operator /= ( float scalar );
	FVector3& operator /= ( const FVector3& other );
	
	float length() const;
	float squaredLength () const;
	float normalize();
	float dotProduct( const FVector3& other ) const;
	FVector3 crossProduct( const FVector3& other ) const;
	FVector3 perpendicular() const;
	float distance( const FVector3& other ) const;
	void clamp( const FVector3& min, const FVector3& max );

	friend FVector3 operator * ( float scalar, const FVector3& vector );
	friend std::ostream& operator << ( std::ostream& os, const FVector3& vector );

	Ogre::Vector3 toOgreVector3() const;
	void toOdeVector3(dVector3& dv) const;
	void toFloatArray( float* array ) const;
};


class FVector4{
public:
	union {
		struct {
			float x, y, z, w;
		};
		float components[4];
	};

//	static const float EPSILON;
	static const FVector4 ZERO;

	FVector4();
	FVector4( float x, float y, float z, float w );
	FVector4( const float components[4] );
//	FVector4( const float* const components );
	FVector4( const FVector4& other );

	float operator [] ( size_t i ) const;
	float& operator [] ( size_t i );
	FVector4& operator = ( const FVector4& other );
	bool operator == ( const FVector4& other ) const;
	bool operator != ( const FVector4& other ) const;

	FVector4 operator + ( const FVector4& other ) const;
	FVector4 operator - ( const FVector4& other ) const;
	FVector4 operator * ( float scalar ) const;
	FVector4 operator * ( const FVector4& other) const;
	FVector4 operator / ( float scalar ) const;
	FVector4 operator / ( const FVector4& other) const;
	FVector4 operator - () const;

	FVector4& operator += ( const FVector4& other );
	FVector4& operator -= ( const FVector4& other );
	FVector4& operator *= ( float scalar );
	FVector4& operator *= ( const FVector4& other );
	FVector4& operator /= ( float scalar );
	FVector4& operator /= ( const FVector4& other );
	
	float length() const;
	float squaredLength () const;
	float normalize();
	float dotProduct( const FVector4& other ) const;

	friend FVector4 operator * ( float scalar, const FVector4& vector );
	friend std::ostream& operator << ( std::ostream& os, const FVector4& vector );
};







// INLINE DEFINITIONS


inline FVector2::FVector2(){}
inline FVector2::FVector2( float fx, float fy ): x(fx), y(fy) {}
inline FVector2::FVector2( const float components[2] ): x(components[0]), y(components[1]) {}
//inline FVector2::FVector2( const float* const components ): x(components[0]), y(components[1]) {}
inline FVector2::FVector2( const FVector2& other ): x(other.x), y(other.y) {}



inline float FVector2::operator [] ( size_t i ) const {
	return components[i];
}

inline float& FVector2::operator [] ( size_t i ){
	return components[i];
}

inline FVector2& FVector2::operator = ( const FVector2& other ){
	x = other.x;
	y = other.y;

	return *this;
}

inline bool FVector2::operator == ( const FVector2& other ) const {
	return ( x == other.x && y == other.y );
}

inline bool FVector2::operator != ( const FVector2& other ) const {
	return ( x != other.x || y != other.y );
}


inline FVector2 FVector2::operator + ( const FVector2& other ) const {
	return FVector2( x + other.x, y + other.y );
}
inline FVector2 FVector2::operator - ( const FVector2& other ) const {
	return FVector2( x - other.x, y - other.y );
}
inline FVector2 FVector2::operator * ( float scalar ) const {
	return FVector2( x * scalar, y * scalar );
}
inline FVector2 FVector2::operator * ( const FVector2& other) const {
	return FVector2( x * other.x, y * other.y );
}
inline FVector2 FVector2::operator / ( float scalar ) const {
	return FVector2( x / scalar, y / scalar );
}
inline FVector2 FVector2::operator / ( const FVector2& other) const {
	return FVector2( x / other.x, y / other.y );
}
inline FVector2 FVector2::operator - () const {
	return FVector2( -x, -y );
}


inline FVector2& FVector2::operator += ( const FVector2& other ){
	x += other.x;
	y += other.y;

	return *this;
}
inline FVector2& FVector2::operator -= ( const FVector2& other ){
	x -= other.x;
	y -= other.y;

	return *this;
}
inline FVector2& FVector2::operator *= ( float scalar ){
	x *= scalar;
	y *= scalar;

	return *this;
}
inline FVector2& FVector2::operator *= ( const FVector2& other ){
	x *= other.x;
	y *= other.y;

	return *this;
}
inline FVector2& FVector2::operator /= ( float scalar ){
	x /= scalar;
	y /= scalar;

	return *this;
}
inline FVector2& FVector2::operator /= ( const FVector2& other ){
	x /= other.x;
	y /= other.y;

	return *this;
}



inline float FVector2::length() const {
	return (float)sqrt( x*x + y*y );
}
inline float FVector2::squaredLength () const{
	return ( x*x + y*y );
}
inline float FVector2::normalize() {
	float length = (float)sqrt( x*x + y*y );

	if( length > FLOAT_EPSILON ){
		x /= length;
		y /= length;
	}

	return length;
}
inline float FVector2::dotProduct( const FVector2& other ) const {
	return ( x*other.x + y*other.y );
}




inline FVector2 operator * ( float scalar, const FVector2& fvector ){
	return FVector2( scalar*fvector.x, scalar*fvector.y );
}

inline std::ostream& operator << ( std::ostream& os, const FVector2& fvector ){
	os << "(" << fvector.x << ", " << fvector.y << ")";
	return os;
}















inline FVector3::FVector3(){}
inline FVector3::FVector3( float fx, float fy, float fz ): x(fx), y(fy), z(fz) {}
inline FVector3::FVector3( const float components[3] ): x(components[0]), y(components[1]), z(components[2]) {}
//inline FVector3::FVector3( const float* const components ): x(components[0]), y(components[1]), z(components[2]) {}

inline FVector3::FVector3( double dx, double dy, double dz ): x(dx), y(dy), z(dz) {}
inline FVector3::FVector3( const double components[3] ): x(components[0]), y(components[1]), z(components[2]) {}
//inline FVector3::FVector3( const double* const components ): x(components[0]), y(components[1]), z(components[2]) {}

inline FVector3::FVector3( const FVector3& other ): x(other.x), y(other.y), z(other.z) {}
inline FVector3::FVector3( const Ogre::Vector3& ogreVector3 ): x(ogreVector3.x), y(ogreVector3.y), z(ogreVector3.z) {}



inline float FVector3::operator [] ( size_t i ) const {
	return components[i];
}
inline float& FVector3::operator [] ( size_t i ){
	return components[i];
}
inline FVector3& FVector3::operator = ( const FVector3& other ){
	x = other.x;
	y = other.y;
	z = other.z;

	return *this;
}
inline bool FVector3::operator == ( const FVector3& other ) const {
	return ( x == other.x && y == other.y && z == other.z );
}
inline bool FVector3::operator != ( const FVector3& other ) const {
	return ( x != other.x || y != other.y || z != other.z );
}


inline FVector3 FVector3::operator + ( const FVector3& other ) const {
	return FVector3( x + other.x, y + other.y, z + other.z );
}
inline FVector3 FVector3::operator - ( const FVector3& other ) const {
	return FVector3( x - other.x, y - other.y, z - other.z );
}
inline FVector3 FVector3::operator * ( float scalar ) const {
	return FVector3( x * scalar, y * scalar, z * scalar );
}
inline FVector3 FVector3::operator * ( const FVector3& other) const {
	return FVector3( x * other.x, y * other.y, z * other.z );
}
inline FVector3 FVector3::operator / ( float scalar ) const {
	return FVector3( x / scalar, y / scalar, z / scalar );
}
inline FVector3 FVector3::operator / ( const FVector3& other) const {
	return FVector3( x / other.x, y / other.y, z / other.z );
}
inline FVector3 FVector3::operator - () const {
	return FVector3( -x, -y, -z );
}


inline FVector3& FVector3::operator += ( const FVector3& other ){
	x += other.x;
	y += other.y;
	z += other.z;

	return *this;
}
inline FVector3& FVector3::operator -= ( const FVector3& other ){
	x -= other.x;
	y -= other.y;
	z -= other.z;

	return *this;
}
inline FVector3& FVector3::operator *= ( float scalar ){
	x *= scalar;
	y *= scalar;
	z *= scalar;

	return *this;
}
inline FVector3& FVector3::operator *= ( const FVector3& other ){
	x *= other.x;
	y *= other.y;
	z *= other.z;

	return *this;
}
inline FVector3& FVector3::operator /= ( float scalar ){
	x /= scalar;
	y /= scalar;
	z /= scalar;

	return *this;
}
inline FVector3& FVector3::operator /= ( const FVector3& other ){
	x /= other.x;
	y /= other.y;
	z /= other.z;

	return *this;
}



inline float FVector3::length() const {
	return (float)sqrt( x*x + y*y + z*z );
}
inline float FVector3::squaredLength () const{
	return ( x*x + y*y + z*z );
}
inline float FVector3::normalize() {
	float length = (float)sqrt( x*x + y*y + z*z );

	if( length > FLOAT_EPSILON ){
		x /= length;
		y /= length;
		z /= length;
	}

	return length;
}
inline float FVector3::dotProduct( const FVector3& other ) const {
	return ( x*other.x + y*other.y + z*other.z );
}

inline FVector3 FVector3::crossProduct( const FVector3& other ) const {
	return FVector3(
			y*other.z - z*other.y,
			z*other.x - x*other.z,
			x*other.y - y*other.x
		);
}
inline FVector3 FVector3::perpendicular() const {
	FVector3 ret = this->crossProduct( UNIT_X );

	if( ret.squaredLength() < FLOAT_EPSILON ){
		ret = this->crossProduct( UNIT_Y );
	}

	return ret;
}
inline float FVector3::distance( const FVector3& other ) const {
	return (float)sqrt( (x-other.x)*(x-other.x) + (y-other.y)*(y-other.y) + (z-other.z)*(z-other.z) );
}

inline void FVector3::clamp( const FVector3& min, const FVector3& max ){
	x = x < min.x ? min.x : x;
	x = x > max.x ? max.x : x;
	y = y < min.y ? min.y : y;
	y = y > max.y ? max.y : y;
	z = z < min.z ? min.z : z;
	z = z > max.z ? max.z : z;
}



inline FVector3 operator * ( float scalar, const FVector3& fvector ){
	return FVector3( scalar*fvector.x, scalar*fvector.y, scalar*fvector.z );
}

inline std::ostream& operator << ( std::ostream& os, const FVector3& fvector ){
	os << "(" << fvector.x << ", " << fvector.y << ", " << fvector.z << ")";
	return os;
}


inline Ogre::Vector3 FVector3::toOgreVector3() const {
	return Ogre::Vector3(x,y,z);
}
inline void FVector3::toOdeVector3(dVector3& dv) const {
	dv[0] = x;
	dv[1] = y;
	dv[2] = z;
}
inline void FVector3::toFloatArray( float* array ) const {
	array[0] = x;
	array[1] = y;
	array[2] = z;
}











inline FVector4::FVector4() {}
inline FVector4::FVector4( float fx, float fy, float fz, float fw ): x(fx), y(fy), z(fz), w(fw) {}
inline FVector4::FVector4( const float components[4] ): x(components[0]), y(components[1]), z(components[2]), w(components[3]) {}
//inline FVector4::FVector4( const float* const components ): x(components[0]), y(components[1]), z(components[2]), w(components[3]) {}
inline FVector4::FVector4( const FVector4& other ): x(other.x), y(other.y), z(other.z), w(other.w) {}



inline float FVector4::operator [] ( size_t i ) const {
	return components[i];
}
inline float& FVector4::operator [] ( size_t i ){
	return components[i];
}
inline FVector4& FVector4::operator = ( const FVector4& other ){
	x = other.x;
	y = other.y;
	z = other.z;
	w = other.w;

	return *this;
}
inline bool FVector4::operator == ( const FVector4& other ) const {
	return ( x == other.x && y == other.y && z == other.z && w == other.w );
}
inline bool FVector4::operator != ( const FVector4& other ) const {
	return ( x != other.x || y != other.y || z != other.z || w != other.w );
}


inline FVector4 FVector4::operator + ( const FVector4& other ) const {
	return FVector4( x + other.x, y + other.y, z + other.z, w + other.w );
}
inline FVector4 FVector4::operator - ( const FVector4& other ) const {
	return FVector4( x - other.x, y - other.y, z - other.z, w - other.w );
}
inline FVector4 FVector4::operator * ( float scalar ) const {
	return FVector4( x * scalar, y * scalar, z * scalar, w * scalar );
}
inline FVector4 FVector4::operator * ( const FVector4& other) const {
	return FVector4( x * other.x, y * other.y, z * other.z, w * other.w );
}
inline FVector4 FVector4::operator / ( float scalar ) const {
	return FVector4( x / scalar, y / scalar, z / scalar, w / scalar );
}
inline FVector4 FVector4::operator / ( const FVector4& other) const {
	return FVector4( x / other.x, y / other.y, z / other.z, w / other.w );
}
inline FVector4 FVector4::operator - () const {
	return FVector4( -x, -y, -z, -w );
}


inline FVector4& FVector4::operator += ( const FVector4& other ){
	x += other.x;
	y += other.y;
	z += other.z;
	w += other.w;

	return *this;
}
inline FVector4& FVector4::operator -= ( const FVector4& other ){
	x -= other.x;
	y -= other.y;
	z -= other.z;
	w -= other.w;

	return *this;
}
inline FVector4& FVector4::operator *= ( float scalar ){
	x *= scalar;
	y *= scalar;
	z *= scalar;
	w *= scalar;

	return *this;
}
inline FVector4& FVector4::operator *= ( const FVector4& other ){
	x *= other.x;
	y *= other.y;
	z *= other.z;
	w *= other.z;

	return *this;
}
inline FVector4& FVector4::operator /= ( float scalar ){
	x /= scalar;
	y /= scalar;
	z /= scalar;
	w /= scalar;

	return *this;
}
inline FVector4& FVector4::operator /= ( const FVector4& other ){
	x /= other.x;
	y /= other.y;
	z /= other.z;
	w /= other.z;

	return *this;
}



inline float FVector4::length() const {
	return (float)sqrt( x*x + y*y + z*z + w*w );
}
inline float FVector4::squaredLength () const{
	return ( x*x + y*y + z*z + w*w );
}
inline float FVector4::normalize() {
	float length = (float)sqrt( x*x + y*y + z*z + w*w );

	if( length > FLOAT_EPSILON ){
		x /= length;
		y /= length;
		z /= length;
		w /= length;
	}

	return length;
}
inline float FVector4::dotProduct( const FVector4& other ) const {
	return ( x*other.x + y*other.y + z*other.z + w*other.w );
}



inline FVector4 operator * ( float scalar, const FVector4& fvector ){
	return FVector4( scalar*fvector.x, scalar*fvector.y, scalar*fvector.z, scalar*fvector.w );
}

inline std::ostream& operator << ( std::ostream& os, const FVector4& fvector ){
	os << "(" << fvector.x << ", " << fvector.y << ", " << fvector.z << ", " << fvector.w << ")";
	return os;
}





#endif // __FVectors_h__
