#include "Game.h"

#include "System.h"
#include "DotSceneLoader.h"
#include "Exception.h"
#include "File.h"
#include "StringTokenizer.h"
#include "Client.h"
#include "Camera.h"
#include "Console.h"
#include "Network.h"
#include "Server.h"
#include "Graphics.h"
#include "Gui.h"
#include "Hud.h"
#include "GameInfoMenu.h"
#include "EquipmentMenu.h"
#include "Area.h"
#include "SpawnArea.h"
#include "ConquestArea.h"
#include "Sound.h"

#include "ArenaFactory.h"
#include "VehicleFactory.h"
#include "WeaponFactory.h"
#include "ShotFactory.h"
#include "GeometryFactory.h"
#include "EffectFactory.h"

#include "Arena.h"
#include "Vehicle.h"
#include "Weapon.h"
#include "Shot.h"
#include "Geometry.h"
#include "Effect.h"

Game::Game(System* system): SubSystem(system){
	this->camera = new Camera();	// create here, because we need camera->mode as cvar
	this->physics = NULL;
	this->arena = NULL;

	this->running = false;
	this->shouldBreakMainLoop = false;

	this->idCounter = 0;

	this->arenaFactory = NULL;
	this->vehicleFactory = NULL;
	this->weaponFactory = NULL;
	this->shotFactory = NULL;
	this->effectFactory = NULL;
	this->geometryFactory = NULL;

	this->vehicles.clear();
	this->geometries.clear();
	this->effects.clear();
	this->shots.clear();
	this->areas.clear();

	this->mode = MODE_DEATHMATCH;

	this->cCmds.game_create = NULL;
	this->cCmds.game_join = NULL;
	this->cCmds.game_disconnect = NULL;
	this->cCmds.game_kill = NULL;

	this->cVars.game_camera_mode = NULL;

	this->resetStats();
}

Game::~Game(){
	delete camera;
}

void Game::initialize(){
	System::log("");
	System::log("*** Initializing Game ***");
	System::log("");

	System::log("Game: Creating arenaFactory...");
	this->arenaFactory = new ArenaFactory(this);

	System::log("Game: Creating vehicleFactory...");
	this->vehicleFactory = new VehicleFactory(this);

	System::log("Game: Creating weaponFactory...");
	this->weaponFactory = new WeaponFactory(this);

	System::log("Game: Creating shotFactory...");
	this->shotFactory = new ShotFactory(this);

	System::log("Game: Creating effectFactory...");
	this->effectFactory = new EffectFactory(this);

	System::log("Game: Creating geometryFactory...");
	this->geometryFactory = new GeometryFactory(this);

	string fwpResourcesConfigFilePath = this->system->getDataDir() + "fwp_resources.cfg";
	System::log("Game: Parsing fwp_resources.cfg...");
	this->parseFwpResources( fwpResourcesConfigFilePath );

	this->initialized = true;

	System::log("Game is ready.");
	System::log("");
}

void Game::shutdown(){
	System::log("");
	System::log("=== Shutting down Game ===");
	System::log("");


	if( this->geometryFactory != NULL ){
		System::log("Game: Destroying geometryFactory...");
		delete this->geometryFactory;
		this->geometryFactory = NULL;
	}
	if( this->effectFactory != NULL ){
		System::log("Game: Destroying effectFactory...");
		delete this->effectFactory;
		this->effectFactory = NULL;
	}
	if( this->shotFactory != NULL ){
		System::log("Game: Destroying shotFactory...");
		delete this->shotFactory;
		this->shotFactory = NULL;
	}
	if( this->weaponFactory != NULL ){
		System::log("Game: Destroying weaponFactory...");
		delete this->weaponFactory;
		this->weaponFactory = NULL;
	}
	if( this->vehicleFactory != NULL ){
		System::log("Game: Destroying vehicleFactory...");
		delete this->vehicleFactory;
		this->vehicleFactory = NULL;
	}
	if( this->arenaFactory != NULL ){
		System::log("Game: Destroying arenaFactory...");
		delete this->arenaFactory;
		this->arenaFactory = NULL;
	}

	this->initialized = false;

	System::log("Game is down.");
	System::log("");
}

void Game::registerCCmds( Console& console ){
	this->cCmds.game_create = new CCmdGameCreate();
	console.registerCCmd( this->cCmds.game_create );

	this->cCmds.game_join = new CCmdGameJoin();
	console.registerCCmd( this->cCmds.game_join );

	this->cCmds.game_disconnect = new CCmdGameDisconnect();
	console.registerCCmd( this->cCmds.game_disconnect );

	this->cCmds.game_kill = new CCmdGameKill();
	console.registerCCmd( this->cCmds.game_kill );

	this->cCmds.game_chatAll = new CCmdGameChatAll();
	console.registerCCmd( this->cCmds.game_chatAll );

	this->cCmds.game_chatTeam = new CCmdGameChatTeam();
	console.registerCCmd( this->cCmds.game_chatTeam );
}
void Game::unregisterCCmds( Console& console ){
	console.unregisterCCmd( this->cCmds.game_create );
	delete this->cCmds.game_create;

	console.unregisterCCmd( this->cCmds.game_join );
	delete this->cCmds.game_join;
	
	console.unregisterCCmd( this->cCmds.game_disconnect );
	delete this->cCmds.game_disconnect;

	console.unregisterCCmd( this->cCmds.game_kill );
	delete this->cCmds.game_kill;

	console.unregisterCCmd( this->cCmds.game_chatAll );
	delete this->cCmds.game_chatAll;

	console.unregisterCCmd( this->cCmds.game_chatTeam );
	delete this->cCmds.game_chatTeam;
}

void Game::registerCVars( Console& console ){
	this->cVars.game_camera_mode = new CVarInt("game.camera.mode", this->camera->getModePtr(), true);
	this->cVars.game_camera_mode->setValueRange( 0, Camera::NUM_MODES-1 );
	this->cVars.game_camera_mode->setFlags( CVar::FLAG_SYSTEM );
	console.registerCVar(this->cVars.game_camera_mode);
}
void Game::unregisterCVars( Console& console ){
	console.unregisterCVar( this->cVars.game_camera_mode );
	delete this->cVars.game_camera_mode;
}


void Game::parseFwpResources( const string& resourceFilePath ){
	const string& dataDir = this->system->getDataDir();

	File f(resourceFilePath, ios::in);
	if( !f.isOpen() ){
		throw Exception("Couldn't open resource config file.", "Game::parseFwpResources()");
	}

	string line;
	while( f.readLine( line, "#" ) ){
		if( line.length() == 0 ){
			continue;
		}

		StringTokenizer st( " \t\r\n=", "\"", false );
		const vector<string>& tokens = st.tokenize( line );

		if( tokens.size() != 2 ){
			System::warn("(in Game::parseFwpResources()): Wrong number of tokens in line %i of file '%s' (expected 2 but read %i). Ignoring line."
				, f.getLineNumber(), f.getPath().c_str(), tokens.size() );
		}

		string resourceType = tokens[0];
		string resourcePath = system->getDataDir() + tokens[1];

		if( resourceType == string("Arena") ){
			this->arenaFactory->addArena( resourcePath );
		}else if( resourceType == string("Vehicle") ){
			this->vehicleFactory->addVehicle( resourcePath );
		}else if( resourceType == string("Weapon") ){
			this->weaponFactory->addWeapon( resourcePath );
		}else if( resourceType == string("Shot") ){
			this->shotFactory->addShot( resourcePath );
		}else if( resourceType == string("Effect") ){
			this->effectFactory->addEffect( resourcePath );
		}else if( resourceType == string("Geometry") ){
			this->geometryFactory->addGeometry( resourcePath );
		}else{
			System::warn( "(Game::parseFwpResources()): Unknown resource type '%s'. Ignoring.", resourceType.c_str() );
		}
	}	//while
	f.close();
}

void Game::setup(){

//	System::log("Game: Applying cvars...");

	this->mode = this->system->getNetwork()->getServer()->properties.gameMode;

	this->idCounter = 0;

	System::log("Game: Preparing GameObject lists...");
	this->effects.clear();
	this->shots.clear();
	this->vehicles.clear();
	this->areas.clear();


	System::log("Game: Creating physics...");
	this->physics = new Physics( this );


	System::log("Game: Creating arena...");
	this->arena = this->arenaFactory->createArena( this->system->getNetwork()->getServer()->properties.arena );

	System::log("Game: Attaching arena...");
	this->arena->attach( this );
	this->arena->collectAreas( this->areas );

	System::log("Game: Setting up camera...");
	camera->attach( this );
	camera->reset();
	


	this->system->getTimer()->calculateDeltaT(); // calc deltaT once to compensate for the loading time


	// set state
	this->running = true;
	this->shouldBreakMainLoop = false;
}

void Game::cleanup(){

	this->running = false;
	this->shouldBreakMainLoop = false;

	if( this->camera->isAttached() ){
		System::log("Game: Detaching camera...");
		this->camera->detach();
	}

	if( !this->effects.empty() ){
		System::log("Game: Destroying effects...");
		for( effects_t::iterator e_iter = this->effects.begin(); e_iter != this->effects.end(); ++e_iter ){
			(*e_iter)->detach();
			delete (*e_iter);
		}
		this->effects.clear();
	}

	if( !this->geometries.empty() ){
		System::log("Game: Destroying geometries...");
		for( geometries_t::iterator g_iter = this->geometries.begin(); g_iter != this->geometries.end(); ++g_iter ){
			(*g_iter)->detach();
			this->geometryFactory->destroyGeometry(*g_iter);
		}
		this->geometries.clear();
	}

	if( !this->shots.empty() ){
		System::log("Game: Destroying shots...");
		for( shots_t::iterator s_iter = this->shots.begin(); s_iter != this->shots.end(); ++s_iter ){
			(*s_iter)->detach();
			this->shotFactory->destroyShot(*s_iter);
		}
		this->shots.clear();
	}

	if( !this->vehicles.empty() ){
		System::log("Game: Destroying vehicles...");
		for( vehicles_t::iterator v_iter = this->vehicles.begin(); v_iter != this->vehicles.end(); ++v_iter ){
			(*v_iter)->detach();
			this->vehicleFactory->destroyVehicle(*v_iter);
		}
		this->vehicles.clear();
	}

//	System::log("Destroying vehicle...");
//	this->unspawnVehicle( NULL );

	this->areas.clear();

	if( this->arena != NULL ){
		if( this->arena->isAttached() ){
			System::log("Game: Detaching arena...");
			this->arena->detach();
		}
		System::log("Game: Destroying arena...");
		this->arenaFactory->destroyArena( this->arena );
		this->arena = NULL;
	}

	if( this->physics != NULL ){
		System::log("Game: Destroying physics...");
		delete this->physics;
		this->physics = NULL;
	}

	Ogre::SceneManager* sm = system->getGraphics()->getSceneManager();
	sm->clearScene();

	System::log("Game: Resetting statistics...");
	this->resetStats();

}


void Game::mainLoop(){

//	this->running = true; states now in setup/cleanup for cleaner exception handling!

	// open eq. and info menu
	this->system->getGui()->getGameInfoMenu()->show();
	this->system->getGui()->getEquipmentMenu()->show();

//	System::log("Transfering control to main loop...");

	while( !shouldBreakMainLoop ){
		this->system->getTimer()->calculateDeltaT();
	
		this->system->getGui()->update();

		this->system->getInput()->processInput();

		this->system->getConsole()->processQueues();

		this->system->getNetwork()->update();

		// do game stuff
		this->update();

		this->system->getSound()->update();

		// update renderTargets
		this->system->getGraphics()->update();
	}
//	this->shouldBreakMainLoop = false;

//	System::log("Returned from main loop...");

//	this->running = false;
}

void Game::setShouldBreakMainLoop(bool newShouldBreakMainLoop){
	this->shouldBreakMainLoop = newShouldBreakMainLoop;
}
bool Game::getShouldBreakMainLoop() const {
	return this->shouldBreakMainLoop;
}

void Game::update(){
	float deltaT = this->system->getTimer()->getDeltaT();

	// simulation
	this->prepareForSimulationStep();
	this->physics->takeSimulationStep( deltaT );
	this->updateAfterSimulationStep();


	// update vehicles
	for( vehicles_t::iterator v_iter = this->vehicles.begin(); v_iter != this->vehicles.end(); ++v_iter ){
		(*v_iter)->update( deltaT );

		// check vehicle against areas
		for( areas_t::iterator a_iter = this->areas.begin(); a_iter != this->areas.end(); ++a_iter ){
			if( (*a_iter)->canAffectVehicle( *v_iter ) ){
				(*a_iter)->affectVehicle( *v_iter );
			}
		}
	}

	// update shots
	for( shots_t::iterator s_iter = this->shots.begin(); s_iter != this->shots.end(); ++s_iter ){
		if( (*s_iter)->shouldBeDestroyed() ){
			(*s_iter)->detach();
			this->shotFactory->destroyShot( *s_iter );
			s_iter = this->shots.erase( s_iter );
		}else{
			(*s_iter)->update( deltaT );
		}
	}

	// update geometries
	for( geometries_t::iterator g_iter = this->geometries.begin(); g_iter != this->geometries.end(); ++g_iter ){
		if( (*g_iter)->shouldBeDetached() ){

			// FIXME: this sucks! check dependent effects and delete them as well
			for( effects_t::iterator iter = this->effects.begin(); iter != this->effects.end(); ++iter ){
				if( (*iter)->getGameObject() == *(g_iter) ){
					(*iter)->detach();
					this->effectFactory->destroyEffect( *iter );
					iter = this->effects.erase( iter );
				}
			}
			
			(*g_iter)->detach();
			this->geometryFactory->destroyGeometry( *g_iter );
			g_iter = this->geometries.erase( g_iter );

		}else{
			(*g_iter)->update( deltaT );
		}
	}

	// update effects
	for( effects_t::iterator e_iter = this->effects.begin(); e_iter != this->effects.end(); ++e_iter ){
		if( (*e_iter)->shouldBeDestroyed() ){
//			this->unspawnEffect( *iter );
			(*e_iter)->detach();
			this->effectFactory->destroyEffect( *e_iter );
			e_iter = this->effects.erase( e_iter );
		}else{
			(*e_iter)->update( deltaT );
		}
	}

	// update camera
	this->camera->update( deltaT );

	this->updateStats();
}

void Game::prepareForSimulationStep(){
	for( geometries_t::iterator g_iter = this->geometries.begin(); g_iter != this->geometries.end(); ++g_iter ){
		(*g_iter)->prepareForSimulationStep();
	}
	for( vehicles_t::iterator v_iter = this->vehicles.begin(); v_iter != this->vehicles.end(); ++v_iter ){
		(*v_iter)->prepareForSimulationStep();
	}
	for( shots_t::iterator s_iter = this->shots.begin(); s_iter != this->shots.end(); ++s_iter ){
		(*s_iter)->prepareForSimulationStep();
	}
}
void Game::updateAfterSimulationStep(){
	for( geometries_t::iterator g_iter = this->geometries.begin(); g_iter != this->geometries.end(); ++g_iter ){
		(*g_iter)->updateAfterSimulationStep();
	}
	for( vehicles_t::iterator v_iter = this->vehicles.begin(); v_iter != this->vehicles.end(); ++v_iter ){
		(*v_iter)->updateAfterSimulationStep();
	}
	for( shots_t::iterator s_iter = this->shots.begin(); s_iter != this->shots.end(); ++s_iter ){
		(*s_iter)->updateAfterSimulationStep();
	}
}







void Game::processInput(Input::actionMap_t& actionMap, Input::mouseState_t& mouseState){
	GameObject* camTarget = this->camera->getTarget();

	if( (this->camera->getMode() == Camera::MODE_FIRST_PERSON || this->camera->getMode() == Camera::MODE_CHASE)
		&& camTarget != NULL && camTarget == this->system->getNetwork()->getClient()->getVehicle() ){
			((Vehicle*)camTarget)->processInput(actionMap, mouseState);
	}else{
		this->camera->processInput(actionMap, mouseState);
	}
}

FVector3 Game::findSpawnPosition( int team ){
	// find spawn area
	int counter = 0;
	areas_t::iterator iter;
	for( iter = this->areas.begin(); iter != this->areas.end(); ++iter ){
		if( (*iter)->getAreaType() == Area::AREA_TYPE_SPAWN && ((SpawnArea*)(*iter))->isUsableForTeam( team ) ){
			counter++;
		}
	}

	int index = this->system->getRandom()->nextInt( counter - 1 );
	counter = 0;

	SpawnArea* sa = NULL;
	for( iter = this->areas.begin(); iter != this->areas.end(); ++iter ){
		if( (*iter)->getAreaType() == Area::AREA_TYPE_SPAWN && ((SpawnArea*)(*iter))->isUsableForTeam( team ) ){
			if( counter == index ){
				sa = (SpawnArea*)(*iter);
				break;
			}else{
				counter ++;
			}
		}
	}

	if( sa == NULL ){
		System::error( "(in Game::findSpawnPosition()): No usable spawn area in list for team %i.", team );
		throw Exception( "Couldn't find spawn position for client.", "Game::findSpawnPosition()" );
	}

	return sa->getSpawnPosition();
}


Vehicle* Game::spawnVehicle( Client* client ){
	Network* network = this->system->getNetwork();

	Vehicle* v = this->vehicleFactory->createVehicle( network->getServer()->lookUpVehicleName(client->info.vehicle) );

	v->attach( this, client );
//	v->position = client->state.position;
//	v->orientation = client->state.orientation;

	// attach weapons (MUST be done AFTER vehicle is attached!)
	int numSlots = v->getPhysicalProperties().weaponSlots.size();
	for( int i=0; i<numSlots; i++ ){
		if( client->info.weapons[i] != -1 ){
			Weapon* w = this->weaponFactory->createWeapon( network->getServer()->lookUpWeaponName(client->info.weapons[i]) );
			w->attach( v, i );
		}
	}

	this->vehicles.push_back( v );

	this->spawnEffect("vehicle_spawn", v);

	if( client == this->system->getNetwork()->getClient() ){
		this->camera->setTarget( v );
		this->cVars.game_camera_mode->setValue( Camera::MODE_CHASE );
	}

	return v;
}

void Game::unspawnVehicle( Vehicle* vehicle ){
	if( this->camera->getTarget() == vehicle ){
		this->camera->setTarget( NULL );
		this->camera->angles = FVector3::ZERO; // THINKABOUTME: better solution...

		this->camera->position = vehicle->position + 50.0f*FVector3::UNIT_Z;
//		this->camera->lookAt( vehicle->position );
	}

//	Effect* e = this->spawnEffect("vehicle_spawn");
//	e->position = vehicle->position;

	// detach waepons
	int numSlots = vehicle->getPhysicalProperties().weaponSlots.size();
	for( int i=0; i<numSlots; i++ ){
		Weapon* w = vehicle->getWeapon(i);
		if( w != NULL ){
			w->detach();
			this->weaponFactory->destroyWeapon( w );
		}
	}


	vehicle->detach();
	this->vehicles.remove( vehicle );

	this->vehicleFactory->destroyVehicle( vehicle );

	// check dependent effects and delete them as well
	for( effects_t::iterator iter = this->effects.begin(); iter != this->effects.end(); ++iter ){
		if( (*iter)->getGameObject() == vehicle ){
			(*iter)->detach();
			this->effectFactory->destroyEffect( *iter );
			iter = this->effects.erase( iter );
		}
	}

}

Effect* Game::spawnEffect( const string& name, GameObject* gameObject ){
	Effect* e = this->effectFactory->createEffect( name );

	e->attach( this, gameObject );

	this->effects.push_back( e );

	return e;
}

Effect* Game::spawnEffect( const string& name, const FVector3& position, const FQuaternion& orientation ){
	Effect* e = this->effectFactory->createEffect( name );

	e->position = position;
	e->orientation = orientation;

	e->attach( this, (GameObject*)NULL );

	this->effects.push_back( e );

	return e;
}

void Game::unspawnEffect( Effect* effect ){
/*
	for( effects_t::iterator iter = this->effects.begin(); iter != this->effects.end(); iter++ ){
		if( *iter == effect ){
			effect->detach();
			delete effect;
			this->effects.erase( iter );
			return;
		}
	}
*/

	// faster version
	effect->detach();
	this->effectFactory->destroyEffect( effect );
	this->effects.remove( effect );

}



Geometry* Game::spawnGeometry( const string& name, const FVector3& position, const FQuaternion& orientation ){
	Geometry* g = this->geometryFactory->createGeometry( name );

	g->position = position;
	g->orientation = orientation;

	g->attach( this );

	this->geometries.push_back( g );

	return g;
}

void Game::unspawnGeometry( Geometry* geometry ){
	geometry->detach();
	this->geometryFactory->destroyGeometry( geometry );
	this->geometries.remove( geometry );

	// check dependent effects and delete them as well
	for( effects_t::iterator iter = this->effects.begin(); iter != this->effects.end(); ++iter ){
		if( (*iter)->getGameObject() == geometry ){
			(*iter)->detach();
			this->effectFactory->destroyEffect( *iter );
			iter = this->effects.erase( iter );
		}
	}
}



Shot* Game::spawnShot( Weapon* weapon, const FVector3& position, const FQuaternion& orientation ){
	// create shot
	Shot* s = this->shotFactory->createShot( weapon->getShot() );

	// set direction, etc.
	s->position = position;
	s->orientation = orientation;
	Ogre::Quaternion q = orientation.toOgreQuaternion();
	s->direction = -q.zAxis();
	s->up = q.yAxis();
	s->right = q.xAxis();
	s->velocity = s->direction * s->getPhysicalProperties().muzzleVelocity;

	// set clientId, muzzle effect, etc. (pos and dir MUST be set, because they are needed for recoil impulse and effect!)
	weapon->setUpShot( s );

	s->attach( this );

	this->shots.push_back( s );

	return s;
}


bool Game::isArenaOwnedBy( Game::teams_e team ) const {
	if( team == TEAM_RED ){
		for( areas_t::const_iterator c_a_iter = this->areas.begin(); c_a_iter != this->areas.end(); ++c_a_iter ){
			if( (*c_a_iter)->getAreaType() == Area::AREA_TYPE_CONQUEST ){
				ConquestArea* ca = (ConquestArea*)(*c_a_iter);
	
				if( ca->getCapturePercentage() > -50.0f ){
					return false;
				}
			}
		}

	}else if( team == TEAM_BLUE ){
		for( areas_t::const_iterator c_a_iter = this->areas.begin(); c_a_iter != this->areas.end(); ++c_a_iter ){
			if( (*c_a_iter)->getAreaType() == Area::AREA_TYPE_CONQUEST ){
				ConquestArea* ca = (ConquestArea*)(*c_a_iter);
	
				if( ca->getCapturePercentage() < 50.0f ){
					return false;
				}
			}
		}

	}else{
	}

	return true;
}



void Game::updateStats(){
	this->stats.position = this->camera->position;
}
void Game::resetStats(){
	this->stats.position = FVector3::ZERO;
}
const Game::stats_s& Game::getStats() const {
	return this->stats;
}

unsigned int Game::getNextId(){
	return ++idCounter;		// THINKABOUTME: Maybe I should to some more checks here...
}


Camera* Game::getCamera() const {
	return this->camera;
}

Physics* Game::getPhysics() const {
	return this->physics;
}
Arena* Game::getArena() const {
	return this->arena;
}
ArenaFactory* Game::getArenaFactory() const {
	return this->arenaFactory;
}
VehicleFactory* Game::getVehicleFactory() const {
	return this->vehicleFactory;
}
WeaponFactory* Game::getWeaponFactory() const {
	return this->weaponFactory;
}
ShotFactory* Game::getShotFactory() const {
	return this->shotFactory;
}
EffectFactory* Game::getEffectFactory() const {
	return this->effectFactory;
}
int Game::getMode() const {
	return this->mode;
}
bool Game::isRunningTeamMode() const {
	return ( this->mode == Game::MODE_TEAM_DEATHMATCH || this->mode == Game::MODE_CONQUEST );
}
Vehicle* Game::getVehicleByClientId( int clientId ) const {
	Client* c = this->system->getNetwork()->getServer()->getClient( clientId );
	if( c != NULL ){
		return c->getVehicle();
	}else{
		return NULL;
	}

}

bool Game::isRunning() const {
	return this->running;
}


const Game::areas_t& Game::getAreas() const {
	return this->areas;
}


string Game::getModeName(int mode){
	if( mode == MODE_EDIT ){
		return string("edit");
	}else if( mode == MODE_DEATHMATCH ){
		return string("deathmatch");
	}else if( mode == MODE_TEAM_DEATHMATCH ){
		return string("team deathmatch");
	}else if( mode == MODE_CONQUEST ){
		return string("conquest");
	}else{
		return string("<unknown mode>");
	}
}

string Game::getTeamName(int team){
	if( team == TEAM_SPECTATORS ){
		return string("spectators");
	}else if( team == TEAM_PLAYERS ){
		return string("players");
	}else if( team == TEAM_RED ){
		return string("red team");
	}else if( team == TEAM_BLUE ){
		return string("blue team");
	}else{
		return string("<unknown team>");
	}
}
