#include "Geometry.h"

#include "Exception.h"
#include "XmlUtils.h"
#include "System.h"
#include "Shot.h"

Geometry::Geometry( geometryTypes_e geometryType ): GameObject( GameObject::GAME_OBJECT_TYPE_GEOMETRY ) {
//	this->sourceType = SOURCE_TYPE_MESH;
//	this->sourceName = "<unitialized>";
	this->geometryType = geometryType;
	this->name = "<unitialized>";

	this->checksum = 0;

	this->instanceId = 0;
	this->instanceName = "<uninitialized>";

	this->graphicalProperties.mesh.clear();
	this->graphicalProperties.castShadows = true;
	this->graphicalProperties.receiveShadows = true;
	this->graphicalProperties.staticGeometry = true;
	this->graphicalProperties.explosionEffect.clear();

	this->physicalProperties.movable = false;
	this->physicalProperties.affectedByGravity = false;

//	this->physicalProperties.collidable = false;
	this->physicalProperties.surfaceType = SURFACE_TYPE_STONE;
	this->physicalProperties.bouncyness = 0.0f;
	this->physicalProperties.friction = 0.0f;

	this->physicalProperties.destroyable = false;
	this->physicalProperties.healthCapacity = 0.0f;
	this->physicalProperties.healthDegeneration = 0.0f;

	this->physicalProperties.inflictDamage = false;
	this->physicalProperties.damagePerSecond = 0.0f;

	this->physicalProperties.shape.clear();
	dMassSetZero( &this->physicalProperties.mass );
	this->physicalProperties.debris.clear();

}

Geometry::~Geometry(){
}



void Geometry::fromXmlElement( TiXmlElement* xmlElement ){
	// name
	const char* tmp = xmlElement->Attribute("name");
	if( tmp != NULL ){
		this->name = string( tmp );
	}else{
		throw Exception( "Geometry element has no name attribute.", "Geometry::fromXmlElement()" );
	}
	if( this->name.length() > 32 ){
		System::error( "(in Geometry::fromXmlElement()): Geometry name '%s' is longer than 32 characters.", this->name.c_str() );
		throw Exception( "XmlElement is not a valid geometry description.", "Geometry::fromXmlElement()" );
	}


	// type and version checked in GeometryFactory...

	for( TiXmlElement* elt = xmlElement->FirstChildElement(); elt != NULL; elt = elt->NextSiblingElement() ){
		string value = string( elt->Value() );
		
		if(value == "position"){
			this->position = XmlUtils::readFVector3( elt );
		}else if(value == "orientation"){
			this->orientation = XmlUtils::readFQuaternion( elt );

		
		}else if(value == "graphicalProperties"){// maybe the user wants to overwrite some properties in the arena file...
			this->parseGraphicalProperties(elt);
		}else if(value == "physicalProperties"){// maybe the user wants to overwrite some properties in the arena file...
			this->parsePhysicalProperties(elt);

		}else{
			System::warn("(in Geometry::fromXmlElement()): Unknown child element: '%s' (row: %i). Ignoring.", value.c_str(), elt->Row() );
		}
	}

}

TiXmlElement Geometry::toXmlElement() const {
	return TiXmlElement("arsch");
}

void Geometry::calculateChecksum(){
	this->checksum = 0;
}


void Geometry::parseGeometryFile( const string& path ){
	TiXmlDocument doc( path.c_str() );
	if( !doc.LoadFile() ){
		throw Exception("Couldn't load file '" + path + "'. (Maybe not a correct xml file?)", "Geometry::parseGeometryFile()");
	}
//	doc.Print();

	TiXmlNode* rootElement = doc.RootElement();
	TiXmlElement* element = rootElement->ToElement();

	string rootElementName = element->Value();
	if( rootElementName != "geometry" ){
		throw Exception("Geometry file '" + path + "' does not have a <geometry> root node.", "Geometry::parseGeometryFile()");
	}

	// check version
	const char* tmp = element->Attribute( "formatVersion" );
	if( tmp == NULL ){
		throw Exception("Geometry file '" + path + "' has no version information.", "Geometry::parseGeometryFile()");
	}else{
		string version = tmp;

		if( version != System::getInstance()->getVersionString() ){
			System::warn("(in Geometry::parseGeometryFile()): file '%s' has version %s. Trying to parse anyway.", path.c_str(), version.c_str() );
		}
//		this->versionString = version;
	}


	// parse graphicalProperties
	element = rootElement->FirstChildElement("graphicalProperties");
	if( element ){
		this->parseGraphicalProperties(element);
	}else{
		throw Exception("Geometry file '" + path + "' has no <graphicalProperties> node.", "Geometry::parseGeometryFile()");
	}

	// parse physicalProperties
	element = rootElement->FirstChildElement("physicalProperties");
	if( element ){
		this->parsePhysicalProperties(element);
	}else{
		throw Exception("Geometry file '" + path + "' has no <physicalProperties> node.", "Geometry::parseGeometryFile()");
	}

}

void Geometry::parseGraphicalProperties( TiXmlElement* graphicalPropertiesElement ){
	XmlUtils::readStringProperty(graphicalPropertiesElement, "mesh", this->graphicalProperties.mesh);
	XmlUtils::readBoolProperty(graphicalPropertiesElement, "castShadows", this->graphicalProperties.castShadows);
	XmlUtils::readBoolProperty(graphicalPropertiesElement, "receiveShadows", this->graphicalProperties.receiveShadows);
	XmlUtils::readBoolProperty(graphicalPropertiesElement, "staticGeometry", this->graphicalProperties.staticGeometry);
	XmlUtils::readStringProperty(graphicalPropertiesElement, "explosionEffect", this->graphicalProperties.explosionEffect);
}


void Geometry::parsePhysicalProperties( TiXmlElement* physicalPropertiesElement ){
	XmlUtils::readBoolProperty(physicalPropertiesElement, "movable", this->physicalProperties.movable);
	XmlUtils::readBoolProperty(physicalPropertiesElement, "affectedByGravity", this->physicalProperties.affectedByGravity);


//	XmlUtils::readBoolProperty(physicalPropertiesElement, "collidable", this->physicalProperties.collidable);
//	this->physicalProperties.surfaceType = SURFACE_TYPE_STONE;
	XmlUtils::readFloatProperty(physicalPropertiesElement, "bouncyness", this->physicalProperties.bouncyness);
	XmlUtils::readFloatProperty(physicalPropertiesElement, "friction", this->physicalProperties.friction);
//	printf("friction: %f\n", this->physicalProperties.friction);

	XmlUtils::readBoolProperty(physicalPropertiesElement, "destroyable", this->physicalProperties.destroyable);
	XmlUtils::readFloatProperty(physicalPropertiesElement, "healthCapacity", this->physicalProperties.healthCapacity);
	XmlUtils::readFloatProperty(physicalPropertiesElement, "healthDegeneration", this->physicalProperties.healthDegeneration);

	XmlUtils::readBoolProperty(physicalPropertiesElement, "inflictDamage", this->physicalProperties.inflictDamage);
	XmlUtils::readFloatProperty(physicalPropertiesElement, "damagePerSecond", this->physicalProperties.damagePerSecond);

	for( TiXmlElement* elt = physicalPropertiesElement->FirstChildElement(); elt != NULL; elt = elt->NextSiblingElement() ){
		string value = string( elt->Value() );

		if( value == string("shape") ){
			this->physicalProperties.shape.fromXmlElement( elt );
		}else if( value == string("mass") ){
			this->parseMass( elt );
		}else if( value == string("debris") ){
			this->parseDebris( elt );
		}else if( value == string("property") ){
			// already read
		}else{
			System::warn("(in Geometry::parsePhysicalProperties()): Unknown child element: '%s' (row: %i). Ignoring.", value.c_str(), elt->Row() );
		}
	}

}

void Geometry::parseMass( TiXmlElement* massElement ){
	for( TiXmlElement* elt = massElement->FirstChildElement(); elt != NULL; elt = elt->NextSiblingElement() ){
		string value = string( elt->Value() );
		if(value == "sphere"){
			float m = elt->Attribute("mass") ? atof(elt->Attribute("mass")) : 0.0f;
			float r = elt->Attribute("radius") ? atof(elt->Attribute("radius")) : 0.0f;

			dMass tempMass;
			dMassSetSphereTotal( &tempMass, m, r );
			dMassAdd( &this->physicalProperties.mass, &tempMass );
		}else if(value == "box"){
			float m = elt->Attribute("mass") ? atof(elt->Attribute("mass")) : 0.0f;
			float w = elt->Attribute("width") ? atof(elt->Attribute("width")) : 0.0f;
			float h = elt->Attribute("height") ? atof(elt->Attribute("height")) : 0.0f;
			float d = elt->Attribute("depth") ? atof(elt->Attribute("depth")) : 0.0f;

			dMass tempMass;
			dMassSetBoxTotal( &tempMass, m, w, h, d );
			dMassAdd( &this->physicalProperties.mass, &tempMass );

		}else if(value == "capsule"){
			float m = elt->Attribute("mass") ? atof(elt->Attribute("mass")) : 0.0f;
			float r = elt->Attribute("radius") ? atof(elt->Attribute("radius")) : 0.0f;
			float l = elt->Attribute("length") ? atof(elt->Attribute("length")) : 0.0f;

			dMass tempMass;
			dMassSetCappedCylinderTotal( &tempMass, m, 3, r, l );
			dMassAdd( &this->physicalProperties.mass, &tempMass );

		}else{
			System::warn("(in Geometry::parseMass()): Unknown child element: '%s' (row: %i). Ignoring.", value.c_str(), elt->Row() );
		}
	}
}

void Geometry::parseDebris( TiXmlElement* debrisElement ){
	physicalProperties_s::debris_s debris;

	debris.initialPosition = FVector3::ZERO;
	debris.initialAngularImpulse = FVector3::ZERO;
	debris.initialLinearImpulse = FVector3::ZERO;

	if( debrisElement->Attribute("geometry") != NULL ){
		debris.geometry = string( debrisElement->Attribute("geometry") );
	}
	if( debrisElement->Attribute("effect") != NULL ){
		debris.effect = string( debrisElement->Attribute("effect") );
	}
	if( debrisElement->Attribute("initialPosition") != NULL ){
		const char* val = debrisElement->Attribute("initialPosition");
		int n = sscanf( val, "x:%f y:%f z:%f", &debris.initialPosition.x, &debris.initialPosition.y, &debris.initialPosition.z );
	}
	if( debrisElement->Attribute("initialAngularImpulse") != NULL ){
		const char* val = debrisElement->Attribute("initialAngularImpulse");
		int n = sscanf( val, "x:%f y:%f z:%f", &debris.initialAngularImpulse.x, &debris.initialAngularImpulse.y, &debris.initialAngularImpulse.z );
	}
	if( debrisElement->Attribute("initialLinearImpulse") != NULL ){
		const char* val = debrisElement->Attribute("initialLinearImpulse");
		int n = sscanf( val, "x:%f y:%f z:%f", &debris.initialLinearImpulse.x, &debris.initialLinearImpulse.y, &debris.initialLinearImpulse.z );
	}

	this->physicalProperties.debris.push_back( debris );
}


const Geometry::physicalProperties_s& Geometry::getPhysicalProperties() const {
	return this->physicalProperties;
}
const Geometry::graphicalProperties_s& Geometry::getGraphicalProperties() const {
	return this->graphicalProperties;
}


Geometry::geometryTypes_e Geometry::getGeometryType() const {
	return this->geometryType;
}

const string& Geometry::getName() const {
	return this->name;
}
