#ifndef __Geometry_h__
#define __Geometry_h__

#include "GameObject.h"
#include "Shape.h"
#include <Ogre.h>

class Shot;

class Geometry: public GameObject {
public:

	enum geometryTypes_e{
		GEOMETRY_TYPE_STATIC = 0,
		GEOMETRY_TYPE_DYNAMIC,

		NUM_GEOMETRY_TYPES
	};

	enum surfaceTypes_e {
		SURFACE_TYPE_METAL = 0,
		SURFACE_TYPE_STONE,
		SURFACE_TYPE_SAND,
		SURFACE_TYPE_WOOD,

		NUM_SURFACE_TYPES
	};

	struct graphicalProperties_s{
		string mesh;
		bool castShadows;
		bool receiveShadows;
		bool staticGeometry;
		string explosionEffect;
	};

	struct physicalProperties_s{
		bool movable;
		bool affectedByGravity;

		float bouncyness;
		float friction;
		surfaceTypes_e surfaceType;

		bool inflictDamage;
		float damagePerSecond;

		bool destroyable;
		float healthCapacity;
		float healthDegeneration;

		dMass mass;
		Shape shape;

		struct debris_s{
			string geometry;
			string effect;
			FVector3 initialPosition;
			FVector3 initialLinearImpulse;
			FVector3 initialAngularImpulse;
		};
		vector<debris_s> debris;

	};



	Geometry( geometryTypes_e geometryType );
	virtual ~Geometry() = 0;

	virtual void reset() = 0;
	virtual Geometry* clone() const = 0;

	virtual void attach(Game* game) = 0;
	virtual void detach() = 0;
	virtual void update( float deltaT ) = 0;

	virtual void fromXmlElement( TiXmlElement* xmlElement );
	virtual TiXmlElement toXmlElement() const;
	virtual void calculateChecksum();

	// Physics interface
	virtual void prepareForSimulationStep() = 0;;
	virtual void updateAfterSimulationStep() = 0;;
	virtual bool collidedWithGameObject(const FVector3& contactPoint, GameObject* other, dContact& contact) = 0;
	virtual bool collidedWithTerrain(const FVector3& contactPoint, dContact& contact) = 0;
	virtual void addLinearImpulse( const FVector3& impulse, const FVector3& position ) = 0;
	virtual void addAngularImpulse( const FVector3& impulse ) = 0;

	virtual void damage( float amount ) = 0;
	virtual void explode() = 0;

	virtual const physicalProperties_s& getPhysicalProperties() const;
	virtual const graphicalProperties_s& getGraphicalProperties() const;
	virtual bool shouldBeDetached() const  = 0;
	virtual geometryTypes_e getGeometryType() const;

	virtual const string& getName() const;

protected:
	geometryTypes_e geometryType;
	string name;
	unsigned int checksum;

	unsigned int instanceId;
	string instanceName;

	graphicalProperties_s graphicalProperties;
	physicalProperties_s physicalProperties;

	virtual void parseGeometryFile( const string& path );
	virtual void parseGraphicalProperties( TiXmlElement* graphicalPropertiesElement );
	virtual void parsePhysicalProperties( TiXmlElement* physicalPropertiesElement );
	virtual void parseMass( TiXmlElement* massElement );
	virtual void parseDebris( TiXmlElement* debrisElement );

};

#endif // __Geometry_h__
