#include "Graphics.h"

#include "System.h"
#include "Exception.h"
#include "File.h"
#include "Input.h"
#include "Console.h"
#include "Gui.h"

// needed for win32 event handling
#if FWP_PLATFORM == FWP_PLATFORM_WIN32
#   define WIN32_LEAN_AND_MEAN
#   include <direct.h>
#   include <windows.h>
#endif


Graphics::Graphics(System* system): SubSystem(system){
	this->ogreRoot = NULL;
	this->renderWindow = NULL;
	this->mainViewport = NULL;
	this->sceneManager = NULL;
	this->mainCamera = NULL;

//	this->frameListener = NULL;

	this->shadowsEnabled = false;
	this->forceOgreConfigDialog = false;

	this->resetStats();

	this->lastUpdateMillis = 0;
}

Graphics::~Graphics(){
}

void Graphics::initialize(){
	System::log("");
	System::log("*** Initializing Graphics ***");
	System::log("");

	string ogrePluginConfigPath = this->system->getConfigDir() + "ogre_plugins_" + this->system->getPlatformString() + ".cfg";
	string ogreConfigPath = this->system->getConfigDir() + "ogre.cfg";
	string ogreLogPath = this->system->getLogDir() + "ogre.log";
	string ogreResourcesConfigFilePath = this->system->getDataDir() + "ogre_resources.cfg";

	if( !File::exists( ogrePluginConfigPath ) ){
		System::warn("(in Graphics::initialize()): '%s' doesn't exist. Creating generic file.\n"
			"(Maybe you'll have to adjust 'PluginFolder'!)", ogrePluginConfigPath.c_str() );
		createOgrePluginConfigFile( ogrePluginConfigPath );
	}

	System::log("Graphics: Creating Ogre::Root...");
	ogreRoot = new Ogre::Root(ogrePluginConfigPath, ogreConfigPath, ogreLogPath);

	System::log("Graphics: Setting up Ogre resources...");
	setupOgreResources( ogreResourcesConfigFilePath );

	System::log("Graphics: Configuring Ogre...");
	configureOgre();

	System::log("Graphics: Initializing Ogre...");
	Ogre::ResourceGroupManager::getSingleton().initialiseAllResourceGroups();
//	Ogre::MeshManager::getSingleton().setPrepareAllMeshesForShadowVolumes( true );

	System::log("Graphics: Creating mainViewport...");
	this->createMainViewport();

	System::log("Graphics: Creating InputListener...");
	this->system->getInput()->createInputListener();

	this->lastUpdateMillis = 0;

	System::log("Graphics is ready.");
	System::log("");

}

void Graphics::shutdown(){
	System::log("");
	System::log("=== Shutting down Graphics ===");
	System::log("");

	System::log("Graphics: Destroying mainViewport...");
	this->destroyMainViewport();

	System::log("Graphics: Destroying InputListener...");
	this->system->getInput()->destroyInputListener();

//	System::log("Destroying FrameListener...");
//	destroyFrameListener();


	if( ogreRoot != NULL ){
		System::log("Graphics: Shutting down Ogre...");
		ogreRoot->shutdown();

		System::log("Graphics: Deleting Ogre::Root object...");
		delete ogreRoot;
	}

	System::log("Graphics is down.");
	System::log("");

}
void Graphics::registerCCmds( Console& console ){
}
void Graphics::unregisterCCmds( Console& console ){
}
void Graphics::registerCVars( Console& console ){
	this->cVars.graphics_shadowsEnabled = new CVarBool("graphics.shadowsEnabled", &this->shadowsEnabled, true);
	this->cVars.graphics_shadowsEnabled->setChangeString("(changes may affect only newly loaded arenas)");
	this->cVars.graphics_shadowsEnabled->setFlags( CVar::FLAG_SYSTEM );
	console.registerCVar(this->cVars.graphics_shadowsEnabled);

	this->cVars.graphics_forceOgreConfigDialog = new CVarBool("graphics.forceOgreConfigDialog", &this->forceOgreConfigDialog, true);
	this->cVars.graphics_forceOgreConfigDialog->setFlags( CVar::FLAG_SYSTEM );
	console.registerCVar(this->cVars.graphics_forceOgreConfigDialog);
}
void Graphics::unregisterCVars( Console& console ){
	console.unregisterCVar( this->cVars.graphics_forceOgreConfigDialog );
	delete this->cVars.graphics_forceOgreConfigDialog;

	console.unregisterCVar( this->cVars.graphics_shadowsEnabled );
	delete this->cVars.graphics_shadowsEnabled;
}


void Graphics::setupOgreResources( const string& ogreResourcesFilePath ){
	Ogre::ConfigFile cf;
	const string& dataDir = this->system->getDataDir();

	cf.load(ogreResourcesFilePath);

// Go through all sections & settings in the file
	Ogre::ConfigFile::SectionIterator seci = cf.getSectionIterator();

	string resName, resType, resGroup;
	while( seci.hasMoreElements() ){
		resGroup = seci.peekNextKey();
		Ogre::ConfigFile::SettingsMultiMap *settings = seci.getNext();
		Ogre::ConfigFile::SettingsMultiMap::iterator i;
		for( i = settings->begin(); i != settings->end(); ++i ){
			resType = i->first;
			resName = i->second;
			Ogre::ResourceGroupManager::getSingleton().addResourceLocation(
				dataDir + resName, resType, resGroup);
		}
	}
}

void Graphics::configureOgre(){

	string windowTitle = "Fight Win Prevail " + this->system->getVersionString();
	if( !this->forceOgreConfigDialog && ogreRoot->restoreConfig() ){
		renderWindow = ogreRoot->initialise(true, windowTitle );
	}else{
		System::warn("(in Graphics::configureOgre()): No working Ogre configuration found or config dialog forced. Must configure by hand.");
		if( ogreRoot->showConfigDialog() ){
			renderWindow = ogreRoot->initialise(true, windowTitle );
		}else{
			throw Exception("User has aborted Ogre configuration.", "Graphics::configureOgre()");
		}
	}
}

void Graphics::createOgrePluginConfigFile( const string& path ){
	File f( path, ios::out );

	if( !f.isOpen() ){
		System::error( "(in Graphics::createOgrePluginConfigFile()): Couldn't create file '%s'.", f.getPath().c_str() );
		throw Exception( "Couldn't create Ogre plugin config file.", "Graphics::createOgrePluginConfigFile()" );
	}

	if( this->system->getPlatform() == System::PLATFORM_WIN32 ){

		f.writeLine("# Defines ogre plugins to load (generated by FWP)");
		f.writeLine("");
		f.writeLine("# Define plugin folder");
		f.writeLine("# Adjust this to match your ogre plugins folder.");
		f.writeLine("PluginFolder=" + string(OGRE_PLUGIN_DIR) );
		f.writeLine("");
		f.writeLine("# Define plugins");
		f.writeLine("Plugin=RenderSystem_Direct3D9");
		f.writeLine("Plugin=RenderSystem_GL");
		f.writeLine("#Plugin=RenderSystem_Direct3D7");
		f.writeLine("Plugin=Plugin_ParticleFX");
		f.writeLine("#Plugin=Plugin_BSPSceneManager");
		f.writeLine("Plugin=Plugin_OctreeSceneManager");
		f.writeLine("#Plugin=Plugin_CgProgramManager");

	}else if( this->system->getPlatform() == System::PLATFORM_LINUX ){

		f.writeLine("# Defines ogre plugins to load (generated by FWP)");
		f.writeLine("");
		f.writeLine("# Define plugin folder");
		f.writeLine("# Adjust this to match your ogre plugins folder.");
		f.writeLine("PluginFolder=" + string(OGRE_PLUGIN_DIR) );
		f.writeLine("");
		f.writeLine("# Define plugins");
		f.writeLine("#Plugin=RenderSystem_Direct3D9");
		f.writeLine("Plugin=RenderSystem_GL");
		f.writeLine("#Plugin=RenderSystem_Direct3D7");
		f.writeLine("Plugin=Plugin_ParticleFX");
		f.writeLine("#Plugin=Plugin_BSPSceneManager");
		f.writeLine("Plugin=Plugin_OctreeSceneManager");
		f.writeLine("#Plugin=Plugin_CgProgramManager");

	}else{
		throw Exception("Tried to create OgrePluginConfigFile for unknown platform!", "Graphics::createOgrePluginConfigFile()");
	}

	f.close();
}

void Graphics::createMainViewport(){
	//create mainViewport AND display loading screen
	this->setSceneManagerByType( Ogre::ST_GENERIC );
	Ogre::SceneManager* sm = this->getSceneManager();

	Ogre::Overlay* ol = Ogre::OverlayManager::getSingleton().getByName("Gui/LoadingScreen");
	ol->show();
	this->renderWindow->update();
	ol->hide();

	sm->clearScene();
}

void Graphics::destroyMainViewport(){
//	this->renderWindow->removeViewport(500);
}


void Graphics::update(){
	Timer* timer = this->system->getTimer();
	unsigned long currentMillis = timer->getMilliseconds();
//	printf("currentMillis: %u\n", currentMillis);
	while( lastUpdateMillis + 2 > currentMillis ){ // 500 fps limit
		currentMillis = timer->getMilliseconds();
	}

	// Pump events on Win32
#if FWP_PLATFORM == FWP_PLATFORM_WIN32
    MSG  msg;
	while( PeekMessage( &msg, NULL, 0U, 0U, PM_REMOVE ) ){
		TranslateMessage( &msg );
		DispatchMessage( &msg );
	}
#endif

	this->ogreRoot->renderOneFrame();

	this->updateStats();

	this->lastUpdateMillis = currentMillis;
}

void Graphics::updateStats(){
	const Ogre::RenderTarget::FrameStats& frameStats = this->renderWindow->getStatistics();

	this->stats.lastFPS = frameStats.lastFPS;
	this->stats.bestFPS = frameStats.bestFPS;
	this->stats.worstFPS = frameStats.worstFPS;
	this->stats.averageFPS = frameStats.avgFPS;
	this->stats.triangleCount = frameStats.triangleCount;
}
void Graphics::resetStats(){
	this->stats.lastFPS = 0.0f;
	this->stats.bestFPS = 0.0f;
	this->stats.worstFPS = 0.0f;
	this->stats.averageFPS = 0.0f;
	this->stats.triangleCount = 0;

}


void Graphics::setSceneManagerByType( Ogre::SceneTypeMask sceneTypeMask ){

	Ogre::SceneManager* newSceneManager = this->ogreRoot->createSceneManager( sceneTypeMask );

	// inform CEGUI that sceneManager has changed
	if( this->system->getGui()->isInitialized() ){
		this->system->getGui()->sceneManagerHasChanged( newSceneManager );
	}

	if( this->sceneManager != NULL ){	// delete old mainViewport and mainCamera
		this->renderWindow->removeAllViewports();
		this->sceneManager->destroyAllCameras();
		this->ogreRoot->destroySceneManager( this->sceneManager );
	}

	this->sceneManager = newSceneManager;
	this->ogreRoot->_setCurrentSceneManager( this->sceneManager );

	// set new mainViewport and mainCamera
	this->mainCamera = this->sceneManager->createCamera( "MainCamera" );
	this->mainViewport = this->renderWindow->addViewport( this->mainCamera, 500 );
	this->mainViewport->setBackgroundColour(Ogre::ColourValue(1,0,0));

}

void Graphics::setShadowTechnique( Ogre::ShadowTechnique newShadowTechnique ){
	this->shadowTechnique = newShadowTechnique;
	if( this->shadowsEnabled ){
		this->sceneManager->setShadowTechnique( this->shadowTechnique );
	}
}

void Graphics::setShadowsEnabled(bool newShadowsEnabled){
	this->shadowsEnabled = newShadowsEnabled;

	if( this->shadowsEnabled ){
		this->sceneManager->setShadowTechnique( this->shadowTechnique );
	}else{
		this->sceneManager->setShadowTechnique( Ogre::SHADOWTYPE_NONE );
	}
}
bool Graphics::getShadowsEnabled() const {
	return this->shadowsEnabled;
}

Ogre::Root* Graphics::getOgreRoot() const {
	return this->ogreRoot;
}
Ogre::RenderWindow* Graphics::getRenderWindow() const {
	return this->renderWindow;
}
Ogre::Viewport* Graphics::getMainViewport() const {
	return this->mainViewport;
}
Ogre::SceneManager* Graphics::getSceneManager() const {
	return this->sceneManager;
}
Ogre::Camera* Graphics::getMainCamera() const {
	return this->mainCamera;
}

const Graphics::stats_s& Graphics::getStats() const {
	return this->stats;
}

