#include "Mech.h"

#include "System.h"
#include "Game.h"
#include "Weapon.h"
#include "Effect.h"

Mech::Mech(): Vehicle(Vehicle::VEHICLE_TYPE_MECH) {
	this->mainUnitYaw = 0.0f;
	this->mainUnitPitch = 0.0f;

	this->standingOnFeet = false;
}

Mech::~Mech(){
}

Vehicle* Mech::clone() const {
	Mech* ret = new Mech();
	*ret = *this;

	return ret;
}

void Mech::reset(){
	this->health = this->physicalProperties.healthCapacity;
	this->energy = this->physicalProperties.energyCapacity;
}



void Mech::takeOverClientState( const Client::state_s& clientState ){
	this->health = clientState.health;
	this->energy = clientState.energy;

	this->position = clientState.position;
	this->orientation = clientState.orientation;
	this->velocity = clientState.velocity;
	this->angularVelocity = clientState.angularVelocity;

	this->mainUnitYaw = *( (float*)(&clientState.vehicleSpecificBytes[0]) );
	this->mainUnitPitch = *( (float*)(&clientState.vehicleSpecificBytes[4]) );
	this->boostersActive = clientState.vehicleSpecificBytes[5] != 0;
}

void Mech::fillInClientState( Client::state_s& clientState ) const {
	clientState.health = (int)this->health;
	clientState.energy = (int)this->energy;

	clientState.position = this->position;
	clientState.orientation = this->orientation;
	clientState.velocity = this->velocity;
	clientState.angularVelocity = this->angularVelocity;

	memcpy( &clientState.vehicleSpecificBytes[0], &this->mainUnitYaw, 4 );
	memcpy( &clientState.vehicleSpecificBytes[4], &this->mainUnitPitch, 4 );
	clientState.vehicleSpecificBytes[5] = this->boostersActive ? 1 : 0;
}

void Mech::processInput(Input::actionMap_t& actionMap, Input::mouseState_t& mouseState){

	FVector3 linearInput = FVector3::ZERO;
	FVector3 angularInput = FVector3::ZERO;
	this->boostersActive = false;
	this->recoveryActive = false;

	float deltaT = game->getSystem()->getTimer()->getDeltaT();

	// keys
	if( actionMap[Input::AC_TURN_LEFT].triggered ){
		if( actionMap[Input::AC_ALIGN_UNITS].triggered ){
			angularInput.y += 1.0f;
		}else{
			this->mainUnitYaw += deltaT * 2.0f;
			if( this->mainUnitYaw > this->physicalProperties.maxMainUnitYaw ){	// for easier aiming...
				angularInput.y += 1.0f;
			}
		}
	}
	if( actionMap[Input::AC_TURN_RIGHT].triggered ){
		if( actionMap[Input::AC_ALIGN_UNITS].triggered ){
			angularInput.y -= 1.0f;
		}else{
			this->mainUnitYaw -= deltaT * 2.0f;
			if( this->mainUnitYaw < this->physicalProperties.minMainUnitYaw ){	// for easier aiming...
				angularInput.y -= 1.0f;
			}
		}
	}
	if( actionMap[Input::AC_TURN_UP].triggered ){
		if( actionMap[Input::AC_ALIGN_UNITS].triggered ){
			angularInput.x += 1.0f;
		}else{
			this->mainUnitPitch += deltaT * 2.0f;
			if( this->mainUnitPitch > this->physicalProperties.maxMainUnitPitch ){	// for easier aiming...
				angularInput.x += 1.0f;
			}
		}
	}
	if( actionMap[Input::AC_TURN_DOWN].triggered ){
		if( actionMap[Input::AC_ALIGN_UNITS].triggered ){
			angularInput.x -= 1.0f;
		}else{
			this->mainUnitPitch -= deltaT * 2.0f;
			if( this->mainUnitPitch < this->physicalProperties.minMainUnitPitch ){	// for easier aiming...
				angularInput.x -= 1.0f;
			}
		}
	}

	if( actionMap[Input::AC_MOVE_LEFT].triggered ){
		linearInput -= FVector3::UNIT_X;
	}
	if( actionMap[Input::AC_MOVE_RIGHT].triggered ){
		linearInput += FVector3::UNIT_X;
	}
	if( actionMap[Input::AC_MOVE_UP].triggered && this->energy > 1.0f ){
		//linearInput += FVector3::UNIT_Y;
		this->boostersActive = true;
	}
	if( actionMap[Input::AC_MOVE_DOWN].triggered ){
		//linearInput -= FVector3::UNIT_Y;
	}
	if( actionMap[Input::AC_MOVE_FORWARD].triggered ){
		linearInput -= FVector3::UNIT_Z;
	}
	if( actionMap[Input::AC_MOVE_BACKWARD].triggered ){
		linearInput += FVector3::UNIT_Z;
	}
	if( actionMap[Input::AC_ROLL_LEFT].triggered ){
		angularInput.y += 1.0f;
	}
	if( actionMap[Input::AC_ROLL_RIGHT].triggered ){
		angularInput.y -= 1.0f;
	}

	if( actionMap[Input::AC_FIRE1].triggered ){
		if( this->weapons[0] != NULL && this->weapons[0]->isReady() ){
			this->weapons[0]->fire();
		}
	}
	if( actionMap[Input::AC_FIRE2].triggered ){
		if( this->weapons[1] != NULL && this->weapons[1]->isReady() ){
			this->weapons[1]->fire();
		}
	}
	if( actionMap[Input::AC_FIRE3].triggered ){
		if( this->weapons[2] != NULL && this->weapons[2]->isReady() ){
			this->weapons[2]->fire();
		}
	}
	if( actionMap[Input::AC_FIRE4].triggered ){
		if( this->weapons[3] != NULL && this->weapons[3]->isReady() ){
			this->weapons[3]->fire();
		}
	}
	

	if( actionMap[Input::AC_ACTIVATE_BOOSTERS].triggered && this->energy > 1.0f ){
		this->boostersActive = true;
	}
	if( actionMap[Input::AC_RECOVER].triggered ){
		this->recoveryActive = true;
	}


	// mouse
	if( actionMap[Input::AC_ALIGN_UNITS].triggered ){
		this->mainUnitYaw -= this->mainUnitYaw * deltaT * 10.0f;
		this->mainUnitPitch -= this->mainUnitPitch * deltaT * 10.0f;
		angularInput.y -= mouseState.xRel * 20.0f * mouseState.sensitivity;
		angularInput.x -= mouseState.yRel * 20.0f * mouseState.sensitivity;
	}else{	// turn
		this->mainUnitYaw -= mouseState.xRel * 0.1f * mouseState.sensitivity;
		if( this->mainUnitYaw > this->physicalProperties.maxMainUnitYaw
			|| this->mainUnitYaw < this->physicalProperties.minMainUnitYaw ){	// for easier aiming...
			if( this->mainUnitYaw * mouseState.xRel < 0.0f ){
				angularInput.y -= mouseState.xRel * 20.0f * mouseState.sensitivity;
			}
		}

		this->mainUnitPitch -= mouseState.yRel * 0.1f * mouseState.sensitivity;
		if( this->mainUnitPitch > this->physicalProperties.maxMainUnitPitch
			|| this->mainUnitPitch < this->physicalProperties.minMainUnitPitch ){	// for easier aiming...
			if( this->mainUnitPitch * mouseState.yRel < 0.0f ){
				angularInput.x -= mouseState.yRel * 20.0f * mouseState.sensitivity;
			}
		}
	}


	this->mainUnitPitch = this->mainUnitPitch > this->physicalProperties.maxMainUnitPitch ? this->physicalProperties.maxMainUnitPitch : this->mainUnitPitch;
	this->mainUnitPitch = this->mainUnitPitch < this->physicalProperties.minMainUnitPitch ? this->physicalProperties.minMainUnitPitch : this->mainUnitPitch;
	this->mainUnitYaw = this->mainUnitYaw > this->physicalProperties.maxMainUnitYaw ? this->physicalProperties.maxMainUnitYaw : this->mainUnitYaw;
	this->mainUnitYaw = this->mainUnitYaw < this->physicalProperties.minMainUnitYaw ? this->physicalProperties.minMainUnitYaw : this->mainUnitYaw;

	linearInput.normalize();
	this->propulsiveForce = linearInput * this->physicalProperties.propulsiveForce;

	angularInput.normalize();
	this->propulsiveTorque = angularInput * this->physicalProperties.propulsiveTorque;


}

void Mech::prepareForSimulationStep(){
//	if( client->isRemote() ){
//		set only vel and ori
//	}
	float deltaT = this->game->getSystem()->getTimer()->getDeltaT();

	dVector3 tmpv3;
	dVector4 tmpv4;

	dBodySetPosition( this->body, this->position.x, this->position.y, this->position.z );
	this->orientation.toOdeVector4( tmpv4 );
	dBodySetQuaternion( this->body, tmpv4 );
	dBodySetLinearVel( this->body, this->velocity.x, this->velocity.y, this->velocity.z );
	dBodySetAngularVel( this->body, this->angularVelocity.x, this->angularVelocity.y, this->angularVelocity.z );

	Physics* physics = this->game->getPhysics();
	const FVector3& g = physics->getGravity();
	const FVector3& ng = physics->getNormalizedGravity();


	if( this->recoveryActive ){
		float mod = 1.0f + this->up.dotProduct(ng);
		mod = mod < 1.0f ? mod : 1.0f;

		dBodyVectorFromWorld( this->body,
			g.x, g.y, g.z, tmpv3);
		FVector3 relGravity(tmpv3);

		// OVERWRITE propulsive force to prevent player from misusing recovery as accel.
		this->propulsiveForce = -relGravity * mod * this->physicalProperties.mass.mass * 2.0f;

		if( this->direction.dotProduct(ng) < 0.0f ){
			this->propulsiveTorque.x -= mod * this->physicalProperties.propulsiveTorque;
		}else{
			this->propulsiveTorque.x += mod * this->physicalProperties.propulsiveTorque;
		}

		if( this->right.dotProduct(ng) < 0.0f ){
			this->propulsiveTorque.z -= mod * this->physicalProperties.propulsiveTorque;
		}else{
			this->propulsiveTorque.z += mod * this->physicalProperties.propulsiveTorque;
		}
	}


	RayIntersectionQuery* riq = physics->createRayIntersectionQuery();
	riq->setRay( this->position, -this->up, this->physicalProperties.hoverHeight );
	riq->setCollisionFlags( IntersectionQuery::RESULT_ENTRY_TYPE_WORLD_GEOMETRY
							| IntersectionQuery::RESULT_ENTRY_TYPE_GAME_OBJECT
							| IntersectionQuery::RESULT_ENTRY_TYPE_ARENA_BOUNDARY);
	riq->setIgnoreGameObject( this );
	riq->execute();
	const IntersectionQuery::result_t& result = riq->getResult();
	this->standingOnFeet = ( result.size() != 0 && this->up.dotProduct(ng) < -0.7f );
//		printf("Ground contact: %i, %f\n", groundContact, result.front().distance);

	// boosters
	if( this->energy < 0.1f ){	// check energy first...
		this->boostersActive = false;
	}
	if( this->boostersActive ){
		dBodyAddRelForce( this->body, 0.0f, this->physicalProperties.boosterForce, 0.0f );
		this->energy -= deltaT * 100.0f;
		this->energy = this->energy < 0.0f ? 0.0f : this->energy;
	}else{
		this->energy += deltaT * 100.0f;
		this->energy = this->energy > this->physicalProperties.energyCapacity ? this->physicalProperties.energyCapacity : this->energy;
	}

	// propulsive force
	if( (this->standingOnFeet || this->recoveryActive) && !this->boostersActive ){
		dBodyAddRelForce( this->body,
			this->propulsiveForce.x, this->propulsiveForce.y, this->propulsiveForce.z );

		// stabilizer (only when booster aren't active (otherwise mech will be clued to the ground...)
			FVector3 v = this->velocity;// - (this->velocity.dotProduct(ng) * ng);
			dBodyVectorFromWorld( this->body,
				v.x, v.y, v.z, tmpv3);
			FVector3 relV(tmpv3);
			float sLinear = relV.normalize() / this->physicalProperties.maxLinearVelocity * this->physicalProperties.propulsiveForce;
			FVector3 sf = -relV * sLinear;
			dBodyAddRelForce( this->body, sf.x, sf.y, sf.z );
	}else{
		dBodyAddRelForce( this->body,
			this->propulsiveForce.x*this->physicalProperties.airControlModifier,
			this->propulsiveForce.y*this->physicalProperties.airControlModifier,
			this->propulsiveForce.z*this->physicalProperties.airControlModifier );
	}

	// propulsive torque
	dBodyAddRelTorque( this->body,
		this->propulsiveTorque.x, this->propulsiveTorque.y, this->propulsiveTorque.z );
	dBodyVectorFromWorld( this->body,
		this->angularVelocity.x, this->angularVelocity.y, this->angularVelocity.z, tmpv3);
	FVector3 relAV(tmpv3);
	float sAngular = relAV.normalize() / this->physicalProperties.maxAngularVelocity * this->physicalProperties.propulsiveTorque;
	FVector3 st = -relAV * sAngular;
	dBodyAddRelTorque( this->body, st.x, st.y, st.z );

	// update anim
	if( this->standingOnFeet && this->propulsionUnitEntity->hasSkeleton() ){
		float deltaT = this->game->getSystem()->getTimer()->getDeltaT();
		Ogre::AnimationState* as = this->propulsionUnitEntity->getAnimationState("walk");
		as->setEnabled(true);
		as->addTime( deltaT*this->velocity.dotProduct(this->direction) );
	}

}

void Mech::updateAfterSimulationStep(){
//	printf("1 %i\n", this->body);
	const dReal* pos = dBodyGetPosition( this->body );
	const dReal* ori = dBodyGetQuaternion( this->body );
	const dReal* vel = dBodyGetLinearVel( this->body );
	const dReal* avel = dBodyGetAngularVel( this->body );

	this->position = FVector3(pos);
	this->orientation = FQuaternion(ori);
	this->velocity = FVector3(vel);
	this->angularVelocity = FVector3(avel);
		
	this->mainUnitSceneNode->setPosition( this->position.toOgreVector3() );
	this->mainUnitSceneNode->setOrientation( this->orientation.toOgreQuaternion() );
	Ogre::Quaternion q = this->mainUnitSceneNode->getOrientation();
	this->right = FVector3(q.xAxis().val);
	this->up = FVector3(q.yAxis().val);
	this->direction = -FVector3(q.zAxis().val);
	this->angles.x = q.getYaw().valueRadians();
	this->angles.y = q.getPitch().valueRadians();
	this->angles.z = q.getRoll().valueRadians();

	this->propulsionUnitSceneNode->setPosition( this->position.toOgreVector3() );
	this->propulsionUnitSceneNode->setOrientation( this->orientation.toOgreQuaternion() );
		
	this->mainUnitSceneNode->rotate( Ogre::Vector3::UNIT_Y, Ogre::Radian(this->mainUnitYaw) );
	this->mainUnitSceneNode->rotate( Ogre::Vector3::UNIT_X, Ogre::Radian(this->mainUnitPitch) );


//	printf("2: %f\n", this->position.y);
	float deltaT = this->game->getSystem()->getTimer()->getDeltaT();

	if( this->boosterEffect != NULL ){
		if( this->boostersActive ){
			if( !this->boosterEffect->isAttached() ){
				this->boosterEffect->attach( this->game, this->propulsionUnitSceneNode );
			}
			this->boosterEffect->update( deltaT );
		}else{
			if( this->boosterEffect->isAttached() ){
				this->boosterEffect->detach();
			}
		}
	}

}

bool Mech::collidedWithGameObject(const FVector3& contactPoint, GameObject* other, dContact& contact){
	if( other == NULL ){ // collision with arena boundary => simply create contact joint
		return true;

	}else if( other->getGameObjectType() == GameObject::GAME_OBJECT_TYPE_SHOT ){
//		Shot* shot = (Shot*)other;
//		return this->collidedWithShot( contactPoint, shot, contact );
		return false;

	}else{

		if(	this->physicalProperties.bouncyness > 0.0f ){
			contact.surface.mode |= dContactBounce;
			contact.surface.bounce += this->physicalProperties.bouncyness*0.5f;
		}
		return true;
	}
}

bool Mech::collidedWithTerrain(const FVector3& contactPoint, dContact& contact){

	// apply friction only when NOT standing on feet and NOT recovering
	if( !this->standingOnFeet && !this->recoveryActive ){
		contact.surface.mu += Physics::frictionToMu(this->physicalProperties.friction) * 0.5f;
	}

	if( this->physicalProperties.bouncyness > 0.0f ){
		contact.surface.mode |= dContactBounce;
		contact.surface.bounce += this->physicalProperties.bouncyness*0.5f;
	}
	return true;
}

