#include "Network.h"

#include "System.h"
#include "Exception.h"
#include "Packets.h"
#include "Socket.h"
#include "Server.h"
#include "Client.h"
#include "Console.h"
#include "Game.h"

//#include <SDL_net.h>

const int Network::PROTOCOL_VERSION = 29;

const int Network::DEFAULT_MASTER_SERVER_PORT = 11111;
const int Network::DEFAULT_MASTER_SERVER_LOCAL_PORT = 0;

Network::Network(System* system): SubSystem(system){

	this->masterServerProperties.host = string("ogc.yz.to");
	this->masterServerProperties.port = DEFAULT_MASTER_SERVER_PORT;
	this->masterServerProperties.localPort = DEFAULT_MASTER_SERVER_LOCAL_PORT;

//	this->connectionSpeed = 10;

	this->server = new Server(this);
	this->client = new Client(this);

	this->cVars.network_protocolVersion = NULL;
//	this->cVars.network_connectionSpeed = NULL;

	this->cVars.network_server_name = NULL;
	this->cVars.network_server_host = NULL;
	this->cVars.network_server_port = NULL;
	this->cVars.network_server_maxClients = NULL;
	this->cVars.network_server_connectionSpeed = NULL;
	this->cVars.network_server_remote = NULL;
	this->cVars.network_server_registerOnMasterServer = NULL;
	this->cVars.network_server_arena = NULL;
	this->cVars.network_server_gameMode = NULL;

	this->cVars.network_client_name = NULL;
	this->cVars.network_client_port = NULL;
	this->cVars.network_client_team = NULL;
	this->cVars.network_client_vehicle = NULL;
	this->cVars.network_client_weapon1 = NULL;
	this->cVars.network_client_weapon2 = NULL;
	this->cVars.network_client_weapon3 = NULL;
	this->cVars.network_client_weapon4 = NULL;

	this->cVars.network_masterServer_host = NULL;
	this->cVars.network_masterServer_port = NULL;
	this->cVars.network_masterServer_localPort = NULL;

	this->resetStats();
}

Network::~Network(){
	delete server;
	delete client;
}

void Network::initialize(){
	System::log("");
	System::log("*** Initializing Network ***");
	System::log("");

	System::log("Network: Initializing SDL_net...");
	if( SDLNet_Init() == -1 ) {
		printf("SDLNet_Init failed: %s", SDLNet_GetError());
		throw Exception("Couldn't initialize SDL_net.", "Network::initialize()");
	}

//	System::log("Network: Creating server..."); // in constr. because we need mirrors for cvars
//	this->server = new Server(this);

//	System::log("Network: Creating client...");
//	this->client = new Client(this);


	this->initialized = true;
	System::log("Network is ready.");
	System::log("");

}

void Network::shutdown(){
	System::log("");
	System::log("=== Shutting down Network ===");
	System::log("");

	System::log("Network: Shutting down SDL_net...");
	SDLNet_Quit();

	this->initialized = false;
	System::log("Network is down.");
	System::log("");

}
void Network::registerCCmds( Console& console ){
}
void Network::unregisterCCmds( Console& console ){
}
void Network::registerCVars( Console& console ){
	this->cVars.network_protocolVersion = new CVarInt("network.protocolVersion", Network::PROTOCOL_VERSION);
	this->cVars.network_protocolVersion->setValueRange(Network::PROTOCOL_VERSION, Network::PROTOCOL_VERSION);
	this->cVars.network_protocolVersion->setChangeString("(this variable is read-only)");
	this->cVars.network_protocolVersion->setFlags( CVar::FLAG_SYSTEM | CVar::FLAG_READ_ONLY );
	console.registerCVar(this->cVars.network_protocolVersion);

	// server stuff
	this->cVars.network_server_name = new CVarString("network.server.name", &this->server->properties.name, false);
	this->cVars.network_server_name->setLengthRange(1, 255);
	this->cVars.network_server_name->setChangeString("(changes will take effect when you start or connect to a new server)");
	this->cVars.network_server_name->setFlags( CVar::FLAG_SYSTEM );
	console.registerCVar(this->cVars.network_server_name);

	this->cVars.network_server_host = new CVarString("network.server.host", &this->server->properties.host, false);
	this->cVars.network_server_host->setLengthRange(1, 255);
	this->cVars.network_server_host->setChangeString("(changes will take effect when you connect to or start a new server)");
	this->cVars.network_server_host->setFlags( CVar::FLAG_SYSTEM );
	console.registerCVar(this->cVars.network_server_host);

	this->cVars.network_server_port = new CVarInt("network.server.port", &this->server->properties.port, false);
	this->cVars.network_server_port->setValueRange(0, 65535);
	this->cVars.network_server_port->setChangeString("(changes will take effect when you start or connect to a new server)");
	this->cVars.network_server_port->setFlags( CVar::FLAG_SYSTEM );
	console.registerCVar(this->cVars.network_server_port);

	this->cVars.network_server_maxClients = new CVarInt("network.server.maxClients", &this->server->properties.maxClients, false);
	this->cVars.network_server_maxClients->setValueRange(1, 16);
	this->cVars.network_server_maxClients->setChangeString("(changes will take effect when you start a new server)");
	this->cVars.network_server_maxClients->setFlags( CVar::FLAG_SYSTEM );
	console.registerCVar(this->cVars.network_server_maxClients);

	this->cVars.network_server_connectionSpeed = new CVarInt("network.server.connectionSpeed", &this->server->properties.connectionSpeed, false);
	this->cVars.network_server_connectionSpeed->setValueRange(1, 1024);
	this->cVars.network_server_connectionSpeed->setChangeString("(changes will take effect when you start a new server)");
	this->cVars.network_server_connectionSpeed->setFlags( CVar::FLAG_SYSTEM );
	console.registerCVar(this->cVars.network_server_connectionSpeed);

	this->cVars.network_server_remote = new CVarBool("network.server.remote", &this->server->properties.remote, false);
	this->cVars.network_server_remote->setChangeString("(changes will take effect when you start or connect to a new server)");
	this->cVars.network_server_remote->setFlags( CVar::FLAG_SYSTEM );
	console.registerCVar(this->cVars.network_server_remote);

	this->cVars.network_server_registerOnMasterServer = new CVarBool("network.server.registerOnMasterServer", &this->server->properties.registerOnMasterServer, false);
	this->cVars.network_server_registerOnMasterServer->setChangeString("(changes will take effect when you start a new server)");
	this->cVars.network_server_registerOnMasterServer->setFlags( CVar::FLAG_SYSTEM );
	console.registerCVar(this->cVars.network_server_registerOnMasterServer);

	this->cVars.network_server_arena = new CVarString("network.server.arena", &this->server->properties.arena, false);
	this->cVars.network_server_arena->setLengthRange(1, 32);
	this->cVars.network_server_arena->setChangeString("(changes will take effect when you start a new server)");
	this->cVars.network_server_arena->setFlags( CVar::FLAG_SYSTEM );
	console.registerCVar(this->cVars.network_server_arena);

	this->cVars.network_server_gameMode = new CVarInt("network.server.gameMode", &this->server->properties.gameMode, false);
	this->cVars.network_server_gameMode->setValueRange(0, Game::NUM_MODES);
	this->cVars.network_server_gameMode->setChangeString("(changes will take effect when you start a new server)");
	this->cVars.network_server_gameMode->setFlags( CVar::FLAG_SYSTEM );
	console.registerCVar(this->cVars.network_server_gameMode);

	this->cVars.network_server_ghostTime = new CVarInt("network.server.ghostTime", &this->server->properties.ghostTime, false);
	this->cVars.network_server_ghostTime->setValueRange(0, 60);
	this->cVars.network_server_ghostTime->setChangeString("(changes will take effect when you start a new server)");
	this->cVars.network_server_ghostTime->setFlags( CVar::FLAG_SYSTEM );
	console.registerCVar(this->cVars.network_server_ghostTime);


	// client stuff
	this->cVars.network_client_name = new CVarString("network.client.name", &this->client->properties.name, false);
	this->cVars.network_client_name->setLengthRange(1, 32);
	this->cVars.network_client_name->setChangeString("(changes will take effect when you connect to a new server)");
	this->cVars.network_client_name->setFlags( CVar::FLAG_SYSTEM );
	console.registerCVar(this->cVars.network_client_name);

	this->cVars.network_client_port = new CVarInt("network.client.port", &this->client->properties.port, false);
	this->cVars.network_client_port->setValueRange(0, 65535);
	this->cVars.network_client_port->setChangeString("(changes will take effect when you connect to a new server)");
	this->cVars.network_client_port->setFlags( CVar::FLAG_SYSTEM );
	console.registerCVar(this->cVars.network_client_port);

	this->cVars.network_client_team = new CVarInt("network.client.team", &this->client->info.team, false);
	this->cVars.network_client_team->setValueRange(0, Game::NUM_TEAMS);
	this->cVars.network_client_team->setChangeString("(changes will take effect when you respawn)");
	this->cVars.network_client_team->setFlags( CVar::FLAG_SYSTEM );
	console.registerCVar(this->cVars.network_client_team);

	this->cVars.network_client_vehicle = new CVarInt("network.client.vehicle", &this->client->info.vehicle, false);
	this->cVars.network_client_vehicle->setValueRange(0, 15);
	this->cVars.network_client_vehicle->setChangeString("(changes will take effect when you respawn)");
	this->cVars.network_client_vehicle->setFlags( CVar::FLAG_SYSTEM );
	console.registerCVar(this->cVars.network_client_vehicle);

	this->cVars.network_client_weapon1 = new CVarInt("network.client.weapon1", &this->client->info.weapons[0], false);
	this->cVars.network_client_weapon1->setValueRange(-1, 15);
	this->cVars.network_client_weapon1->setChangeString("(changes will take effect when you respawn)");
	this->cVars.network_client_weapon1->setFlags( CVar::FLAG_SYSTEM );
	console.registerCVar(this->cVars.network_client_weapon1);

	this->cVars.network_client_weapon2 = new CVarInt("network.client.weapon2", &this->client->info.weapons[1], false);
	this->cVars.network_client_weapon2->setValueRange(-1, 15);
	this->cVars.network_client_weapon2->setChangeString("(changes will take effect when you respawn)");
	this->cVars.network_client_weapon2->setFlags( CVar::FLAG_SYSTEM );
	console.registerCVar(this->cVars.network_client_weapon2);

	this->cVars.network_client_weapon3 = new CVarInt("network.client.weapon3", &this->client->info.weapons[2], false);
	this->cVars.network_client_weapon3->setValueRange(-1, 15);
	this->cVars.network_client_weapon3->setChangeString("(changes will take effect when you respawn)");
	this->cVars.network_client_weapon3->setFlags( CVar::FLAG_SYSTEM );
	console.registerCVar(this->cVars.network_client_weapon3);

	this->cVars.network_client_weapon4 = new CVarInt("network.client.weapon4", &this->client->info.weapons[3], false);
	this->cVars.network_client_weapon4->setValueRange(-1, 15);
	this->cVars.network_client_weapon4->setChangeString("(changes will take effect when you respawn)");
	this->cVars.network_client_weapon4->setFlags( CVar::FLAG_SYSTEM );
	console.registerCVar(this->cVars.network_client_weapon4);


	// master server stuff
	this->cVars.network_masterServer_host = new CVarString("network.masterServer.host", &this->masterServerProperties.host, false);
	this->cVars.network_masterServer_host->setLengthRange(1, 255);
	this->cVars.network_masterServer_host->setChangeString("(changes will take effect when you connect to the new masterServer)");
	this->cVars.network_masterServer_host->setFlags( CVar::FLAG_SYSTEM );
	console.registerCVar(this->cVars.network_masterServer_host);

	this->cVars.network_masterServer_port = new CVarInt("network.masterServer.port", &this->masterServerProperties.port, false);
	this->cVars.network_masterServer_port->setValueRange(0, 65535);
	this->cVars.network_masterServer_port->setChangeString("(changes will take effect when you connect to the new masterServer)");
	this->cVars.network_masterServer_port->setFlags( CVar::FLAG_SYSTEM );
	console.registerCVar(this->cVars.network_masterServer_port);

	this->cVars.network_masterServer_localPort = new CVarInt("network.masterServer.localPort", &this->masterServerProperties.localPort, false);
	this->cVars.network_masterServer_localPort->setValueRange(0, 65535);
	this->cVars.network_masterServer_localPort->setChangeString("(changes will take effect when you connect to the new masterServer)");
	this->cVars.network_masterServer_localPort->setFlags( CVar::FLAG_SYSTEM );
	console.registerCVar(this->cVars.network_masterServer_localPort);

}
void Network::unregisterCVars( Console& console ){
	console.unregisterCVar( cVars.network_protocolVersion );
	delete cVars.network_protocolVersion;

//	console.unregisterCVar( cVars.network_connectionSpeed);
//	delete cVars.network_connectionSpeed;

	console.unregisterCVar( cVars.network_server_name );
	delete cVars.network_server_name;

	console.unregisterCVar( cVars.network_server_host );
	delete cVars.network_server_host;

	console.unregisterCVar( cVars.network_server_port );
	delete cVars.network_server_port;

	console.unregisterCVar( cVars.network_server_maxClients );
	delete cVars.network_server_maxClients;

	console.unregisterCVar( cVars.network_server_connectionSpeed );
	delete cVars.network_server_connectionSpeed;

	console.unregisterCVar( cVars.network_server_remote );
	delete cVars.network_server_remote;

	console.unregisterCVar( cVars.network_server_registerOnMasterServer );
	delete cVars.network_server_registerOnMasterServer;

	console.unregisterCVar( cVars.network_server_arena );
	delete cVars.network_server_arena;

	console.unregisterCVar( cVars.network_server_gameMode );
	delete cVars.network_server_gameMode;

	console.unregisterCVar( cVars.network_server_ghostTime );
	delete cVars.network_server_ghostTime;

	
	console.unregisterCVar( cVars.network_client_name );
	delete cVars.network_client_name;

	console.unregisterCVar( cVars.network_client_port );
	delete cVars.network_client_port;

	console.unregisterCVar( cVars.network_client_team );
	delete cVars.network_client_team;

	console.unregisterCVar( cVars.network_client_vehicle );
	delete cVars.network_client_vehicle;

	console.unregisterCVar( cVars.network_client_weapon1 );
	delete cVars.network_client_weapon1;
	console.unregisterCVar( cVars.network_client_weapon2 );
	delete cVars.network_client_weapon2;
	console.unregisterCVar( cVars.network_client_weapon3 );
	delete cVars.network_client_weapon3;
	console.unregisterCVar( cVars.network_client_weapon4 );
	delete cVars.network_client_weapon4;


	console.unregisterCVar( cVars.network_masterServer_host );
	delete cVars.network_masterServer_host;

	console.unregisterCVar( cVars.network_masterServer_port );
	delete cVars.network_masterServer_port;

	console.unregisterCVar( cVars.network_masterServer_localPort );
	delete cVars.network_masterServer_localPort;
}

void Network::onGameCreation(){
	System::log("Network: Applying cvars...");
//	this->cVars.network_connectionSpeed->updateReferencedVariable();

	this->cVars.network_server_remote->updateReferencedVariable();
	this->cVars.network_server_registerOnMasterServer->updateReferencedVariable();
	this->cVars.network_server_name->updateReferencedVariable();
	this->cVars.network_server_host->updateReferencedVariable();
	this->cVars.network_server_port->updateReferencedVariable();
	this->cVars.network_server_maxClients->updateReferencedVariable();
	this->cVars.network_server_connectionSpeed->updateReferencedVariable();
	this->cVars.network_server_arena->updateReferencedVariable();
	this->cVars.network_server_gameMode->updateReferencedVariable();
	this->cVars.network_server_ghostTime->updateReferencedVariable();

	this->cVars.network_client_name->updateReferencedVariable();
	this->cVars.network_client_port->updateReferencedVariable();
	this->cVars.network_client_team->setValue( 0 );
	this->cVars.network_client_vehicle->setValue( 0 );
	this->cVars.network_client_weapon1->setValue( -1 );
	this->cVars.network_client_weapon2->setValue( -1 );
	this->cVars.network_client_weapon3->setValue( -1 );
	this->cVars.network_client_weapon4->setValue( -1 );

	this->cVars.network_masterServer_host->updateReferencedVariable();
	this->cVars.network_masterServer_port->updateReferencedVariable();
	this->cVars.network_masterServer_localPort->updateReferencedVariable();

//	System::log("server port: %i, %i", this->cVars.network_server_port->getValue(), this->server->properties.port );

	System::log("Network: setting up server...");
	this->server->setUp();

	System::log("Network: setting up client...");
	this->client->setUp();

	if( this->server->properties.remote ){ // remote server => make remote connection
		System::log("Network: Establishing remote connection to '%s:%i'...", this->server->properties.host.c_str(), this->server->properties.port);

		this->client->establishRemoteConnection( this->server );
		
		System::log("Network: Remote connection to '%s:%i' established.", this->server->properties.host.c_str(), this->server->properties.port);
	}else{ // local server => make loopback connection
//		this->cVars.network_server_host->setValue("localhost"); // make sure we're using localhost as server
//		this->cVars.network_server_host->updateReferencedVariable();

		System::log("Network: Establishing loopback connection to '%s:%i'...", this->server->properties.host.c_str(), this->server->properties.port);

		this->client->establishLoopbackConnection( this->server );
		this->server->establishLoopbackConnection( this->client );

		System::log("Network: Loopback connection to '%s:%i' established.", this->server->properties.host.c_str(), this->server->properties.port);
	}
}

void Network::onGameDestruction(){
//	if( this->client != NULL ){
		System::log("Network: Tearing down client...");
		this->client->tearDown();

//		System::log("Network: Destroying client...");
//		delete this->client;
//		this->client = NULL;
//	}

//	if( this->server != NULL ){
		System::log("Network: Tearing down server...");
		this->server->tearDown();

//		System::log("Network: Destroying server...");
//		delete this->server;
//		this->server = NULL;
//	}

	System::log("Network: Resetting statistics...");
	this->resetStats();
}

void Network::update(){
	this->server->update();
	this->client->update();

	this->updateStats();
}


void Network::updateStats(){
	float deltaT = this->system->getTimer()->getDeltaT();

	this->stats.lastPing = this->client->info.ping;
	this->stats.bestPing = this->stats.lastPing < this->stats.bestPing ? this->stats.lastPing : this->stats.bestPing;
	this->stats.worstPing = this->stats.lastPing > this->stats.worstPing ? this->stats.lastPing : this->stats.worstPing;

	this->stats.stateInterval = this->client->clientStateIntervalMillis;

	this->stats.secondsSinceLastTrafficUpdate += deltaT;
	if( this->stats.secondsSinceLastTrafficUpdate > 1.0 ){
		this->stats.incomingTraffic = this->stats.incomingBytes / 1024.0f / this->stats.secondsSinceLastTrafficUpdate;
		this->stats.outgoingTraffic = this->stats.outgoingBytes / 1024.0f / this->stats.secondsSinceLastTrafficUpdate;
		this->stats.traffic = this->stats.incomingTraffic + this->stats.outgoingTraffic;

		this->stats.incomingBytes = 0;
		this->stats.outgoingBytes = 0;

		this->stats.secondsSinceLastTrafficUpdate = 0.0f;
	}
}
void Network::resetStats(){
	this->stats.lastPing = 0;
	this->stats.bestPing = 0;
	this->stats.worstPing = 0;
	this->stats.stateInterval = 0;

	this->stats.traffic = 0.0f;
	this->stats.incomingTraffic = 0.0f;
	this->stats.outgoingTraffic = 0.0f;

	this->stats.secondsSinceLastTrafficUpdate = 0.0f;
	this->stats.incomingBytes = 0;
	this->stats.outgoingBytes = 0;
}


void Network::notifyPacketSent( Packet* packet ){
	this->stats.outgoingBytes += packet->udpPacket->len;
//	if( packet->udpPacket->len < 0 ){
//		System::warn("SCHEISSE 1!");
//	}
}
void Network::notifyPacketReceived( Packet* packet ){
	this->stats.incomingBytes += packet->dataSize; // not accurate for MS packets, but saver...
//	this->stats.incomingBytes += packet->udpPacket->len;
//	if( packet->udpPacket->len < 0 ){
//		System::warn("SCHEISSE 2!");
//	}
}


Server* Network::getServer() const {
	return this->server;
}
Client* Network::getClient() const {
	return this->client;
}

const Network::stats_s& Network::getStats() const {
	return this->stats;
}
const Network::masterServerProperties_s& Network::getMasterServerProperties() const {
	return this->masterServerProperties;
}


Packet::ipAddress_t Network::resolveHost( const string& host, int port ){
	Packet::ipAddress_t ret;

//	System::log("inc: %s:%X", host.c_str(), port);
	int b = SDLNet_ResolveHost( &ret, host.c_str(), port);
	if( b != 0 ){
		System::error("(in Network::resolveHost()): SDLNet_ResolveHost() for %s:%i failed: %s.", host.c_str(), port, SDLNet_GetError());
		throw Exception("Couldn't resolve host.", "Network::resolveHost()");
	}

//	System::log("ret: %X:%X", ret.host, ret.port);

	return ret;
}
string Network::ipAddressToString( const Packet::ipAddress_t ipAddress ){
	throw Exception("Not yet implemented.", "Network::ipAddressToString()");
	return string("1.2.3.4:5678");
}

