#include "Shape.h"

#include "System.h"
#include "Exception.h"

Shape::Shape(){
	this->geomInfos.clear();

	this->geoms.clear();
}

Shape::~Shape(){
}

void Shape::fromXmlElement(TiXmlElement* xmlElement){
	for( TiXmlElement* elt = xmlElement->FirstChildElement(); elt != NULL; elt = elt->NextSiblingElement() ){
		string value = string( elt->Value() );
		
		if(value == "box"){
			this->parseBox( elt );
		}else if(value == "sphere"){
			this->parseSphere( elt );
		}else if(value == "capsule"){
			this->parseCapsule( elt );

		}else{
			System::warn("(in Shape::fromXmlElement()): Unknown child element: '%s' (row: %i). Ignoring.", value.c_str(), elt->Row() );
		}
	}

}


void Shape::clear(){
	this->geomInfos.clear();

	this->destroyGeoms();
	this->geoms.clear();
}

void Shape::createGeoms( void* data ){

	for( vector<geomInfo_s>::iterator iter = this->geomInfos.begin(); iter != this->geomInfos.end(); iter++ ){
		dGeomID geom;
		if( iter->geomClass == dBoxClass ){
			geom = dCreateBox( 0, iter->edgeLengths.x, iter->edgeLengths.y, iter->edgeLengths.z );

		}else if( iter->geomClass == dSphereClass ){
			geom = dCreateSphere( 0, iter->radius );

		}else if( iter->geomClass == dCCylinderClass ){
			geom = dCreateCCylinder( 0, iter->radius, iter->length );

		}else{
			throw Exception("Unknown geom class.", "Shape::createGeoms()");
		}

		dGeomSetPosition( geom, iter->center.x, iter->center.y, iter->center.z );
		dGeomID  wrapper = dCreateGeomTransform( 0 );
		dGeomTransformSetGeom( wrapper, geom );
		dGeomTransformSetCleanup( wrapper, 1 );

		dGeomSetData( wrapper, data );

		this->geoms.push_back( wrapper );
	}

}

void Shape::destroyGeoms(){
	for( vector<dGeomID>::iterator iter = this->geoms.begin(); iter != this->geoms.end(); iter++ ){
		dGeomDestroy( *iter );
	}
}

void Shape::attachGeomsToBody( dBodyID body ){
	for( vector<dGeomID>::iterator iter = this->geoms.begin(); iter != this->geoms.end(); iter++ ){
		dGeomSetBody( *iter, body );
//		dGeomSetData( *iter, dBodyGetData(body) );
	}
}
void Shape::detachGeomsFromBody(){
	for( vector<dGeomID>::iterator iter = this->geoms.begin(); iter != this->geoms.end(); iter++ ){
//		dGeomSetData( *iter, NULL );
		dGeomSetBody( *iter, 0 );
	}
}

void Shape::addGeomsToSpace( dSpaceID space ){
	for( vector<dGeomID>::iterator iter = this->geoms.begin(); iter != this->geoms.end(); iter++ ){
		dSpaceAdd( space, *iter );
	}
}
void Shape::removeGeomsFromSpace( dSpaceID space ){
	for( vector<dGeomID>::iterator iter = this->geoms.begin(); iter != this->geoms.end(); iter++ ){
		dSpaceRemove( space, *iter );
	}
}

void Shape::parseBox(TiXmlElement* boxElement){
	geomInfo_s gi;

	gi.geomClass = dBoxClass;

	gi.center.x = boxElement->Attribute("centerX") ? atof(boxElement->Attribute("centerX")) : 0.0f;
	gi.center.y = boxElement->Attribute("centerY") ? atof(boxElement->Attribute("centerY")) : 0.0f;
	gi.center.z = boxElement->Attribute("centerZ") ? atof(boxElement->Attribute("centerZ")) : 0.0f;

	gi.edgeLengths.x = boxElement->Attribute("width") ? atof(boxElement->Attribute("width")) : 1.0f;
	gi.edgeLengths.y = boxElement->Attribute("height") ? atof(boxElement->Attribute("height")) : 1.0f;
	gi.edgeLengths.z = boxElement->Attribute("depth") ? atof(boxElement->Attribute("depth")) : 1.0f;

	this->geomInfos.push_back(gi);
}

void Shape::parseSphere(TiXmlElement* sphereElement){
	geomInfo_s gi;

	gi.geomClass = dSphereClass;

	gi.center.x = sphereElement->Attribute("centerX") ? atof(sphereElement->Attribute("centerX")) : 0.0f;
	gi.center.y = sphereElement->Attribute("centerY") ? atof(sphereElement->Attribute("centerY")) : 0.0f;
	gi.center.z = sphereElement->Attribute("centerZ") ? atof(sphereElement->Attribute("centerZ")) : 0.0f;

	gi.radius = sphereElement->Attribute("radius") ? atof(sphereElement->Attribute("radius")) : 1.0f;

	this->geomInfos.push_back(gi);
}

void Shape::parseCapsule(TiXmlElement* sphereElement){
	geomInfo_s gi;

	gi.geomClass = dCCylinderClass;

	gi.center.x = sphereElement->Attribute("centerX") ? atof(sphereElement->Attribute("centerX")) : 0.0f;
	gi.center.y = sphereElement->Attribute("centerY") ? atof(sphereElement->Attribute("centerY")) : 0.0f;
	gi.center.z = sphereElement->Attribute("centerZ") ? atof(sphereElement->Attribute("centerZ")) : 0.0f;

	gi.radius = sphereElement->Attribute("radius") ? atof(sphereElement->Attribute("radius")) : 1.0f;
	gi.length = sphereElement->Attribute("length") ? atof(sphereElement->Attribute("length")) : 1.0f;

	this->geomInfos.push_back(gi);
}


void Shape::setGeomsPosition( const FVector3& pos ){
	for( vector<dGeomID>::iterator iter = this->geoms.begin(); iter != this->geoms.end(); iter++ ){
		dGeomSetPosition( *iter, pos.x, pos.y, pos.z );
	}
}
void Shape::setGeomsOrientation( const FQuaternion& ori ){
	dVector4 o;
	ori.toOdeVector4(o);
	for( vector<dGeomID>::iterator iter = this->geoms.begin(); iter != this->geoms.end(); iter++ ){
		dGeomSetQuaternion( *iter, o );
	}
}


const vector<dGeomID>& Shape::getGeoms() const {
	return this->geoms;
}
