#include "Shot.h"

#include "System.h"
#include "XmlUtils.h"
#include "Exception.h"
#include "Game.h"
#include "Vehicle.h"

Shot::Shot( shotTypes_e shotType ): GameObject( GameObject::GAME_OBJECT_TYPE_SHOT  ) {
	this->shotType = shotType;
	this->clientId = -1;

	this->name = "<uninitialized>";
	this->checksum = 0;

	this->id = 0;
	this->lifeTime = 0.0f;
	this->exploded = false;

	this->travelEffect = NULL;

	this->graphicalProperties.explosionEffect.clear();
	this->graphicalProperties.travelEffect.clear();

	this->physicalProperties.damage = 0.0f;
	this->physicalProperties.explosionRadius = 0.0f;
	this->physicalProperties.impactImpulse = 0.0f;
	this->physicalProperties.lifeTime = 0.0f;
	this->physicalProperties.muzzleVelocity = 0.0f;
	this->physicalProperties.propulsiveForce = FVector3::ZERO;
	this->physicalProperties.causesThermalShock = false;

	this->physicalProperties.bouncyness = 0.0f;
	this->physicalProperties.affectedByGravity = false;
	this->physicalProperties.shape.clear();
	dMassSetZero( &this->physicalProperties.mass );

}

Shot::~Shot(){
}



void Shot::fromXmlElement( TiXmlElement* xmlElement ){
	// name
	const char* tmp = xmlElement->Attribute("name");
	if( tmp != NULL ){
		this->name = string( tmp );
	}else{
		throw Exception( "shot element has no name attribute.", "Shot::fromXmlElement()" );
	}
	if( this->name.length() > 32 ){
		System::error( "(in Shot::fromXmlElement()): Shot name '%s' is longer than 32 characters.", this->name.c_str() );
		throw Exception( "XmlElement is not a valid shot description.", "Shot::fromXmlElement()" );
	}

	
	for( TiXmlElement* elt = xmlElement->FirstChildElement(); elt != NULL; elt = elt->NextSiblingElement() ){
		string value = string( elt->Value() );

		if(value == "graphicalProperties"){
			this->parseGraphicalProperties(elt);
		}else if(value == "physicalProperties"){
			this->parsePhysicalProperties(elt);
		}else if(value == "acousticalProperties"){
			this->parseAcousticalProperties(elt);

		}else{
			System::warn("(in Shot::fromXmlElement()): Unknown child element: '%s' (row: %i). Ignoring.", value.c_str(), elt->Row() );
		}
	}
}
TiXmlElement Shot::toXmlElement() const {
	return TiXmlElement("dummy");
}
void Shot::calculateChecksum(){
	this->checksum = 0;
}




void Shot::explode( GameObject* ignoreGameObject ){
	if( this->exploded ){
		return;
	}

	this->exploded = true; // set to true to avoid endless recursion

	if( this->physicalProperties.explosionRadius > 0.0f ){

		SphereIntersectionQuery siq( this->game->getPhysics(), this->position, this->physicalProperties.explosionRadius );

		siq.setCollisionFlags( IntersectionQuery::RESULT_ENTRY_TYPE_GAME_OBJECT );
		siq.setIgnoreGameObject( this );
		siq.execute();
		const IntersectionQuery::result_t& result = siq.getResult();

		for( IntersectionQuery::result_t::const_iterator c_iter = result.begin(); c_iter != result.end(); ++c_iter ){
			if( c_iter->gameObject == ignoreGameObject ){
				continue;
			}

//			printf("HIT: d: %f\n", c_iter->distance);
			// impulse
			FVector3 impulseDir = c_iter->intersectionPoint - this->position;
			float mod = (1.0f - impulseDir.length() / this->physicalProperties.explosionRadius);
			impulseDir.normalize();
			FVector3 impulse = impulseDir * mod * this->physicalProperties.impactImpulse;
//			cout << "mod: " << mod << endl;
			c_iter->gameObject->addLinearImpulse( impulse, c_iter->intersectionPoint );


			if( this->physicalProperties.damage >= 0.0f ){
				c_iter->gameObject->damage( mod * this->physicalProperties.damage );
			}else{ // negative damage => heal game object
				c_iter->gameObject->heal( -mod * this->physicalProperties.damage );
			}

			if( c_iter->gameObject->getGameObjectType() == GameObject::GAME_OBJECT_TYPE_VEHICLE ){
				Vehicle* v = (Vehicle*)(c_iter->gameObject);
				v->setAttackerClientId( this->clientId );

				if( this->physicalProperties.causesThermalShock ){
					v->thermalShock();
				}
			}
		}

	}

	// effect
	if( !this->graphicalProperties.explosionEffect.empty() ){
		Effect* e = this->game->spawnEffect( this->graphicalProperties.explosionEffect, this->position, this->orientation );
	}
}

void Shot::impact( const FVector3& contactPoint, GameObject* other ){
	FVector3 impulse = this->velocity;
	impulse.normalize();
	impulse = impulse * this->physicalProperties.impactImpulse;
	other->addLinearImpulse( impulse, contactPoint );

	if( this->physicalProperties.damage >= 0.0f ){
		other->damage( this->physicalProperties.damage );
	}else{ // negative damage => heal game object
		other->heal( -this->physicalProperties.damage );
	}

	if( other->getGameObjectType() == GameObject::GAME_OBJECT_TYPE_VEHICLE ){
		Vehicle* v = (Vehicle*)(other);
		v->setAttackerClientId( this->clientId );
		if( this->physicalProperties.causesThermalShock ){
			v->thermalShock();
		}
	}

	this->explode( other ); // to avoid double damage
//	// impactEffect
//	if( !this->graphicalProperties.explosionEffect.empty() ){
//		Effect* e = this->game->spawnEffect( this->graphicalProperties.explosionEffect, contactPoint, this->orientation );
//	}

}

void Shot::damage( float amount ){
	if( this->physicalProperties.explosionRadius > 0.0f ){
		this->explode();
	}

	this->lifeTime = -1.0f;
}



void Shot::parseGraphicalProperties( TiXmlElement* graphicalPropertiesElement ){
	XmlUtils::readStringProperty(graphicalPropertiesElement, "travelEffect", this->graphicalProperties.travelEffect);
	XmlUtils::readStringProperty(graphicalPropertiesElement, "explosionEffect", this->graphicalProperties.explosionEffect);
}


void Shot::parsePhysicalProperties( TiXmlElement* physicalPropertiesElement ){
	XmlUtils::readFloatProperty(physicalPropertiesElement, "damage", this->physicalProperties.damage);
	XmlUtils::readFloatProperty(physicalPropertiesElement, "lifeTime", this->physicalProperties.lifeTime);
	XmlUtils::readFloatProperty(physicalPropertiesElement, "explosionRadius", this->physicalProperties.explosionRadius);
	XmlUtils::readFloatProperty(physicalPropertiesElement, "impactImpulse", this->physicalProperties.impactImpulse);
	XmlUtils::readFloatProperty(physicalPropertiesElement, "muzzleVelocity", this->physicalProperties.muzzleVelocity);
	XmlUtils::readBoolProperty(physicalPropertiesElement, "causesThermalShock", this->physicalProperties.causesThermalShock);
	
	XmlUtils::readFVector3Property(physicalPropertiesElement, "propulsiveForce", this->physicalProperties.propulsiveForce);

	XmlUtils::readBoolProperty(physicalPropertiesElement, "affectedByGravity", this->physicalProperties.affectedByGravity);
	XmlUtils::readFloatProperty(physicalPropertiesElement, "bouncyness", this->physicalProperties.bouncyness);
	XmlUtils::readFloatProperty(physicalPropertiesElement, "friction", this->physicalProperties.friction);

	TiXmlElement* element = physicalPropertiesElement->FirstChildElement("shape");
	if( element != 0 ){
		this->physicalProperties.shape.clear();
		this->physicalProperties.shape.fromXmlElement( element );
	}

	element = physicalPropertiesElement->FirstChildElement("mass");
	if( element != 0 ){
		this->parseMass( element );
	}

}

void Shot::parseAcousticalProperties( TiXmlElement* acousticalPropertiesElement ){
}

void Shot::parseMass( TiXmlElement* massElement ){
	for( TiXmlElement* elt = massElement->FirstChildElement(); elt != NULL; elt = elt->NextSiblingElement() ){
		string value = string( elt->Value() );
		if(value == "sphere"){
			float m = elt->Attribute("mass") ? atof(elt->Attribute("mass")) : 0.0f;
			float r = elt->Attribute("radius") ? atof(elt->Attribute("radius")) : 0.0f;

			dMass tempMass;
			dMassSetSphereTotal( &tempMass, m, r );
			dMassAdd( &this->physicalProperties.mass, &tempMass );
		}else if(value == "box"){
			float m = elt->Attribute("mass") ? atof(elt->Attribute("mass")) : 0.0f;
			float w = elt->Attribute("width") ? atof(elt->Attribute("width")) : 0.0f;
			float h = elt->Attribute("height") ? atof(elt->Attribute("height")) : 0.0f;
			float d = elt->Attribute("depth") ? atof(elt->Attribute("depth")) : 0.0f;

			dMass tempMass;
			dMassSetBoxTotal( &tempMass, m, w, h, d );
			dMassAdd( &this->physicalProperties.mass, &tempMass );

		}else if(value == "capsule"){
			float m = elt->Attribute("mass") ? atof(elt->Attribute("mass")) : 0.0f;
			float r = elt->Attribute("radius") ? atof(elt->Attribute("radius")) : 0.0f;
			float l = elt->Attribute("length") ? atof(elt->Attribute("length")) : 0.0f;

			dMass tempMass;
			dMassSetCappedCylinderTotal( &tempMass, m, 3, r, l );
			dMassAdd( &this->physicalProperties.mass, &tempMass );

		}else{
			System::warn("(in Shot::parseMass()): Unknown child element: '%s' (row: %i). Ignoring.", value.c_str(), elt->Row() );
		}
	}
}



const string& Shot::getName() const {
	return this->name;
}
bool Shot::shouldBeDestroyed() const {
	return (this->lifeTime < 0.0f);
}
int Shot::getClientId() const {
	return this->clientId;
}
unsigned int Shot::getChecksum() const {
	return this->checksum;
}

const Shot::physicalProperties_s& Shot::getPhysicalProperties() const {
	return this->physicalProperties;
}

const Shot::graphicalProperties_s& Shot::getGraphicalProperties() const {
	return this->graphicalProperties;
}
