#include "SpawnArea.h"

#include "System.h"
#include "Exception.h"
#include "XmlUtils.h"
#include "Game.h"

SpawnArea::SpawnArea(): Area( Area::AREA_TYPE_SPAWN ) {
	this->teamFlags = 0;
}

SpawnArea::~SpawnArea(){
}


Area* SpawnArea::clone() const {
	SpawnArea* ret = new SpawnArea();

	*ret = *this;

	return ret;
}


void SpawnArea::fromXmlElement( TiXmlElement* xmlElement ){
	// name
	const char* tmp = xmlElement->Attribute("name");
	if( tmp != NULL ){
		this->name = string( tmp );
	}else{
		throw Exception( "SpawnArea element has no name attribute.", "SpawnArea::fromXmlElement()" );
	}
	if( this->name.length() > 32 ){
		System::error( "(in SpawnArea::fromXmlElement()): SpawnArea name '%s' is longer than 32 characters.", this->name.c_str() );
		throw Exception( "XmlElement is not a valid spawn area description.", "SpawnArea::fromXmlElement()" );
	}

	// teamFlags
	tmp = xmlElement->Attribute("teamFlags");
	if( tmp != NULL ){
		string flags = string( tmp );
		if( flags.find("s") != string::npos ){
			this->teamFlags |= Game::TEAM_FLAG_SPECTATORS;
		}
		if( flags.find("p") != string::npos ){
			this->teamFlags |= Game::TEAM_FLAG_PLAYERS;
		}
		if( flags.find("r") != string::npos ){
			this->teamFlags |= Game::TEAM_FLAG_RED;
		}
		if( flags.find("b") != string::npos ){
			this->teamFlags |= Game::TEAM_FLAG_BLUE;
		}
	}else{
		throw Exception( "SpawnArea element has no teamFlags attribute.", "SpawnArea::fromXmlElement()" );
	}



	for( TiXmlElement* elt = xmlElement->FirstChildElement(); elt != NULL; elt = elt->NextSiblingElement() ){
		string value = string( elt->Value() );
		
		if(value == "edgeLengths"){
			this->edgeLengths = XmlUtils::readFVector3( elt );

		}else if(value == "position"){
			this->position = XmlUtils::readFVector3( elt );
		}else if(value == "orientation"){
			this->orientation = XmlUtils::readFQuaternion( elt );


		}else{
			System::warn("(in SpawnArea::fromXmlElement()): Unknown child element: '%s' (row: %i). Ignoring.", value.c_str(), elt->Row() );
		}
	}


}

TiXmlElement SpawnArea::toXmlElement() const {
	return TiXmlElement("dummy");
}
void SpawnArea::calculateChecksum(){
	this->checksum = 0;
}




void SpawnArea::attach( Game* game ){

	if( this->game != NULL ){
		throw Exception("SpawnArea is already attached to a game.", "SpawnArea::attach()");
	}

	this->game = game;
}

void SpawnArea::detach(){
	if( this->game == NULL ){
		throw Exception("SpawnArea is not attached to a game.", "SpawnArea::detach()");
	}

	this->game = NULL;
}


bool SpawnArea::isUsableForTeam( int team ){
	unsigned int mask = ( 0x00000001 << team );

	return ( (this->teamFlags & mask) != 0 );
}

FVector3 SpawnArea::getSpawnPosition(){
	FVector3 ret( this->position );

	Random* random = this->game->getSystem()->getRandom();
	
	ret.x += random->nextFloat( -this->edgeLengths.x*0.5f, this->edgeLengths.x*0.5f );
	ret.y += random->nextFloat( -this->edgeLengths.y*0.5f, this->edgeLengths.y*0.5f );
	ret.z += random->nextFloat( -this->edgeLengths.z*0.5f, this->edgeLengths.z*0.5f );

	return ret;
}

