#include "VehicleFactory.h"

#include "System.h"
#include "Exception.h"

#include "Drone.h"
#include "Mech.h"
#include "Hovercraft.h"

VehicleFactory::VehicleFactory(Game* game){
	this->game = game;

	this->vehiclesByName.clear();
}

VehicleFactory::~VehicleFactory(){
	// delete vehicles
	for( vehiclesByName_t::iterator iter = vehiclesByName.begin(); iter != vehiclesByName.end(); iter++ ){
		delete iter->second;
	}

}

void VehicleFactory::addVehicle(const string& vehicleFilePath){
//	System::log( "VehicleFactory: Parsing vehicle '%s'...", vehicleFilePath.c_str() );

	try{
		TiXmlDocument doc( vehicleFilePath.c_str() );
		if( !doc.LoadFile() ){
			throw Exception("Couldn't load file '" + vehicleFilePath + "'. (Maybe not a correct xml file?)", "VehicleFactory::addVehicle()");
		}
//		doc.Print();

		TiXmlNode* rootElement = doc.RootElement();
		TiXmlElement* element = rootElement->ToElement();

		string rootElementName = element->Value();
		if( rootElementName != "vehicle" ){
			throw Exception("Vehicle file '" + vehicleFilePath + "' does not have a <vehicle> root node.", "VehicleFactory::addVehicle()");
		}

		// check version
		const char* tmp = element->Attribute( "formatVersion" );
		if( tmp == NULL ){
			throw Exception("Vehicle file '" + vehicleFilePath + "' has no version information.", "VehicleFactory::addVehicle()");
		}else{
			string version = tmp;

			if( version != System::getInstance()->getVersionString() ){
				System::warn("(in VehicleFactory::addVehicle()): file '%s' has version %s. Trying to parse anyway.", vehicleFilePath.c_str(), version.c_str() );
			}
		}


		Vehicle* v = NULL;

		// check type and create a vehicle of the proper subclass
		tmp = element->Attribute( "type" );
		if( tmp == NULL ){
			throw Exception("Vehicle file '" + vehicleFilePath + "' has no type information.", "VehicleFactory::addVehicle()");
		}else{
			string type = tmp;

			// v = new sowieso
			if( type == string("drone") ){
				v = new Drone();
			}else if( type == string("hovercraft") ){
				v = new Hovercraft();
			}else if( type == string("mech") ){
				v = new Mech();

			}else{
				System::warn("(in VehicleFactory::addVehicle()): Vehicle file '%s' references unknown vehicle type ('%s'). Ignoring vehicle.", vehicleFilePath.c_str(), type.c_str() );
			}
		}


		if( v != NULL ){
			v->fromXmlElement( element );
//			v->applyProperties();
			v->calculateChecksum();

			vehiclesByName_t::const_iterator c_iter = vehiclesByName.find( v->getName() );
			if( c_iter != vehiclesByName.end() ){ // another vehicle with that name is already registered -> throw exception
				throw Exception("A vehicle with name '" + v->getName() + "' is already registered.", "VehicleFactory::addVehicle()");
			}
			vehiclesByName.insert( vehiclesByName_t::value_type(v->getName(), v) );
			System::log("VehicleFactory: Added vehicle '%s' to list.", v->getName().c_str());
		}

	}catch( Exception& e ){
		System::warn("(VehicleFactory::addVehicle()): An exception occured while parsing vehicle file '%s'. Ignoring vehicle.", vehicleFilePath.c_str());
		System::log("VehicleFactory: The exception was: %s", e.getDescription().c_str());
	}

}

void VehicleFactory::removeVehicle(const string& name){
	vehiclesByName_t::iterator iter = vehiclesByName.find( name );
	if( iter == vehiclesByName.end() ){
		throw Exception("A vehicle with name '" + name +"' is not registered.", "VehicleFactory::removeVehicle()");
	}
	this->vehiclesByName.erase(iter);
}






Vehicle* VehicleFactory::createVehicle(const string& name) const {

	vehiclesByName_t::const_iterator c_iter = vehiclesByName.find( name );
	if( c_iter == vehiclesByName.end() ){
		throw Exception("A vehicle with name '" + name +"' is not registered.", "VehicleFactory::createVehicle()");
	}

	Vehicle* ret = c_iter->second->clone();

	ret->reset();

	return ret;
}

void VehicleFactory::destroyVehicle(Vehicle* vehicle) const {
	delete vehicle;
}



bool VehicleFactory::isVehicleAvailable( const string& name ) const {
	vehiclesByName_t::const_iterator c_iter = vehiclesByName.find( name );

	return (c_iter != this->vehiclesByName.end());
}


vector<string> VehicleFactory::getVehicleNames() const {
	vector<string> ret;

	for( vehiclesByName_t::const_iterator c_iter = this->vehiclesByName.begin(); c_iter != this->vehiclesByName.end(); c_iter++ ){
		ret.push_back( c_iter->first );
	}

	return ret;
}

const string& VehicleFactory::getVehicleDescription( const string& name ) const {
	vehiclesByName_t::const_iterator c_iter = vehiclesByName.find( name );
	if( c_iter == vehiclesByName.end() ){
		throw Exception("A vehicle with name '" + name +"' is not registered.", "VehicleFactory::getVehicleDescription()");
	}else{
		return c_iter->second->getDescription();
	}

}
int VehicleFactory::getVehicleNumWeaponSlots( const string& name ) const {
	vehiclesByName_t::const_iterator c_iter = vehiclesByName.find( name );
	if( c_iter == vehiclesByName.end() ){
		throw Exception("A vehicle with name '" + name +"' is not registered.", "VehicleFactory::getVehicleNumWeaponSlots()");
	}else{
		return c_iter->second->getPhysicalProperties().weaponSlots.size();
	}

}
