#include "Weapon.h"

#include "Vehicle.h"
#include "System.h"
#include "Exception.h"
#include "StringConverter.h"
#include "XmlUtils.h"
#include "Game.h"
#include "Effect.h"
#include "Shot.h"
#include "Graphics.h"
#include "EffectFactory.h"
#include "Camera.h"

Weapon::Weapon(){
//	this->weaponType = weaponType;

	this->vehicle = NULL;
	this->slot = -1;

	this->name = "<uninitialized>";
	this->shot = "<uninitialized>";
	this->checksum = 0;

	this->info.description.clear();

	this->graphicalProperties.mesh.clear();
	this->graphicalProperties.muzzleEffect.clear();
	this->graphicalProperties.hudIcon.clear();
	this->graphicalProperties.overheatedEffect.clear();

	this->physicalProperties.ammoCapacity = 0;
	this->physicalProperties.heatCapacity = 0;
	this->physicalProperties.heatOutput = 0.0f;
	this->physicalProperties.reloadTime = 0.0f;
	this->physicalProperties.recoilImpulse = 0.0f;

	this->physicalProperties.muzzlePoint = FVector3::ZERO;

	this->secondsUntilReloaded = 0.0f;
	this->ammo = 0;
	this->heat = 0.0f;
	this->overheated = false;

	this->sceneNode = NULL;
	this->entity = NULL;
	this->animationState = NULL;

	this->overheatedEffect = NULL;
}

Weapon::~Weapon(){
}

Weapon* Weapon::clone() const {
	Weapon* w = new Weapon();

	*w = *this;

	return w;
}


void Weapon::reset(){
	this->ammo = this->physicalProperties.ammoCapacity;
	this->heat = 0.0f;
	this->overheated = false;
}

void Weapon::fromXmlElement( TiXmlElement* xmlElement ){
	// name
	const char* tmp = xmlElement->Attribute("name");
	if( tmp != NULL ){
		this->name = string( tmp );
	}else{
		throw Exception( "weapon element has no name attribute.", "Weapon::fromXmlElement()" );
	}
	if( this->name.length() > 32 ){
		System::error( "(in Weapon::fromXmlElement()): Weapon name '%s' is longer than 32 characters.", this->name.c_str() );
		throw Exception( "XmlElement is not a valid weapon description.", "Weapon::fromXmlElement()" );
	}

	// shot
	tmp = xmlElement->Attribute("shot");
	if( tmp != NULL ){
		this->shot = string( tmp );
	}else{
		throw Exception( "weapon element has no shot attribute.", "Weapon::fromXmlElement()" );
	}

	for( TiXmlElement* elt = xmlElement->FirstChildElement(); elt != NULL; elt = elt->NextSiblingElement() ){
		string value = string( elt->Value() );

		if(value == "info"){
			this->parseInfo(elt);
		}else if(value == "graphicalProperties"){
			this->parseGraphicalProperties(elt);
		}else if(value == "physicalProperties"){
			this->parsePhysicalProperties(elt);
		}else if(value == "acousticalProperties"){
			this->parseAcousticalProperties(elt);

		}else{
			System::warn("(in Weapon::fromXmlElement()): Unknown child element: '%s' (row: %i). Ignoring.", value.c_str(), elt->Row() );
		}
	}
}
TiXmlElement Weapon::toXmlElement() const {
	return TiXmlElement("dummy");
}

void Weapon::attach(Vehicle* vehicle, int slot){
	if( this->vehicle != NULL ){
		System::error("(in Weapon::attach()): This weapon is already attached to a vehicle.");
		throw Exception("Couldn't attach weapon.", "Weapon::attach()");
	}

	Ogre::SceneManager* sm = vehicle->game->getSystem()->getGraphics()->getSceneManager();

	string fullName = vehicle->client->properties.name + ":" + vehicle->name + "/Weapon" + StringConverter::toString(slot);
	this->sceneNode = vehicle->mainUnitSceneNode->createChildSceneNode( fullName );
	this->sceneNode->setPosition( vehicle->physicalProperties.weaponSlots[slot].toOgreVector3() );
//	this->mainUnitSceneNode->setOrientation( FQuaternion::IDENTITY );
	if( !this->graphicalProperties.mesh.empty() ){
		this->entity = sm->createEntity( fullName, this->graphicalProperties.mesh );
		this->entity->setCastShadows( true );
		this->sceneNode->attachObject( this->entity );

		if( this->entity->hasSkeleton() ){
			this->animationState = this->entity->getAnimationState("reload");
			this->animationState->setLoop(false);
		}

	}

	if( !this->graphicalProperties.overheatedEffect.empty() ){
		this->overheatedEffect = vehicle->game->getEffectFactory()->createEffect( this->graphicalProperties.overheatedEffect );
//		this->overheatedEffect->attach( vehicle->game, this->sceneNode );
	}

	// add mass to vehcile's mass
	float newMass = vehicle->physicalProperties.mass.mass + this->physicalProperties.mass;
	dMassAdjust( &vehicle->physicalProperties.mass, newMass );
	dBodySetMass( vehicle->body, &vehicle->physicalProperties.mass );

	vehicle->weapons[slot] = this;
	this->vehicle = vehicle;
	this->slot = slot;
}
void Weapon::detach(){
	if( this->vehicle == NULL ){
		System::error("(in Weapon::detach()): This weapon is not attached to a vehicle.");
		throw Exception("Couldn't detach weapon.", "Weapon::detach()");
	}

	float newMass = this->vehicle->physicalProperties.mass.mass - this->physicalProperties.mass;
	dMassAdjust( &this->vehicle->physicalProperties.mass, newMass );
	dBodySetMass( vehicle->body, &vehicle->physicalProperties.mass );

	if( this->overheatedEffect != NULL ){
		if( this->overheatedEffect->isAttached() ){
			this->overheatedEffect->detach();
		}

		this->vehicle->game->getEffectFactory()->destroyEffect( this->overheatedEffect );
		this->overheatedEffect = NULL;
	}

	Ogre::SceneManager* sm = this->vehicle->game->getSystem()->getGraphics()->getSceneManager();

	if( this->entity ){
		sm->destroyEntity( this->entity );
	}
	this->sceneNode->getParentSceneNode()->removeAndDestroyChild( this->sceneNode->getName() );
	this->sceneNode = NULL;


	this->vehicle->weapons[this->slot] = NULL;
	this->vehicle = NULL;
}

void Weapon::update( float deltaT ){

	this->heat -= deltaT;
	if( this->heat <= 0.0f ){
		if( this->overheated ){
			this->overheated = false;
		}
		this->heat = 0.0f;
	}

	if( this->secondsUntilReloaded > 0.0f ){
		this->secondsUntilReloaded -= deltaT;
		this->secondsUntilReloaded = this->secondsUntilReloaded > 0.0f ? this->secondsUntilReloaded : 0.0f;
	}

	// update anim
	if( this->animationState != NULL ){
		if( this->secondsUntilReloaded > 0.0f ){
			this->animationState->setEnabled( true );
			this->animationState->addTime( deltaT );
		}else{
			this->animationState->setEnabled( false );
			this->animationState->setTimePosition( 0.0f );
		}
	}

	// update overheated effect
	if( this->overheatedEffect != NULL ){
		if( this->overheated ){
			if( !this->overheatedEffect->isAttached() ){
				this->overheatedEffect->attach( this->vehicle->game, this->sceneNode );
			}
			this->overheatedEffect->update( deltaT );
		}else{
			if( this->overheatedEffect->isAttached() ){
				this->overheatedEffect->detach();
			}
		}
	}
}

bool Weapon::isReady() const {
	return !(
		this->physicalProperties.ammoCapacity > 0 && this->ammo <= 0
		|| this->overheated
		|| this->secondsUntilReloaded > 0.0f
		);
}



void Weapon::fire(){
	// return if no ammo or overheated
//	if( !this->isReady() ){
//		return;
//	}

	this->secondsUntilReloaded = this->physicalProperties.reloadTime;

	// send packet

	// set up shot, so that it flies into main unit's direction
	const Ogre::Quaternion& derivedOri = this->sceneNode->_getDerivedOrientation();
	Ogre::Matrix3 m;
	derivedOri.ToRotationMatrix( m );
	FVector3 position = m * this->physicalProperties.muzzlePoint.toOgreVector3() + this->sceneNode->_getDerivedPosition();
	FQuaternion orientation = derivedOri;
	FVector3 direction = -derivedOri.zAxis();
	FVector3 up = derivedOri.yAxis();
	FVector3 right = derivedOri.xAxis();

	// get target point
	Camera* cam = this->vehicle->getGame()->getCamera();
	RayIntersectionQuery riq( this->vehicle->game->getPhysics(), cam->position, cam->direction, true );
	riq.setIgnoreGameObject( this->vehicle );
	riq.setCollisionFlags( IntersectionQuery::RESULT_ENTRY_TYPE_GAME_OBJECT | IntersectionQuery::RESULT_ENTRY_TYPE_WORLD_GEOMETRY );
	riq.execute();
	const IntersectionQuery::result_t& result = riq.getResult();
	if( result.size() > 0 ){ // correct shot's orientation to face target point
//		cout << "TARGET: " << result.front().intersectionPoint << endl;

		const FVector3& targetPoint = result.front().intersectionPoint;

		FVector3 dir = targetPoint - position;
		dir.normalize();

		if( dir.dotProduct( direction ) > 0.9f ){
			direction = dir;
			right = dir.crossProduct( up );
			up = right.crossProduct( direction );

			orientation = Ogre::Quaternion( right.toOgreVector3(), up.toOgreVector3(), -direction.toOgreVector3() );
		}
	}


	this->vehicle->client->sendFirePacket( this->slot, position, orientation );
}


void Weapon::setUpShot( Shot* s ){
	// do ammo, heat, etc here because of network sync.
//	this->secondsUntilReloaded = this->physicalProperties.reloadTime;

	this->heat += this->physicalProperties.heatOutput;
	if( this->heat > this->physicalProperties.heatCapacity ){
		this->heat = this->physicalProperties.heatCapacity;
		this->overheated = true;
	}

	this->ammo--;


	s->clientId = this->vehicle->client->properties.clientId;

	// spawn muzzleEffect
	if( !this->graphicalProperties.muzzleEffect.empty() ){
		this->vehicle->game->spawnEffect( this->graphicalProperties.muzzleEffect, s->position, s->orientation );
	}

	// convert recoil impulse into force and apply it
	if( this->physicalProperties.recoilImpulse > 0.0f ){
	//	float deltaT = this->vehicle->game->getSystem()->getTimer()->getDeltaT();
		FVector3 impulse = -s->direction * this->physicalProperties.recoilImpulse;
		this->vehicle->addLinearImpulse( impulse, s->position );
	}

}





void Weapon::restockAmmoRel( float relAmount ){
	if( this->physicalProperties.ammoCapacity == 0 ){
		return;
	}

	int amount = (int)( this->physicalProperties.ammoCapacity * relAmount );
	if( amount < 1 ){
		amount = 1;
	}

	this->ammo += amount;

	if( this->ammo > this->physicalProperties.ammoCapacity ){
		this->ammo = this->physicalProperties.ammoCapacity;
	}
}

void Weapon::overheat(){
	this->heat = this->physicalProperties.heatCapacity;
	this->overheated = true;
}


void Weapon::parseInfo(TiXmlElement* infoElement){
	XmlUtils::readStringProperty(infoElement, "description", this->info.description);

/*
	TiXmlElement* elt = infoElement->FirstChildElement("description");
	if( elt != NULL ){
		this->parseDescription(elt);
	}
*/
}

void Weapon::parseDescription(TiXmlElement* descriptionElement){

	TiXmlNode* n = descriptionElement->FirstChild();
	if( n != NULL && n->ToText() != NULL ){
		this->info.description = string( n->ToText()->Value() );
	}
}


void Weapon::parseGraphicalProperties( TiXmlElement* graphicalPropertiesElement ){
	XmlUtils::readStringProperty(graphicalPropertiesElement, "mesh", this->graphicalProperties.mesh);
	XmlUtils::readStringProperty(graphicalPropertiesElement, "hudIcon", this->graphicalProperties.hudIcon);
	XmlUtils::readStringProperty(graphicalPropertiesElement, "muzzleEffect", this->graphicalProperties.muzzleEffect);
	XmlUtils::readStringProperty(graphicalPropertiesElement, "overheatedEffect", this->graphicalProperties.overheatedEffect);
//	XmlUtils::readStringProperty(graphicalPropertiesElement, "impactEffect", this->graphicalProperties.impactEffect);
}


void Weapon::parsePhysicalProperties( TiXmlElement* physicalPropertiesElement ){
	XmlUtils::readFloatProperty(physicalPropertiesElement, "mass", this->physicalProperties.mass);

	XmlUtils::readIntProperty(physicalPropertiesElement, "ammoCapacity", this->physicalProperties.ammoCapacity);
	XmlUtils::readIntProperty(physicalPropertiesElement, "heatCapacity", this->physicalProperties.heatCapacity);
	XmlUtils::readFloatProperty(physicalPropertiesElement, "heatOutput", this->physicalProperties.heatOutput);
	XmlUtils::readFloatProperty(physicalPropertiesElement, "recoilImpulse", this->physicalProperties.recoilImpulse);

//	XmlUtils::readFloatProperty(physicalPropertiesElement, "damage", this->physicalProperties.damage);
	XmlUtils::readFloatProperty(physicalPropertiesElement, "reloadTime", this->physicalProperties.reloadTime);
//	XmlUtils::readFloatProperty(physicalPropertiesElement, "shotLifeTime", this->physicalProperties.shotLifeTime);

	XmlUtils::readFVector3Property(physicalPropertiesElement, "muzzlePoint", this->physicalProperties.muzzlePoint);
//	XmlUtils::readFVector3Property(physicalPropertiesElement, "muzzleVelocity", this->physicalProperties.muzzleVelocity);
//	XmlUtils::readFVector3Property(physicalPropertiesElement, "propulsiveForce", this->physicalProperties.propulsiveForce);
}

void Weapon::parseAcousticalProperties( TiXmlElement* acousticalPropertiesElement ){
}




void Weapon::calculateChecksum(){
	this->checksum = 0;
}
unsigned int Weapon::getChecksum() const {
	return this->checksum;
}
const string& Weapon::getName() const {
	return this->name;
}
const string& Weapon::getShot() const {
	return this->shot;
}
const string& Weapon::getDescription() const {
	return this->info.description;
}
const Weapon::graphicalProperties_s& Weapon::getGraphicalProperties() const {
	return this->graphicalProperties;
}
const Weapon::physicalProperties_s& Weapon::getPhysicalProperties() const {
	return this->physicalProperties;
}
float Weapon::getHeat() const {
	return this->heat;
}
int Weapon::getAmmo() const {
	return this->ammo;
}
bool Weapon::isOverheated() const {
	return this->overheated;
}
