#include "BallisticShot.h"

#include "System.h"
#include "Exception.h"
#include "Game.h"
#include "EffectFactory.h"
#include "Vehicle.h"
//#include <ode/ode.h>

BallisticShot::BallisticShot(): Shot( Shot::SHOT_TYPE_BALLISTIC ) {
	this->body = 0;
}

BallisticShot::~BallisticShot(){
}

Shot* BallisticShot::clone(){
	BallisticShot* ret = new BallisticShot();

	*ret = *this;

	return ret;
}

void BallisticShot::attach( Game* game ){
	if( this->game != NULL ){
		throw Exception("Shot is already attached to a game.", "Shot::attach()");
	}

	this->id = game->getNextId();

	this->lifeTime = this->physicalProperties.lifeTime;

	// attach physics
	Physics* physics = game->getPhysics();
	this->physicalProperties.shape.createGeoms( this );
	this->body = dBodyCreate( physics->getWorld() );
	dBodySetMass( this->body, &this->physicalProperties.mass );
	dBodySetPosition( this->body, this->position.x, this->position.y, this->position.z );
	dVector4 tmpv4;
	this->orientation.toOdeVector4(tmpv4);
	dBodySetQuaternion( this->body, tmpv4 );
	dBodySetLinearVel( this->body, this->velocity.x, this->velocity.y, this->velocity.z );

	dBodySetGravityMode( this->body, this->physicalProperties.affectedByGravity );
	dBodySetData( this->body, this );
	this->physicalProperties.shape.attachGeomsToBody( this->body );
	this->physicalProperties.shape.addGeomsToSpace( physics->getDynamicGeometrySpace() );


	// create effect
	if( !this->graphicalProperties.travelEffect.empty() ){
		this->travelEffect = game->getEffectFactory()->createEffect( this->graphicalProperties.travelEffect );
		this->travelEffect->attach( game, this );
	}

	this->game = game;
}
void BallisticShot::detach(){
	if( this->game == NULL ){
		throw Exception("Shot is not attached to a game.", "Shot::detach()");
	}

	if( this->travelEffect != NULL ){
		this->travelEffect->detach();
		this->game->getEffectFactory()->destroyEffect( this->travelEffect );
	}

	Physics* physics = this->game->getPhysics();
	this->physicalProperties.shape.detachGeomsFromBody( );
	dBodyDestroy( this->body );
	this->physicalProperties.shape.removeGeomsFromSpace( physics->getDynamicGeometrySpace() );
	this->physicalProperties.shape.destroyGeoms();

	this->game = NULL;


}
void BallisticShot::update( float deltaT ){

	if( this->travelEffect != NULL ){
		this->travelEffect->update( deltaT );
	}

	this->lifeTime -= deltaT;

	if( this->lifeTime < 0.0f && this->physicalProperties.explosionRadius > 0.0f ){
		this->explode();
	}
}

void BallisticShot::prepareForSimulationStep(){
	dBodyAddRelForce( this->body, 
		this->physicalProperties.propulsiveForce.x, this->physicalProperties.propulsiveForce.y, this->physicalProperties.propulsiveForce.z);

	if( this->angularVelocity.length() > 1000.0f ){
		dBodySetAngularVel( this->body, 0.0f, 0.0f, 0.0f );
	}
}
void BallisticShot::updateAfterSimulationStep(){
	const dReal* pos = dBodyGetPosition( this->body );
	const dReal* ori = dBodyGetQuaternion( this->body );
	const dReal* vel = dBodyGetLinearVel( this->body );
	const dReal* avel = dBodyGetAngularVel( this->body );

	this->position = FVector3(pos);
	this->orientation = FQuaternion(ori);
	this->velocity = FVector3(vel);
	this->angularVelocity = FVector3(avel);
}

bool BallisticShot::collidedWithGameObject(const FVector3& contactPoint, GameObject* other, dContact& contact){
//	printf("GO\n");
	
	if( other == NULL || other == this->game->getVehicleByClientId( this->clientId ) ){
		return false;
	}

	contact.surface.mu += Physics::frictionToMu(this->physicalProperties.friction) * 0.5f;
	if( this->physicalProperties.bouncyness > 0.0f && other->getGameObjectType() != GameObject::GAME_OBJECT_TYPE_VEHICLE ){
		// bounce
		contact.surface.mode |= dContactBounce;
		contact.surface.bounce += this->physicalProperties.bouncyness*0.5f;
		return true;
	}else{
		// explode
		this->impact( contactPoint, other );
//		this->explode( other );
	}


	this->lifeTime = -1.0f;

	return false;
}
bool BallisticShot::collidedWithTerrain(const FVector3& contactPoint, dContact& contact){
//	printf("TE\n");

	contact.surface.mu += Physics::frictionToMu(this->physicalProperties.friction) * 0.5f;
	if( this->physicalProperties.bouncyness > 0.0f ){
		// bounce
		contact.surface.mode |= dContactBounce;
		contact.surface.bounce += this->physicalProperties.bouncyness*0.5f;
		return true;
	}else{
		// explode
		if( this->physicalProperties.explosionRadius > 0.0f ){
			this->explode();
		}
	}

	this->lifeTime = -1.0f;

	return false;
}

