#include "Camera.h"

#include "System.h"
#include "Exception.h"
#include "Vehicle.h"
#include "Graphics.h"
#include "Game.h"

Camera::Camera(): GameObject(GameObject::GAME_OBJECT_TYPE_CAMERA){
	this->mode = MODE_FREE;
	this->ogreCamera = NULL;
	this->target = NULL;

	this->turnSpeed = 20.0f;
	this->moveSpeed = 100.0f;

	this->zoom = 10.0f;
}

Camera::~Camera(){
}

void Camera::processInput(Input::actionMap_t& actionMap, Input::mouseState_t& mouseState){

	if( this->mode == MODE_FREE || this->mode == MODE_THIRD_PERSON ){ // FIXME: implement this in a better way
		this->angles.z = 0.0f;		// don't allow roll when camera is free
	}

	this->velocity = FVector3::ZERO;
	this->angularVelocity = FVector3::ZERO;

	if( this->mode == MODE_FREE || this->target == NULL ){
		if( actionMap[Input::AC_TURN_LEFT].triggered ){
			angularVelocity[0] += turnSpeed * 0.1f;
		}
		if( actionMap[Input::AC_TURN_RIGHT].triggered ){
			angularVelocity[0] -= turnSpeed * 0.1f;
		}
		if( actionMap[Input::AC_TURN_UP].triggered ){
			angularVelocity[1] += turnSpeed * 0.1f;
		}
		if( actionMap[Input::AC_TURN_DOWN].triggered ){
			angularVelocity[1] -= turnSpeed * 0.1f;
		}

		if( actionMap[Input::AC_MOVE_LEFT].triggered ){
			this->velocity -= this->moveSpeed * this->right;
		}
		if( actionMap[Input::AC_MOVE_RIGHT].triggered ){
			this->velocity += this->moveSpeed * this->right;
		}
		if( actionMap[Input::AC_MOVE_UP].triggered ){
			this->velocity += this->moveSpeed * FVector3::UNIT_Y;
		}
		if( actionMap[Input::AC_MOVE_DOWN].triggered ){
			this->velocity -= this->moveSpeed * FVector3::UNIT_Y;
		}
		if( actionMap[Input::AC_MOVE_FORWARD].triggered ){
			this->velocity += this->moveSpeed * this->direction;
		}
		if( actionMap[Input::AC_MOVE_BACKWARD].triggered ){
			this->velocity -= this->moveSpeed * this->direction;
		}

		// mouse
		this->angularVelocity[0] -= mouseState.xRel * mouseState.sensitivity * this->turnSpeed;
		this->angularVelocity[1] -= mouseState.yRel * mouseState.sensitivity * this->turnSpeed;

	}else if( this->mode == MODE_THIRD_PERSON ){
		if( actionMap[Input::AC_TURN_LEFT].triggered ){
			angularVelocity[0] -= turnSpeed * 0.1f;
		}
		if( actionMap[Input::AC_TURN_RIGHT].triggered ){
			angularVelocity[0] += turnSpeed * 0.1f;
		}
		if( actionMap[Input::AC_TURN_UP].triggered ){
			angularVelocity[1] -= turnSpeed * 0.1f;
		}
		if( actionMap[Input::AC_TURN_DOWN].triggered ){
			angularVelocity[1] += turnSpeed * 0.1f;
		}

		if( actionMap[Input::AC_MOVE_LEFT].triggered ){
			angularVelocity[0] -= turnSpeed * 0.1f;
		}
		if( actionMap[Input::AC_MOVE_RIGHT].triggered ){
			angularVelocity[0] += turnSpeed * 0.1f;
		}
		if( actionMap[Input::AC_MOVE_UP].triggered ){
			angularVelocity[1] -= turnSpeed * 0.1f;
		}
		if( actionMap[Input::AC_MOVE_DOWN].triggered ){
			angularVelocity[1] += turnSpeed * 0.1f;
		}
		if( actionMap[Input::AC_MOVE_FORWARD].triggered ){
			this->velocity += this->moveSpeed * this->direction;
		}
		if( actionMap[Input::AC_MOVE_BACKWARD].triggered ){
			this->velocity -= this->moveSpeed * this->direction;
		}

		// mouse
		this->angularVelocity[0] -= mouseState.xRel * mouseState.sensitivity * this->turnSpeed;
		this->angularVelocity[1] -= mouseState.yRel * mouseState.sensitivity * this->turnSpeed;
	}else{
		// do nothing
	}
}


void Camera::turn(float deltaT){
//	GameObject::turn(deltaT);

	this->angles += deltaT * this->angularVelocity;

	if( this->angles[0] > FLOAT_PI * 2.0f ){
		this->angles[0] = 0.0f;
	}
	if( this->angles[0] < 0.0f ){
		this->angles[0] = FLOAT_PI * 2.0f;
	}
	if( this->angles[1] > FLOAT_PI*0.5f ){
		this->angles[1] = FLOAT_PI*0.5f;
	}
	if( this->angles[1] < -FLOAT_PI*0.5f ){
		this->angles[1] = -FLOAT_PI*0.5f;
	}

	if( ogreCamera != NULL ){
		this->ogreCamera->setOrientation( Ogre::Quaternion::IDENTITY );
		this->ogreCamera->yaw( Ogre::Radian(this->angles[0]) );
		this->ogreCamera->pitch( Ogre::Radian(this->angles[1]) );
		this->ogreCamera->roll( Ogre::Radian(this->angles[2]) );

		const Ogre::Quaternion& q = this->ogreCamera->getOrientation();
		this->orientation = q;
		this->direction = -q.zAxis();
		this->up = q.yAxis();
		this->right = q.xAxis();

		this->updateOgreCamera();
	}
}

void Camera::move(float deltaT){
	this->position += deltaT * this->velocity;

	if( ogreCamera != NULL ){
		this->updateOgreCamera();
	}
}
void Camera::update(float deltaT){
	if( this->mode == MODE_FREE || this->target == NULL ){
		this->turn(deltaT);
		this->move(deltaT);
	}else if( this->mode == MODE_THIRD_PERSON ){
		this->turn(deltaT);
		this->position = this->target->position - this->zoom*this->direction;
		this->move(deltaT);
		this->zoom = this->position.distance( this->target->position );
		this->zoom = this->zoom > 0.1f ? this->zoom : 0.1f;
//		if( ogreCamera != NULL ){
//			this->updateOgreCamera();
//		}
	}else if( this->mode == MODE_FIRST_PERSON || this->mode == MODE_CHASE ){
		if( this->target->getGameObjectType() == GameObject::GAME_OBJECT_TYPE_VEHICLE ){
			((Vehicle*)this->target)->setupCamera(this);

			if( ogreCamera != NULL ){
				this->updateOgreCamera();
			}
		}
	}
}

void Camera::reset(){
	this->position = FVector3::ZERO;
	this->orientation = FQuaternion::IDENTITY;
	this->angles = FVector3::ZERO;

	this->direction = - FVector3::UNIT_Z;
	this->up = FVector3::UNIT_Y;
	this->right = FVector3::UNIT_X;

	this->velocity = FVector3::ZERO;
	this->angularVelocity = FVector3::ZERO;

}

void Camera::attach(Game* game){
	if( this->game != NULL ){
		throw Exception("Camera is already attached to a game.", "Camera::attach()");
	}

	Graphics* graphics = game->getSystem()->getGraphics();
	Ogre::SceneManager* sm = graphics->getSceneManager();
	Ogre::Viewport* vp = graphics->getMainViewport();
	Ogre::Camera* oc = graphics->getMainCamera();

	this->setOgreCamera( oc );
	vp->setBackgroundColour( Ogre::ColourValue::Black );
	ogreCamera->setAspectRatio( Ogre::Real(vp->getActualWidth()) / Ogre::Real(vp->getActualHeight()) );
	ogreCamera->setFOVy( Ogre::Degree(45) );
	ogreCamera->setNearClipDistance( 0.1 );
	ogreCamera->setFarClipDistance( 2000 );

	this->reset();

	this->game = game;
}

void Camera::detach(){
	if( this->game == NULL ){
		throw Exception("Camera is not attached to a game.", "Camera::detach()");
	}

//	Graphics* graphics = game->getSystem()->getGraphics();
	this->target = NULL;
	
	this->setOgreCamera( NULL );

	this->game = NULL;
}

bool Camera::isAttached() const {
	return (this->game != NULL);
}

void Camera::updateOgreCamera(){
	this->ogreCamera->setPosition( this->position.toOgreVector3() );
	this->ogreCamera->setOrientation( this->orientation.toOgreQuaternion() );
}

void Camera::lookAt( const FVector3& point ){
	if( this->ogreCamera == NULL ){
		this->updateOgreCamera();
		this->ogreCamera->lookAt( point.toOgreVector3() );
		this->orientation = this->ogreCamera->getOrientation();
	}

}

void Camera::targetNextVehicle(){
}
void Camera::targetPreviousVehicle(){
}



Camera::modes_e Camera::getMode() const {
	return this->mode;
}
void Camera::setMode(Camera::modes_e newMode){
	this->mode = newMode;
}
int* Camera::getModePtr() const {
	return (int*)&this->mode;
}

Ogre::Camera* Camera::getOgreCamera() const{
	return this->ogreCamera;
}
void Camera::setOgreCamera(Ogre::Camera* newOgreCamera){
	this->ogreCamera = newOgreCamera;
	if( this->ogreCamera != NULL ){
//		this->ogreCamera->setFixedYawAxis( true, Ogre::Vector3::UNIT_Y );
		this->ogreCamera->setFixedYawAxis( false );
	}
}

void Camera::setTarget(GameObject* newTarget){
	this->target = newTarget;
}
GameObject* Camera::getTarget() const {
	return this->target;
}
