#include "Client.h"

#include "System.h"
#include "Exception.h"
#include "Vehicle.h"
#include "Game.h"
#include "Network.h"
#include "Socket.h"
#include "Server.h"
#include "Camera.h"
#include "Gui.h"
#include "EquipmentMenu.h"

const int Client::DEFAULT_PORT = 0;
const int Client::DEFAULT_CLIENT_STATE_INTERVAL = 20;

Client::Client( Network* network ){
	this->network = network;

	this->properties.name = string("Player");
	this->properties.host = string("localhost");
	this->properties.port = Client::DEFAULT_PORT;
//	this->properties.clientStateInterval = Client::DEFAULT_CLIENT_STATE_INTERVAL;
	this->properties.remote = false; // THINKABOUTME: 
	this->properties.ipAddress = Packet::IP_ADDRESS_NONE;

	this->resetInfo();

	this->resetState();

	this->lastDeathMillis = 0;
	this->lastClientStateMillis = 0;
	this->clientStateIntervalMillis = 100;
	this->lastPingMillis = 0;

	this->ready = false;
	this->connectionInterrupted = false;

//	this->packetsReceived = 0;
//	this->packetsSent = 0;

	this->socket = NULL;
	this->server = NULL;
	this->vehicle = NULL;
}

Client::~Client(){
//	delete this->socket;
}

void Client::resetInfo(){
	this->info.ping = 0;
	this->info.secondsOnServer = 0;
	this->info.kills = 0;
	this->info.deaths = 0;
	this->info.score = 0;
	this->info.damageInflicted = 0;
	this->info.team = Game::TEAM_SPECTATORS;
	this->info.vehicle = -1;
	this->info.weapons[0] = -1;
	this->info.weapons[1] = -1;
	this->info.weapons[2] = -1;
	this->info.weapons[3] = -1;
}

void Client::resetState(){
	this->state.position = FVector3::ZERO;
	this->state.orientation = FQuaternion::IDENTITY;
	this->state.velocity = FVector3::ZERO;
	this->state.angularVelocity = FVector3::ZERO;
//	this->state.mainUnitYaw = 0.0f;
//	this->state.mainUnitPitch = 0.0f;
	this->state.health = 0;
	this->state.energy = 0;
	memset( &this->state.vehicleSpecificBytes, 0, 16 );
}

void Client::setUp(){
	System::log("Client: Opening socket on port %i...", this->properties.port);
	this->socket = new Socket( this->network, this->properties.port );
	this->socket->open();
	this->socket->emptyIncomingPacketQueue();

	this->properties.ipAddress = Network::resolveHost( this->properties.host, this->properties.port );

	// reset info and state
	this->resetInfo();
	this->resetState();

	this->ready = false;
	this->connectionInterrupted = false;

	unsigned long currentMillis = this->network->getSystem()->getTimer()->getMilliseconds();
	this->lastClientStateMillis = currentMillis;
	this->lastPingMillis = currentMillis;
	this->lastDeathMillis = 0; // instant spawn
	
//	this->packetsReceived = 0;
//	this->packetsSent = 0;

	this->vehicle = NULL;
}

void Client::tearDown(){
	if( this->server != NULL ){
		System::log("Client: Disconnecting from server...");
		this->disconnect();
	}

	if( this->socket != NULL && this->socket->isOpen() ){
		System::log("Client: Closing socket...");
		this->socket->close();
		delete this->socket;
		this->socket = NULL;
	}
}

void Client::establishLoopbackConnection( Server* server ){
	// send connect packet
	ConnectPacket cp;
	cp.data.clientId = 0;
	Packet::stringToString32( cp.data.clientName, this->properties.name );
	cp.data.protocolVersion = Network::PROTOCOL_VERSION;
	cp.pack( server->properties.ipAddress );

	System::log( "Client: Sending connect packet to '%s:%i'...", server->properties.host.c_str(), server->properties.port );
	this->socket->send( &cp );

	this->server = server;
}

void Client::establishRemoteConnection( Server* server ){
	// send connect packet
	ConnectPacket cp;
	cp.data.clientId = -1;
	Packet::stringToString32( cp.data.clientName, this->properties.name );
	cp.data.protocolVersion = Network::PROTOCOL_VERSION;
	cp.pack( server->properties.ipAddress );

	System::log("Client: Sending connect packet to '%s:%i'...", server->properties.host.c_str(), server->properties.port);
	this->socket->send(&cp);

	// wait for answer(s)
	unsigned long receiveStartMillis = this->network->getSystem()->getTimer()->getMilliseconds();
	unsigned long currentMillis = receiveStartMillis;
	ConnectPacket answer_cp;
	bool answer_cp_received = false;
	DisconnectPacket answer_dp;
	bool answer_dp_received = false;
	ServerPropertiesPacket answer_spp;
	bool answer_spp_received = false;
//	ServerInfoPacket* answer_sip = NULL;

	Packet::packetList_t packetList;
	do{
		this->socket->receive( packetList, 16, 2048 ); // server properties packet is bigger than normal game packets...

		for( list<Packet*>::iterator iter = packetList.begin(); iter != packetList.end(); ++iter ){
			int packetType = (*iter)->packetType;

			switch( packetType ){
				case Packet::PACKET_TYPE_DISCONNECT:
					answer_dp.data = ((DisconnectPacket*)(*iter))->data;
					answer_dp_received = true;
					break;

				case Packet::PACKET_TYPE_CONNECT:
					answer_cp.data = ((ConnectPacket*)(*iter))->data;
					answer_cp_received = true;
//					System::log("Client: Received connect packet packet.");
					break;

				case Packet::PACKET_TYPE_SERVER_PROPERTIES:
					answer_spp.data = ((ServerPropertiesPacket*)(*iter))->data;
					answer_spp_received = true;
//					System::log("Client: Received server properties packet.");
					break;
/*
				case Packet::PACKET_TYPE_SERVER_INFO:
					answer_sip = (ServerInfoPacket*)(*iter);
					break;
*/
				default:
					System::warn( "(in Client::establishRemoteConnection()): Received packet of unexpected packetType (%i). Ignoring", packetType );
					break;
			}
		}

		Packet::clearPacketList( packetList );
		
		currentMillis = this->network->getSystem()->getTimer()->getMilliseconds();
	}while( !answer_dp_received
		  && ( !answer_cp_received || !answer_spp_received /*|| answer_sip == NULL*/ )
		  && ( currentMillis < receiveStartMillis + 5000 )
		);

	if( answer_dp_received ){ // server has sent disconnect packet
		string reason;
		Packet::string256ToString( reason, answer_dp.data.reason );
		System::error( "(in Client::establishRemoteConnection()): Server rejected connection request. Reason: %s", reason.c_str() );
		throw Exception( "Couldn't establish remote connection. (Server rejected connection request.)", "Client::establishRemoteConnection()" );
	}

	if( currentMillis >= receiveStartMillis + 5000 ){ // time limit exceeded
		System::error("(in Client::establishRemoteConnection()): Time limit exceeded while waiting for packets from server.");
		throw Exception("Couldn't establish remote connection. (No answer from server.)", "Client::establishRemoteConnection()");
	}

	if( answer_cp_received && answer_spp_received /*&& answer_sip != NULL*/ ){ // all necessary packets received => extract information
		// spp
		server->applyServerPropertiesPacket( &answer_spp );

		// cp
		server->addClient( this, answer_cp.data.clientId );
		this->properties.clientId = answer_cp.data.clientId;
		Packet::string32ToString( this->properties.name, answer_cp.data.clientName );

		// sip
//		server->applyServerInfoPacket( answer_sip );

		System::log( "Client: Connected to '%s' as '%s'.", server->properties.name.c_str(), this->properties.name.c_str() );
	}

	this->server = server;
}

void Client::disconnect(){
	if( this->server->properties.remote ){
		// send disconnect packet
		DisconnectPacket cp;
		cp.data.clientId = this->properties.clientId;
		strcpy(cp.data.reason, "client-side disconnect");
		cp.pack( server->properties.ipAddress );

		System::log("Client: Sending disconnect packet to '%s:%i'...", server->properties.host.c_str(), server->properties.port);
		this->socket->send( &cp );
	}

	this->server = NULL; // THINKABOUTME: dangerous
}


void Client::update(){
	this->sendPackets();
	this->receivePackets();

}

void Client::receivePackets(){
	Packet::packetList_t packetList;
	this->socket->receive( packetList );

//	this->socket->emptyIncomingPacketQueue(); // THINKABOUTME: ...

	for( list<Packet*>::iterator iter = packetList.begin(); iter != packetList.end(); ++iter ){
		int packetType = (*iter)->packetType;

		switch( packetType ){
			case Packet::PACKET_TYPE_CONNECT:
				this->handleConnectPacket( (ConnectPacket*)(*iter) );
				break;
			case Packet::PACKET_TYPE_DISCONNECT:
				this->handleDisconnectPacket( (DisconnectPacket*)(*iter) );
				break;
			case Packet::PACKET_TYPE_PING:
				this->handlePingPacket( (PingPacket*)(*iter) );
				break;

			case Packet::PACKET_TYPE_SPAWN:
				this->handleSpawnPacket( (SpawnPacket*)(*iter) );
				break;
			case Packet::PACKET_TYPE_KILL:
				this->handleKillPacket( (KillPacket*)(*iter) );
				break;
			case Packet::PACKET_TYPE_FIRE:
				this->handleFirePacket( (FirePacket*)(*iter) );
				break;
			case Packet::PACKET_TYPE_CHAT_MESSAGE:
				this->handleChatMessagePacket( (ChatMessagePacket*)(*iter) );
				break;

			case Packet::PACKET_TYPE_CLIENT_STATE:
				this->handleClientStatePacket( (ClientStatePacket*)(*iter) );
				break;

			case Packet::PACKET_TYPE_SERVER_INFO:
				this->handleServerInfoPacket( (ServerInfoPacket*)(*iter) );
				break;

			default:
				System::warn("(in Client::receivePackets()): Received packet with unexpected packetType (%i). Ignoring.", packetType);
				break;
		}

	}		
	
//	this->packetsReceived += packetList.size(); // count for stats
	Packet::clearPacketList( packetList );
}

void Client::handleConnectPacket( ConnectPacket* cp ){
	if( this->server->properties.remote ){ // remote server => update shadow server
		this->server->handleConnectPacket( cp );
	}
}
void Client::handleDisconnectPacket( DisconnectPacket* dp ){
	Client *c = this->server->clients[dp->data.clientId];
/* check not needed (this is != NULL )
	if( c == NULL ){
		System::warn( "(in Client::handleDisconnectPacket()): Slot %i not in use. Ignoring packet.", dp->data.clientId );
		return;
	}
*/
	// first check whether it's we who got disconnected...
	if( c == this ){
		string reason;
		Packet::string256ToString( reason, dp->data.reason );

		System::error( "(in Client::handleDisconnectPacket()): The server disconnected you. Reason: %s.", reason.c_str() );
		throw Exception( "You were disconnected from the server.", "Client::handleDisconnectPacket()" );
	}

	if( this->server->properties.remote ){ // remote server => update shadow server
		this->server->handleDisconnectPacket( dp );
	}

}
void Client::handlePingPacket( PingPacket* pp ){

	// just answer
	PingPacket p;
	p.data.clientId = this->properties.clientId;
	p.data.startMillis = pp->data.startMillis;
	p.pack( pp->udpPacket->address );
	this->socket->send( &p );
	
	unsigned long currentMillis = this->network->getSystem()->getTimer()->getMilliseconds();
	this->lastPingMillis = currentMillis;
	this->connectionInterrupted = false;

	if( this->server->properties.remote ){
		this->server->lastPingMillis = currentMillis;
	}
}


void Client::handleSpawnPacket( SpawnPacket* sp ){
	if( this->server->properties.remote ){ // remote server => update shadow server
		this->server->handleSpawnPacket( sp );
	}

	Client *c = this->server->clients[sp->data.clientId];
	if( c == NULL ){
		System::warn( "(in Client::handleSpawnPacket()): Slot not in use. Ignoring packet." );
		return;
	}

	// update game object
	Game* game = this->network->getSystem()->getGame();

	// let the vehicle spawn (if there's one to spawn...)
	if( c->vehicle != NULL ){ // if there's already a vehicle unspawn it first
		game->unspawnVehicle( c->vehicle );
		c->vehicle = NULL;
	}
	if( c->info.team != Game::TEAM_SPECTATORS && c->info.vehicle != -1 ){
		c->vehicle = game->spawnVehicle( c );
	}

	// if this client has joined spectators set up camera pos
	if( c == this ){
		game->getCamera()->position = FVector3( sp->data.position );
	}
}
void Client::handleKillPacket( KillPacket* kp ){
	if( this->server->properties.remote ){ // remote server => update shadow server
		this->server->handleKillPacket( kp );
	}

	int killedId = kp->data.killedId;
	Client *killedClient = this->server->clients[killedId];
	
	if( killedClient == NULL ){
		System::warn("(in Client::handleKillPacket()): Slot %i not in use. Ignoring packet.", killedId);
		return;
	}
	
	// if there's a vehicle unspawn it
	Game* game = this->network->getSystem()->getGame();
	if( killedClient->vehicle != NULL ){
		killedClient->vehicle->explode();
		game->unspawnVehicle( killedClient->vehicle );
		killedClient->vehicle = NULL;
	}
	// if this client has been killed, open equipment menu
	Gui* gui = this->network->getSystem()->getGui();
	if( killedClient == this && gui->getOpenEquipmentMenuUponDeath() ){
		gui->getEquipmentMenu()->show();
	}
}
void Client::handleFirePacket( FirePacket* fp ){
	int cId = fp->data.clientId;
	int slot = fp->data.weaponSlot;
	Client *c = this->server->clients[cId];
	
	if( c == NULL ){
		System::warn("(in Client::handleFirePacket()): Slot %i not in use. Ignoring packet.", cId);
		return;
	}
	
	if( slot < 0 || slot > 3  ){
		System::warn("(in Client::handleFirePacket()): Invalid weapon slot: %i. Ignoring packet.", slot);
		return;
	}

	Game* game = this->network->getSystem()->getGame();
	if( c->vehicle != NULL && c->vehicle->getWeapon(slot) != NULL ){
		FVector3 pos = FVector3(fp->data.position);
		FQuaternion ori = FQuaternion(fp->data.orientation);
		game->spawnShot( c->vehicle->getWeapon(slot), pos, ori );
	}
}
void Client::handleChatMessagePacket( ChatMessagePacket* cmp ){
	int cId = cmp->data.clientId;
	Client *c = this->server->clients[cId];
	if( c == NULL ){
		System::warn( "(in Client::handleChatMessagePacket()): Slot %i not in use. Ignoring packet.", cId );
		return;
	}

	string message;
	Packet::string256ToString( message, cmp->data.message );

	if( cmp->data.mode == ChatMessagePacket::MODE_TEAM ){
		System::log(FColor::CYAN, "[%s]: %s", c->properties.name.c_str(), message.c_str() );
	}else{
		System::log(FColor::GREEN, "%s: %s", c->properties.name.c_str(), message.c_str() );
	}

}

void Client::handleServerInfoPacket( ServerInfoPacket* sip ){

//	this->connectionInterrupted = false;

	if( this->server->properties.remote ){ // remote server => update shadow server
		this->server->applyServerInfoPacket( sip );
	}

}
/*
void Client::handleServerStatePacket( ServerStatePacket* ssp ){

	if( this->server->properties.remote ){ // remote server => update shadow server
		this->server->applyServerStatePacket( ssp );
	}

}
*/
void Client::handleClientStatePacket( ClientStatePacket* csp ){
	int cId = csp->data.clientId;
	Client* client = this->server->clients[cId];

	if( client == NULL ){
//		System::warn("(in Client::handleClientStatePacket()): Slot %i not in use. Ignoring packet.", cId);
		return;
	}

	ClientStatePacket::data_s* packetData = &csp->data;

	if( client != this ){ // only apply if other client
		client->state.health = packetData->health;
		client->state.energy = packetData->energy;
//		client->state.mainUnitPitch = packetData->mainUnitPitch;
//		client->state.mainUnitYaw = packetData->mainUnitYaw;
		client->state.position = FVector3( packetData->position );
		client->state.orientation = FQuaternion( packetData->orientation );
		client->state.velocity = FVector3( packetData->velocity );
		client->state.angularVelocity = FVector3( packetData->angularVelocity );

		memcpy( &client->state.vehicleSpecificBytes, &packetData->vehicleSpecificBytes, 16 );

		// update vehicle if there is one
		if( client->vehicle != NULL ){
			client->vehicle->takeOverClientState( client->state );
		}
	}
}

void Client::sendPackets(){
//	Game* game = this->network->getSystem()->getGame();

	unsigned long currentMillis = this->network->getSystem()->getTimer()->getMilliseconds();

	// send a vehicle spawn if player wants to change team and time is right
	if( currentMillis > this->lastDeathMillis + this->server->properties.ghostTime*1000 ){
		if( !this->ready // not ready => send spawn as ready signal to server
			||
		    ( this->vehicle == NULL // client has no vehicle AND...
			  && (    this->network->cVars.network_client_team->getValue() != this->info.team // ...either wants to change team...
			       || this->info.team != Game::TEAM_SPECTATORS // ...or is in a playing team
		         )
		    )
		){

			this->sendSpawnPacket();
			this->lastDeathMillis = currentMillis;
		}
	}

	if( currentMillis > this->lastClientStateMillis + this->clientStateIntervalMillis ){
		if( this->vehicle != NULL ){
			this->sendClientStatePacket();
		}

		this->lastClientStateMillis = currentMillis;

		// recalculate clientStateInterval to match network bandwidth
		unsigned int cs = this->server->properties.connectionSpeed;
		this->clientStateIntervalMillis = (100 * 2 * this->server->info.numClients + 100) / cs;

		this->clientStateIntervalMillis = this->clientStateIntervalMillis < 20 ? 20 : this->clientStateIntervalMillis;
	}

}

void Client::sendClientStatePacket(){
	ClientStatePacket csp;

	csp.data.clientId = this->properties.clientId;
	csp.data.flags = 0;

	if( this->vehicle != NULL ){
//		state_s temp;

		this->vehicle->fillInClientState( this->state );
		csp.data.health = this->state.health;
		csp.data.energy = this->state.energy;
		this->state.position.toFloatArray( csp.data.position );
		this->state.orientation.toFloatArray( csp.data.orientation );
		this->state.velocity.toFloatArray( csp.data.velocity );
		this->state.angularVelocity.toFloatArray( csp.data.angularVelocity );

		memcpy( csp.data.vehicleSpecificBytes, this->state.vehicleSpecificBytes, 16 );

	}else{
		csp.data.health = 0;
		csp.data.energy = 0;
	}

	csp.pack( this->server->properties.ipAddress );
	this->socket->send( &csp );
}

void Client::sendSpawnPacket(){
	SpawnPacket sp;
	sp.data.clientId = this->properties.clientId;
	sp.data.team = this->network->cVars.network_client_team->getValue();
	sp.data.vehicle = this->network->cVars.network_client_vehicle->getValue();
	sp.data.weapons[0] = this->network->cVars.network_client_weapon1->getValue();
	sp.data.weapons[1] = this->network->cVars.network_client_weapon2->getValue();
	sp.data.weapons[2] = this->network->cVars.network_client_weapon3->getValue();
	sp.data.weapons[3] = this->network->cVars.network_client_weapon4->getValue();

	FVector3 pos = this->network->getSystem()->getGame()->findSpawnPosition( sp.data.team );
	sp.data.position[0] = pos.components[0];
	sp.data.position[1] = pos.components[1];
	sp.data.position[2] = pos.components[2];

//	System::log("sendS: t: %i, v: %i\n", sp.data.team, sp.data.vehicle);

	sp.pack( this->server->properties.ipAddress );
	this->socket->send( &sp );
}

void Client::sendKillPacket( int killedClientId, int killerClientId ){
	KillPacket kp;
	kp.data.killedId = killedClientId;
	kp.data.killerId = killerClientId;

	kp.pack( this->server->properties.ipAddress );
	this->socket->send( &kp );
}
void Client::sendFirePacket( int weaponSlot, const FVector3& position, const FQuaternion& orientation ){
	FirePacket fp;
	fp.data.clientId = this->properties.clientId;
	fp.data.weaponSlot = weaponSlot;
	position.toFloatArray( fp.data.position );
	orientation.toFloatArray( fp.data.orientation );

	fp.pack( this->server->properties.ipAddress );
	this->socket->send( &fp );
}
void Client::sendChatMessagePacket( const string& message, ChatMessagePacket::modes_e mode ){
	ChatMessagePacket cmp;

	cmp.data.clientId = this->properties.clientId;
	cmp.data.mode = mode;
	Packet::stringToString256( cmp.data.message, message );

	cmp.pack( this->server->properties.ipAddress );
	this->socket->send( &cmp );
}


bool Client::isConnected() const {
	return (this->server != NULL);
}
bool Client::isConnectionInterrupted() const {
	return this->connectionInterrupted;
}

Vehicle* Client::getVehicle() const {
	return this->vehicle;
}
