#include "DotSceneLoader.h"

#include "Exception.h"
#include "System.h"
#include "XmlUtils.h"

DotSceneLoader::DotSceneLoader(const string& dotSceneFilePath
							   , Ogre::SceneManager* sceneManager
							   , Ogre::SceneNode* rootNode
							   , bool staticMeshes
							   , bool castShadows){

	this->dotSceneFilePath = dotSceneFilePath;
	this->renderWindow = NULL;
	this->sceneManager = sceneManager;
	this->rootNode = NULL;

	this->staticMeshes = staticMeshes;
	this->castShadows = castShadows;

	this->nodeCounter = 0;
	this->entityCounter = 0;
	this->lightCounter = 0;
	this->billboardSetCounter = 0;
	this->particleSystemCounter = 0;
}

DotSceneLoader::~DotSceneLoader(){
}

void DotSceneLoader::load(){
	if( this->rootNode == NULL ){
		this->rootNode = sceneManager->getRootSceneNode();
	}
//	this->rootNode->rotate( Ogre::Vector3(1,0,0), Ogre::Degree(-90) );

	TiXmlDocument doc( dotSceneFilePath.c_str() );
	if( !doc.LoadFile() ){
		throw Exception("Couldn't load file '" + dotSceneFilePath + "'. (Maybe not a correct xml file?)", "DotSceneLoader::load()");
	}
//	doc.Print();

	TiXmlNode* rootElement = doc.RootElement();
	TiXmlElement* element = rootElement->ToElement();

	string rootElementName = element->Value();
	if( rootElementName != "scene" ){
		throw Exception("File '" + dotSceneFilePath + "' does not have a <scene> root node.", "DotSceneLoader::load()");
	}

	// check version
	const char* tmp = element->Attribute( "formatVersion" );
	if( tmp == NULL ){
		System::warn("(in DotSceneLoader::load()): file '%s' has no version information. Trying to parse anyway.", dotSceneFilePath.c_str());
	}else{
		string version = tmp;
//		System::log("VERSION: %s", version.c_str());
		if( version != "0.2.0" ){
			System::warn("(in DotSceneLoader::load()): file '%s' has version %s. Trying to parse anyway.", dotSceneFilePath.c_str(), version.c_str() );
		}
	}

	// parse environment
	element = rootElement->FirstChildElement("environment");
	if( element ){
		this->parseEnvironment(element);
	}

	// parse nodes
	element = rootElement->FirstChildElement("nodes");
	if( element != NULL ){
		this->parseNodes(element);
	}

	// ignore externals...
//	element = rootElement->FirstChildElement("externals");
//	if( element != NULL ){
//		this->parseExternals(element);
//	}
}





void DotSceneLoader::parseEnvironment(TiXmlElement* environmentElement){

	Ogre::ColourValue amb = Ogre::ColourValue::Black;
	Ogre::ColourValue bkg = Ogre::ColourValue::Black;

	TiXmlElement* elt = NULL;

	elt = environmentElement->FirstChildElement("colourAmbient");
	if( elt != NULL ){
		amb = XmlUtils::readOgreColourValue(elt);
	}

	elt = environmentElement->FirstChildElement("colourBackground");
	if( elt != NULL ){
		bkg = XmlUtils::readOgreColourValue(elt);
	}

	this->sceneManager->setAmbientLight(amb);
//	for( int n=0; n<this->renderWindow->getNumViewports(); n++ ){
//		this->renderWindow->getViewport(n)->setBackgroundColour(bkg);
//	}


	elt = environmentElement->FirstChildElement("skyBox");
	if( elt != NULL ){
		this->parseSkyBox(elt);
	}
	elt = environmentElement->FirstChildElement("skyDome");
	if( elt != NULL ){
		this->parseSkyDome(elt);
	}
	elt = environmentElement->FirstChildElement("skyPlane");
	if( elt != NULL ){
		this->parseSkyPlane(elt);
	}
	elt = environmentElement->FirstChildElement("fog");
	if( elt != NULL ){
		this->parseFog(elt);
	}

}

void DotSceneLoader::parseSkyBox(TiXmlElement* skyBoxElement){
	// material
	string mat = skyBoxElement->Attribute("material") ? skyBoxElement->Attribute("material") : "BaseWhite";

	// distance
	float dist = skyBoxElement->Attribute("distance") ? atof( skyBoxElement->Attribute("distance") ) : 1000.0f;

	// drawFirst
	bool drawFirst = skyBoxElement->Attribute("drawFirst") ? string( skyBoxElement->Attribute("drawFirst") ) == string("true") : false;

	// orientation
	TiXmlElement* orientationElement = skyBoxElement->FirstChildElement("rotation");
	Ogre::Quaternion ori = Ogre::Quaternion::IDENTITY;
	if( orientationElement != NULL ){
		ori = XmlUtils::readOgreQuaternion(orientationElement);
	}
//	this->sceneManager->setSkyDome(false, "");
//	this->sceneManager->setSkyPlane(false, Plane(), "");
	this->sceneManager->setSkyBox(true, mat, dist, drawFirst, ori);
}

void DotSceneLoader::parseSkyDome(TiXmlElement* skyDomeElement){
}

void DotSceneLoader::parseSkyPlane(TiXmlElement* skyPlaneElement){
	// material
	string mat = skyPlaneElement->Attribute("material") ? skyPlaneElement->Attribute("material") : "BaseWhite";

	// drawFirst
	bool drawFirst = skyPlaneElement->Attribute("drawFirst") ? string( skyPlaneElement->Attribute("distance") ) == string("true") : false;

	// plane
	Ogre::Plane plane( Ogre::Vector3(0.0, -1.0, 0.0), 1000.0 );
	if( skyPlaneElement->Attribute("planeX") != NULL ){
		plane = XmlUtils::readOgrePlane(skyPlaneElement);
	}

	// scale
	float scale = skyPlaneElement->Attribute("scale") ? atof( skyPlaneElement->Attribute("scale") ) : 1000.0f;

	// tiling
	float tiling = skyPlaneElement->Attribute("tiling") ? atof( skyPlaneElement->Attribute("tiling") ) : 10.0f;

	// bow
	float bow = skyPlaneElement->Attribute("bow") ? atof( skyPlaneElement->Attribute("bow") ) : 0.0f;

	// set it
	this->sceneManager->setSkyPlane(true, plane, mat, scale, tiling, drawFirst, bow);
}

void DotSceneLoader::parseFog(TiXmlElement* fogElement){
}







void DotSceneLoader::parseNodes(TiXmlElement* nodesElement){
	for( TiXmlElement* elt = nodesElement->FirstChildElement(); elt != 0; elt = elt->NextSiblingElement() ){
		string value = string( elt->Value() );

		if (value == "position"){
			this->rootNode->setPosition( XmlUtils::readOgreVector3(elt) );
			this->rootNode->setInitialState();
		}else if( value == "rotation" ){
			this->rootNode->setOrientation( XmlUtils::readOgreQuaternion(elt) );
			this->rootNode->setInitialState();
		}else if (value == "scale"){
			this->rootNode->setScale( XmlUtils::readOgreVector3(elt) );
			this->rootNode->setInitialState();
		}else if( value == "node" ){
			this->parseNode( this->rootNode, elt );
		}else{
			System::warn("(in DotSceneLoader::parseNodes()): Unknown element: '%s' (row: %i). Ignoring.", value.c_str(), elt->Row() );
		}
	}
}

void DotSceneLoader::parseNode(Ogre::SceneNode* parentNode, TiXmlElement* nodeElement){
	string name;
	const char* tmp = nodeElement->Attribute("name");
	if( tmp == NULL ){
		name = "unnamed node " + nodeCounter;
	}else{
		name = tmp;
	}

	// create the node
	Ogre::SceneNode* sceneNode = parentNode->createChildSceneNode( name );
	this->nodeCounter++;

	// ignore isTargetAttribute...

	// do node PRS values first
	TiXmlElement* elt = NULL;
	for( elt = nodeElement->FirstChildElement(); elt != 0; elt = elt->NextSiblingElement() ){
//		System::log("parsing PRS %s", elt->Value() );

		string value = string( elt->Value() );
		if( value == "position" ){
			sceneNode->setPosition( XmlUtils::readOgreVector3(elt) );
		}else if( value == "scale" ){
			sceneNode->setScale( XmlUtils::readOgreVector3(elt) );
		}else if( value == "rotation" ){
			sceneNode->setOrientation( XmlUtils::readOgreQuaternion(elt) );
		}else if( value == "lookTarget" ){
			// ignore
		}else if (value == "trackTarget"){
			// ignore
		}
	}

	// do movable objects now
	for( elt = nodeElement->FirstChildElement(); elt != 0; elt = elt->NextSiblingElement() ){
//		System::log("parsing mo %s", elt->Value() );

		string value = string( elt->Value() );
		if( value == "entity" ){
			this->parseEntity(sceneNode, elt);
		}else if( value == "light" ){
			this->parseLight(sceneNode, elt);
		}else if( value == "camera" ){
			this->parseCamera(sceneNode, elt);
		}else if( value == "billboardSet" ){
			this->parseBillboardSet(sceneNode, elt);
		}else if( value == "particleSystem" ){
			this->parseParticleSystem(sceneNode, elt);
		}
	}

	// do child nodes as last !!!
	for( elt = nodeElement->FirstChildElement(); elt != 0; elt = elt->NextSiblingElement() ){
//		System::log("parsing cn %s", elt->Value() );

		string value = string( elt->Value() );
		if( value == "node" ){
			this->parseNode(sceneNode, elt);
		}
	}
}

void DotSceneLoader::parseEntity(Ogre::SceneNode* parentNode, TiXmlElement* entityElement){
	string name;
	const char* tmp = entityElement->Attribute("name");
	if( tmp == NULL ){
		name = "unnamed entity " + entityCounter;
	}else{
		name = tmp;
	}
//	System::log("parsing entity '%s'", name.c_str());

	Ogre::Entity *ent = NULL;

	Ogre::String meshName = entityElement->Attribute("meshFile") ? entityElement->Attribute("meshFile") : Ogre::StringUtil::BLANK;
	if( meshName != Ogre::StringUtil::BLANK ){
		//Ogre::MeshManager::getSingleton.load(meshName, );
		ent = this->sceneManager->createEntity(name, meshName);
		this->entityCounter++;

//		if( entityElement->Attribute("static") != NULL && string(entityElement->Attribute("static")) == string("true") ){
//		if( true ){
		if( this->staticMeshes ){
			Ogre::StaticGeometry* sg = this->sceneManager->createStaticGeometry( "static geometry for " + name );
			sg->addEntity( ent, parentNode->_getDerivedPosition(), parentNode->_getDerivedOrientation(), parentNode->_getDerivedScale());
			sg->setCastShadows(this->castShadows);
			sg->build();
			// THINKABOUTME: oder mit node?
		}else{
			ent->setCastShadows(this->castShadows);
			parentNode->attachObject(ent);
		}
	}
}

void DotSceneLoader::parseLight(Ogre::SceneNode* parentNode, TiXmlElement* lightElement){
	string name;
	const char* tmp = lightElement->Attribute("name");
	if( tmp == NULL ){
		name = "unnamed light " + lightCounter;
	}else{
		name = tmp;
	}
//	System::log("parsing light '%s'", name.c_str());


	Ogre::Light* light = this->sceneManager->createLight(name);
	parentNode->attachObject(light);
	this->lightCounter++;

	// visibility
	bool visible = true;
	if( lightElement->Attribute("visible") != NULL && string( lightElement->Attribute("visible") ) == string("false") ){
		visible = false;
	}
	light->setVisible(visible);

	// type
	string type = lightElement->Attribute("type") ? lightElement->Attribute("type") : "point";
	if (type == "point"){
		light->setType(Ogre::Light::LT_POINT);
	}else if (type == "directional" || type == "targetDirectional"){
		light->setType(Ogre::Light::LT_DIRECTIONAL);
	}else if (type == "spot" || type == "targetSpot"){
		light->setType(Ogre::Light::LT_SPOTLIGHT);
	}

	// shadows
	bool castShadows = true;
	if( lightElement->Attribute("castShadows") != NULL && string( lightElement->Attribute("castShadows") ) == string("false") ){
		castShadows = false;
	}
	light->setCastShadows(castShadows);


	TiXmlElement* elt = NULL;
	// colours
	elt = lightElement->FirstChildElement("colourDiffuse");
	if( elt ){
		light->setDiffuseColour( XmlUtils::readOgreColourValue(elt) );
	}
	elt = lightElement->FirstChildElement("colourSpecular");
	if( elt ){
		light->setSpecularColour( XmlUtils::readOgreColourValue(elt) );
	}

	// attenuation
	elt = lightElement->FirstChildElement("lightAttenuation");
	if( elt ){
		Ogre::Real range = elt->Attribute("range") ? atof(elt->Attribute("range")) : 1000.0;
		Ogre::Real constant = elt->Attribute("constant") ? atof(elt->Attribute("constant")) : 1.0;
		Ogre::Real linear = elt->Attribute("linear") ? atof(elt->Attribute("linear")) : 1.0;
		Ogre::Real quadratic = elt->Attribute("quadratic") ? atof(elt->Attribute("quadratic")) : 0.0;
		light->setAttenuation(range, constant, linear, quadratic);
	}

	// spotlight range
	elt = lightElement->FirstChildElement("lightRange");
	if( elt ){
		Ogre::Real inner = elt->Attribute("inner") ? atof(elt->Attribute("inner")) : 1.0;
		Ogre::Real outer = elt->Attribute("outer") ? atof(elt->Attribute("outer")) : 1.0;
		Ogre::Real falloff = elt->Attribute("falloff") ? atof(elt->Attribute("falloff")) : 0.0;
		light->setSpotlightRange(Ogre::Angle(inner), Ogre::Angle(outer), falloff);
	}

	// direction
	elt = lightElement->FirstChildElement("normal");
	if( elt ){
		light->setDirection( XmlUtils::readOgreVector3(elt) );
	}

	// position
	elt = lightElement->FirstChildElement("position");
	if( elt ){
		light->setPosition( XmlUtils::readOgreVector3(elt) );
	}
}

void DotSceneLoader::parseCamera(Ogre::SceneNode* parentNode, TiXmlElement* cameraElement){
	System::warn("(DotSceneLoader::parseCamera()): Ignoring camera.");
}

void DotSceneLoader::parseBillboardSet(Ogre::SceneNode* parentNode, TiXmlElement* billboardSetElement){
	string name;
	const char* tmp = billboardSetElement->Attribute("name");
	if( tmp == NULL ){
		name = "unnamed billboard set " + billboardSetCounter;
	}else{
		name = tmp;
	}
//	System::log("parsing billboardSet '%s'", name.c_str());

	Ogre::BillboardSet* bs = this->sceneManager->createBillboardSet(name);
	this->billboardSetCounter++;

	// type
	Ogre::BillboardType bt = Ogre::BBT_POINT;
	if( billboardSetElement->Attribute("type") != NULL ){
		string type = billboardSetElement->Attribute("type");
		if( type == "point" ){
			bt = Ogre::BBT_POINT;
		}else if(type == "orientedCommon" ){
			bt = Ogre::BBT_ORIENTED_COMMON;
		}else if(type == "orientedSelf" ){
			bt = Ogre::BBT_ORIENTED_SELF;
		}else{
			bt = Ogre::BBT_POINT;
			// warn!
		}
	}
	bs->setBillboardType(bt);
	bs->setCommonDirection(Ogre::Vector3(0.0, 0.0, 1.0));

	// origin
	Ogre::BillboardOrigin bo = Ogre::BBO_CENTER;
	if( billboardSetElement->Attribute("origin") != NULL ){
		string origin = billboardSetElement->Attribute("origin");
		if(origin == "topLeft")
			bo = Ogre::BBO_TOP_LEFT;
		else if(origin == "topCenter")
			bo = Ogre::BBO_TOP_CENTER;
		else if(origin == "topRight")
			bo = Ogre::BBO_TOP_RIGHT;
		else if(origin == "centerLeft")
			bo = Ogre::BBO_CENTER_LEFT;
		else if(origin == "center")
			bo = Ogre::BBO_CENTER;
		else if(origin == "centerRight")
			bo = Ogre::BBO_CENTER_RIGHT;
		else if(origin == "bottomLeft")
			bo = Ogre::BBO_BOTTOM_LEFT;
		else if(origin == "bottomCenter")
			bo = Ogre::BBO_BOTTOM_CENTER;
		else if(origin == "bottomRight")
			bo = Ogre::BBO_BOTTOM_RIGHT;
		else
			bo = Ogre::BBO_CENTER;
	}
	bs->setBillboardOrigin(bo);

	// dimensions
	if( billboardSetElement->Attribute("width") != NULL ){
		bs->setDefaultWidth( atof(billboardSetElement->Attribute("width")) );
	}
	if( billboardSetElement->Attribute("height") != NULL ){
		bs->setDefaultHeight( atof(billboardSetElement->Attribute("height")) );
	}

	// material
	if( billboardSetElement->Attribute("material") ){
		string matName = billboardSetElement->Attribute("material");
		bs->setMaterialName(matName);
	}

	// do all billboards
	for( TiXmlElement* elt = billboardSetElement->FirstChildElement(); elt != 0; elt = elt->NextSiblingElement() ){
		if( string(elt->Value()) == string("billboard") ){
			this->parseBillboard(bs, elt);
		}else{
			// warn!
		}
	}

	parentNode->attachObject( bs );
//	parentNode->rotate(Ogre::Vector3(1,0,0), Ogre::Degree(90));
}

void DotSceneLoader::parseBillboard(Ogre::BillboardSet* parentBillboardSet, TiXmlElement* billboardElement){
	string name;
	const char* tmp = billboardElement->Attribute("name");
	if( tmp == NULL ){
		char numStr[256];
		sprintf(numStr, "%i", parentBillboardSet->getNumBillboards() );
		name = "unnamed billboard " + string(numStr) + " of set " + string(parentBillboardSet->getName());
	}else{
		name = tmp;
	}
//	System::log("parsing billboard '%s'", name.c_str());


	// dimensions

	// read position
	Ogre::Vector3 pos = Ogre::Vector3::ZERO;
	TiXmlElement* elt = billboardElement->FirstChildElement("position");
	if( elt != NULL ){
		pos = XmlUtils::readOgreVector3(elt);
	}

	// read diffuse color
	Ogre::ColourValue dif = Ogre::ColourValue::White;
	elt = billboardElement->FirstChildElement("colourDiffuse");
	if( elt != NULL ){
		dif = XmlUtils::readOgreColourValue(elt);
	}

	// rotation
	Ogre::Quaternion rot = Ogre::Quaternion::IDENTITY;
	elt = billboardElement->FirstChildElement("rotation");
	if( elt != NULL ){
		rot = XmlUtils::readOgreQuaternion(elt);
	}

	// dimensions
	float width = -1.0f;
	if( billboardElement->Attribute("width") != NULL ){
		width = atof(billboardElement->Attribute("width"));
	}
	float height = -1.0f;
	if( billboardElement->Attribute("height") != NULL ){
		height = atof(billboardElement->Attribute("height"));
	}

	// create
	Ogre::Billboard* b = parentBillboardSet->createBillboard(pos, dif);
	b->setRotation( Ogre::Angle(rot.w) );
	if( width != -1.0f && height != -1.0f ){
		b->setDimensions(width, height);
	}

}

void DotSceneLoader::parseParticleSystem(Ogre::SceneNode* parentNode, TiXmlElement* particleSystemElement){
	string name;
	const char* tmp = particleSystemElement->Attribute("name");
	if( tmp == NULL ){
		name = "unnamed particle system " + particleSystemCounter;
	}else{
		name = tmp;
	}
//	System::log("parsing particleSystem '%s'", name.c_str());

	if( particleSystemElement->Attribute("template") ){
		string templateName = particleSystemElement->Attribute("template");
		Ogre::ParticleSystem* ps = this->sceneManager->createParticleSystem(name, templateName);
//		cout << "ps: " << ps << endl;
		parentNode->attachObject( ps );
	}
}



