#include "DynamicGeometry.h"

#include "Exception.h"
#include "XmlUtils.h"
#include "System.h"
#include "Shot.h"
#include "Game.h"
#include "Graphics.h"

DynamicGeometry::DynamicGeometry(): Geometry( Geometry::GEOMETRY_TYPE_DYNAMIC ) {

	this->staticGeometry = NULL;
	this->sceneNode = NULL;
	this->entity = NULL;

	this->health = 0.0f;
	this->exploded = false;
}

DynamicGeometry::~DynamicGeometry(){
}

void DynamicGeometry::reset(){
	this->health = this->physicalProperties.healthCapacity;
	this->exploded = false;
}

Geometry* DynamicGeometry::clone() const {
	DynamicGeometry* ret = new DynamicGeometry();

	*ret = *this;

	// TODO: copy ogreSceneNode, etc...
	ret->reset();

	return ret;
}

void DynamicGeometry::attach( Game* game ){
	if( this->game != NULL ){
		throw Exception("Geometry is already attached to a game.", "DynamicGeometry::attach()");
	}

	this->instanceId = game->getNextId();
//	printf("id: %d\n", id);
	char tmp[512];
	sprintf(tmp, "effect '%s' (instance id: %d)", this->name.c_str(), this->instanceId);
	this->instanceName = string(tmp);

	Ogre::SceneManager* sm = game->getSystem()->getGraphics()->getSceneManager();
	Ogre::SceneNode* rootSceneNode = sm->getRootSceneNode();

	this->sceneNode = rootSceneNode->createChildSceneNode( this->instanceName + "/SceneNode" );
	this->sceneNode->setPosition( this->position.toOgreVector3() );
	this->sceneNode->setOrientation( this->orientation.toOgreQuaternion() );

	if( !this->graphicalProperties.mesh.empty() ){
		this->entity = sm->createEntity( this->instanceName + "/Mesh", this->graphicalProperties.mesh );
		this->entity->setCastShadows( this->graphicalProperties.castShadows );
		this->sceneNode->attachObject( this->entity );
	}

	if( this->graphicalProperties.staticGeometry ){
		// TODO
		this->staticGeometry = sm->createStaticGeometry( this->instanceName + "/StaticGeometry" );
		this->staticGeometry->addSceneNode( this->sceneNode );
		this->staticGeometry->setCastShadows( this->graphicalProperties.castShadows );
		this->staticGeometry->build();

		rootSceneNode->removeAndDestroyChild( this->sceneNode->getName() );
		this->sceneNode = NULL;
	}else{
	}



	Physics* physics = game->getPhysics();
	this->physicalProperties.shape.createGeoms( this );

	this->body = dBodyCreate( physics->getWorld() );
	dBodySetMass( this->body, &this->physicalProperties.mass );
	dBodySetPosition( this->body, this->position.x, this->position.y, this->position.z );
	dVector4 tmpv4;
	this->orientation.toOdeVector4(tmpv4);
	dBodySetQuaternion( this->body, tmpv4 );
	dBodySetGravityMode( this->body, this->physicalProperties.affectedByGravity );
	dBodySetData( this->body, this );
	this->physicalProperties.shape.attachGeomsToBody( this->body );
	this->physicalProperties.shape.addGeomsToSpace( physics->getDynamicGeometrySpace() );

	this->game = game;
}

void DynamicGeometry::detach(){
	if( this->game == NULL ){
		throw Exception("Geometry is not attached to a game.", "DynamicGeometry::detach()");
	}

	Physics* physics = this->game->getPhysics();
	this->physicalProperties.shape.detachGeomsFromBody( );
	dBodyDestroy( this->body );
	this->physicalProperties.shape.removeGeomsFromSpace( physics->getDynamicGeometrySpace() );
	this->physicalProperties.shape.destroyGeoms();

	Ogre::SceneManager* sm = this->game->getSystem()->getGraphics()->getSceneManager();

	if( this->staticGeometry != NULL ){
		sm->destroyStaticGeometry( this->staticGeometry );
		this->staticGeometry = NULL;
	}

	if( this->entity != NULL ){
		sm->destroyEntity( this->entity );
		this->entity = NULL;
	}

	if( this->sceneNode != NULL ){
		//sm->destroySceneNode( this->ogreSceneNode->getName() );
		this->sceneNode->getParentSceneNode()->removeAndDestroyChild( this->sceneNode->getName() );
		this->sceneNode = NULL;
	}

	this->game = NULL;
}


void DynamicGeometry::update( float deltaT ){
	if( this->physicalProperties.healthDegeneration > 0.0f ){
		this->damage( this->physicalProperties.healthDegeneration * deltaT );
	}
}




//void DynamicGeometry::calculateChecksum(){
//	this->checksum = 0;
//}


void DynamicGeometry::prepareForSimulationStep(){
//	dVector3 tmpv3;
	dVector4 tmpv4;

	dBodySetPosition( this->body, this->position.x, this->position.y, this->position.z );
	this->orientation.toOdeVector4(tmpv4);
	dBodySetQuaternion( this->body, tmpv4 );
	dBodySetLinearVel( this->body, this->velocity.x, this->velocity.y, this->velocity.z );
	dBodySetAngularVel( this->body, this->angularVelocity.x, this->angularVelocity.y, this->angularVelocity.z );
}

void DynamicGeometry::updateAfterSimulationStep(){
	const dReal* pos = dBodyGetPosition( this->body );
	const dReal* ori = dBodyGetQuaternion( this->body );
	const dReal* vel = dBodyGetLinearVel( this->body );
	const dReal* avel = dBodyGetAngularVel( this->body );
	this->position = FVector3(pos);
	this->orientation = FQuaternion(ori);
	this->velocity = FVector3(vel);
	this->angularVelocity = FVector3(avel);

	this->sceneNode->setPosition(this->position.toOgreVector3());
	this->sceneNode->setOrientation(this->orientation.toOgreQuaternion());
}


bool DynamicGeometry::collidedWithGameObject(const FVector3& contactPoint, GameObject* other, dContact& contact){

	if( other == NULL ){
		return true;

	}else if( other->getGameObjectType() == GameObject::GAME_OBJECT_TYPE_SHOT ){
//		Shot* shot = (Shot*)other;
//		return this->collidedWithShot( contactPoint, shot, contact );
		return false;

	}else{
		contact.surface.mu += Physics::frictionToMu(this->physicalProperties.friction) * 0.5f;
		if( this->physicalProperties.bouncyness > 0.0f ){
			contact.surface.mode |= dContactBounce;
			contact.surface.bounce += this->physicalProperties.bouncyness*0.5f;
		}
		return true;
	}
}
bool DynamicGeometry::collidedWithTerrain(const FVector3& contactPoint, dContact& contact){
	contact.surface.mu += Physics::frictionToMu(this->physicalProperties.friction) * 0.5f;
	if( this->physicalProperties.bouncyness > 0.0f ){
		contact.surface.mode |= dContactBounce;
		contact.surface.bounce += this->physicalProperties.bouncyness*0.5f;
	}
	return true;
}

void DynamicGeometry::addLinearImpulse( const FVector3& impulse, const FVector3& position ){

	// convert impulse into force and apply it
	float deltaT = this->game->getSystem()->getTimer()->getDeltaT();
	dVector3 force;
	dWorldImpulseToForce( this->game->getPhysics()->getWorld(), deltaT,
		impulse.x, impulse.y, impulse.z, force);

	dBodyAddForceAtPos( this->body,
		force[0], force[1], force[2],
		position.x, position.y, position.z );

	dBodyEnable( this->body );

//			printf("FORCE: %f, %f, %f\n", force[0], force[1], force[2]);

}
void DynamicGeometry::addAngularImpulse( const FVector3& impulse ){
	// convert impulse into force and apply it
	float deltaT = this->game->getSystem()->getTimer()->getDeltaT();
	dVector3 torque;
	dWorldImpulseToForce( this->game->getPhysics()->getWorld(), deltaT,
		impulse.x, impulse.y, impulse.z, torque);

	dBodyAddTorque( this->body,
		torque[0], torque[1], torque[2] );

	dBodyEnable( this->body );

}

void DynamicGeometry::damage( float amount ){
	if( this->physicalProperties.destroyable && !this->exploded ){
		this->health -= amount;

		if( this->health < 0.0 ){
			this->explode();
			this->health = 0.0f;
		}
	}
}

void DynamicGeometry::explode(){
	if( !this->exploded ){
		this->exploded = true; // set to true to avoid endless recursion

	}

	// effect
	if( !this->graphicalProperties.explosionEffect.empty() ){
		Effect* e = this->game->spawnEffect( this->graphicalProperties.explosionEffect, this->position, this->orientation );
	}

	// debris
	for( vector<physicalProperties_s::debris_s>::const_iterator c_iter = this->physicalProperties.debris.begin();
		c_iter != this->physicalProperties.debris.end(); ++c_iter )
		{

		Ogre::Matrix3 m;
		m.FromAxes( this->right.toOgreVector3(), this->up.toOgreVector3(), -this->direction.toOgreVector3() );
		FVector3 pos = this->position + FVector3( m * c_iter->initialPosition.toOgreVector3() );

		if( !c_iter->geometry.empty() ){
			Geometry* g = this->game->spawnGeometry( c_iter->geometry, pos, this->orientation );
			if( !c_iter->effect.empty() ){
				this->game->spawnEffect( c_iter->effect, g );
			}
			g->velocity = this->velocity;
			g->angularVelocity = this->angularVelocity;

			g->addLinearImpulse( c_iter->initialLinearImpulse, pos );
			g->addAngularImpulse( c_iter->initialAngularImpulse );

		}else{ // only spawn effect if there is one
			if( !c_iter->effect.empty() ){
				this->game->spawnEffect( c_iter->effect, pos, this->orientation );
			}
		}
	}
}

bool DynamicGeometry::shouldBeDetached() const {
	return this->exploded;
}


