#include "EffectFactory.h"

#include "System.h"
#include "Exception.h"

EffectFactory::EffectFactory(Game* game){
	this->game = game;

	this->effectsByName.clear();
}

EffectFactory::~EffectFactory(){
	// delete effects
	for( effectsByName_t::iterator iter = effectsByName.begin(); iter != effectsByName.end(); iter++ ){
		delete iter->second;
	}

}

void EffectFactory::addEffect(const string& effectFilePath){
//	System::log( "EffectFactory: Parsing effect '%s'...", effectFilePath.c_str() );

	try{
		TiXmlDocument doc( effectFilePath.c_str() );
		if( !doc.LoadFile() ){
			throw Exception("Couldn't load file '" + effectFilePath + "'. (Maybe not a correct xml file?)", "EffectFactory::addEffect()");
		}
//		doc.Print();

		TiXmlNode* rootElement = doc.RootElement();
		TiXmlElement* element = rootElement->ToElement();

		string rootElementName = element->Value();
		if( rootElementName != "effect" ){
			throw Exception("Effect file '" + effectFilePath + "' does not have an <effect> root node.", "EffectFactory::addEffect()");
		}

		// check version
		const char* tmp = element->Attribute( "formatVersion" );
		if( tmp == NULL ){
			throw Exception("Effect file '" + effectFilePath + "' has no version information.", "EffectFactory::addEffect()");
		}else{
			string version = tmp;

			if( version != System::getInstance()->getVersionString() ){
				System::warn("(in EffectFactory::addEffect()): file '%s' has version %s. Trying to parse anyway.", effectFilePath.c_str(), version.c_str() );
			}
		}


		Effect* e = new Effect();
		e->loadFromFile(effectFilePath);

		effectsByName_t::const_iterator c_iter = effectsByName.find( e->getName() );
		if( c_iter != effectsByName.end() ){ // another effect with that name is already registered -> throw exception
			throw Exception("An effect with name '" + e->getName() +"' is already registered.", "EffectFactory::addEffect()");
		}


		// insert the new effect into the list
		effectsByName. insert( effectsByName_t::value_type(e->getName(), e) );
		System::log("EffectFactory: Added effect '%s' to list.", e->getName().c_str());
	}catch( Exception& e ){
		System::warn("(EffectFactory::addEffect()): An exception occured while parsing effect file '%s'. Ignoring effect.", effectFilePath.c_str() );
		System::log("EffectFactory: The exception was: %s", e.getDescription().c_str());
	}

}

void EffectFactory::removeEffect(const string& name){
	effectsByName_t::iterator iter = effectsByName.find( name );
	if( iter == effectsByName.end() ){
		throw Exception("An effect with name '" + name +"' is not registered.", "EffectFactory::removeEffect()");
	}
	this->effectsByName.erase(iter);

}


Effect* EffectFactory::createEffect(const string& name) const {

	effectsByName_t::const_iterator c_iter = effectsByName.find( name );
	if( c_iter == effectsByName.end() ){
		throw Exception("An effect with name '" + name +"' is not registered.", "EffectFactory::createEffect()");
	}

	Effect* ret = c_iter->second->clone();

	return ret;
}

void EffectFactory::destroyEffect(Effect* effect) const {
	delete effect;
}
