#ifndef __Game_h__
#define __Game_h__

#include "GameCCmds.h"
#include "SubSystem.h"

#include "FVectors.h"
#include "FQuaternion.h"

#include "CVar.h"
#include "Input.h"

class Camera;
class Physics;
class GameObject;
class Arena;
class Vehicle;
class Weapon;
class Shot;
class Geometry;
class Effect;
class Area;

class ArenaFactory;
class VehicleFactory;
class WeaponFactory;
class ShotFactory;
class EffectFactory;
class GeometryFactory;

class Console;
class Client;

class Game: public SubSystem{
public:

	enum modes_e{
		MODE_EDIT = 0,
		MODE_DEATHMATCH,
		MODE_TEAM_DEATHMATCH,
		MODE_CONQUEST,

		NUM_MODES
	};

	enum teams_e{
		TEAM_SPECTATORS = 0,
		TEAM_PLAYERS,

		TEAM_RED,
		TEAM_BLUE,

		NUM_TEAMS
	};

	enum teamFlags_e{
		TEAM_FLAG_NONE = 0x00,

		TEAM_FLAG_SPECTATORS = 1 << TEAM_SPECTATORS,
		TEAM_FLAG_PLAYERS = 1 << TEAM_PLAYERS,
		TEAM_FLAG_RED = 1 << TEAM_RED,
		TEAM_FLAG_BLUE = 1 << TEAM_BLUE,

		TEAM_FLAG_ALL = 0x0F,

	};

	struct stats_s{
		FVector3 position;
	};

	Game(System* system);
	~Game();

	void initialize();
	void shutdown();

	struct cCmds_s {
		CCmdGameCreate* game_create;
		CCmdGameJoin* game_join;
		CCmdGameDisconnect* game_disconnect;
		CCmdGameKill* game_kill;
		CCmdGameChatAll* game_chatAll;
		CCmdGameChatTeam* game_chatTeam;
	} cCmds;
	void registerCCmds( Console& console );
	void unregisterCCmds( Console& console );

	struct cVars_s{
		CVarInt* game_camera_mode;

	} cVars;
	void registerCVars( Console& console );
	void unregisterCVars( Console& console );

	void setup();
	void cleanup();
	void mainLoop();
	void setShouldBreakMainLoop(bool newShouldBreakMainLoop);
	bool getShouldBreakMainLoop() const;
	void update();

	void processInput(Input::actionMap_t& actionMap, Input::mouseState_t& mouseState);

	FVector3 findSpawnPosition( int team );
	Vehicle* spawnVehicle( Client* client );
	void unspawnVehicle( Vehicle* vehicle );

	Effect* spawnEffect( const string& name, GameObject* gameObject = NULL );
	Effect* spawnEffect( const string& name, const FVector3& position, const FQuaternion& orientation = FQuaternion::IDENTITY );
	void unspawnEffect( Effect* effect );

	Geometry* spawnGeometry( const string& name, const FVector3& position, const FQuaternion& orientation = FQuaternion::IDENTITY );
	void unspawnGeometry( Geometry* geometry );

	Shot* spawnShot( Weapon* weapon, const FVector3& position, const FQuaternion& orientation  );

	bool isArenaOwnedBy( teams_e team ) const;

	unsigned int getNextId();

	Camera* getCamera() const;
	Physics* getPhysics() const;
	Arena* getArena() const;
	ArenaFactory* getArenaFactory() const;
	VehicleFactory* getVehicleFactory() const;
	WeaponFactory* getWeaponFactory() const;
	ShotFactory* getShotFactory() const;
	EffectFactory* getEffectFactory() const;
	GeometryFactory* getGeometryFactory() const;

	int getMode() const;
	bool isRunningTeamMode() const;
//	const string& getArenaName() const;
	bool isRunning() const;
	const stats_s& getStats() const;
	Vehicle* getVehicleByClientId( int clientId ) const;

	typedef list<Area*> areas_t;
	const areas_t& getAreas() const;

	static string getModeName(int mode);
	static string getTeamName(int team);

protected:
	bool running;
	bool shouldBreakMainLoop;

	unsigned int idCounter;

	Camera* camera;
	Physics* physics;
	Arena* arena;

	int mode;


	typedef list<Vehicle*> vehicles_t;
	vehicles_t vehicles;

	typedef list<Geometry*> geometries_t;
	geometries_t geometries;

	typedef list<Effect*> effects_t;
	effects_t effects;

	typedef list<Shot*> shots_t;
	shots_t shots;

//	typedef list<Area*> areas_t;
	areas_t areas;

	ArenaFactory* arenaFactory;
	VehicleFactory* vehicleFactory;
	WeaponFactory* weaponFactory;
	ShotFactory* shotFactory;
	EffectFactory* effectFactory;
	GeometryFactory* geometryFactory;

	stats_s stats;
	void updateStats();
	void resetStats();

	void parseFwpResources( const string& resourcesFilePath );

	void prepareForSimulationStep();
	void updateAfterSimulationStep();
	void checkAreas();

};

#endif // __Game_h__
