#include "GameCCmds.h"

#include "System.h"
#include "Console.h"
#include "Network.h"
#include "Game.h"
#include "Client.h"
#include "Server.h"
#include "Gui.h"
#include "MessagesMenu.h"

CCmdGameCreate::CCmdGameCreate(): CCmd("game.create"){
	this->usageString = "game.create <arena> <gamemode> <maxPlayers>";
	this->infoString = "creates a new game (i.e. starts a server)";
}
CCmdGameCreate::~CCmdGameCreate(){
}
void CCmdGameCreate::exec( const vector<string>& arguments ){
	Console* console = System::getInstance()->getConsole();

	Network* network = console->getSystem()->getNetwork();

	if( arguments.size() >= 4 ){
		System::log("game.create: usage: %s", usageString.c_str());
		return;
	}

	if( arguments.size() >= 3 ){
		network->cVars.network_server_maxClients->setValueString( arguments[2] );
	}
	if( arguments.size() >= 2 ){
		network->cVars.network_server_gameMode->setValueString( arguments[1] );
	}
	if( arguments.size() >= 1 ){
		network->cVars.network_server_arena->setValueString( arguments[0] );
	}

	// we want a server running on this machine...
	network->cVars.network_server_remote->setValue( false );

	console->getSystem()->startGame();

}

CCmdGameJoin::CCmdGameJoin(): CCmd("game.join"){
	usageString = "game.join <host> <port>";
	infoString = "joins a game running on a remote host";
}
CCmdGameJoin::~CCmdGameJoin(){
}
void CCmdGameJoin::exec( const vector<string>& arguments ){
	Console* console = System::getInstance()->getConsole();

	Network* network = console->getSystem()->getNetwork();

	if( arguments.size() >= 3 ){
		System::log("game.join: usage: %s", usageString.c_str());
		return;
	}

	if( arguments.size() >= 2 ){
		network->cVars.network_server_port->setValueString( arguments[1] );
	}
	if( arguments.size() >= 1 ){
		network->cVars.network_server_host->setValueString( arguments[0] );
	}

	network->cVars.network_server_remote->setValue( true );

	console->getSystem()->startGame();
}

CCmdGameDisconnect::CCmdGameDisconnect(): CCmd("game.disconnect"){
	usageString = "game.disconnect";
	infoString = "disconnects you from a remote server or shuts down a local server";
}
CCmdGameDisconnect::~CCmdGameDisconnect(){
}
void CCmdGameDisconnect::exec( const vector<string>& arguments ){
	Console* console = System::getInstance()->getConsole();

	if( arguments.size() == 0 ){
		if( console->getSystem()->getGame()->isRunning() ){
			console->getSystem()->endGame();
		}else{
			System::log("game.disconnect: no game running!");
		}
	}else{
		System::log("game.disconnect: usage: %s", usageString.c_str());
		return;
	}
}



CCmdGameKill::CCmdGameKill(): CCmd("game.kill"){
	usageString = "game.kill <clientId>";
	infoString = "kills a client (you can only kill other clients if you're running the server)";
}
CCmdGameKill::~CCmdGameKill(){
}
void CCmdGameKill::exec( const vector<string>& arguments ){
	Console* console = System::getInstance()->getConsole();

	if( !console->getSystem()->getGame()->isRunning() ){
		System::log("game.kill: no game running!");
		return;
	}

	Client* c = console->getSystem()->getNetwork()->getClient();
	if( arguments.size() == 0 ){
		c->sendKillPacket( c->properties.clientId, c->properties.clientId ); // suicide
	}else if( arguments.size() == 1 ){
		if( !console->getSystem()->getNetwork()->getServer()->properties.remote ){
			c->sendKillPacket( c->properties.clientId, -1 ); // admin kill
		}else{
			System::log("game.kill: you can only kill other clients if you're running the server!");
		}

	}else{
		System::log("game.kill: usage: %s", usageString.c_str());
		return;
	}
}




CCmdGameChatAll::CCmdGameChatAll(): CCmd("game.chatAll"){
	usageString = "game.chatAll <message>";
	infoString = "sends a chat message to all players (or - if no message is present - opens an editbox where you can type one)";
}
CCmdGameChatAll::~CCmdGameChatAll(){
}
void CCmdGameChatAll::exec( const vector<string>& arguments ){
	Console* console = System::getInstance()->getConsole();

	if( !console->getSystem()->getGame()->isRunning() ){
		System::log("game.chatAll: no game running!");
		return;
	}

	Client* c = console->getSystem()->getNetwork()->getClient();
	if( arguments.size() == 0 ){ // => open editbox
		console->getSystem()->getGui()->getMessagesMenu()->showComposePanel( ChatMessagePacket::MODE_ALL );
	}else if( arguments.size() == 1 ){ // => send message
		Client* c = console->getSystem()->getNetwork()->getClient();
		c->sendChatMessagePacket( arguments[0], ChatMessagePacket::MODE_ALL );
	}else{
		System::log("game.chatAll: usage: %s", usageString.c_str());
		return;
	}
}

CCmdGameChatTeam::CCmdGameChatTeam(): CCmd("game.chatTeam"){
	usageString = "game.chatTeam <message>";
	infoString = "sends a chat message to your team (or - if no message is present - opens an editbox where you can type one)";
}
CCmdGameChatTeam::~CCmdGameChatTeam(){
}
void CCmdGameChatTeam::exec( const vector<string>& arguments ){
	Console* console = System::getInstance()->getConsole();

	if( !console->getSystem()->getGame()->isRunning() ){
		System::log("game.chatTeam: no game running!");
		return;
	}

	Client* c = console->getSystem()->getNetwork()->getClient();
	if( arguments.size() == 0 ){ // => open editbox
		console->getSystem()->getGui()->getMessagesMenu()->showComposePanel( ChatMessagePacket::MODE_TEAM );
	}else if( arguments.size() == 1 ){ // => send message
		Client* c = console->getSystem()->getNetwork()->getClient();
		c->sendChatMessagePacket( arguments[0], ChatMessagePacket::MODE_TEAM );
	}else{
		System::log("game.chatTeam: usage: %s", usageString.c_str());
		return;
	}
}



