#include "GeometryFactory.h"

#include "System.h"
#include "Exception.h"

#include "StaticGeometry.h"
#include "DynamicGeometry.h"

GeometryFactory::GeometryFactory(Game* game){
	this->game = game;

	this->geometriesByName.clear();
}

GeometryFactory::~GeometryFactory(){
	// delete geometry
	for( geometriesByName_t::iterator iter = geometriesByName.begin(); iter != geometriesByName.end(); iter++ ){
		delete iter->second;
	}

}

void GeometryFactory::addGeometry( const string& geometryFilePath ){

	try{
		TiXmlDocument doc( geometryFilePath.c_str() );
		if( !doc.LoadFile() ){
			throw Exception("Couldn't load file '" + geometryFilePath + "'. (Maybe not a correct xml file?)", "GeometryFactory::addGeometry()");
		}

		TiXmlNode* rootElement = doc.RootElement();
		TiXmlElement* element = rootElement->ToElement();

		string rootElementName = element->Value();
		if( rootElementName != "geometry" ){
			throw Exception("Geometry file '" + geometryFilePath + "' does not have a <geometry> root node.", "GeometryFactory::addGeometry()");
		}

		// check version
		const char* tmp = element->Attribute( "formatVersion" );
		if( tmp == NULL ){
			throw Exception("Geometry file '" + geometryFilePath + "' has no version information.", "GeometryFactory::addGeometry()");
		}else{
			string version = tmp;

			if( version != System::getInstance()->getVersionString() ){
				System::warn("(in GeometryFactory::addGeometry()): file '%s' has version %s. Trying to parse anyway.", geometryFilePath.c_str(), version.c_str() );
			}
		}


		Geometry* g = NULL;

		// check type and create a geoemtry of the proper subclass
		tmp = element->Attribute( "type" );
		if( tmp == NULL ){
			throw Exception("Geometry file '" + geometryFilePath + "' has no type information.", "GeometryFactory::addGeometry()");
		}else{
			string type = tmp;

			if( type == string("static") ){
				g = new StaticGeometry();
			}else if( type == string("dynamic") ){
				g = new DynamicGeometry();

			}else{
				System::warn("(in GeometryFactory::addGeometry()): Geometry file '%s' references unknown geometry type ('%s'). Ignoring vehicle.", geometryFilePath.c_str(), type.c_str() );
			}
		}


		if( g != NULL ){
			g->fromXmlElement( element );
			g->calculateChecksum();

			geometriesByName_t::const_iterator c_iter = geometriesByName.find( g->getName() );
			if( c_iter != geometriesByName.end() ){ // another geometry with that name is already registered -> throw exception
				throw Exception("A geometry with name '" + g->getName() + "' is already registered.", "GeometryFactory::addGeometry()");
			}
			this->geometriesByName.insert( geometriesByName_t::value_type(g->getName(), g) );
			System::log("GeometryFactory: Added geometry '%s' to list.", g->getName().c_str());
		}

	}catch( Exception& e ){
		System::warn("(GeometryFactory::addGeometry()): An exception occured while parsing geometry file '%s'. Ignoring vehicle.", geometryFilePath.c_str());
		System::log("GeometryFactory: The exception was: %s", e.getDescription().c_str());
	}

}

void GeometryFactory::removeGeometry(const string& name){
	geometriesByName_t::iterator iter = this->geometriesByName.find( name );
	if( iter == this->geometriesByName.end() ){
		throw Exception("A geometry with name '" + name +"' is not registered.", "GeometryFactory::removeGeometry()");
	}
	this->geometriesByName.erase(iter);
}






Geometry* GeometryFactory::createGeometry(const string& name) const {

	geometriesByName_t::const_iterator c_iter = this->geometriesByName.find( name );
	if( c_iter == this->geometriesByName.end() ){
		throw Exception("A geometry with name '" + name +"' is not registered.", "GeometryFactory::createGeometry()");
	}

	Geometry* ret = c_iter->second->clone();

	ret->reset();

	return ret;
}

void GeometryFactory::destroyGeometry(Geometry* geometry) const {
	delete geometry;
}



bool GeometryFactory::isGeometryAvailable( const string& name ) const {
	geometriesByName_t::const_iterator c_iter = this->geometriesByName.find( name );

	return (c_iter != this->geometriesByName.end());
}


vector<string> GeometryFactory::getGeometryNames() const {
	vector<string> ret;

	for( geometriesByName_t::const_iterator c_iter = this->geometriesByName.begin(); c_iter != this->geometriesByName.end(); ++c_iter ){
		ret.push_back( c_iter->first );
	}

	return ret;
}

