#include "Gui.h"

//#include <CEGUIBase.h>

#include "System.h"
#include "Exception.h"
#include "DotSceneLoader.h"
#include "Graphics.h"
#include "Camera.h"
#include "Game.h"
#include "Console.h"

#include "Hud.h"
#include "MainMenu.h"
#include "OptionsMenu.h"
#include "ConsoleMenu.h"
#include "StartServerMenu.h"
#include "JoinServerMenu.h"
#include "DebugMenu.h"
#include "MessageBoxMenu.h"
#include "HelpMenu.h"
#include "EquipmentMenu.h"
#include "GameInfoMenu.h"
#include "ScoresMenu.h"
#include "MessagesMenu.h"


Gui::Gui(System* system): SubSystem(system){
	this->ceguiRenderer = NULL;
	this->ceguiSystem = NULL;
	this->guiSheet = NULL;

	this->hud = NULL;

	this->debugMenu = NULL;
	this->messageBoxMenu = NULL;
	this->mainMenu = NULL;
	this->consoleMenu = NULL;
	this->startServerMenu = NULL;
	this->joinServerMenu = NULL;
	this->helpMenu = NULL;
	this->equipmentMenu = NULL;
	this->gameInfoMenu = NULL;
	this->scoresMenu = NULL;
	this->messagesMenu = NULL;

	this->escToggledMenus.clear();
	this->visible = true;
	this->drawHudMarkers = true;
	this->drawHudTeamStatus = true;
	this->openEquipmentMenuUponDeath = true;

	this->backgroundSceneManager = NULL;
	this->backgroundScene_camera_moveRight = true;
	this->shouldBreakBackgroundSceneMainLoop = false;
}

Gui::~Gui(){
}

void Gui::initialize(){
	System::log("");
	System::log("*** Initializing Gui ***");
	System::log("");

	System::log("Gui: Initializing CEGUI...");
	this->initializeCegui();

	System::log("Gui: Creating Menus...");
	this->createMenus();

	System::log("Gui: Creating Hud...");
	this->createHud();

//	System::log("Loading music...");
//	this->backgroundScene_backgroundMusic = this->system->getSound()->loadMusic( system->getDataDir() + "gui/background_scene/music.mid" );

	this->initialized = true;
	System::log("Gui is ready.");
	System::log("");

}

void Gui::shutdown(){
	System::log("");
	System::log("=== Shutting down Gui ===");
	System::log("");

	System::log("Gui: Destroying Hud...");
	this->destroyHud();

	System::log("Gui: Destroying Menus...");
	this->destroyMenus();

//	this->system->getSound()->freeMusic( this->backgroundScene_backgroundMusic );

	System::log("Gui: Shutting down CEGUI...");
	this->shutdownCegui();

	this->initialized = false;
	System::log("Gui is down.");
	System::log("");
}

void Gui::registerCCmds( Console& console ){
}
void Gui::unregisterCCmds( Console& console ){
}
void Gui::registerCVars( Console& console ){
	this->cVars.gui_hud_drawMarkers = new CVarBool( "gui.hud.drawMarkers", &this->drawHudMarkers, true );
	this->cVars.gui_hud_drawMarkers->setFlags( CVar::FLAG_SYSTEM );
	console.registerCVar( this->cVars.gui_hud_drawMarkers );

	this->cVars.gui_hud_drawTeamStatus = new CVarBool( "gui.hud.drawTeamStatus", &this->drawHudTeamStatus, true );
	this->cVars.gui_hud_drawTeamStatus->setFlags( CVar::FLAG_SYSTEM );
	console.registerCVar( this->cVars.gui_hud_drawTeamStatus );

	this->cVars.gui_openEquipmentMenuUponDeath = new CVarBool( "gui.openEquipmentMenuUponDeath", &this->openEquipmentMenuUponDeath, true );
	this->cVars.gui_openEquipmentMenuUponDeath->setFlags( CVar::FLAG_SYSTEM );
	console.registerCVar( this->cVars.gui_openEquipmentMenuUponDeath );

}
void Gui::unregisterCVars( Console& console ){
	console.unregisterCVar( this->cVars.gui_hud_drawMarkers );
	delete this->cVars.gui_hud_drawMarkers;

	console.unregisterCVar( this->cVars.gui_hud_drawTeamStatus );
	delete this->cVars.gui_hud_drawTeamStatus;

	console.unregisterCVar( this->cVars.gui_openEquipmentMenuUponDeath );
	delete this->cVars.gui_openEquipmentMenuUponDeath;
}

// setup GUI system
void Gui::initializeCegui(){
//	Ogre::Root* root = this->system.getGraphics().getOgreRoot();
	Ogre::RenderWindow* renderWindow = this->system->getGraphics()->getRenderWindow();
	Ogre::SceneManager* sm = this->system->getGraphics()->getSceneManager();

	this->ceguiRenderer = new CEGUI::OgreCEGUIRenderer(
		renderWindow, Ogre::RENDER_QUEUE_OVERLAY, true, 0, sm );

	string ceguiLogFile = this->system->getLogDir() + "cegui.log";
	this->ceguiSystem = new CEGUI::System( ceguiRenderer, (CEGUI::utf8*)ceguiLogFile.c_str() );
	CEGUI::Logger::getSingleton().setLoggingLevel( CEGUI::Informative );

	// load scheme and set up defaults
	CEGUI::SchemeManager::getSingleton().loadScheme( (CEGUI::utf8*)"TaharezLook.scheme" );
	ceguiSystem->setDefaultMouseCursor( (CEGUI::utf8*)"TaharezLook", (CEGUI::utf8*)"MouseArrow" );
	ceguiSystem->setDefaultFont((CEGUI::utf8*)"MenuBigFont");

	this->guiSheet = CEGUI::WindowManager::getSingleton().loadWindowLayout( (CEGUI::utf8*)"gui.layout.xml" );
	this->ceguiSystem->setGUISheet( this->guiSheet );
}

void Gui::shutdownCegui(){
	CEGUI::WindowManager::getSingleton().destroyWindow(this->guiSheet);
	delete ceguiSystem;
	delete ceguiRenderer;
}



void Gui::createMenus(){
	System::log("Gui: Creating DebugMenu...");
	this->debugMenu = new DebugMenu(this);
	this->debugMenu->registerEventHandlers();
	this->debugMenu->hide();

	System::log("Gui: Creating MessageBoxMenu...");
	this->messageBoxMenu = new MessageBoxMenu(this);
	this->messageBoxMenu->registerEventHandlers();
	this->messageBoxMenu->hide();

	System::log("Gui: Creating MainMenu...");
	this->mainMenu = new MainMenu(this);
	this->mainMenu->disableGameTab();
	this->mainMenu->disableEditTab();
	this->mainMenu->registerEventHandlers();
	this->escToggledMenus.push_back( menuVisibilityReminder_t(this->mainMenu, true) );

	System::log("Gui: Creating OptionsMenu...");
	this->optionsMenu = new OptionsMenu(this);
	this->optionsMenu->registerEventHandlers();
	this->optionsMenu->hide();
	this->escToggledMenus.push_back( menuVisibilityReminder_t(this->optionsMenu, true) );

	System::log("Gui: Creating ConsoleMenu...");
	this->consoleMenu = new ConsoleMenu(this);
	this->consoleMenu->registerEventHandlers();
	this->consoleMenu->hide();
	this->escToggledMenus.push_back( menuVisibilityReminder_t(this->consoleMenu, true) );

	System::log("Gui: Creating StartServerMenu...");
	this->startServerMenu = new StartServerMenu(this);
	this->startServerMenu->registerEventHandlers();
	this->startServerMenu->hide();
	this->escToggledMenus.push_back( menuVisibilityReminder_t(this->startServerMenu, true) );

	System::log("Gui: Creating JoinServerMenu...");
	this->joinServerMenu = new JoinServerMenu(this);
	this->joinServerMenu->registerEventHandlers();
	this->joinServerMenu->hide();
	this->escToggledMenus.push_back( menuVisibilityReminder_t(this->joinServerMenu, true) );

	System::log("Gui: Creating HelpMenu...");
	this->helpMenu = new HelpMenu(this);
	this->helpMenu->registerEventHandlers();
	this->helpMenu->hide();
	this->escToggledMenus.push_back( menuVisibilityReminder_t(this->helpMenu, true) );

	System::log("Gui: Creating EquipmentMenu...");
	this->equipmentMenu = new EquipmentMenu(this);
	this->equipmentMenu->registerEventHandlers();
	this->equipmentMenu->hide();
	this->escToggledMenus.push_back( menuVisibilityReminder_t(this->equipmentMenu, true) );

	System::log("Gui: Creating GameInfoMenu...");
	this->gameInfoMenu = new GameInfoMenu(this);
	this->gameInfoMenu->registerEventHandlers();
	this->gameInfoMenu->hide();
	this->escToggledMenus.push_back( menuVisibilityReminder_t(this->gameInfoMenu, true) );

	System::log("Gui: Creating ScoresMenu...");
	this->scoresMenu = new ScoresMenu(this);
	this->scoresMenu->registerEventHandlers();
	this->scoresMenu->hide();
	this->escToggledMenus.push_back( menuVisibilityReminder_t(this->scoresMenu, true) );

	System::log("Gui: Creating MessagesMenu...");
	this->messagesMenu = new MessagesMenu(this);
	this->messagesMenu->registerEventHandlers();
	this->messagesMenu->hide();
//	this->escToggledMenus.push_back( menuVisibilityReminder_t(this->messagesMenu, true) );

	// this must be done after all menus are loaded
	this->guiSheet->setAlpha(0.6f);
}

void Gui::destroyMenus(){
	delete this->messagesMenu;
	this->messagesMenu = NULL;

	delete this->scoresMenu;
	this->scoresMenu = NULL;

	delete this->gameInfoMenu;
	this->gameInfoMenu = NULL;

	delete this->equipmentMenu;
	this->equipmentMenu = NULL;

	delete this->helpMenu;
	this->helpMenu = NULL;

	delete this->joinServerMenu;
	this->joinServerMenu = NULL;

	delete this->startServerMenu;
	this->startServerMenu = NULL;

	delete this->consoleMenu;
	this->consoleMenu = NULL;

	delete this->optionsMenu;
	this->optionsMenu = NULL;

	delete this->mainMenu;
	this->mainMenu = NULL;

	delete this->messageBoxMenu;
	this->messageBoxMenu = NULL;

	delete this->debugMenu;
	this->debugMenu = NULL;
}



void Gui::createHud(){
	this->hud = new Hud( this );
}
void Gui::destroyHud(){
	delete this->hud;
}



void Gui::update(){
	if( this->hud->isVisible() ){
		this->hud->update();
	}

	if( this->debugMenu->isVisible() ){
		this->debugMenu->update();
	}

	if( this->messagesMenu->isVisible() ){
		this->messagesMenu->update();
	}

	if( this->scoresMenu->isVisible() ){
		this->scoresMenu->update();
	}

	if( this->joinServerMenu->isUpdating() ){
		this->joinServerMenu->update();
	}
}


void Gui::setupBackgroundScene(){
	System::log("Gui: Setting up background scene...");

	this->hide();
	this->showLoadingScreen();

	Graphics* graphics = this->system->getGraphics();
	graphics->setSceneManagerByType( Ogre::ST_GENERIC );
	this->backgroundSceneManager = system->getGraphics()->getSceneManager();

	graphics->setShadowTechnique( Ogre::SHADOWTYPE_STENCIL_ADDITIVE );
	this->backgroundSceneManager->setShadowFarDistance( 100.0 );

	// set up the camera
	Camera* camera = this->system->getGame()->getCamera();
	camera->attach( this->system->getGame() );
	camera->reset();
	camera->position = FVector3(-5.0f, 25.0f, 30.0f);
	camera->angles.x = 0.0f;
	camera->angles.y = -0.75f;
	camera->angles.z = 0.0f;

	DotSceneLoader dsl(this->system->getDataDir() + "gui/background_scene/background_scene.scene.xml"
		, this->backgroundSceneManager);
	dsl.load();

	// to compensate for the loading time
	this->system->getTimer()->calculateDeltaT();
//	this->system->getSound()->playMusic( this->backgroundScene_backgroundMusic );

	this->hideLoadingScreen();
	this->show();
}

void Gui::cleanupBackgroundScene(){
	System::log("Gui: Cleaning up background scene...");
//	Ogre::Root* root = this->system->getGraphics()->getOgreRoot();

	this->system->getGame()->getCamera()->detach();
	this->backgroundSceneManager->clearScene();

	this->backgroundSceneManager = NULL;

//	this->system->getSound()->stopMusic( this->backgroundScene_backgroundMusic );
}


void Gui::backgroundSceneMainLoop(){
//	System::log("Transfering control to rendering loop...");

	while( !shouldBreakBackgroundSceneMainLoop ){
		this->system->getTimer()->calculateDeltaT();

		this->update();

		this->system->getInput()->processInput();

		this->system->getConsole()->processQueues();

		this->updateBackgroundScene();

		// update renderTargets
		this->system->getGraphics()->update();
	}
	this->shouldBreakBackgroundSceneMainLoop = false;

//	System::log("Returned from background scene main loop.");
}

void Gui::setShouldBreakBackgroundSceneMainLoop(bool newShouldBreakBackgroundSceneMainLoop){
	this->shouldBreakBackgroundSceneMainLoop = newShouldBreakBackgroundSceneMainLoop;
}
bool Gui::getShouldBreakBackgroundSceneMainLoop() const {
	return this->shouldBreakBackgroundSceneMainLoop;
}

void Gui::updateBackgroundScene(){
//	Ogre::Root* root = this->system->getGraphics()->getOgreRoot();
//	Ogre::SceneManager* sceneMgr = root->getSceneManager( Ogre::ST_GENERIC );

//	Ogre::Camera* camera = sceneMgr->getCamera("Camera");
	if( this->system->getInput()->getInputMode() == Input::INPUT_MODE_GUI ){
		float deltaT = this->system->getTimer()->getDeltaT();
		Camera* camera = this->system->getGame()->getCamera();
		unsigned long currentMillis = this->system->getTimer()->getMilliseconds();
		float t = ( currentMillis % 10000 ) / 10000.0f;
		float turnSpeed = (fabs(t-0.5f) * 2.0f - 0.5f) * 0.3f;
		camera->angularVelocity[0] = turnSpeed;
		camera->velocity = FVector3::ZERO;
		camera->update(deltaT);
	}

	try{
		Ogre::Light* fireLight1 = this->backgroundSceneManager->getLight("FireLight1");

		float r = this->system->getRandom()->nextFloat();
		fireLight1->setPosition( Ogre::Vector3(0.0, 0.0, r*0.2f) );
		fireLight1->setDiffuseColour( 1.0, 0.4+r*0.1, 0.1 );
		fireLight1->setSpecularColour( 1.0, 0.2+r*0.2, 0.0 );
		fireLight1->setAttenuation(40.0, 1.0, 1.0/40.0, 0.0);

	}catch(Ogre::Exception& e){
		e.getDescription();
		// ignore if fireLight1 doesn't exist in scene...
	}

}







void Gui::show(){
	for( escToggledMenus_t::iterator iter = escToggledMenus.begin(); iter != escToggledMenus.end(); iter++ ){
		if( iter->second ){
			iter->first->show();
		}
	}
//	showMouseCursor();
//	this->system->getInput()->setInputMode( Input::INPUT_MODE_GUI );
	this->makeToggleChecks();

	this->visible = true;
}
void Gui::hide(){
	for( escToggledMenus_t::iterator iter = escToggledMenus.begin(); iter != escToggledMenus.end(); iter++ ){
		iter->second = iter->first->isVisible();
		iter->first->hide();
	}

//	hideMouseCursor();
//	this->system->getInput()->setInputMode( Input::INPUT_MODE_GAME );
	this->makeToggleChecks();

	this->visible = false;
}
void Gui::toggleVisibility(){
	if( this->visible ){
		this->hide();
	}else{
		this->show();
	}
}

bool Gui::isVisible() const {
	return this->visible;
}

void Gui::makeToggleChecks(){
	if( !this->initialized ){
		return;
	}

	if( ( this->mainMenu != NULL && this->mainMenu->isVisible() )
		|| ( this->optionsMenu != NULL && this->optionsMenu->isVisible() )
		|| ( this->consoleMenu != NULL && this->consoleMenu->isVisible() )
		|| ( this->startServerMenu != NULL && this->startServerMenu->isVisible() )
		|| ( this->joinServerMenu != NULL && this->joinServerMenu->isVisible() )

		|| ( this->messageBoxMenu != NULL && this->messageBoxMenu->isVisible() )
		|| ( this->helpMenu != NULL && this->helpMenu->isVisible() )
		|| ( this->equipmentMenu != NULL && this->equipmentMenu->isVisible() )
		|| ( this->gameInfoMenu != NULL && this->gameInfoMenu->isVisible() )
		|| ( this->scoresMenu != NULL && this->scoresMenu->isVisible() )
		){
		this->showMouseCursor();
		this->system->getInput()->setInputMode( Input::INPUT_MODE_GUI );
	}else{
		this->hideMouseCursor();
		this->system->getInput()->setInputMode( Input::INPUT_MODE_GAME );
	}
}

void Gui::showMouseCursor(){
//	ceguiSystem->setDefaultMouseCursor( (CEGUI::utf8*)"TaharezLook", (CEGUI::utf8*)"MouseArrow" );
	CEGUI::MouseCursor::getSingleton().show();
}
void Gui::hideMouseCursor(){
//	ceguiSystem->setDefaultMouseCursor( NULL );
	CEGUI::MouseCursor::getSingleton().hide();
}
void Gui::toggleMouseCursorVisibility(){
	if( CEGUI::MouseCursor::getSingleton().isVisible() ){
		hideMouseCursor();
	}else{
		showMouseCursor();
	}
}


void Gui::showLoadingScreen(){
	Ogre::Overlay* ol = Ogre::OverlayManager::getSingleton().getByName("Gui/LoadingScreen");
	ol->show();
	this->system->getGraphics()->getRenderWindow()->update();
}

void Gui::hideLoadingScreen(){
	Ogre::Overlay* ol = Ogre::OverlayManager::getSingleton().getByName("Gui/LoadingScreen");
	ol->hide();
}




void Gui::onGameCreation(){
	if( this->isVisible() ){
		this->hide();
	}

	this->mainMenu->enableGameTab();
//	if( editmode ) this->mainMenu->enableEditTab();

	this->hud->show();
	this->messagesMenu->show();

//	this->equipmentMenu->show();	// do this after game.running is true
}
void Gui::onGameDestruction(){
	this->messagesMenu->hide();
	this->hud->hide();

	if( !this->isVisible() ){
		this->show();
	}

	this->mainMenu->disableGameTab();
	this->mainMenu->disableEditTab();
	if( !this->mainMenu->isVisible() ){
		this->mainMenu->show();
	}

	if( this->equipmentMenu->isVisible() ){
		this->equipmentMenu->hide();
	}
	if( this->gameInfoMenu->isVisible() ){
		this->gameInfoMenu->hide();
	}
	if( this->scoresMenu->isVisible() ){
		this->scoresMenu->hide();
	}
}



void Gui::sceneManagerHasChanged( Ogre::SceneManager* newSceneManager ){
	this->ceguiRenderer->setTargetSceneManager( newSceneManager );
}



CEGUI::Renderer* Gui::getCeguiRenderer() const {
	return this->ceguiRenderer;
}

Hud* Gui::getHud() const {
	return this->hud;
}

DebugMenu* Gui::getDebugMenu() const {
	return this->debugMenu;
}
MessageBoxMenu* Gui::getMessageBoxMenu() const {
	return this->messageBoxMenu;
}
MainMenu* Gui::getMainMenu() const {
	return this->mainMenu;
}
OptionsMenu* Gui::getOptionsMenu() const {
	return this->optionsMenu;
}
ConsoleMenu* Gui::getConsoleMenu() const {
	return this->consoleMenu;
}
StartServerMenu* Gui::getStartServerMenu() const {
	return this->startServerMenu;
}
JoinServerMenu* Gui::getJoinServerMenu() const {
	return this->joinServerMenu;
}
HelpMenu* Gui::getHelpMenu() const {
	return this->helpMenu;
}
EquipmentMenu* Gui::getEquipmentMenu() const {
	return this->equipmentMenu;
}
GameInfoMenu* Gui::getGameInfoMenu() const {
	return this->gameInfoMenu;
}
ScoresMenu* Gui::getScoresMenu() const {
	return this->scoresMenu;
}
MessagesMenu* Gui::getMessagesMenu() const {
	return this->messagesMenu;
}


bool Gui::getOpenEquipmentMenuUponDeath() const {
	return this->openEquipmentMenuUponDeath;
}
bool Gui::getDrawHudMarkers() const {
	return this->drawHudMarkers;
}
bool Gui::getDrawHudTeamStatus() const {
	return this->drawHudTeamStatus;
}
