#include "Hud.h"

#include "System.h"
#include "StringConverter.h"
#include "GameObject.h"
#include "Gui.h"
#include "Game.h"
#include "Vehicle.h"
#include "Camera.h"
#include "Weapon.h"
#include "Network.h"
#include "Server.h"
#include "ConquestArea.h"

Hud::Hud( Gui* gui ){
	int i;

	this->gui = gui;
	this->centerOfScreenMessageExpireMillis = 0;

	Ogre::OverlayManager& overlayMgr = Ogre::OverlayManager::getSingleton();

	this->coreOverlay = overlayMgr.getByName("Gui/Hud");

	this->crosshair = overlayMgr.getOverlayElement("Gui/Hud/Crosshair");
	this->centerOfScreenMessage = overlayMgr.getOverlayElement("Gui/Hud/CenterOfScreenMessage");

	this->status = overlayMgr.getOverlayElement("Gui/Hud/Status");
	this->energyBar = overlayMgr.getOverlayElement("Gui/Hud/Status/EnergyBar");
	this->healthBar = overlayMgr.getOverlayElement("Gui/Hud/Status/HealthBar");

	for( i=0; i<4; i++ ){
		string baseName = "Gui/Hud/Status/Weapon" + StringConverter::toString(i+1);

 		this->weaponInfos[i].panel = overlayMgr.getOverlayElement( baseName );
		this->weaponInfos[i].icon = overlayMgr.getOverlayElement( baseName + "/Icon" );
		this->weaponInfos[i].ammoIndicator = overlayMgr.getOverlayElement( baseName + "/AmmoIndicator" );
		this->weaponInfos[i].heatIndicator = overlayMgr.getOverlayElement( baseName + "/HeatIndicator" );
	}


	this->teamStatusOverlay = overlayMgr.getByName("Gui/Hud/TeamStatus");
	for( i=0; i<16; i++ ){
		this->teamStatusEntries[i].panel = overlayMgr.getOverlayElement( "Gui/Hud/TeamStatus/Entry" + StringConverter::toString(i) + "/Panel" );
		this->teamStatusEntries[i].panel->hide();

		this->teamStatusEntries[i].healthBar = overlayMgr.getOverlayElement( "Gui/Hud/TeamStatus/Entry" + StringConverter::toString(i) + "/HealthBar" );
		this->teamStatusEntries[i].name = overlayMgr.getOverlayElement( "Gui/Hud/TeamStatus/Entry" + StringConverter::toString(i) + "/Name" );
	}

	this->playerMarkersOverlay = overlayMgr.getByName("Gui/Hud/PlayerMarkers");
	for( i=0; i<16; i++ ){
		this->playerMarkers[i].rectangle = overlayMgr.getOverlayElement( "Gui/Hud/PlayerMarker" + StringConverter::toString(i) + "/Rectangle" );
		this->playerMarkers[i].rectangle->hide();

		this->playerMarkers[i].topText = overlayMgr.getOverlayElement( "Gui/Hud/PlayerMarker" + StringConverter::toString(i) + "/TopText"  );
		this->playerMarkers[i].topText->hide();

		this->playerMarkers[i].bottomText = overlayMgr.getOverlayElement( "Gui/Hud/PlayerMarker" + StringConverter::toString(i) + "/BottomText"  );
		this->playerMarkers[i].bottomText->hide();
	}


	this->conquestAreaMarkersOverlay = overlayMgr.getByName("Gui/Hud/ConquestAreaMarkers");
	for( i=0; i<16; i++ ){
		this->conquestAreaMarkers[i].rectangle = overlayMgr.getOverlayElement( "Gui/Hud/ConquestAreaMarker" + StringConverter::toString(i) + "/Rectangle" );
		this->conquestAreaMarkers[i].rectangle->hide();

		this->conquestAreaMarkers[i].topText = overlayMgr.getOverlayElement( "Gui/Hud/ConquestAreaMarker" + StringConverter::toString(i) + "/TopText"  );
		this->conquestAreaMarkers[i].topText->hide();

		this->conquestAreaMarkers[i].bottomText = overlayMgr.getOverlayElement( "Gui/Hud/ConquestAreaMarker" + StringConverter::toString(i) + "/BottomText"  );
		this->conquestAreaMarkers[i].bottomText->hide();
	}
}

Hud::~Hud(){

}

void Hud::show(){
	this->coreOverlay->show();
	this->teamStatusOverlay->show();
	this->playerMarkersOverlay->show();
	this->conquestAreaMarkersOverlay->show();
}

void Hud::hide(){
	this->coreOverlay->hide();
	this->teamStatusOverlay->hide();
	this->playerMarkersOverlay->hide();
	this->conquestAreaMarkersOverlay->hide();
}
bool Hud::isVisible() const {
	return this->coreOverlay->isVisible();
}

void Hud::update(){

	Game* game = this->gui->getSystem()->getGame();

	this->updateCenterOfScreenMessage();

	this->updateStatusBar();

	// team status
	if( game->isRunningTeamMode()
		&& this->gui->getDrawHudTeamStatus()
		){
		this->teamStatusOverlay->show();
		this->updateTeamStatus();
	}else{
		this->teamStatusOverlay->hide();
	}

	// player markers
	if( this->gui->getDrawHudMarkers() ){
		this->playerMarkersOverlay->show();
		this->updatePlayerMarkers();
	}else{
		this->playerMarkersOverlay->hide();
	}

	// conquest area markers
	if( game->getMode() == Game::MODE_CONQUEST
		&& this->gui->getDrawHudMarkers()
		){
		this->conquestAreaMarkersOverlay->show();
		this->updateConquestAreaMarkers();
	}else{
		this->conquestAreaMarkersOverlay->hide();
	}
}


void Hud::updateCenterOfScreenMessage(){
	unsigned long currentMillis = this->gui->getSystem()->getTimer()->getMilliseconds();
	if( currentMillis < this->centerOfScreenMessageExpireMillis ){

		float alpha = (this->centerOfScreenMessageExpireMillis - currentMillis) / 1000.0f;
		alpha = alpha > 1.0f ? 1.0f : alpha;
		string bottomColor = "1.0 1.0 1.0 " + StringConverter::toString(alpha);
		string topColor = "0.7 0.7 0.7 " + StringConverter::toString(alpha);
		this->centerOfScreenMessage->setParameter( "colour_top", topColor );
		this->centerOfScreenMessage->setParameter( "colour_bottom", bottomColor );

		float l = this->centerOfScreenMessage->getCaption().size() * 0.5f * 0.017f;
		this->centerOfScreenMessage->setLeft( -l );

		this->centerOfScreenMessage->show();
	}else{
		this->centerOfScreenMessage->hide();
		this->centerOfScreenMessage->setCaption( "" );
	}
}

void Hud::updateStatusBar(){
	Camera* cam = this->gui->getSystem()->getGame()->getCamera();
	GameObject* camTarget = cam->getTarget();
	Camera::modes_e camMode = cam->getMode();

	if( camMode != Camera::MODE_FREE
		&& camTarget != NULL && camTarget->getGameObjectType() == GameObject::GAME_OBJECT_TYPE_VEHICLE
		){

		Vehicle* v = (Vehicle*)camTarget;

		// health and energy bars
		float energyRatio = v->getEnergy() / v->getPhysicalProperties().energyCapacity;
		float healthRatio = v->getHealth() / v->getPhysicalProperties().healthCapacity;

		this->energyBar->setWidth( energyRatio*0.169f );
		this->healthBar->setWidth( healthRatio*0.169f );

		// weapons
		for( int i=0; i<4; i++ ){ // must iterate over all slots
			Weapon* w = v->getWeapon(i);
			if( w != NULL ){
				if( !w->getGraphicalProperties().hudIcon.empty() ){
					this->weaponInfos[i].icon->setMaterialName( w->getGraphicalProperties().hudIcon );
					this->weaponInfos[i].icon->show();
				}else{
					this->weaponInfos[i].icon->hide();
				}

				const Weapon::physicalProperties_s& pp = w->getPhysicalProperties();
				if( pp.ammoCapacity > 0 ){ // do ammo
					float relAmmo = w->getAmmo() / pp.ammoCapacity;
					FColor c = FColor::WHITE;

					if( !w->isReady() ){ // darken if weapon is not ready
						c = c * 0.5f;
					}
					string colorString = StringConverter::toString(c.r) + " " + StringConverter::toString(c.g) + " " + StringConverter::toString(c.b);
					this->weaponInfos[i].ammoIndicator->setParameter( "colour", colorString );
					this->weaponInfos[i].ammoIndicator->setCaption( StringConverter::toString( w->getAmmo() ) );

					this->weaponInfos[i].ammoIndicator->show();
				}else{
					this->weaponInfos[i].ammoIndicator->hide();
				}

				if( pp.heatCapacity > 0 ){ // do heat
					float relHeat = w->getHeat() / pp.heatCapacity;
					this->weaponInfos[i].heatIndicator->setWidth( relHeat * 0.044f );
					if( w->isOverheated() ){
						this->weaponInfos[i].heatIndicator->setMaterialName( "gui/hud/status/weapons/overheated_bar" );
					}else{
						this->weaponInfos[i].heatIndicator->setMaterialName( "gui/hud/status/weapons/heat_bar" );
					}

					this->weaponInfos[i].heatIndicator->show();
				}else{
					this->weaponInfos[i].heatIndicator->hide();
				}

				this->weaponInfos[i].panel->show();
			}else{
				this->weaponInfos[i].panel->hide();
			}
		}

		this->status->show();
	}else{
		this->status->hide();
	}


}


void Hud::updateTeamStatus(){
	int i;

	for( i=0; i<16; i++ ){
		this->teamStatusEntries[i].panel->hide();
	}


	Game* game = this->gui->getSystem()->getGame();
	Network* network = this->gui->getSystem()->getNetwork();
	Camera* camera = this->gui->getSystem()->getGame()->getCamera();
	Client* localClient = this->gui->getSystem()->getNetwork()->getClient();

	if( localClient->info.team == Game::TEAM_SPECTATORS ){
		return;
	}

	int maxClients = network->getServer()->properties.maxClients;
	Client** clientArray = network->getServer()->getClientArray();
	int counter = 0;
	for( i=0; i<maxClients; i++ ){
		Client* c = clientArray[i];

		if( c == NULL
			|| c == localClient
			|| c->info.team != localClient->info.team
			){
			continue;
		}

		Vehicle* v = c->getVehicle();
		if( v == NULL ){ // => dead
			this->teamStatusEntries[counter].healthBar->setWidth( 0.0 );
			this->teamStatusEntries[counter].name->setParameter( "colour_top", "0.3 0.3 0.3" );
			this->teamStatusEntries[counter].name->setParameter( "colour_bottom", "0.5 0.5 0.5" );
			this->teamStatusEntries[counter].name->setCaption( c->properties.name );
		}else{ // alive
			float relativeHealth = v->getHealth() / v->getPhysicalProperties().healthCapacity;
			this->teamStatusEntries[counter].healthBar->setWidth( relativeHealth * 0.146f);
			this->teamStatusEntries[counter].name->setParameter( "colour_top", "0.5 0.5 0.5" );
			this->teamStatusEntries[counter].name->setParameter( "colour_bottom", "0.7 0.7 0.7" );
			this->teamStatusEntries[counter].name->setCaption( c->properties.name );
		}
		this->teamStatusEntries[counter].panel->show();
		counter++;
	}

}

void Hud::updatePlayerMarkers(){
	int i;

	for( i=0; i<16; i++ ){
		this->playerMarkers[i].rectangle->hide();
		this->playerMarkers[i].topText->hide();
		this->playerMarkers[i].bottomText->hide();
	}


	Game* game = this->gui->getSystem()->getGame();
	Camera* camera = this->gui->getSystem()->getGame()->getCamera();
	FVector3 referencePosition = camera->position;
//	if( camTarget != NULL ){
//		referencePosition = camTarget->position;
//	}

	Client* localClient = this->gui->getSystem()->getNetwork()->getClient();
	int maxClients = this->gui->getSystem()->getNetwork()->getServer()->properties.maxClients;
	for( i=0; i<maxClients; i++ ){
		Vehicle* v = game->getVehicleByClientId( i );

		if( v == NULL
			|| (camera->getMode() != Camera::MODE_FREE && v->getClient() == localClient )
			){
//			this->playerMarkers[i].rectangle->hide();
//			this->playerMarkers[i].topText->hide();
//			this->playerMarkers[i].bottomText->hide();
			continue;
		}

		Ogre::Sphere sphere( v->position.toOgreVector3(), 5.0f );
		Ogre::Real left, top, right, bottom;
		bool b = camera->getOgreCamera()->projectSphere( sphere, &left, &top, &right, &bottom );
		if( b ){
//			printf("l:%f t:%f r:%f b:%f\n", left, top, right, bottom);

			left = left * 0.5;
			right = right * 0.5;
			top = top * 0.5;
			bottom = bottom * 0.5;

			float w = right - left;
			float h = top - bottom;
			h = h > 0.04f ? h : 0.04f;
			w = w > 0.03f ? w : 0.03f;
			float cx = (right + left) * 0.5f;
			float cy = (top + bottom) * 0.5f;

			int distance = (int)( v->position.distance( referencePosition ) );
			string topTextCaption = v->getClient()->properties.name + " - " + StringConverter::toString(distance) + "m";
			float l = cx - topTextCaption.length()*0.5f * 0.0087f;
			this->playerMarkers[i].topText->setCaption( topTextCaption );
			this->playerMarkers[i].topText->setPosition( l, -top - 0.025 );
			this->playerMarkers[i].topText->setDimensions( 1.0, 0.1 );

			int healthPercent = (int)( v->getHealth() / v->getPhysicalProperties().healthCapacity * 100.0f );
			string bottomTextCaption = v->getName() + " - " + StringConverter::toString(healthPercent) + "%";
			l = cx - bottomTextCaption.length()*0.5f * 0.0087f;
			this->playerMarkers[i].bottomText->setCaption( bottomTextCaption );
			this->playerMarkers[i].bottomText->setPosition( l, -bottom );
			this->playerMarkers[i].bottomText->setDimensions( right-left, top-bottom );

			this->playerMarkers[i].rectangle->setDimensions( w, h );
			this->playerMarkers[i].rectangle->setPosition( cx - w*0.5f, -cy - h*0.5 );

			if( v->getClient()->info.team == Game::TEAM_RED ){
				this->playerMarkers[i].rectangle->setParameter( "border_material", "gui/hud/player_marker/rectangle_red" );
			}else if( v->getClient()->info.team == Game::TEAM_BLUE ){
				this->playerMarkers[i].rectangle->setParameter( "border_material", "gui/hud/player_marker/rectangle_blue" );
			}else{
				this->playerMarkers[i].rectangle->setParameter( "border_material", "gui/hud/player_marker/rectangle_yellow" );
			}

			this->playerMarkers[i].rectangle->show();
			this->playerMarkers[i].topText->show();
			this->playerMarkers[i].bottomText->show();

		}else{

//			this->playerMarkers[i].rectangle->hide();
//			this->playerMarkers[i].topText->hide();
//			this->playerMarkers[i].bottomText->hide();

		}
	}
}

void Hud::updateConquestAreaMarkers(){
	int i;

	for( i=0; i<16; i++ ){
		this->conquestAreaMarkers[i].rectangle->hide();
		this->conquestAreaMarkers[i].topText->hide();
		this->conquestAreaMarkers[i].bottomText->hide();
	}


	Game* game = this->gui->getSystem()->getGame();
	Camera* camera = this->gui->getSystem()->getGame()->getCamera();
	FVector3 referencePosition = camera->position;
//	if( camTarget != NULL ){
//		referencePosition = camTarget->position;
//	}

	const Game::areas_t& areas = game->getAreas();
	int counter = 0;
	for( Game::areas_t::const_iterator c_a_iter = areas.begin(); c_a_iter != areas.end(); ++c_a_iter ){

		if( (*c_a_iter)->getAreaType() != Area::AREA_TYPE_CONQUEST ){
			continue;
		}

		ConquestArea* ca = (ConquestArea*)(*c_a_iter);

		Ogre::Sphere sphere( ca->position.toOgreVector3(), 3.0f );
		Ogre::Real left, top, right, bottom;
		bool b = camera->getOgreCamera()->projectSphere( sphere, &left, &top, &right, &bottom );
		if( b ){ // => sphere was projected onto viewport
//			printf("c:%i l:%f t:%f r:%f b:%f\n", counter, left, top, right, bottom);

			left = left * 0.5;
			right = right * 0.5;
			top = top * 0.5;
			bottom = bottom * 0.5;

			float w = right - left;
			float h = top - bottom;
			h = h > 0.02f ? h : 0.02f;
			w = w > 0.015f ? w : 0.015f;
			float cx = (right + left) * 0.5f;
			float cy = (top + bottom) * 0.5f;


			int distance = (int)( ca->position.distance( referencePosition ) );
			string topTextCaption = ca->getName() + " - " + StringConverter::toString(distance) + "m";
			float l = cx - topTextCaption.length()*0.5f * 0.0087f;
			this->conquestAreaMarkers[counter].topText->setCaption( topTextCaption );
			this->conquestAreaMarkers[counter].topText->setPosition( l, -top - 0.025 );
			this->conquestAreaMarkers[counter].topText->setDimensions( 1.0, 0.1 );

			string bottomTextCaption = "neutral - 0%";
			string materialName = "gui/hud/conquest_area_marker/rectangle_neutral";
			int capturePercentage = (int)( ca->getCapturePercentage() );
			if( capturePercentage < -50.0f ){ // => red
				materialName = "gui/hud/conquest_area_marker/rectangle_red";
				bottomTextCaption = "red - " + StringConverter::toString(-capturePercentage) + "%";
			}else if( capturePercentage > 50.0f ){ // => blue
				materialName = "gui/hud/conquest_area_marker/rectangle_blue";
				bottomTextCaption = "blue - " + StringConverter::toString(capturePercentage) + "%";
			}else{ // => neutral
				if( capturePercentage < 0.0f ){
					materialName = "gui/hud/conquest_area_marker/rectangle_red_up";
					bottomTextCaption = "neutral - " + StringConverter::toString(-capturePercentage) + "% red";
				}else if( capturePercentage > 0.0f ){
					materialName = "gui/hud/conquest_area_marker/rectangle_blue_up";
					bottomTextCaption = "neutral - " + StringConverter::toString(capturePercentage) + "% blue";
				}
			}

			l = cx - bottomTextCaption.length()*0.5f * 0.0087f;
			this->conquestAreaMarkers[counter].bottomText->setCaption( bottomTextCaption );
			this->conquestAreaMarkers[counter].bottomText->setPosition( l, -bottom );
			this->conquestAreaMarkers[counter].bottomText->setDimensions( right-left, top-bottom );


			this->conquestAreaMarkers[counter].rectangle->setDimensions( w, h );
			this->conquestAreaMarkers[counter].rectangle->setPosition( cx - w*0.5f, -cy - h*0.5 );
			this->conquestAreaMarkers[counter].rectangle->setMaterialName( materialName );

			this->conquestAreaMarkers[counter].rectangle->show();
			this->conquestAreaMarkers[counter].topText->show();
			this->conquestAreaMarkers[counter].bottomText->show();

		}

		counter++;
	}

}

void Hud::showMarkers(){
	this->playerMarkersOverlay->show();
	this->conquestAreaMarkersOverlay->show();
}
void Hud::hideMarkers(){
	this->playerMarkersOverlay->hide();
	this->conquestAreaMarkersOverlay->hide();
}

void Hud::setCenterOfScreenMessage( const string& newCenterOfScreenMessage, unsigned long durationMillis ){
	this->centerOfScreenMessage->setCaption( newCenterOfScreenMessage );
	this->centerOfScreenMessageExpireMillis = this->gui->getSystem()->getTimer()->getMilliseconds() + durationMillis;
}
