#include "IntersectionQueries.h"

#include "Physics.h"
#include "System.h"
#include "Exception.h"
#include "Game.h"
#include "Graphics.h"

#include <algorithm>
#include <functional>

struct sortResultEntriesByDistance: public std::binary_function<const IntersectionQuery::resultEntry_s&, const IntersectionQuery::resultEntry_s&, bool>
{
   result_type operator()( first_argument_type a, second_argument_type b )
   {
      return a.distance < b.distance;
   }
};



IntersectionQuery::IntersectionQuery( Physics* physics ){
	this->physics = physics;
	this->ignoreGameObject = NULL;
	this->collisionFlags = RESULT_ENTRY_TYPE_ALL;
	this->result.clear();
}

IntersectionQuery::~IntersectionQuery(){
}

const IntersectionQuery::result_t& IntersectionQuery::getResult() const {
	return this->result;
}
void IntersectionQuery::setCollisionFlags( unsigned int flags ){
	this->collisionFlags = flags;
}
void IntersectionQuery::setIgnoreGameObject(GameObject* ignoreGameObject){
	this->ignoreGameObject = ignoreGameObject;
}






RayIntersectionQuery::RayIntersectionQuery( Physics* physics ): IntersectionQuery(physics) {
	this->origin = FVector3::ZERO;
	this->direction = FVector3::UNIT_Z;
	this->length = dInfinity;
	this->sortByDistance = false;
}
RayIntersectionQuery::RayIntersectionQuery( Physics* physics, const FVector3& origin, const FVector3& direction, bool sortByDistance ): IntersectionQuery(physics) {
	this->origin = origin;
	this->direction = direction;
	this->length = dInfinity;
	this->sortByDistance = sortByDistance;
}

RayIntersectionQuery::~RayIntersectionQuery(){
}

void RayIntersectionQuery::execute(){
	this->result.clear();

	// collide against geoms
	FVector3 x = this->direction.perpendicular();
	x.normalize();
	FVector3 y = this->direction.crossProduct( x );
	y.normalize();
	Ogre::Quaternion q( x.toOgreVector3(), y.toOgreVector3(), this->direction.toOgreVector3() );
	dMatrix3 m;
	dRFrom2Axes(m, x.x, x.y, x.z, y.x, y.y, y.z);

//	printf("length: %f\n", length);
	dGeomID ray = dCreateRay(0, this->length );
	dGeomSetPosition( ray, this->origin.x, this->origin.y, this->origin.z );
	dGeomSetRotation( ray, m );
	dGeomSetData( ray, this );

	dSpaceCollide2( ray, (dGeomID)physics->getStaticGeometrySpace(), this, &RayIntersectionQuery::geomNearGeomCallback );
	dSpaceCollide2( ray, (dGeomID)physics->getDynamicGeometrySpace(), this, &RayIntersectionQuery::geomNearGeomCallback );

	dGeomDestroy( ray );


	// collide against terrain
	// THINKABOUTME: Fix for casting bug
	if( physics->getCollideAgainstTerrain() && this->direction.y < 0.99f 
		&& (this->collisionFlags & IntersectionQuery::RESULT_ENTRY_TYPE_WORLD_GEOMETRY) != 0 ){

		Ogre::SceneManager* sm = this->physics->getGame()->getSystem()->getGraphics()->getSceneManager();
		Ogre::RaySceneQuery* rsq = sm->createRayQuery( Ogre::Ray( this->origin.toOgreVector3(), this->direction.toOgreVector3() ) );
		rsq->setSortByDistance( this->sortByDistance );
		rsq->setWorldFragmentType(Ogre::SceneQuery::WFT_SINGLE_INTERSECTION);

		Ogre::RaySceneQueryResult& rsqResult = rsq->execute();
		Ogre::RaySceneQueryResult::iterator iter = rsqResult.begin();
		resultEntry_s re;
		while( iter != rsqResult.end() ){
			if( iter->worldFragment != NULL && iter->distance <= this->length ){
				// sort in world fragments
				re.type = RESULT_ENTRY_TYPE_WORLD_GEOMETRY;
				re.intersectionPoint = FVector3( iter->worldFragment->singleIntersection.val );
				re.distance = iter->distance;
//			re.distance = (this->origin - re.intersectionPoint).length();
				this->result.push_back( re );
			}
			iter ++;
		}
		sm->destroyQuery(rsq);
	}

	if( this->sortByDistance ){
		this->result.sort( sortResultEntriesByDistance() );
	}
}

void RayIntersectionQuery::setRay( const FVector3& origin, const FVector3& direction, float length ){
	this->origin = origin;
	this->direction = direction;
	this->length = length;
}
void RayIntersectionQuery::setSortByDistance( bool newSortByDistance ){
	this->sortByDistance = newSortByDistance;
}

void RayIntersectionQuery::geomNearGeomCallback(void* data, dGeomID g1, dGeomID g2){
//	printf("### HURRA: g1: %i (b1: %i), g2: %i (b2: %i)\n", g1, dGeomGetBody(g1), g2, dGeomGetBody(g2));
	RayIntersectionQuery* instance = (RayIntersectionQuery*)data;

	void* data1 = dGeomGetData(g1);
	void* data2 = dGeomGetData(g2);
	dGeomID ray;
	dGeomID other;
	void* go;
	if( data1 == instance ){
		ray = g1;
		other = g2;
		go = data2;
	}else{
		ray = g2;
		other = g1;
		go = data1;
	}

	// get the exact contacts
	dContact contacts[1];
	int numContacts = dCollide(ray, other, 1, &contacts[0].geom, sizeof(dContact));
	if( numContacts > 0 ){
//		printf("HIT: %d!\n", go);

		IntersectionQuery::resultEntry_s re;

		re.type = go != NULL ? RESULT_ENTRY_TYPE_GAME_OBJECT : RESULT_ENTRY_TYPE_ARENA_BOUNDARY;
		re.distance = contacts[0].geom.depth;
		re.geom = other;
		re.gameObject = (GameObject*)go;

		if( (re.type & instance->collisionFlags) == 0 || re.distance > instance->length
			|| (instance->ignoreGameObject != NULL && re.gameObject == instance->ignoreGameObject) ){
			return;
		}

		re.intersectionPoint = FVector3(contacts[0].geom.pos);
		re.intersectionNormal = FVector3(contacts[0].geom.normal);

		instance->result.push_back( re );
	} // if
}











SphereIntersectionQuery::SphereIntersectionQuery( Physics* physics ): IntersectionQuery(physics) {
	this->origin = FVector3::ZERO;
	this->radius = 0.0f;
}
SphereIntersectionQuery::SphereIntersectionQuery( Physics* physics, const FVector3& origin, float radius ): IntersectionQuery(physics) {
	this->origin = origin;
	this->radius = radius;
}

SphereIntersectionQuery::~SphereIntersectionQuery(){
}

void SphereIntersectionQuery::execute(){
	this->result.clear();

	// collide against geoms
	// FIXME: if sphere.origin is inside box no collosion is detected!
	dGeomID sphere = dCreateSphere(0, this->radius );
	dGeomSetPosition( sphere, this->origin.x, this->origin.y, this->origin.z );
	dGeomSetData( sphere, this );

	dSpaceCollide2( sphere, (dGeomID)physics->getStaticGeometrySpace(), this, &SphereIntersectionQuery::geomNearGeomCallback );
	dSpaceCollide2( sphere, (dGeomID)physics->getDynamicGeometrySpace(), this, &SphereIntersectionQuery::geomNearGeomCallback );

	dGeomDestroy( sphere );


	// collide against terrain
	// FIXME: not supported
	if( physics->getCollideAgainstTerrain()
		&& (this->collisionFlags & IntersectionQuery::RESULT_ENTRY_TYPE_WORLD_GEOMETRY) != 0 ){

		throw Exception( "Collision against terrain not yet supported.", "SphereIntersectionQuery::execute()" );
	}

}

void SphereIntersectionQuery::setSphere( const FVector3& origin, float radius ){
	this->origin = origin;
	this->radius = radius;
}

void SphereIntersectionQuery::geomNearGeomCallback(void* data, dGeomID g1, dGeomID g2){
//	printf("### HURRA: g1: %i (b1: %i), g2: %i (b2: %i)\n", g1, dGeomGetBody(g1), g2, dGeomGetBody(g2));
	SphereIntersectionQuery* instance = (SphereIntersectionQuery*)data;

	void* data1 = dGeomGetData(g1);
	void* data2 = dGeomGetData(g2);
	dGeomID sphere;
	dGeomID other;
	void* go;
	if( data1 == instance ){
		sphere = g1;
		other = g2;
		go = data2;
	}else{
		sphere = g2;
		other = g1;
		go = data1;
	}

	// get the exact contacts
	dContact contacts[1];
	int numContacts = dCollide( sphere, other, 1, &contacts[0].geom, sizeof(dContact) );
	if( numContacts > 0 ){
//		printf("HIT: %d!\n", go);

		IntersectionQuery::resultEntry_s re;

		re.type = go != NULL ? RESULT_ENTRY_TYPE_GAME_OBJECT : RESULT_ENTRY_TYPE_ARENA_BOUNDARY;
		re.distance = instance->radius - contacts[0].geom.depth;
		re.geom = other;
		re.gameObject = (GameObject*)go;

		if( (re.type & instance->collisionFlags) == 0 || re.distance < 0.0f
			|| (instance->ignoreGameObject != NULL && re.gameObject == instance->ignoreGameObject) ){
			return;
		}

		re.intersectionPoint = FVector3(contacts[0].geom.pos);
		re.intersectionNormal = FVector3(contacts[0].geom.normal);

		instance->result.push_back( re );
	} // if
}

