#include "JoinServerMenu.h"

#include "System.h"
#include "Exception.h"
#include "Socket.h"
#include "StringConverter.h"
#include "Gui.h"
#include "Network.h"
#include "Game.h"
#include "MessageBoxMenu.h"
#include "OptionsMenu.h"

const unsigned long JoinServerMenu::MAX_UPDATE_DURATION_MILLIS = 3000;

JoinServerMenu::JoinServerMenu(Gui* gui): Menu(gui, "join_server_menu.layout.xml") {
	this->updating = false;
	this->updatingSingleServer = false;
	this->lastPacketReceivedMillis = 0;
	this->masterServerIpAddress = Packet::IP_ADDRESS_NONE;

	this->serverList.clear();
	this->socket = NULL;

	this->serverMultiColumnList = static_cast<CEGUI::MultiColumnList*>(
		CEGUI::WindowManager::getSingleton().getWindow( (CEGUI::utf8*)"GuiSheet/JoinServerMenuWindow/ServerMultiColumnList")
		);
	this->serverMultiColumnList->setSelectionMode( CEGUI::MultiColumnList::RowSingle );

	this->serverMultiColumnList->addColumn((CEGUI::utf8*)"Name", 0, 0.4f);
	this->serverMultiColumnList->addColumn((CEGUI::utf8*)"Arena", 1, 0.2f);
	this->serverMultiColumnList->addColumn((CEGUI::utf8*)"Mode", 2, 0.2f);
	this->serverMultiColumnList->addColumn((CEGUI::utf8*)"Players", 3, 0.1f);
	this->serverMultiColumnList->addColumn((CEGUI::utf8*)"Ping", 4, 0.1f);


	this->serverInfoMultiColumnList = static_cast<CEGUI::MultiColumnList*>(
		CEGUI::WindowManager::getSingleton().getWindow( (CEGUI::utf8*)"GuiSheet/JoinServerMenuWindow/ServerInfoMultiColumnList")
		);
	this->serverInfoMultiColumnList->addColumn((CEGUI::utf8*)"Property", 0, 0.5f);
	this->serverInfoMultiColumnList->addColumn((CEGUI::utf8*)"Value", 1, 0.5f);

}

JoinServerMenu::~JoinServerMenu(){
}

void JoinServerMenu::show(){
	//this->ceguiWindow->show();
	Menu::show();

	// init gui elements
	// is done upon update...
}

void JoinServerMenu::hide(){
	if( this->updating ){
		this->stopUpdate();
	}
	Menu::hide();
}

void JoinServerMenu::registerEventHandlers(){
	CEGUI::WindowManager& wmgr = CEGUI::WindowManager::getSingleton();

	CEGUI::FrameWindow* fm = (CEGUI::FrameWindow*)(
			wmgr.getWindow( (CEGUI::utf8*)"GuiSheet/JoinServerMenuWindow")
		);
	fm->subscribeEvent(
			CEGUI::FrameWindow::EventCloseClicked,
			CEGUI::Event::Subscriber( &JoinServerMenu::handleCloseClicked, this )
		);

	wmgr.getWindow( (CEGUI::utf8*)"GuiSheet/JoinServerMenuWindow/JoinServerButton" )
		->subscribeEvent(
			CEGUI::PushButton::EventClicked,
			CEGUI::Event::Subscriber( &JoinServerMenu::handleJoinServerButtonClicked, this )
		);
	wmgr.getWindow( (CEGUI::utf8*)"GuiSheet/JoinServerMenuWindow/UpdateServerButton" )
		->subscribeEvent(
			CEGUI::PushButton::EventClicked,
			CEGUI::Event::Subscriber( &JoinServerMenu::handleUpdateServerButtonClicked, this )
		);
	wmgr.getWindow( (CEGUI::utf8*)"GuiSheet/JoinServerMenuWindow/UpdateServerListButton" )
		->subscribeEvent(
			CEGUI::PushButton::EventClicked,
			CEGUI::Event::Subscriber( &JoinServerMenu::handleUpdateServerListButtonClicked, this )
		);
	wmgr.getWindow( (CEGUI::utf8*)"GuiSheet/JoinServerMenuWindow/ConfigureMasterServerButton" )
		->subscribeEvent(
			CEGUI::PushButton::EventClicked,
			CEGUI::Event::Subscriber( &JoinServerMenu::handleConfigureMasterServerButtonClicked, this )
		);

	this->serverMultiColumnList->subscribeEvent(
			CEGUI::MultiColumnList::EventSelectionChanged,
			CEGUI::Event::Subscriber( &JoinServerMenu::handleServerMultiColumnListSelectionChanged, this )
		);

}


void JoinServerMenu::startUpdate(){

	// apply cvars!
	Network* network = this->gui->getSystem()->getNetwork();
	network->cVars.network_masterServer_host->updateReferencedVariable();
	network->cVars.network_masterServer_port->updateReferencedVariable();
	network->cVars.network_masterServer_localPort->updateReferencedVariable();

	const Network::masterServerProperties_s& masterServerProperties = network->getMasterServerProperties();

	// resolve master server
	this->masterServerIpAddress = Network::resolveHost( masterServerProperties.host, masterServerProperties.port );

	// open messageBox
	MessageBoxMenu* mbm = this->gui->getMessageBoxMenu();
	mbm->setButtonClickedCallback(JoinServerMenu::messageBoxMenuCallback, this);
	mbm->show("Updating...", "Please wait...", "Cancel");

	// block gui
	this->ceguiWindow->setEnabled(false);

	// open socket
	this->socket = new Socket( network, masterServerProperties.localPort );
	this->socket->open();
	this->socket->emptyIncomingPacketQueue(); // just to be sure


	// set state
	this->lastPacketReceivedMillis = this->gui->getSystem()->getTimer()->getMilliseconds();
	this->updating = true;

	System::log("JoinServermenu: update started.");
}
void JoinServerMenu::stopUpdate(){
	
	if( this->socket != NULL && this->socket->isOpen() ){
		if( !this->updatingSingleServer ){ // send listquit!
			MSStopListPacket slp;
			slp.pack( this->masterServerIpAddress );
			this->socket->send( &slp );
		}

		// close socket
		this->socket->close();
		delete this->socket;
		this->socket = NULL;
	}

	// unblock gui
	this->ceguiWindow->setEnabled(true);
	this->ceguiWindow->activate();

	// close messageBox
	this->gui->getMessageBoxMenu()->hide();

	this->updating = false;

	System::log( "JoinServermenu: update stopped, %u servers in list.", this->serverList.size() );
}
bool JoinServerMenu::isUpdating() const {
	return this->updating;
}

void JoinServerMenu::update(){
	try{

	// receive packets
	Packet::packetList_t packetList;
	this->socket->receive( packetList, 16, 2048 ); // server properties packet is bigger than normal game packets...

	for( list<Packet*>::iterator iter = packetList.begin(); iter != packetList.end(); ++iter ){
		int packetType = (*iter)->packetType;

		switch( packetType ){
			case Packet::PACKET_TYPE_MS_ENTRY:
				this->handleMSEntryPacket( (MSEntryPacket*)(*iter) );
//				System::log("JoinServerMenu: Received MSEntryPacket.");
				break;

			case Packet::PACKET_TYPE_PING:
				this->handlePingPacket( (PingPacket*)(*iter) );
//				System::log("JoinServerMenu: Received PingPacket.");
				break;
			case Packet::PACKET_TYPE_SERVER_PROPERTIES:
				this->handleServerPropertiesPacket( (ServerPropertiesPacket*)(*iter) );
//				System::log("JoinServerMenu: Received ServerPropertiesPacket.");
				break;
			case Packet::PACKET_TYPE_SERVER_INFO:
				this->handleServerInfoPacket( (ServerInfoPacket*)(*iter) );
//				System::log("JoinServerMenu: Received ServerInfoPacket.");
				break;

			default:
				System::warn("(in JoinServerMenu::receivePackets()): Received packet with unexpected packetType (%i). Ignoring.", packetType);
				break;
		}

	}		
	
	unsigned long currentMillis = this->gui->getSystem()->getTimer()->getMilliseconds();

	if( packetList.size() > 0 ){
		this->lastPacketReceivedMillis = currentMillis;
	}

	if( this->lastPacketReceivedMillis + JoinServerMenu::MAX_UPDATE_DURATION_MILLIS < currentMillis ){
		this->stopUpdate();
	}

	Packet::clearPacketList( packetList );

	this->updateAppearance();


	}catch( Exception& e ){
		System::warn("(in JoinServerMenu::update()): An exception has been caught while updating. Aborting update.");
		System::log("JoinServerMenu: The exception was: %s", e.getDescription().c_str());
		this->stopUpdate();
	}
}


void JoinServerMenu::updateAppearance(){
	// update list
	this->serverMultiColumnList->resetList();

	for( serverList_t::const_iterator c_iter = this->serverList.begin(); c_iter != this->serverList.end(); ++c_iter ){
		
		int rowIndex = this->serverMultiColumnList->addRow();
		this->serverMultiColumnList->setItem(new MultiColumnListboxItem( c_iter->name, c_iter->id ), 0, rowIndex);
		this->serverMultiColumnList->setItem(new MultiColumnListboxItem( c_iter->arena, c_iter->id ), 1, rowIndex);
		this->serverMultiColumnList->setItem(new MultiColumnListboxItem( c_iter->mode, c_iter->id ), 2, rowIndex);
		this->serverMultiColumnList->setItem(new MultiColumnListboxItem( c_iter->currentPlayers + "/" + c_iter->maxPlayers, c_iter->id ), 3, rowIndex);
		this->serverMultiColumnList->setItem(new MultiColumnListboxItem( c_iter->ping, c_iter->id ), 4, rowIndex);

	}

	this->serverInfoMultiColumnList->resetList();
}


JoinServerMenu::serverListEntry_s* JoinServerMenu::getServerListEntryById(int id){
	for( serverList_t::iterator iter = this->serverList.begin(); iter != this->serverList.end(); iter++ ){
		if( iter->id == id ){
			return &(*iter);
		}
	}

//	System::error("(in JoinServerMenu::getServerListEntryById()): No item with id %i found in serverList.", id);
//	throw Exception("Couldn't find serverListItem.", "JoinServerMenu::getServerListEntryById()");
	return NULL;
}
JoinServerMenu::serverListEntry_s* JoinServerMenu::getServerListEntryByIpAddress(const Packet::ipAddress_t& ipAddress){
	for( serverList_t::iterator iter = this->serverList.begin(); iter != this->serverList.end(); iter++ ){
		if( iter->ipAddress == ipAddress ){
			return &(*iter);
		}
	}

	return NULL;
}




void JoinServerMenu::handleMSEntryPacket( MSEntryPacket* ep ){
	// check version and add to list
	string version;
	Packet::string32ToString( version, ep->data.version );
	if( version != this->gui->getSystem()->getVersionString() ){
		System::warn( "(in JoinServerMenu::handleMSEntryPacket()): Received packet with incompatible version ('%s'). Ignoring"
			,version.c_str() );
		return;
	}
	
	string serverName;
	Packet::string256ToString( serverName, ep->data.serverName );

	serverListEntry_s sle;
	Packet::string32ToString( sle.host, ep->data.host );
	sle.port = ep->data.port;
	sle.id = ep->data.id;
	sle.name = serverName;
	sle.arena = "???";
	sle.mode = "???";
	sle.currentPlayers = StringConverter::toString( ep->data.currentPlayers );
	sle.maxPlayers = StringConverter::toString( ep->data.maxPlayers );
	sle.ping = "???";

	sle.ipAddress = Network::resolveHost( sle.host, ep->data.port );

	// send serverUpdateRequest
	ServerUpdateRequestPacket surp;
	surp.pack( sle.ipAddress );
	this->socket->send( &surp );
	
	// send ping
	PingPacket pp;
	pp.data.startMillis = this->gui->getSystem()->getTimer()->getMilliseconds();
	pp.data.clientId = -1;
	pp.pack( sle.ipAddress );
	this->socket->send( &pp );

	// add to list
	this->serverList.push_back( sle );
}


void JoinServerMenu::handleServerPropertiesPacket( ServerPropertiesPacket* spp ){
	serverListEntry_s* sle = this->getServerListEntryByIpAddress( spp->udpPacket->address );

	if( sle == NULL ){
		System::warn("(in JoinServerMenu::handleServerPropertiesPacket()): Received ServerPropertiesPacket from unknown server. Ignoring.");
		return;
	}

	Packet::string256ToString( sle->name, spp->data.name );
	Packet::string32ToString( sle->arena, spp->data.arena );
	sle->mode = Game::getModeName( spp->data.gameMode );
	sle->maxPlayers = StringConverter::toString( (int)spp->data.maxClients );
}
void JoinServerMenu::handleServerInfoPacket( ServerInfoPacket* sip ){
	serverListEntry_s* sle = this->getServerListEntryByIpAddress( sip->udpPacket->address );

	if( sle == NULL ){
		System::warn("(in JoinServerMenu::handleServerInfoPacket()): Received ServerInfoPacket from unknown server. Ignoring.");
		return;
	}

	sle->currentPlayers = StringConverter::toString( (int)sip->data.numClients );
	sle->players.clear();
	for( int i=0; i<16; i++ ){
		if( (sip->data.clientArray.usedSlots & (0x01 << i)) != 0 ){
			string clientName;
			Packet::string32ToString( clientName, sip->data.clientArray.entries[i].name );
			sle->players.push_back( clientName );
		}
	}
}
void JoinServerMenu::handlePingPacket( PingPacket* pp ){
	serverListEntry_s* sle = this->getServerListEntryByIpAddress( pp->udpPacket->address );

	if( sle == NULL ){
		System::warn("(in JoinServerMenu::handlePingPacket()): Received PingPacket from unknown server. Ignoring.");
		return;
	}

	unsigned long currentMillis = this->gui->getSystem()->getTimer()->getMilliseconds();
	sle->ping = StringConverter::toString( (int)( currentMillis - pp->data.startMillis ) );
}



bool JoinServerMenu::handleCloseClicked(const CEGUI::EventArgs& e){

	this->hide();

	return true;
}

bool JoinServerMenu::handleJoinServerButtonClicked(const CEGUI::EventArgs& e){
	// check whether an entry is selected
	CEGUI::ListboxItem* lbi = this->serverMultiColumnList->getFirstSelectedItem();
	if( lbi == NULL ){ // no entry selected => do nothing
		return true;
	}

	// find selected server in list
	int serverListEntryId = lbi->getID();
	serverListEntry_s* sle = this->getServerListEntryById( serverListEntryId );

	if( sle == NULL ){
		return true;
	}
	

	Network* network = this->gui->getSystem()->getNetwork();
	network->cVars.network_server_host->setValueString( sle->host );
	network->cVars.network_server_port->setValueString( sle->port );
	network->cVars.network_server_remote->setValue( true );

	this->gui->getSystem()->startGame();

	return true;
}

bool JoinServerMenu::handleUpdateServerButtonClicked(const CEGUI::EventArgs& e){
	// check whether an entry is selected
	CEGUI::ListboxItem* lbi = this->serverMultiColumnList->getFirstSelectedItem();
	if( lbi == NULL ){
		return true;
	}

	// find selected server in list
	int serverListEntryId = lbi->getID();
	serverListEntry_s* sle = this->getServerListEntryById( serverListEntryId );
	if( sle == NULL ){
		return true;
	}

	// prepare update
	this->updatingSingleServer = true;

	try{

		this->startUpdate();

		// send serverUpdateRequest
		ServerUpdateRequestPacket surp;
		surp.pack( sle->ipAddress );
		this->socket->send( &surp );

		// send ping
		PingPacket pp;
		pp.data.startMillis = this->gui->getSystem()->getTimer()->getMilliseconds();
		pp.data.clientId = -1;
		pp.pack( sle->ipAddress );
		this->socket->send( &pp );

	}catch( Exception& e ){
		System::warn("(in JoinServerMenu::handleUpdateServerButtonClicked()): An exception has been caught. Description: %s", e.getDescription().c_str() );

		System::log("JoinServerMenu: Trying to recover...");
		this->updatingSingleServer = true; // FIXME: Find a better way to prevent stopUpdate from sending list quit to ms...
		this->stopUpdate();
		System::log("JoinServerMenu: Recovery done.");

		this->gui->getMessageBoxMenu()->show("An Exception has occurred",
				"An Exception has been caught in JoinServerMenu::handleUpdateServerButtonClicked(). Details:\n"
				+ e.getFullInfo()
				+ "\n(The console may provide more information.)", "Ok");
	}

	return true;
}
bool JoinServerMenu::handleUpdateServerListButtonClicked(const CEGUI::EventArgs& e){
	this->serverList.clear();
	this->serverMultiColumnList->resetList();

	this->updatingSingleServer = false;

	try{

		this->startUpdate();
	
		// send listgames...
		MSListGamesPacket lgp;
		lgp.pack( this->masterServerIpAddress );
		this->socket->send( &lgp );

	}catch( Exception& e ){
		System::warn("(in JoinServerMenu::handleUpdateServerListButtonClicked()): An exception has been caught. Description: %s", e.getDescription().c_str() );

		System::log("JoinServerMenu: Trying to recover...");
		this->updatingSingleServer = true; // FIXME: Find a better way to prevent stopUpdate from sending list quit to ms...
		this->stopUpdate();
		System::log("JoinServerMenu: Recovery done.");

		this->gui->getMessageBoxMenu()->show("An Exception has occurred",
				"An Exception has been caught in JoinServerMenu::handleUpdateServerListButtonClicked(). Details:\n"
				+ e.getFullInfo()
				+ "\n(The console may provide more information.)", "Ok");
	}

	return true;
}
bool JoinServerMenu::handleConfigureMasterServerButtonClicked(const CEGUI::EventArgs& e){
	OptionsMenu* om = this->gui->getOptionsMenu();
	om->show();
	om->activateNetworkTab();

	return true;
}

bool JoinServerMenu::handleServerMultiColumnListSelectionChanged(const CEGUI::EventArgs& e){
	this->serverInfoMultiColumnList->resetList();

	// check whether an entry is selected
	CEGUI::ListboxItem* lbi = this->serverMultiColumnList->getFirstSelectedItem();
	if( lbi == NULL ){ // no entry selected => do nothing
		return true;
	}

	// find selected server in list
	int serverListEntryId = lbi->getID();
	serverListEntry_s* sle = this->getServerListEntryById( serverListEntryId );
	if( sle == NULL ){
		return true;
	}

	string players;
	for( list<string>::const_iterator c_iter = sle->players.begin(); c_iter != sle->players.end(); ++c_iter ){
		players = players + ", " + (*c_iter);
	}

	this->serverInfoMultiColumnList->addRow();
	this->serverInfoMultiColumnList->setItem(new MultiColumnListboxItem( "Players" ), 0, 0);
	this->serverInfoMultiColumnList->setItem(new MultiColumnListboxItem( players ), 1, 0);

	return true;
}



void JoinServerMenu::messageBoxMenuCallback(void* object){
	((JoinServerMenu*)object)->stopUpdate();
}
