#include "Packets.h"

#include "System.h"
#include "Exception.h"
#include "StringTokenizer.h"
#include "Network.h"

const int Packet::GAME_PACKETS_MAX_DATA_BUFFER_SIZE = 1024;
//const int Packet::FWP_PACKET_ID = 27;
const char* Packet::FWP_GAME_NAME = "fightwinprevail";

const Packet::ipAddress_t Packet::IP_ADDRESS_NONE = {0x00000000, 0};
const Packet::ipAddress_t Packet::IP_ADDRESS_ANY = {0xFFFFFFFF, 0};


Packet::Packet(packetTypes_e packetType, int dataSize){
	this->packetType = packetType;
//	printf("dataSize: %i\n", dataSize);
	this->dataSize = dataSize;

	this->udpPacket = SDLNet_AllocPacket(this->dataSize);
	if( this->udpPacket == NULL ){
		System::error("(in Packet::Paket()): SDLNet_AllocPacket() failed: %s.", SDLNet_GetError());
		throw Exception("Couldn't alloc udpPacket.", "Packet::Paket()");
	}
}

Packet::~Packet(){
	if( this->udpPacket != NULL ){
		SDLNet_FreePacket( this->udpPacket );
	}
}

Packet* Packet::wrapUdpPacket( const udpPacket_t* udpPacket ){
	Packet* ret = NULL;

	int packetType = udpPacket->data[0];
	if( packetType > 'A' && packetType < 'Z' ){ // this is text message from the master server
		string packetString = string( (char*)udpPacket->data );

		if( packetString.find("GAME:") == 0 ){
			ret = new MSRegisterGamePacket();
		}else if( packetString.find("QUIT:") == 0 ){
			ret = new MSRemoveGamePacket();
		}else if( packetString.find("LISTGAMES:") == 0 ){
			ret = new MSListGamesPacket();
		}else if( packetString.find("LISTQUIT:") == 0 ){
			ret = new MSStopListPacket();
		}else if( packetString.find("ENTRY:") == 0 ){
			ret = new MSEntryPacket();
		}else if( packetString.find("DELENTRY:") == 0 ){
			ret = new MSEntryRemovedPacket();

		}else{
			System::error("(in Packet::wrapUdpPacket()): UdpPacket string does not start with master server command (%s).", packetString.c_str());
			throw Exception("Couldn't wrap packet.", "Packet::wrapUdpPacket()");
		}
	}else{ // a normal game packet
		switch( packetType ){
			case PACKET_TYPE_CONNECT:
				ret = new ConnectPacket();
				break;
			case PACKET_TYPE_DISCONNECT:
				ret = new DisconnectPacket();
				break;
			case PACKET_TYPE_PING:
				ret = new PingPacket();
				break;

			case PACKET_TYPE_SPAWN:
				ret = new SpawnPacket();
				break;
			case PACKET_TYPE_KILL:
				ret = new KillPacket();
				break;
			case PACKET_TYPE_FIRE:
				ret = new FirePacket();
				break;
			case PACKET_TYPE_CHAT_MESSAGE:
				ret = new ChatMessagePacket();
				break;

			case PACKET_TYPE_CLIENT_STATE:
				ret = new ClientStatePacket();
				break;


			case PACKET_TYPE_SERVER_UPDATE_REQUEST:
				ret = new ServerUpdateRequestPacket();
				break;
			case PACKET_TYPE_SERVER_PROPERTIES:
				ret = new ServerPropertiesPacket();
				break;
			case PACKET_TYPE_SERVER_INFO:
				ret = new ServerInfoPacket();
				break;
//			case PACKET_TYPE_SERVER_STATE:
//				ret = new ServerStatePacket();
//				break;

			default:
				System::error("(in Packet::wrapUdpPacket()): UdpPacket has unknown packetType field (%i).", packetType);
				throw Exception("Couldn't wrap packet.", "Packet::wrapUdpPacket()");
		}
	}

	if( ret == NULL ){
		System::error("(in Packet::wrapUdpPacket()): Packet was not wrapped properly (ret is NULL).");
		throw Exception("Couldn't wrap packet.", "Packet::wrapUdpPacket()");
	}

	ret->unpack( udpPacket );

	return ret;
}

void Packet::clearPacketList( packetList_t& packetList ){
	for( packetList_t::iterator iter = packetList.begin(); iter != packetList.end(); ++iter ){
		delete (*iter);
	}
	packetList.clear();
}

void Packet::stringToString32( string32 destination, const string& source ){
	strncpy( destination, source.c_str(), 32 );
}
void Packet::stringToString256( string256 destination, const string& source ){
	strncpy( destination, source.c_str(), 256 );
}
void Packet::string32ToString( string& destination, const string32 source ){
	char buff[33];
	strncpy( buff, source, 33 );
	destination = string(buff);
}
void Packet::string256ToString( string& destination, const string256 source ){
	char buff[257];
	strncpy( buff, source, 257 );
	destination = string(buff);
}









ConnectPacket::ConnectPacket(): Packet( Packet::PACKET_TYPE_CONNECT, sizeof(data_s) ){
	this->data.packetType = Packet::PACKET_TYPE_CONNECT;
	this->data.protocolVersion = Network::PROTOCOL_VERSION;
}
ConnectPacket::~ConnectPacket(){
}
void ConnectPacket::pack(const ipAddress_t& ipAddress){
	memcpy(this->udpPacket->data, &this->data, sizeof(data_s));

	this->udpPacket->len = sizeof(data_s);
	this->udpPacket->address = ipAddress;
}
void ConnectPacket::unpack(const udpPacket_t* udpPacket){
	memcpy(&this->data, udpPacket->data, udpPacket->len);
	this->udpPacket->address = udpPacket->address;
}


DisconnectPacket::DisconnectPacket(): Packet( Packet::PACKET_TYPE_DISCONNECT, sizeof(data_s) ){
	this->data.packetType = Packet::PACKET_TYPE_DISCONNECT;
}
DisconnectPacket::~DisconnectPacket(){
}
void DisconnectPacket::pack(const ipAddress_t& ipAddress){
	memcpy(this->udpPacket->data, &this->data, sizeof(data_s));

	this->udpPacket->len = sizeof(data_s);
	this->udpPacket->address = ipAddress;
}
void DisconnectPacket::unpack(const udpPacket_t* udpPacket){
	memcpy(&this->data, udpPacket->data, udpPacket->len);
	this->udpPacket->address = udpPacket->address;
}

PingPacket::PingPacket(): Packet( Packet::PACKET_TYPE_PING, sizeof(data_s) ){
	this->data.packetType = Packet::PACKET_TYPE_PING;
}
PingPacket::~PingPacket(){
}
void PingPacket::pack(const ipAddress_t& ipAddress){
	memcpy(this->udpPacket->data, &this->data, sizeof(data_s));

	this->udpPacket->len = sizeof(data_s);
	this->udpPacket->address = ipAddress;
}
void PingPacket::unpack(const udpPacket_t* udpPacket){
	memcpy(&this->data, udpPacket->data, udpPacket->len);
	this->udpPacket->address = udpPacket->address;
}






ClientStatePacket::ClientStatePacket(): Packet( Packet::PACKET_TYPE_CLIENT_STATE, sizeof(data_s) ){
	this->data.packetType = Packet::PACKET_TYPE_CLIENT_STATE;
}
ClientStatePacket::~ClientStatePacket(){
}
void ClientStatePacket::pack(const ipAddress_t& ipAddress){
	memcpy(this->udpPacket->data, &this->data, sizeof(data_s));

	this->udpPacket->len = sizeof(data_s);
	this->udpPacket->address = ipAddress;
}
void ClientStatePacket::unpack(const udpPacket_t* udpPacket){
	memcpy(&this->data, udpPacket->data, udpPacket->len);
	this->udpPacket->address = udpPacket->address;
}






ServerUpdateRequestPacket::ServerUpdateRequestPacket(): Packet( Packet::PACKET_TYPE_SERVER_UPDATE_REQUEST, sizeof(data_s) ){
	this->data.packetType = Packet::PACKET_TYPE_SERVER_UPDATE_REQUEST;
	this->data.protocolVersion = Network::PROTOCOL_VERSION;
}
ServerUpdateRequestPacket::~ServerUpdateRequestPacket(){
}
void ServerUpdateRequestPacket::pack(const ipAddress_t& ipAddress){
	memcpy(this->udpPacket->data, &this->data, sizeof(data_s));

	this->udpPacket->len = sizeof(data_s);
	this->udpPacket->address = ipAddress;
}
void ServerUpdateRequestPacket::unpack(const udpPacket_t* udpPacket){
	memcpy(&this->data, udpPacket->data, udpPacket->len);
	this->udpPacket->address = udpPacket->address;
}

ServerPropertiesPacket::ServerPropertiesPacket(): Packet( Packet::PACKET_TYPE_SERVER_PROPERTIES, sizeof(data_s) ){
	this->data.packetType = Packet::PACKET_TYPE_SERVER_PROPERTIES;
	this->data.protocolVersion = Network::PROTOCOL_VERSION;
}
ServerPropertiesPacket::~ServerPropertiesPacket(){
}
void ServerPropertiesPacket::pack(const ipAddress_t& ipAddress){
	memcpy(this->udpPacket->data, &this->data, sizeof(data_s));

	this->udpPacket->len = sizeof(data_s);
	this->udpPacket->address = ipAddress;
}
void ServerPropertiesPacket::unpack(const udpPacket_t* udpPacket){
	memcpy(&this->data, udpPacket->data, udpPacket->len);
	this->udpPacket->address = udpPacket->address;
}

ServerInfoPacket::ServerInfoPacket(): Packet( Packet::PACKET_TYPE_SERVER_INFO, sizeof(data_s) ){
	this->data.packetType = Packet::PACKET_TYPE_SERVER_INFO;
}
ServerInfoPacket::~ServerInfoPacket(){
}
void ServerInfoPacket::pack(const ipAddress_t& ipAddress){
	memcpy(this->udpPacket->data, &this->data, sizeof(data_s));

	this->udpPacket->len = sizeof(data_s);
	this->udpPacket->address = ipAddress;
}
void ServerInfoPacket::unpack(const udpPacket_t* udpPacket){
	memcpy(&this->data, udpPacket->data, udpPacket->len);
	this->udpPacket->address = udpPacket->address;
}





SpawnPacket::SpawnPacket(): Packet( Packet::PACKET_TYPE_SPAWN, sizeof(data_s) ){
	this->data.packetType = Packet::PACKET_TYPE_SPAWN;
}
SpawnPacket::~SpawnPacket(){
}
void SpawnPacket::pack(const ipAddress_t& ipAddress){
	memcpy(this->udpPacket->data, &this->data, sizeof(data_s));

	this->udpPacket->len = sizeof(data_s);
	this->udpPacket->address = ipAddress;
}
void SpawnPacket::unpack(const udpPacket_t* udpPacket){
	memcpy(&this->data, udpPacket->data, udpPacket->len);
	this->udpPacket->address = udpPacket->address;
}

KillPacket::KillPacket(): Packet( Packet::PACKET_TYPE_KILL, sizeof(data_s) ){
	this->data.packetType = Packet::PACKET_TYPE_KILL;
}
KillPacket::~KillPacket(){
}
void KillPacket::pack(const ipAddress_t& ipAddress){
	memcpy(this->udpPacket->data, &this->data, sizeof(data_s));

	this->udpPacket->len = sizeof(data_s);
	this->udpPacket->address = ipAddress;
}
void KillPacket::unpack(const udpPacket_t* udpPacket){
	memcpy(&this->data, udpPacket->data, udpPacket->len);
	this->udpPacket->address = udpPacket->address;
}

FirePacket::FirePacket(): Packet( Packet::PACKET_TYPE_FIRE, sizeof(data_s) ){
	this->data.packetType = Packet::PACKET_TYPE_FIRE;
}
FirePacket::~FirePacket(){
}
void FirePacket::pack(const ipAddress_t& ipAddress){
	memcpy(this->udpPacket->data, &this->data, sizeof(data_s));

	this->udpPacket->len = sizeof(data_s);
	this->udpPacket->address = ipAddress;
}
void FirePacket::unpack(const udpPacket_t* udpPacket){
	memcpy(&this->data, udpPacket->data, udpPacket->len);
	this->udpPacket->address = udpPacket->address;
}

ChatMessagePacket::ChatMessagePacket(): Packet( Packet::PACKET_TYPE_CHAT_MESSAGE, sizeof(data_s) ){
	this->data.packetType = Packet::PACKET_TYPE_CHAT_MESSAGE;
}
ChatMessagePacket::~ChatMessagePacket(){
}
void ChatMessagePacket::pack(const ipAddress_t& ipAddress){
	memcpy(this->udpPacket->data, &this->data, sizeof(data_s));

	this->udpPacket->len = sizeof(data_s);
	this->udpPacket->address = ipAddress;
}
void ChatMessagePacket::unpack(const udpPacket_t* udpPacket){
	memcpy(&this->data, udpPacket->data, udpPacket->len);
	this->udpPacket->address = udpPacket->address;
}













/*

  MASTER SERVER COMMUNICATION

*/




MSRegisterGamePacket::MSRegisterGamePacket(): Packet( Packet::PACKET_TYPE_MS_REGISTER_GAME, sizeof(data_s) ){
}
MSRegisterGamePacket::~MSRegisterGamePacket(){
}
void MSRegisterGamePacket::pack(const ipAddress_t& ipAddress){
	sprintf((char*)this->udpPacket->data, "GAME:%s\t%s\t%s\tIPv4\t%i\t%i\t%s", Packet::FWP_GAME_NAME,
		System::getInstance()->getVersionString().c_str(),
		data.serverName, /*data.port,*/ data.currentPlayers, data.maxPlayers, data.status);

	this->udpPacket->len = strlen((char*)this->udpPacket->data)+1;
	this->udpPacket->address = ipAddress;
//	this->udpPacket->channel = -1;
}
void MSRegisterGamePacket::unpack( const udpPacket_t* udpPacket ){
	throw Exception("Method not implemented.", "MSRegisterGamePacket::unpack()");
}


MSRemoveGamePacket::MSRemoveGamePacket(): Packet( Packet::PACKET_TYPE_MS_REMOVE_GAME, sizeof(data_s) ){
}
MSRemoveGamePacket::~MSRemoveGamePacket(){
}
void MSRemoveGamePacket::pack(const ipAddress_t& ipAddress){
	sprintf((char*)this->udpPacket->data, "QUIT:");
	this->udpPacket->len = strlen((char*)this->udpPacket->data)+1;
	this->udpPacket->address = ipAddress;
}
void MSRemoveGamePacket::unpack( const udpPacket_t* udpPacket ){
	throw Exception("Method not implemented.", "MSRemoveGamePacket::unpack()");
}


MSListGamesPacket::MSListGamesPacket(): Packet( Packet::PACKET_TYPE_MS_LIST_GAMES, sizeof(data_s) ){
}
MSListGamesPacket::~MSListGamesPacket(){
}
void MSListGamesPacket::pack(const ipAddress_t& ipAddress){
	sprintf((char*)this->udpPacket->data, "LISTGAMES:%s", Packet::FWP_GAME_NAME);
	this->udpPacket->len = strlen((char*)this->udpPacket->data)+1;
	this->udpPacket->address = ipAddress;
}
void MSListGamesPacket::unpack( const udpPacket_t* udpPacket ){
	throw Exception("Method not implemented.", "MSListGamesPacket::unpack()");
}


MSStopListPacket::MSStopListPacket(): Packet( Packet::PACKET_TYPE_MS_STOP_LIST, sizeof(data_s) ){
}
MSStopListPacket::~MSStopListPacket(){
}
void MSStopListPacket::pack(const ipAddress_t& ipAddress){
	sprintf((char*)this->udpPacket->data, "LISTQUIT:");
	this->udpPacket->len = strlen((char*)this->udpPacket->data)+1;
	this->udpPacket->address = ipAddress;
}
void MSStopListPacket::unpack( const udpPacket_t* udpPacket ){
	throw Exception("Method not implemented.", "MSStopListPacket::unpack()");
}


MSEntryPacket::MSEntryPacket(): Packet( Packet::PACKET_TYPE_MS_ENTRY, sizeof(data_s) ){
}
MSEntryPacket::~MSEntryPacket(){
}
void MSEntryPacket::pack(const ipAddress_t& ipAddress){
	throw Exception("Method not implemented.", "MSEntryPacket::pack()");
}
void MSEntryPacket::unpack( const udpPacket_t* udpPacket ){

	unsigned int l = strlen( (char*)udpPacket->data );
	if( l > sizeof(data_s) ){
		System::error("(in MSEntryPacket::unpack()): udpPacket->data too long (%u). Invalid udpPacket.", l);
		throw Exception("Couldn't unpack udpPacket", "MSEntryPacket::unpack()");
	}

	string str = string( (char*)udpPacket->data );
	StringTokenizer t("\t");
	const StringTokenizer::tokens_t& tokens = t.tokenize(str);
	
	if( tokens.size() != 10 ){
		System::error("(in MSEntryPacket::unpack()): udpPacket->data has wrong number of tokens (expected %i but read %i). Invalid udpPacket.", 10, tokens.size());
		throw Exception("Couldn't unpack udpPacket", "MSEntryPacket::unpack()");
	}

	// read id
	int numRead = sscanf(tokens[0].c_str(), "ENTRY:%u", &this->data.id);
	if( numRead != 1 ){
		System::error("(in MSEntryPacket::unpack()): Couldn't get id out of string '%s'.", tokens[0].c_str());
		throw Exception("Couldn't unpack udpPacket", "MSEntryPacket::unpack()");
	}

	// read host
	Packet::stringToString32( this->data.host, tokens[1] );
//	if( tokens[1].size() > 31 ){
//		System::error("(in MSEntryPacket::unpack()): Host field is too long ('%s').", tokens[1].c_str());
//		throw Exception("Couldn't unpack udpPacket", "MSEntryPacket::unpack()");
//	}
//	sprintf(data.host, "%s", tokens[1].c_str());

	// read port
	data.port = atoi(tokens[2].c_str());
	if( data.port == 0 ){
		System::error("(in MSEntryPacket::unpack()): Couldn't get port out of string '%s'.", tokens[2].c_str());
		throw Exception("Couldn't unpack udpPacket", "MSEntryPacket::unpack()");
	}

	// read version
	Packet::stringToString32( data.version, tokens[4] );
//	if( tokens[4].length() > 31 ){
//		System::error("(in MSEntryPacket::unpack()): Version field is too long ('%s').", tokens[4].c_str());
//		throw Exception("Couldn't unpack udpPacket", "MSEntryPacket::unpack()");
//	}

	// read serverName
	Packet::stringToString256( this->data.serverName, tokens[6] );
//	if( tokens[6].size() > 255 ){
//		System::error("(in MSEntryPacket::unpack()): serverName field is too long ('%s').", tokens[6].c_str());
//		throw Exception("Couldn't unpack udpPacket", "MSEntryPacket::unpack()");
//	}
//	sprintf(data.serverName, "%s", tokens[6].c_str());

	// read status
	Packet::stringToString32( this->data.status, tokens[7] );
//	if( tokens[7].size() > 31 ){
//		System::error("(in MSEntryPacket::unpack()): status field is too long ('%s').", tokens[7].c_str());
//		throw Exception("Couldn't unpack udpPacket", "MSEntryPacket::unpack()");
//	}
//	sprintf(data.status, "%s", tokens[7].c_str());

	// read currentPlayers
	data.currentPlayers = atoi(tokens[8].c_str());

	// read maxPlayers
	data.maxPlayers = atoi(tokens[9].c_str());

}


MSEntryRemovedPacket::MSEntryRemovedPacket(): Packet( Packet::PACKET_TYPE_MS_ENTRY_REMOVED, sizeof(data_s) ){
}
MSEntryRemovedPacket::~MSEntryRemovedPacket(){
}
void MSEntryRemovedPacket::pack(const ipAddress_t& ipAddress){
	throw Exception("Method not implemented.", "MSEntryPacket::pack()");
}
void MSEntryRemovedPacket::unpack( const udpPacket_t* udpPacket ){
	unsigned int l = strlen( (char*)udpPacket->data );
	if( l > sizeof(data_s) ){
		System::error("(in MSEntryRemovedPacket::unpack()): udpPacket->data too long (%u). Invalid udpPacket.", l);
		throw Exception("Couldn't unpack udpPacket", "MSEntryRemovedPacket::unpack()");
	}

	int numRead = sscanf((char*)udpPacket->data, "DELENTRY:%u", &data.id);
	if( numRead != 1 ){
		System::error("(in MSEntryRemovedPacket::unpack()): Couldn't get id out of string '%s'.", (char*)udpPacket->data);
		throw Exception("Couldn't unpack udpPacket", "MSEntryRemovedPacket::unpack()");
	}

}

