#ifndef __Packets_h__
#define __Packets_h__

#include <OgreNoMemoryMacros.h>
#include <SDL_net.h>
#include <OgreMemoryMacros.h>

#include <list>
#include <string>
using namespace std;

class Packet{
public:
	enum packetTypes_e{
		PACKET_TYPE_PING = 0,
		PACKET_TYPE_CONNECT,
		PACKET_TYPE_DISCONNECT,

		PACKET_TYPE_SERVER_PROPERTIES,
		PACKET_TYPE_SERVER_INFO,
		PACKET_TYPE_SERVER_UPDATE_REQUEST,

//		PACKET_TYPE_CLIENT_INFO,
		PACKET_TYPE_CLIENT_STATE,

		PACKET_TYPE_SPAWN,
		PACKET_TYPE_KILL,
		PACKET_TYPE_FIRE,
		PACKET_TYPE_CHAT_MESSAGE,

		PACKET_TYPE_MS_REGISTER_GAME,
		PACKET_TYPE_MS_REMOVE_GAME,
		PACKET_TYPE_MS_LIST_GAMES,
		PACKET_TYPE_MS_STOP_LIST,
		PACKET_TYPE_MS_ENTRY,
		PACKET_TYPE_MS_ENTRY_REMOVED,

		NUM_PACKET_TYPES
	};

	typedef IPaddress ipAddress_t;
	typedef UDPpacket udpPacket_t;
	typedef list< Packet* > packetList_t;

	typedef Uint8 uint8;
	typedef Uint16 uint16;
	typedef Uint32 uint32;
	typedef char string32[32];
	typedef char string256[256];

	static const int GAME_PACKETS_MAX_DATA_BUFFER_SIZE;
//	static const int FWP_PACKET_ID;
	static const char* FWP_GAME_NAME;

	static const ipAddress_t IP_ADDRESS_NONE;
	static const ipAddress_t IP_ADDRESS_ANY;

	packetTypes_e packetType;
	udpPacket_t* udpPacket;
	int dataSize;

	Packet(packetTypes_e packetType, int dataSize);
	virtual ~Packet();

	virtual void pack(const ipAddress_t& ipAddress = IP_ADDRESS_NONE) = 0;
	virtual void unpack(const udpPacket_t* udpPacket) = 0;

	static Packet* wrapUdpPacket( const udpPacket_t* udpPacket );

	static void clearPacketList( packetList_t& packetList );
	static void stringToString32( string32 destination, const string& source );
	static void stringToString256( string256 destination, const string& source );
	static void string32ToString( string& destination, const string32 source );
	static void string256ToString( string& destination, const string256 source );
//	static udpPacket_t* allocUdpPacket(int size);
//	static void freeUdpPacket(udpPacket_t* udpPacket);

protected:
	// everything public for faster access
};

inline bool operator == ( const Packet::ipAddress_t& left, const Packet::ipAddress_t& right ){
	return ( left.host == right.host && left.port == right.port );
}

inline bool operator != ( const Packet::ipAddress_t& left, const Packet::ipAddress_t& right ){
	return ( left.host != right.host || left.port != right.port );
}





class ConnectPacket: public Packet {
public:
	struct data_s {
		uint8 packetType;
		uint8 protocolVersion;

		char clientId; // can be -1 if client asks for a slot
		string32 clientName;
	} data;

	ConnectPacket();
	~ConnectPacket();

	void pack(const ipAddress_t& ipAddress = IP_ADDRESS_NONE);
	void unpack(const udpPacket_t* udpPacket);
};

class DisconnectPacket: public Packet {
public:
	struct data_s {
		uint8 packetType;

		uint8 clientId;
		string256 reason;
	} data;

	DisconnectPacket();
	~DisconnectPacket();

	void pack(const ipAddress_t& ipAddress = IP_ADDRESS_NONE);
	void unpack(const udpPacket_t* udpPacket);
};

class PingPacket: public Packet {
public:
	struct data_s {
		uint8 packetType;

		char clientId; // can be -1 for ping packets from server
		uint32 startMillis;
	} data;

	PingPacket();
	~PingPacket();

	void pack(const ipAddress_t& ipAddress = IP_ADDRESS_NONE);
	void unpack(const udpPacket_t* udpPacket);
};





class ClientStatePacket: public Packet {
public:
	struct data_s{
		uint8 packetType;

		uint8 clientId;

		uint16 energy;
		uint16 health;

		float position[3];
		float orientation[4];
		float velocity[3];
		float angularVelocity[3];

		uint8 vehicleSpecificBytes[16];
		uint8 flags;
	} data; // total size: 75

	ClientStatePacket();
	~ClientStatePacket();

	void pack(const ipAddress_t& ipAddress = IP_ADDRESS_NONE);
	void unpack(const udpPacket_t* udpPacket);
};



class ServerUpdateRequestPacket: public Packet {
public:
	struct data_s{
		uint8 packetType;

		uint8 protocolVersion;
	} data; // total size: 2

	ServerUpdateRequestPacket();
	~ServerUpdateRequestPacket();

	void pack(const ipAddress_t& ipAddress = IP_ADDRESS_NONE);
	void unpack(const udpPacket_t* udpPacket);
};

class ServerPropertiesPacket: public Packet {
public:
	struct data_s{
		uint8 packetType;

		uint8 protocolVersion;
		uint8 flags;
		uint8 connectionSpeed;
		string256 name;

		string32 arena;
		uint8 gameMode;
		uint8 ghostTime;
		uint8 maxClients;

		struct vehicleList_s {
			uint8 size;
			string32 entries[16];
		} vehicleList; // size: 513

		struct weaponList_s {
			uint16 size;
			string32 entries[16];
		} weaponList;
	} data; // total size: 1289

	ServerPropertiesPacket();
	~ServerPropertiesPacket();

	void pack(const ipAddress_t& ipAddress = IP_ADDRESS_NONE);
	void unpack(const udpPacket_t* udpPacket);
};

class ServerInfoPacket: public Packet {
public:
	struct data_s{
		uint8 packetType;

		uint16 secondsRunning;
		uint8 numClients;
/*
		struct geometryList_s {
			uint16 usedSlots;
			string32 geometries[16];
		} geometryList; // size: 514
*/
		struct clientArray_s {
			uint16 usedSlots;
			struct entry_s {
				string32 name;
				uint8 team;
				char vehicle; // can be -1 to indicate no vehicle
				char weapons[4];
//				char weapon2;
//				char weapon3;
//				char weapon4;
				uint16 score;
				uint16 kills;
				uint16 deaths;
				uint16 damageInflicted;
				uint16 secondsOnServer;
				uint16 ping;
			} entries[16];
		} clientArray; // size: 802
	} data; // total size: 1320

	ServerInfoPacket();
	~ServerInfoPacket();

	void pack(const ipAddress_t& ipAddress = IP_ADDRESS_NONE);
	void unpack(const udpPacket_t* udpPacket);
};






class SpawnPacket: public Packet {
public:
	struct data_s {
		uint8 packetType;

		uint8 clientId;

		uint8 team;
		char vehicle;
		char weapons[4];
//		char weapon2;
//		char weapon3;
//		char weapon4;

		float position[3];
	} data;

	SpawnPacket();
	~SpawnPacket();

	void pack(const ipAddress_t& ipAddress = IP_ADDRESS_NONE);
	void unpack(const udpPacket_t* udpPacket);
};

class KillPacket: public Packet {
public:
	struct data_s {
		uint8 packetType;

		uint8 killedId;
		char killerId; // can be -1 if admin killed client
	} data;

	KillPacket();
	~KillPacket();

	void pack(const ipAddress_t& ipAddress = IP_ADDRESS_NONE);
	void unpack(const udpPacket_t* udpPacket);
};

class FirePacket: public Packet {
public:
	struct data_s {
		uint8 packetType;

		uint8 clientId;
		uint8 weaponSlot;

		float position[3];
		float orientation[4];
	} data;

	FirePacket();
	~FirePacket();

	void pack(const ipAddress_t& ipAddress = IP_ADDRESS_NONE);
	void unpack(const udpPacket_t* udpPacket);
};

class ChatMessagePacket: public Packet {
public:
	enum modes_e {
		MODE_ALL = 0,
		MODE_TEAM,

		NUM_MODES
	};

	struct data_s {
		uint8 packetType;

		uint8 clientId;
		uint8 mode;
		string256 message;
	} data;

	ChatMessagePacket();
	~ChatMessagePacket();

	void pack(const ipAddress_t& ipAddress = IP_ADDRESS_NONE);
	void unpack(const udpPacket_t* udpPacket);
};






/*

  MASTER SERVER COMMUNICATION

*/

/**
 * Registers or updates a game on the master server.
 */
class MSRegisterGamePacket: public Packet {
public:
	struct data_s{
//		string32 command;
//		string32 gameName;
//		string32 version;
		string256 serverName;
//		uint16 port;
//		string32 netType;
		uint8 currentPlayers;
		uint8 maxPlayers;
		string32 status;

		string256 placeHolder;
	};
	data_s data;

	MSRegisterGamePacket();
	~MSRegisterGamePacket();

	void pack(const ipAddress_t& ipAddress = IP_ADDRESS_NONE);
	void unpack(const udpPacket_t* udpPacket);
};

/**
 * Removes a game from the master server.
 */
class MSRemoveGamePacket: public Packet {
public:
	struct data_s{
		string32 placeHolder;
	};
	data_s data;

	MSRemoveGamePacket();
	~MSRemoveGamePacket();

	void pack(const ipAddress_t& ipAddress = IP_ADDRESS_NONE);
	void unpack(const udpPacket_t* udpPacket);
};


/**
 * Requests a game list from the master server.
 */
class MSListGamesPacket: public Packet {
public:
	struct data_s{
		string32 placeHolder;
	};
	data_s data;

	MSListGamesPacket();
	~MSListGamesPacket();

	void pack(const ipAddress_t& ipAddress = IP_ADDRESS_NONE);
	void unpack(const udpPacket_t* udpPacket);
};

/**
 * Tells the master server to stop sending list updates.
 */
class MSStopListPacket: public Packet {
public:
	struct data_s{
		string32 placeHolder;
	};
	data_s data;

	MSStopListPacket();
	~MSStopListPacket();

	void pack(const ipAddress_t& ipAddress = IP_ADDRESS_NONE);
	void unpack(const udpPacket_t* udpPacket);
};


/**
 * An entry (game) sent by the master server.
 */
class MSEntryPacket: public Packet {
public:
	struct data_s{
		uint32 id;
		string32 host;
		uint16 port;
//		string32 gameName;
		string32 version;
//		string32 netType
		string256 serverName;
		string32 status;
		uint8 currentPlayers;
		uint8 maxPlayers;

		string256 placeHolder;
	};
	data_s data;

	MSEntryPacket();
	~MSEntryPacket();

	void pack(const ipAddress_t& ipAddress = IP_ADDRESS_NONE);
	void unpack(const udpPacket_t* udpPacket);
};

/**
 * Notification that an entry (game) was removed from the master server.
 */
class MSEntryRemovedPacket: public Packet {
public:
	struct data_s{
		uint32 id;

		string32 placeHolder;
	};
	data_s data;

	MSEntryRemovedPacket();
	~MSEntryRemovedPacket();

	void pack(const ipAddress_t& ipAddress = IP_ADDRESS_NONE);
	void unpack(const udpPacket_t* udpPacket);
};





#endif // __Packets_h__
