#include "Physics.h"


#include "System.h"
#include "Game.h"
#include "GameObject.h"
#include "Graphics.h"
#include "Vehicle.h"

Physics::Physics( Game* game ){
	this->game = game;


	this->world = dWorldCreate();
	dWorldSetGravity( this->world, 0.0f, 0.0f, -9.81f );
	dWorldSetERP( this->world, 0.2f );
	dWorldSetCFM( this->world, 0.001f );
	dWorldSetQuickStepNumIterations( this->world, 10 );
	dWorldSetAutoDisableFlag( this->world, true );
	dWorldSetAutoDisableLinearThreshold( this->world, 0.01f );
	dWorldSetAutoDisableAngularThreshold( this->world, 0.01f );
	dWorldSetAutoDisableSteps( this->world, 10 );
	dWorldSetAutoDisableTime( this->world, 0 );
	dWorldSetContactMaxCorrectingVel( this->world, 1000.0f );
	dWorldSetContactSurfaceLayer( this->world, 0.001f );

	this->staticGeometrySpace = dHashSpaceCreate(0);
	this->dynamicGeometrySpace = dHashSpaceCreate(0);

	this->contactJointGroup = dJointGroupCreate( 0 );

	this->gravity = FVector3::ZERO;
	this->normalizedGravity = FVector3::ZERO;
	this->collideAgainstTerrain = false;
}

Physics::~Physics(){
	dJointGroupDestroy( this->contactJointGroup );
	dSpaceDestroy( this->staticGeometrySpace );
	dSpaceDestroy( this->dynamicGeometrySpace );
	dWorldDestroy( this->world );
	dCloseODE();
}


void Physics::createSpaces(int quadTreeLevels, const FVector3& min, const FVector3& max){
	dSpaceDestroy( this->staticGeometrySpace );
	dSpaceDestroy( this->dynamicGeometrySpace );

	this->staticGeometrySpace = dHashSpaceCreate(0);

	if( false /*quadTreeLevels > 0*/ ){
		FVector3 extents = (max-min);
		FVector3 center = extents*0.5f + min;
		dVector3 c, e;
		extents.toOdeVector3(e);
		center.toOdeVector3(c);
		this->dynamicGeometrySpace = dQuadTreeSpaceCreate(0, c, e, quadTreeLevels);
	}else{
		this->dynamicGeometrySpace = dHashSpaceCreate(0);
	}

//	cout << "min: " << min << endl;
//	cout << "max: " << max << endl;
	dCreatePlane(this->staticGeometrySpace, 1.0f, 0.0f, 0.0f, min.x);
	dCreatePlane(this->staticGeometrySpace, -1.0f, 0.0f, 0.0f, -max.x);
	dCreatePlane(this->staticGeometrySpace, 0.0f, 1.0f, 0.0f, min.y);
	dCreatePlane(this->staticGeometrySpace, 0.0f, -1.0f, 0.0f, -max.y);
	dCreatePlane(this->staticGeometrySpace, 0.0f, 0.0f, 1.0f, min.z);
	dCreatePlane(this->staticGeometrySpace, 0.0f, 0.0f, -1.0f, -max.z);
}

void Physics::collisionDetection(){
	collideShots();
	collideGeoms();
}
void Physics::collideShots(){
	// hmm, i dont need this any more...
}




float Physics::getTerrainHeightAt( const FVector3& position ){
//	float ret = -FLT_MAX;
	float ret = 0;	// THINKABOUTME:

	Ogre::SceneManager* sm = this->game->getSystem()->getGraphics()->getSceneManager();
	FVector3 ori = position + FVector3(0.0f, 999999.0f, 0.0f);
	Ogre::Ray ray(ori.toOgreVector3(), Ogre::Vector3(0.0f, -1.0f, 0.0f));
	Ogre::RaySceneQuery* rsq = sm->createRayQuery(ray);
	rsq->setSortByDistance(false);
	rsq->setWorldFragmentType(Ogre::SceneQuery::WFT_SINGLE_INTERSECTION);

	Ogre::RaySceneQueryResult& rsqResult = rsq->execute();
	Ogre::RaySceneQueryResult::iterator iter = rsqResult.begin();
	while( iter != rsqResult.end() ){
		if( iter->worldFragment != NULL ){
//			System::log( "raySceneQuery: hit worldFragment (x: %.2f, y: %.2f, z: %.2f, dist: %.2f).",
//				iter->worldFragment->singleIntersection.x, iter->worldFragment->singleIntersection.y, iter->worldFragment->singleIntersection.z, iter->distance );
			ret = iter->worldFragment->singleIntersection.y;
			break;
		}
		iter ++;
	}
	sm->destroyQuery(rsq);
	return ret;
}

bool Physics::aabbIntersectsTerrain( const Ogre::AxisAlignedBox& aabb ){
	Ogre::SceneManager* sm = this->game->getSystem()->getGraphics()->getSceneManager();
	Ogre::AxisAlignedBoxSceneQuery* aabbsq = sm->createAABBQuery(aabb);
//	aabbsq->setSortByDistance(false);
//	printf("###supp: %i", aabbsq->getSupportedWorldFragmentTypes()->size() );
	aabbsq->setWorldFragmentType(Ogre::SceneQuery::WFT_NONE);

	Ogre::SceneQueryResult& aabbsqResult = aabbsq->execute();
	bool ret = !aabbsqResult.worldFragments.empty();
	sm->destroyQuery(aabbsq);
	return ret;

}


void Physics::collideGeomsAgainstTerrain(){
	int numDynamicGeoms = dSpaceGetNumGeoms( this->dynamicGeometrySpace );
	for( int i=0; i<numDynamicGeoms; i++ ){
		dGeomID g = dSpaceGetGeom( this->dynamicGeometrySpace, i );

		bool isMech = false;	// mechs are treated differently
		void* data = dGeomGetData( g );
		if( data != NULL
			&& ((GameObject*)data)->getGameObjectType() == GameObject::GAME_OBJECT_TYPE_VEHICLE
			&& ((Vehicle*)data)->getVehicleType() == Vehicle::VEHICLE_TYPE_MECH
			){
			isMech = true;
		}

		FVector3 geomPos = FVector3( dGeomGetPosition(g) );
		FVector3 p = FVector3(geomPos.x, this->getTerrainHeightAt(geomPos), geomPos.z);

		dGeomID plane = 0;
		// calc the plane
		if( isMech ){
			plane = dCreatePlane( 0, 0.0f, 1.0f, 0.0f, p.y);
		}else{
			dReal aabb[6];
			dGeomGetAABB( g, aabb );

			FVector3 p1 = FVector3((float)aabb[0], geomPos.y, geomPos.z);
			p1.y = this->getTerrainHeightAt( p1 );
			FVector3 p2 = FVector3((float)aabb[1], geomPos.y, geomPos.z);
			p2.y = this->getTerrainHeightAt( p2 );
			FVector3 p3 = FVector3(geomPos.x, geomPos.y, (float)aabb[4]);
			p3.y = this->getTerrainHeightAt( p3 );
			FVector3 p4 = FVector3(geomPos.x, geomPos.y, (float)aabb[5]);
			p4.y = this->getTerrainHeightAt( p4 );

			FVector3 v1 = p2 - p1;
			FVector3 v2 = p4 - p3;
			FVector3 n = v2.crossProduct(v1);
			n.normalize();
			float d = n.dotProduct( p );

			plane = dCreatePlane( 0, n.x, n.y, n.z, d);
		}

		// check for collision with plane
		int maxContacts = 5;
		dContact* contacts = new dContact[maxContacts];
		int numContacts = dCollide(g, plane, maxContacts, &contacts[0].geom, sizeof(dContact));

		if( numContacts > 0 ){
			bool gWantsContactJoint = false;
			dContact c;
			c.surface.mode = 0;
			c.surface.mu = 0.0f;
			c.surface.bounce = 0.0f;
			GameObject* go = (GameObject*)dGeomGetData(g);
			if( go != NULL ){
				gWantsContactJoint = go->collidedWithTerrain( FVector3(contacts[0].geom.pos), c );
			}

			for( int i=0; i<numContacts; i++ ){
				contacts[i].surface.mode = c.surface.mode;// | dContactSoftCFM;
				contacts[i].surface.mu = c.surface.mu;
				contacts[i].surface.bounce = c.surface.bounce;
//					printf("mu: %f, bounce: %f\n", contacts[i].surface.mu, contacts[i].surface.bounce);

				if( gWantsContactJoint ){
					dJointID j = dJointCreateContact( this->world, this->contactJointGroup, &contacts[i]);
					dBodyID b = dGeomGetBody(g);
					dJointAttach( j, b, 0);
				}
			} // for all contacts
		}// if numContacts > 0

		dGeomDestroy( plane );
		delete[] contacts;
	} // for all geoms
}

void Physics::collideGeoms(){
	if( collideAgainstTerrain ){
		collideGeomsAgainstTerrain();
	}
	dSpaceCollide(this->dynamicGeometrySpace, this, &Physics::geomNearGeomCallback);
	dSpaceCollide2((dGeomID)this->dynamicGeometrySpace, (dGeomID)this->staticGeometrySpace, this, &Physics::geomNearGeomCallback);
}

void Physics::takeSimulationStep(float deltaT){
	collisionDetection();

    if( deltaT < 0.001f ){
		System::warn("(Physics::takeSimulationStep()): DeltaT for simulation step is %.2f. Setting it to 0.001 to ensure stability.", deltaT);
		deltaT = 0.001f;
    }
	if( deltaT > 1.0f ){
		System::warn("(Physics::takeSimulationStep()): DeltaT for simulation step is %.2f. Setting it to 1.0 to ensure stability.", deltaT);
		deltaT = 1.0f;
	}

	dWorldQuickStep( this->world, deltaT );

	dJointGroupEmpty( this->contactJointGroup );
}




void Physics::geomNearGeomCallback(void* data, dGeomID g1, dGeomID g2){
//	printf("### HURRA: g1: %i (b1: %i), g2: %i (b2: %i)\n", g1, dGeomGetBody(g1), g2, dGeomGetBody(g2));
	Physics* instance = (Physics*)data;

	// get the exact contacts
	int maxContacts = 5;
	dContact* contacts = new dContact[maxContacts];
	int numContacts = dCollide(g1, g2, maxContacts, &contacts[0].geom, sizeof(dContact));


	// do the callbacks only for the first contact
	if( numContacts > 0 ){
		bool g1WantsContactJoint = false;
		bool g2WantsContactJoint = false;

		dContact c; // make one template
		c.surface.mode = 0;
		c.surface.mu = 0.0f;
		c.surface.bounce = 0.0f;
		GameObject* go1 = (GameObject*)dGeomGetData(g1);
		GameObject* go2 = (GameObject*)dGeomGetData(g2);
		if( go1 != NULL ){
			g1WantsContactJoint = go1->collidedWithGameObject(FVector3(contacts[0].geom.pos), go2, c);
		}
		if( go2 != NULL ){
			g2WantsContactJoint = go2->collidedWithGameObject(FVector3(contacts[0].geom.pos), go1, c);
		}


		// create joints for all (if desired)
		if( g1WantsContactJoint || g2WantsContactJoint ){
			for( int i=0; i<numContacts; i++ ){

				contacts[i].surface.mode = c.surface.mode;
				contacts[i].surface.mu = c.surface.mu;
				contacts[i].surface.bounce = c.surface.bounce;

//			contacts[i].geom.g1 = g1;
//			contacts[i].geom.g2 = g2;

					dJointID j = dJointCreateContact( instance->world, instance->contactJointGroup, &contacts[i]);
					dBodyID b1 = dGeomGetBody(g1);
					dBodyID b2 = dGeomGetBody(g2);
					dJointAttach(j, b1, b2);
//		printf("CONTACT %i: d: %f, g1: %i (b1: %i), g2: %i (b2: %i)\n", i, contacts[i].depth, contacts[i].g1, dGeomGetBody(contacts[i].g1), contacts[i].g2, dGeomGetBody(contacts[i].g2));
//		printf("CONTACT %i: n1: %f, n2: %f, n3: %f\n", i, contacts[i].normal[0], contacts[i].normal[1], contacts[i].normal[2]);
//		printf("JOINT %i: j: %i, b1: %i, b2: %i\n", i, j, dJointGetBody (j,0), dJointGetBody (j,1));
			} // for
		} // if
	} // if

	delete[] contacts;
}



RayIntersectionQuery* Physics::createRayIntersectionQuery(){
	RayIntersectionQuery* ret = new RayIntersectionQuery( this );
	return ret;
}


void Physics::setGravity(const FVector3& newGravity){
	this->gravity = newGravity;
	this->normalizedGravity = newGravity;
	this->normalizedGravity.normalize();

	dWorldSetGravity( this->world, this->gravity.x, this->gravity.y, this->gravity.z );
}
const FVector3& Physics::getGravity() const {
	return this->gravity;
}
const FVector3& Physics::getNormalizedGravity() const {
	return this->normalizedGravity;
}

void Physics::setCollideAgainstTerrain( bool newCollideAgainstTerrain ){
	this->collideAgainstTerrain = newCollideAgainstTerrain;
}
bool Physics::getCollideAgainstTerrain() const {
	return this->collideAgainstTerrain;
}

dWorldID Physics::getWorld() const {
	return this->world;
}
dSpaceID Physics::getStaticGeometrySpace() const {
	return this->staticGeometrySpace;
}
dSpaceID Physics::getDynamicGeometrySpace() const {
	return this->dynamicGeometrySpace;
}
dJointGroupID Physics::getContactJointGroup() const {
	return this->contactJointGroup;
}

Game* Physics::getGame() const {
	return this->game;
}



float Physics::frictionToMu( float friction ){
	return friction * 100000.0f;// * 1000000000;
}

