#include "RegenerationArea.h"

#include "System.h"
#include "Exception.h"
#include "XmlUtils.h"
#include "Game.h"
#include "Vehicle.h"
#include "Graphics.h"

const unsigned long RegenerationArea::REGENERATION_INTERVAL = 1000;

RegenerationArea::RegenerationArea(): Area( Area::AREA_TYPE_REGENERATION ) {
	this->teamFlags = 0;
	this->lastRegenerationMillis = 0;

	this->particleSystem = NULL;
}

RegenerationArea::~RegenerationArea(){
}


Area* RegenerationArea::clone() const {
	RegenerationArea* ret = new RegenerationArea();

	*ret = *this;

	return ret;
}




void RegenerationArea::fromXmlElement( TiXmlElement* xmlElement ){
	// name
	const char* tmp = xmlElement->Attribute("name");
	if( tmp != NULL ){
		this->name = string( tmp );
	}else{
		throw Exception( "RegenerationArea element has no name attribute.", "RegenerationArea::fromXmlElement()" );
	}
	if( this->name.length() > 32 ){
		System::error( "(in RegenerationArea::fromXmlElement()): RegenerationArea name '%s' is longer than 32 characters.", this->name.c_str() );
		throw Exception( "XmlElement is not a valid regeneration area description.", "RegenerationArea::fromXmlElement()" );
	}

	// teamFlags
	tmp = xmlElement->Attribute("teamFlags");
	if( tmp != NULL ){
		string flags = string( tmp );
		if( flags.find("s") != string::npos ){
			this->teamFlags |= Game::TEAM_FLAG_SPECTATORS;
		}
		if( flags.find("p") != string::npos ){
			this->teamFlags |= Game::TEAM_FLAG_PLAYERS;
		}
		if( flags.find("r") != string::npos ){
			this->teamFlags |= Game::TEAM_FLAG_RED;
		}
		if( flags.find("b") != string::npos ){
			this->teamFlags |= Game::TEAM_FLAG_BLUE;
		}
	}else{
		throw Exception( "RegenerationArea element has no teamFlags attribute.", "RegenerationArea::fromXmlElement()" );
	}



	for( TiXmlElement* elt = xmlElement->FirstChildElement(); elt != NULL; elt = elt->NextSiblingElement() ){
		string value = string( elt->Value() );

		if(value == "edgeLengths"){
			this->edgeLengths = XmlUtils::readFVector3( elt );

		}else if(value == "position"){
			this->position = XmlUtils::readFVector3( elt );
		}else if(value == "orientation"){
			this->orientation = XmlUtils::readFQuaternion( elt );


		}else{
			System::warn("(in RegenerationArea::fromXmlElement()): Unknown child element: '%s' (row: %i). Ignoring.", value.c_str(), elt->Row() );
		}
	}


}

TiXmlElement RegenerationArea::toXmlElement() const {
	return TiXmlElement("dummy");
}

void RegenerationArea::calculateChecksum(){
	this->checksum = 0;
}



void RegenerationArea::attach( Game* game ){

	if( this->game != NULL ){
		throw Exception("RegenerationArea is already attached to a game.", "RegenerationArea::attach()");
	}

	Ogre::SceneManager* sm = game->getSystem()->getGraphics()->getSceneManager();
	Ogre::SceneNode* rootNode = sm->getRootSceneNode();

	this->sceneNode = rootNode->createChildSceneNode( this->name );
	this->sceneNode->setPosition( this->position.toOgreVector3() );
	this->sceneNode->setOrientation( this->orientation.toOgreQuaternion() );
//	this->ogreSceneNode->setScale( this->edgeLengths.toOgreVector3() );
/*
	// attach base plane
	Ogre::Plane plane(Ogre::Vector3(0.0f, 1.0f, 0.0f), -this->edgeLengths.y * 0.5f);
	Ogre::MeshManager::getSingleton().createPlane(this->name + " base mesh", Ogre::ResourceGroupManager::DEFAULT_RESOURCE_GROUP_NAME, plane, this->edgeLengths.x, this->edgeLengths.y, 1, 1, false, 1, 1, 1, Ogre::Vector3::UNIT_Z);
	Ogre::SceneManager* sm = this->game->getSystem()->getGraphics()->getSceneManager();
	Ogre::Entity* ent = sm->createEntity( this->name + " base entity", this->name + " base mesh");
	ent->setMaterialName("are/regeneration_area_base");
	ent->setCastShadows(false);
	this->ogreSceneNode->attachObject( ent );
*/

//	Ogre::ParticleSystemManager& psm = Ogre::ParticleSystemManager::getSingleton();
	this->particleSystem = sm->createParticleSystem( this->name + " particle system", "are/regeneration_area_sparkle" );
	Ogre::ParticleEmitter* e = this->particleSystem->getEmitter(0);
	e->setParameter("position", Ogre::StringConverter::toString(Ogre::Vector3(0.0, -this->edgeLengths.y*0.5f, 0.0)));
	e->setParameter("width", Ogre::StringConverter::toString(Ogre::Real(this->edgeLengths.x)));
	e->setParameter("height", Ogre::StringConverter::toString(Ogre::Real(this->edgeLengths.z)));
	e->setParameter("depth", Ogre::StringConverter::toString(Ogre::Real(1.0)));
	e->setParameter("time_to_live", Ogre::StringConverter::toString(Ogre::Real(this->edgeLengths.y*0.1)));
	e->setParameter("emission_rate", Ogre::StringConverter::toString(Ogre::Real(this->edgeLengths.x * this->edgeLengths.y * 0.02)));
	this->sceneNode->attachObject( this->particleSystem );

	this->lastRegenerationMillis = game->getSystem()->getTimer()->getMilliseconds();

	this->game = game;
}

void RegenerationArea::detach(){
	if( this->game == NULL ){
		throw Exception("RegenerationArea is not attached to a game.", "RegenerationArea::detach()");
	}

	Ogre::SceneManager* sm = this->game->getSystem()->getGraphics()->getSceneManager();
//	Ogre::ParticleSystemManager& psm = Ogre::ParticleSystemManager::getSingleton();
	sm->destroyParticleSystem( this->particleSystem );

	this->sceneNode->getParentSceneNode()->removeAndDestroyChild( this->sceneNode->getName() );
	this->sceneNode = NULL;

	this->game = NULL;
}

bool RegenerationArea::canAffectVehicle( Vehicle* vehicle ) const {
	unsigned long currentMillis = this->game->getSystem()->getTimer()->getMilliseconds();

	if( this->lastRegenerationMillis + RegenerationArea::REGENERATION_INTERVAL > currentMillis ){
		return false;
	}

	FVector3 doubleDiff = (vehicle->position - this->position) * 2.0f;
	return ( fabs(doubleDiff.x) < this->edgeLengths.x
			&& fabs(doubleDiff.y) < this->edgeLengths.y
			&& fabs(doubleDiff.z) < this->edgeLengths.z );
}

void RegenerationArea::affectVehicle( Vehicle* vehicle ){
	unsigned long currentMillis = this->game->getSystem()->getTimer()->getMilliseconds();

	vehicle->heal( 5.0f ); // heal for 5 hp
	vehicle->restockAmmoRel( 0.05f ); // restock 5% of ammo for all weapons

//	this->game->spawnEffect( "vehicle_regeneration", vehicle );

	this->lastRegenerationMillis = currentMillis;
}
