#include "Server.h"

#include "System.h"
#include "Exception.h"
#include "Socket.h"
#include "Network.h"
#include "Game.h"
#include "Client.h"
#include "Vehicle.h"
#include "VehicleFactory.h"
#include "WeaponFactory.h"
#include "Gui.h"
#include "Hud.h"

const int Server::DEFAULT_PORT = 27680;

const unsigned long Server::SERVER_INFO_INTERVAL_MILLIS = 3333;
const unsigned long Server::PING_INTERVAL_MILLIS = 1000;
const unsigned long Server::MASTER_SERVER_UPDATE_INTERVAL_MILLIS = 60000;

const int Server::MAX_CLIENT_IDLE_TIME = 40;
const int Server::MAX_SERVER_IDLE_TIME = 40;

Server::Server( Network* network ){
	this->network = network;

	this->properties.name = string("Player's Server");
	this->properties.host = string("localhost");
	this->properties.port = Server::DEFAULT_PORT;
	this->properties.maxClients = 16;
	this->properties.connectionSpeed = 10;
	this->properties.remote = false;
	this->properties.registerOnMasterServer = false;
	this->properties.arena = "spacechess";
	this->properties.ghostTime = 5;
	this->properties.availableVehicles.clear();
	this->properties.availableWeapons.clear();
	this->properties.ipAddress = Packet::IP_ADDRESS_NONE;

	this->clients = NULL;
	this->socket = NULL;

	this->info.secondsRunning = 0;
	this->info.numClients = 0;

	this->lastServerInfoMillis = 0;
	this->lastPingMillis = 0;
	this->lastMasterServerUpdateMillis = 0;

	this->masterServerIpAddress = Packet::IP_ADDRESS_NONE;
//	this->packetsReceived = 0;
//	this->packetsSent = 0;

}

Server::~Server(){
}

void Server::setUp(){
	int i;


	this->masterServerIpAddress = Packet::IP_ADDRESS_NONE;

	if( !this->properties.remote ){ // local server => open socket, etc...

		System::log("Server: Opening socket on port %i...", this->properties.port);
		this->socket = new Socket( this->network, this->properties.port );
		this->socket->open();
		this->socket->emptyIncomingPacketQueue();

		System::log("Server: Preparing clients array for %i clients...", this->properties.maxClients);
		this->clients = new Client* [this->properties.maxClients];
		for( i=0; i<this->properties.maxClients; i++ ){
			this->clients[i] = NULL;
		}


		System::log("Server: Creating available vehicles list...");
		this->properties.availableVehicles = this->network->getSystem()->getGame()->getVehicleFactory()->getVehicleNames();
		System::log("Server: Creating available weapons list...");
		this->properties.availableWeapons = this->network->getSystem()->getGame()->getWeaponFactory()->getWeaponNames();

		// prepare server properties packet
		Packet::stringToString32( this->spp.data.arena, this->properties.arena );
		this->spp.data.flags = 0;
		this->spp.data.gameMode = this->properties.gameMode;
		this->spp.data.ghostTime = this->properties.ghostTime;
		this->spp.data.maxClients = this->properties.maxClients;
		this->spp.data.connectionSpeed = this->properties.connectionSpeed;
		Packet::stringToString256( this->spp.data.name, this->properties.name );

		this->spp.data.vehicleList.size = this->properties.availableVehicles.size();
		for( i=0; i<(signed int)this->properties.availableVehicles.size(); i++ ){
			Packet::stringToString32( this->spp.data.vehicleList.entries[i], this->properties.availableVehicles[i] );
		}

		this->spp.data.weaponList.size = this->properties.availableWeapons.size();
		for( i=0; i<(signed int)this->properties.availableWeapons.size(); i++ ){
			Packet::stringToString32( this->spp.data.weaponList.entries[i], this->properties.availableWeapons[i] );
		}
		spp.pack();

		// resolve master server
		if( this->properties.registerOnMasterServer ){
			const Network::masterServerProperties_s& msprops = this->network->getMasterServerProperties();

			System::log( "Server: Resolving master server (%s:%i)...", msprops.host.c_str(), msprops.port );
			this->masterServerIpAddress = Network::resolveHost( msprops.host, msprops.port );

		}
	}

	this->properties.ipAddress = Network::resolveHost( this->properties.host, this->properties.port );

	// reset info
	this->info.secondsRunning = 0;
	this->info.numClients = 0;

	unsigned long currentMillis = this->network->getSystem()->getTimer()->getMilliseconds();
	this->lastServerInfoMillis = currentMillis;
	this->lastPingMillis = currentMillis; // important for shadow server (to check whether server is idle)
	this->lastMasterServerUpdateMillis = 0; // instantly update ms (in the first iteration)

//	this->packetsReceived = 0;
//	this->packetsSent = 0;

}

void Server::tearDown(){
	if( !this->properties.remote && this->socket != NULL && this->socket->isOpen() ){ // local server -> disconnect clients, close socket
		System::log("Server: Disconnecting remote clients...");
		this->disconnectAll( "server is shutting down" );

		// remove game from master server if it was registered
		if( this->properties.registerOnMasterServer
			&& this->masterServerIpAddress != Packet::IP_ADDRESS_NONE
			&& this->lastMasterServerUpdateMillis != 0
			){

			System::log("Server: Removing game from master server...");
			MSRemoveGamePacket rgp;
			rgp.pack( this->masterServerIpAddress );
			this->socket->send( &rgp );
		}

		System::log("Server: Closing socket...");
		this->socket->close();
		delete this->socket;
		this->socket = NULL;
	}

	if( this->clients != NULL ){
		System::log("Server: Destroying clients array...");
		for( int i=0; i<this->properties.maxClients; i++ ){
			if( this->clients[i] != NULL && this->clients[i] != this->network->getClient() ){
				this->removeAndDestroyClient( i );
			}
		}
		delete[] this->clients;
		this->clients = NULL;
	}

	this->properties.availableVehicles.clear();
}

void Server::update(){
	if( !this->properties.remote ){ // local server => send and receive server stuff
		this->receivePackets();
		this->sendPackets();

	}else{ // shadow server

		// check whether server is idle
		// local client's connectionInterrupted is used to indicate this...
		unsigned int currentMillis = this->network->getSystem()->getTimer()->getMilliseconds();

		if( !this->network->getClient()->connectionInterrupted
			&& currentMillis > this->lastPingMillis + Server::MAX_SERVER_IDLE_TIME/2 * 1000 ){

			this->network->getClient()->connectionInterrupted = true;
			System::warn( "(in Server::update()): Remote server has been idle for over %i seconds. Disconnecting after %i seconds."
				, Server::MAX_SERVER_IDLE_TIME/2, Server::MAX_SERVER_IDLE_TIME );

		}

		if( currentMillis > this->lastPingMillis + Server::MAX_SERVER_IDLE_TIME * 1000 ){
			System::error( "(in Server::update()): Remote server has been idle for over %i seconds. Disconnecting.", Server::MAX_SERVER_IDLE_TIME );
			throw Exception( "Connection to server lost.", "Server::update()" );
		}
	}
}



void Server::establishLoopbackConnection(Client* client){
	// receive connect packet
	unsigned long receiveStartMillis = this->network->getSystem()->getTimer()->getMilliseconds();
	unsigned long currentMillis = receiveStartMillis;
	bool connectPacketReceived = false;

	System::log("Server: Waiting for connect packet from client...");
	do{
		Packet::packetList_t packetList;
		this->socket->receive( packetList );

		for( list<Packet*>::iterator iter = packetList.begin(); iter != packetList.end(); ++iter ){
			if( (*iter)->packetType == Packet::PACKET_TYPE_CONNECT ){
				System::log("Server: Received connect packet for loopback connection. Adding client to list...");

				client->properties.ipAddress = (*iter)->udpPacket->address;

				client->lastClientStateMillis = currentMillis;
				client->lastPingMillis = currentMillis;
				client->ready = false; // not sending any packets to this client, until it has spawned (and thus finished loading)

				// add client to list
				this->addClient( client );
				System::log( "Server: Client '%s' added to list.", client->properties.name.c_str() );

				connectPacketReceived = true;
			}else{
				// ignore other packets
			}
		}
		
		Packet::clearPacketList( packetList );
		currentMillis = this->network->getSystem()->getTimer()->getMilliseconds();
	}while( !connectPacketReceived && currentMillis < receiveStartMillis + 5000 );

	if( !connectPacketReceived ){ // no packet received -> throw exception
		System::error("(in Server::establishLoopbackConnection()): Time limit exceeded while waiting for connect packet from client.");
		throw Exception("Couldn't establish loopback connection. (Maybe we're firewalled?)", "Server::establishLoopbackConnection()");
	}
}




void Server::addClient( Client* client ){

	// find free slot
	int slot = 0;
	for( slot=0; slot<this->properties.maxClients; slot++ ){
		if( this->clients[slot] == NULL )
			break;
	}

	if( slot == this->properties.maxClients ){
		System::error("(in Server::addClient()): No free slot found.");
		throw Exception("Couldn't add client.", "Server::addClient()");
	}

	client->properties.clientId = slot;
	client->properties.name = this->createValidClientName( client->properties.name );
	client->server = this; // THINKABOUTME: maybe I shoukld do this somewhere else...

	this->info.numClients++;

	this->clients[slot] = client;
}
void Server::addClient( Client* client, int slot ){

	if( this->clients[slot] != NULL ){
		System::error("(in Server::addClient()): Slot %i already in use.", slot);
		throw Exception("Couldn't add client.", "Server::addClient()");
	}

	client->properties.clientId = slot;
//	client->properties.name = this->createValidClientName( client->properties.name ); THINKABOUTME: dangerous if shadow server has an old (invalid) slot with the same name...
	client->server = this;

	this->info.numClients++;

	this->clients[slot] = client;
}

string Server::createValidClientName( const string& desiredName ){
	bool nameValid = false;
	string ret = desiredName;

	do{
		nameValid = true;
		for( int i=0; i<this->properties.maxClients; i++ ){
			if( this->clients[i] != NULL && this->clients[i]->properties.name == ret ){
				ret = ret + "*";
				nameValid = false;
			}
		}
	}while( !nameValid );

	if( ret.length() > 32 ){
		System::warn( "(in Server::createValidClientName()): Generated name too long ('%s').", ret.c_str() );
		ret = string( "<name too long>" );
	}

	return ret;
}


void Server::removeAndDestroyClient( int slot ){
	if( this->clients[slot] == NULL ){
		System::error("(in Server::removeClient()): Slot %i is not in use.", slot);
		throw Exception("Couldn't remove client.", "Server::removeClient()");
	}

	Client* c = this->clients[slot];
	// TODO: delete vehicle...
//	c->server = NULL;
//	c->properties.clientId = -1;

	delete c;

	this->clients[slot] = NULL;

//	if( c != this->network->getClient() ){
//		delete c;
//	}

	this->info.numClients--;
}
void Server::removeAndDestroyClient( Client* client ){
	int slot = 0;
	for( slot=0; slot<this->properties.maxClients; slot++ ){
		if( this->clients[slot] == client ){
			break;
		}
	}

	if( slot < this->properties.maxClients ){
		this->removeAndDestroyClient( slot );
	}else{
		System::error( "(in Server::removeClient()): client '%s' not in list.", client->properties.name.c_str() );
		throw Exception( "Couldn't remove client.", "Server::removeClient()" );
	}
}

void Server::disconnect( Client* client, const string& reason ){
	// unspawn vehicle
	Game* game = this->network->getSystem()->getGame();
	if( client->vehicle != NULL ){ // if there's a vehicle unspawn it
		game->unspawnVehicle( client->vehicle );
		client->vehicle = NULL;
	}

	DisconnectPacket dp;

	dp.data.clientId = client->properties.clientId;
	Packet::stringToString256( dp.data.reason, reason );
	dp.pack();

	this->sendToAllClients( &dp );
}
void Server::disconnectAll( const string& reason ){
	// send disconnect packet to all REMOTE clients
	DisconnectPacket cp;

	for(int i=1; i<this->properties.maxClients; i++){
		if( this->clients[i] != NULL ){
			cp.data.clientId = i;
			Packet::stringToString256( cp.data.reason, reason );
			cp.pack( this->clients[i]->properties.ipAddress );
			this->socket->send( &cp );
		}
	}
}



void Server::applyServerPropertiesPacket( ServerPropertiesPacket* spp ){
	int i;

	this->properties.connectionSpeed = spp->data.connectionSpeed;
	this->properties.gameMode = spp->data.gameMode;
	this->properties.ghostTime = spp->data.ghostTime;
	Packet::string256ToString( this->properties.name, spp->data.name );
	Packet::string32ToString( this->properties.arena, spp->data.arena );

	// update clients array
	this->properties.maxClients = spp->data.maxClients;
	System::log("Server: Preparing clients array for %i clients...", this->properties.maxClients);
	this->clients = new Client* [this->properties.maxClients];
	for( i=0; i<this->properties.maxClients; i++ ){
		this->clients[i] = NULL;
	}

	Game* game = this->network->getSystem()->getGame();

	// update vehicle list
	this->properties.availableVehicles.clear();
	for( i=0; i<spp->data.vehicleList.size; i++ ){
		string v;
		Packet::string256ToString( v, spp->data.vehicleList.entries[i] );

		// check whether v is available
		if( !game->getVehicleFactory()->isVehicleAvailable( v ) ){
			System::error( "(in Server::applyServerProperties()): Server requires vehicle '%s' which is not available.", v.c_str() );
			throw Exception( "Server requires a vehicle which you don't have.", "Server::applyServerProperties()" );
		}
		this->properties.availableVehicles.push_back( v );
	}

	// update weapon list
	this->properties.availableWeapons.clear();
	for( i=0; i<spp->data.weaponList.size; i++ ){
		string w;
		Packet::string256ToString( w, spp->data.weaponList.entries[i] );

		// check whether w is available
		if( !game->getWeaponFactory()->isWeaponAvailable( w ) ){
			System::error( "(in Server::applyServerProperties()): Server requires weapon '%s' which is not available.", w.c_str() );
			throw Exception( "Server requires a weapon which you don't have.", "Server::applyServerProperties()" );
		}
		this->properties.availableWeapons.push_back( w );
	}
}

void Server::applyServerInfoPacket( ServerInfoPacket* sip ){
	this->lastServerInfoMillis = this->network->getSystem()->getTimer()->getMilliseconds();

	this->info.secondsRunning = sip->data.secondsRunning;
	this->info.numClients = sip->data.numClients;

	for( int i=0; i<this->properties.maxClients; i++ ){
		if( (sip->data.clientArray.usedSlots & (0x01 << i)) != 0 ){ // slot i is in use
			if( this->clients[i] == NULL ){ // create new client if slot is empty (maybe we simply missed the connect packet)
				Client* client = new Client( this->network );

				client->properties.ipAddress = sip->udpPacket->address; // NOTE: points to server for remote clients,
																		// but since clients aren't allowed to
																		// communicate directly with each other
																		// that doesn't matter
				Packet::string32ToString( client->properties.name, sip->data.clientArray.entries[i].name );
				client->properties.remote = true;
				// these remain empty
				client->properties.host = string( "<empty>" );
				client->properties.port = 0;
			
				// add client to list
//				System::warn( "(in Server::applyServerInfoPacket()): Slot %i is used in server info packet but not in local copy. Adding client '%s'.", i, client->properties.name.c_str() );
				this->addClient( client, i );
			}

			// apply client info
			Client* client = this->clients[i];
			ServerInfoPacket::data_s::clientArray_s::entry_s* entry = &(sip->data.clientArray.entries[i]);

//			System::log("applyServerInfo.p: i: %i, t: %i, v: %i", i, entry->team, entry->vehicle);

			client->info.secondsOnServer = entry->secondsOnServer;
			client->info.ping = entry->ping;
			client->info.kills = entry->kills;
			client->info.kills = entry->deaths;
			client->info.score = entry->score;
			client->info.damageInflicted = entry->damageInflicted;

			client->info.team = entry->team;
			client->info.vehicle = entry->vehicle;
			client->info.weapons[0] = entry->weapons[0];
			client->info.weapons[1] = entry->weapons[1];
			client->info.weapons[2] = entry->weapons[2];
			client->info.weapons[3] = entry->weapons[3];

//			System::log("applyServerInfo.c: i: %i, t: %i, v: %i", i, client->info.team, client->info.vehicle);

			Game* game = this->network->getSystem()->getGame();
			if( client->info.team != Game::TEAM_SPECTATORS && client->info.vehicle != -1 && client->vehicle == NULL ){
//				System::warn( "(in Server::applyServerInfoPacket()): Client '%s' has no vehicle attached, but references vehicle %i in client info. Spawning new vehicle.", client->properties.name.c_str(), client->info.vehicle );
				client->vehicle = game->spawnVehicle( client );
			}

		}else{ // slot i NOT in use
			if( this->clients[i] != NULL ){ // remove client
				if( this->clients[i] == this->network->getClient() ){
					System::warn( "(in Server::applyServerInfoPacket()): Slot %i is empty in server info packet but references local client in local copy. Ignoring slot.", i );
				}else{
					System::warn( "(in Server::applyServerInfoPacket()): Slot %i is empty in server info packet but not in local copy. Removing client '%s'.", i, this->clients[i]->properties.name.c_str() );
					this->removeAndDestroyClient( i );
				}
			}
		}
	}
}


void Server::receivePackets(){
	int loopCounter;

	Packet::packetList_t packetList;

	for( loopCounter=0; loopCounter<=this->info.numClients; loopCounter++ ){
		this->socket->receive( packetList );

		if( packetList.size() == 0 ){
			break;
		}

		for( list<Packet*>::iterator iter = packetList.begin(); iter != packetList.end(); ++iter ){
			int packetType = (*iter)->packetType;

			switch( packetType ){
				case Packet::PACKET_TYPE_CONNECT:
//					System::log("Server: ConnectPacket received.");
					this->handleConnectPacket( (ConnectPacket*)(*iter) );
					break;
				case Packet::PACKET_TYPE_DISCONNECT:
//					System::log("Server: DisconnectPacket received.");
					this->handleDisconnectPacket( (DisconnectPacket*)(*iter) );
					break;
				case Packet::PACKET_TYPE_PING:
//					System::log("Server: PingPacket received.");
					this->handlePingPacket( (PingPacket*)(*iter) );
					break;

				case Packet::PACKET_TYPE_SPAWN:
//					System::log("Server: SpawnPacket received.");
					this->handleSpawnPacket( (SpawnPacket*)(*iter) );
					break;

				case Packet::PACKET_TYPE_KILL:
//					System::log("Server: KillPacket received.");
					this->handleKillPacket( (KillPacket*)(*iter) );
					break;

				case Packet::PACKET_TYPE_FIRE:
//					System::log("Server: FirePacket received.");
					this->handleFirePacket( (FirePacket*)(*iter) );
					break;

				case Packet::PACKET_TYPE_CHAT_MESSAGE:
//					System::log("Server: ChatMessagePacket received.");
					this->handleChatMessagePacket( (ChatMessagePacket*)(*iter) );
					break;

				case Packet::PACKET_TYPE_CLIENT_STATE:
//					System::log("Server: ClientStatePacket received.");
					this->handleClientStatePacket( (ClientStatePacket*)(*iter) );
					break;

				case Packet::PACKET_TYPE_SERVER_UPDATE_REQUEST:
//					System::log("Server: ServerUpdateRequestPacket received.");
					this->handleServerUpdateRequestPacket( (ServerUpdateRequestPacket*)(*iter) );
					break;

				default:
					System::warn("(in Server::receivePackets()): Received packet with unexpected packetType (%i). Ignoring.", packetType);
					break;
			}

		}		
	
//		this->packetsReceived += packetList.size();

		Packet::clearPacketList( packetList );
	}

	if( loopCounter > this->info.numClients ){ // too many packets
		System::warn("(in Server::receivePackets()): Too many packets to receive. Emptying packet queue to prevent overflow...");
		this->socket->emptyIncomingPacketQueue();
	}
}

void Server::handleConnectPacket( ConnectPacket* cp ){
//	System::log("Server: HandleConnectPacket()");

	int cId = cp->data.clientId;

	if( !this->properties.remote ){ // a real server => do some more checks
		if( cp->data.protocolVersion != Network::PROTOCOL_VERSION ){
			System::warn("(in Server::handleConnectPacket()): Received connect packet with wrong protocol version (%i). Rejecting connection.", cp->data.protocolVersion);

			DisconnectPacket dp;
			dp.data.clientId = 0;
			Packet::stringToString256(dp.data.reason, "Wrong protocol version.");
			dp.pack( cp->udpPacket->address );
			this->socket->send(&dp);
	
			return;
		}

		if( this->info.numClients >= this->properties.maxClients ){
			System::warn("(in Server::handleConnectPacket()): Received connect packet but server is already full. Rejecting connection.");

			DisconnectPacket dp;
			dp.data.clientId = 0;
			Packet::stringToString256(dp.data.reason, "Server is full.");
			dp.pack( cp->udpPacket->address );
			this->socket->send(&dp);

			return;
		}

		if( cId >= 0 && this->clients[cId] != NULL ){
			System::warn("(in Server::handleConnectPacket()): Received connect packet with clientId (%i) that is already in use. Ignoring packet.", cId);
			return;
		}
	}

	// CONNECTION ACCEPTED

	// create new client
	Client* client = new Client( this->network );

	client->properties.ipAddress = cp->udpPacket->address;
	Packet::string32ToString( client->properties.name, cp->data.clientName );
	client->properties.remote = true;
	// these remain empty
	client->properties.host = string( "<empty>" );
	client->properties.port = 0;

	unsigned long currentMillis = this->network->getSystem()->getTimer()->getMilliseconds();
	client->lastClientStateMillis = currentMillis;
	client->lastPingMillis = currentMillis;
	client->connectionInterrupted = false;
	client->ready = false; // net sending any packets to this client, until it has spawned (and thus finished loading)

	// add client to list
	if( this->properties.remote && cp->data.clientId != -1 ){ // client has already a id from remote server
		this->addClient( client, cp->data.clientId );
	}else{
		this->addClient( client );
	}

	if( !this->properties.remote ){ // not a shadow server => notify clients
		ConnectPacket p;

		p.data.clientId = client->properties.clientId;
		Packet::stringToString32( p.data.clientName, client->properties.name );
		p.pack();
		this->sendToAllClients( &p );

		// and send connect packet and server properties to newly connected client
		p.udpPacket->address = client->properties.ipAddress;
		this->socket->send( &p );
		this->sendServerPropertiesPacket( client->properties.ipAddress );
	}

	System::log("Server: '%s' connected (clientId: %i).", client->properties.name.c_str(), client->properties.clientId);
}

void Server::handleDisconnectPacket( DisconnectPacket* dp ){
	int cId = dp->data.clientId;
	string reason;
	Packet::string256ToString( reason, dp->data.reason );

	Client* c = this->clients[cId];

	if( c == NULL ){ // slot not in use
		System::warn("(in Server::handleDisconnectPacket()): Slot %i not in use. Ignoring packet.", cId);
		return;
	}

	// unspawn vehicle
	Game* game = this->network->getSystem()->getGame();
	if( c->vehicle != NULL ){ // if there's a vehicle unspawn it
		game->unspawnVehicle( c->vehicle );
		c->vehicle = NULL;
	}

	// remember name for message and then destroy client
	string cName = c->properties.name;
	this->removeAndDestroyClient( cId );

	if( !this->properties.remote ){ // not a shadow server => notify (remaining) clients
		DisconnectPacket p;

		p.data = dp->data;

		p.pack();
		this->sendToAllClients( &p );
	}

	System::log( "Server: '%s' disconnected. Reason: %s.", cName.c_str(), reason.c_str() );
}


void Server::handlePingPacket( PingPacket* pp ){

	if( pp->data.clientId == -1 ){ // not a connected client => just answer
		PingPacket p;
		p.data = pp->data;
		p.pack( pp->udpPacket->address );
		this->socket->send( &p );

	}else{

		int cId = pp->data.clientId;
		Client* c = this->clients[cId];
	
		if( c == NULL ){ // slot not in use
			System::warn("(in Server::handlePingPacket()): Slot %i not in use. Ignoring packet.", cId);
			return;
		}

		unsigned long currentMillis = this->network->getSystem()->getTimer()->getMilliseconds();
		c->lastPingMillis = currentMillis;
		c->connectionInterrupted = false;

		c->info.ping = (int)( currentMillis - pp->data.startMillis );
	}
}


void Server::handleSpawnPacket( SpawnPacket* sp ){
	int cId = sp->data.clientId;
	Client *c = this->clients[cId];

	if( c == NULL ){
		System::warn("(in Server::handleSpawnPacket()): Slot %i not in use. Ignoring packet.", cId);
		return;
	}

	c->ready = true;

	// update client info
	if( this->network->getSystem()->getGame()->isRunningTeamMode() && sp->data.team == Game::TEAM_PLAYERS ){
		bool b = (this->network->getSystem()->getRandom()->nextFloat() < 0.5f);
		if( b ){
			sp->data.team = Game::TEAM_RED;
		}else{
			sp->data.team = Game::TEAM_BLUE;
		}
	}
	if( !this->network->getSystem()->getGame()->isRunningTeamMode() && sp->data.team > Game::TEAM_PLAYERS ){
		sp->data.team = Game::TEAM_PLAYERS;
	}

	c->info.team = sp->data.team;
	c->info.vehicle = sp->data.vehicle;
	c->info.weapons[0] = sp->data.weapons[0];
	c->info.weapons[1] = sp->data.weapons[1];
	c->info.weapons[2] = sp->data.weapons[2];
	c->info.weapons[3] = sp->data.weapons[3];

	c->state.position = FVector3( sp->data.position );
	c->state.orientation = FQuaternion::IDENTITY;

//	System::log("s::handleSpawn: t: %i, v: %i\n", c->info.team, c->info.vehicle);


	if( !this->properties.remote ){ // not a shadow server => notify clients
		SpawnPacket p;
		p.data = sp->data;
		p.pack();

		this->sendToAllClients( &p );
	}

	if( sp->data.team == Game::TEAM_SPECTATORS ){
		System::log( "Server: '%s' joined spectators. ", c->properties.name.c_str() );
		this->network->getSystem()->getGui()->getHud()->setCenterOfScreenMessage(
			"Spectating. Use the equipment menu (F4) to join battle." );
	}else{
		System::log( "Server: '%s' spawned for %s."
			, c->properties.name.c_str(), Game::getTeamName(c->info.team).c_str() );
	}
}

void Server::handleKillPacket( KillPacket* kp ){
	int killedId = kp->data.killedId;
	Client *killedClient = this->clients[killedId];
	
	int killerId = kp->data.killerId;
	Client *killerClient = NULL;
	if( killerId != -1 ){
		killerClient = this->clients[killerId];
	}

	if( killedClient == NULL ){
		System::warn("(in Server::handleKillPacket()): Slot %i not in use. Ignoring packet.", killedId);
		return;
	}

	if( killedClient->info.vehicle == -1 || killedClient->info.team == Game::TEAM_SPECTATORS ){ // alread dead
		return;
	}

	killedClient->info.vehicle = -1;
	killedClient->lastDeathMillis = this->network->getSystem()->getTimer()->getMilliseconds();
	killedClient->info.deaths += 1;

	// update clients info
	if( killerClient != NULL ){
		if( killerClient != killedClient ){ // normal kill
			killerClient->info.score += 1;
			killerClient->info.kills += 1;

			// TODO: team kill penalty
			System::log( "Server: '%s' was killed by '%s'.", killedClient->properties.name.c_str(), killerClient->properties.name.c_str() );
			if( killerClient == this->network->getClient() ){
				this->network->getSystem()->getGui()->getHud()->setCenterOfScreenMessage(
					"You killed " + killedClient->properties.name );
			}
			if( killedClient == this->network->getClient() ){
				this->network->getSystem()->getGui()->getHud()->setCenterOfScreenMessage(
					"You were killed  by " + killerClient->properties.name );
			}
		}else{ // suicide
			killerClient->info.score -= 1;

			System::log( "Server: '%s' commited suicide.", killedClient->properties.name.c_str() );
		}
	}else{ // killed by admin or already disconnected client
		System::log( "Server: '%s' was killed.", killedClient->properties.name.c_str() );
	}

	if( !this->properties.remote ){ // not a shadow server => notify clients
		KillPacket p;
		p.data = kp->data;
		p.pack();

		this->sendToAllClients( &p );
	}

}

void Server::handleFirePacket( FirePacket* fp ){

	// THINKABOUTME: more checks here...

	if( !this->properties.remote ){ // not a shadow server => notify clients
		FirePacket p;
		p.data = fp->data;
		p.pack();

		this->sendToAllClients( &p );
	}
}

void Server::handleChatMessagePacket( ChatMessagePacket* cmp ){
	// THINKABOUTME: more checks here...

	if( !this->properties.remote ){ // not a shadow server => pass on to clients
		ChatMessagePacket p;
		p.data = cmp->data;
		p.pack();

		this->sendToAllClients( &p );
	}
}


void Server::handleClientStatePacket( ClientStatePacket* csp ){
	int cId = csp->data.clientId;
	Client *client = this->clients[cId];

	if( client == NULL ){
		System::warn("(in Server::handleClientStatePacket()): Slot %i not in use. Ignoring packet.", cId);
		return;
	}

	client->lastClientStateMillis = this->network->getSystem()->getTimer()->getMilliseconds();
//	client->connectionInterrupted = false; now in ping

	client->state.health = csp->data.health;
	client->state.energy = csp->data.energy;

	client->state.position = FVector3( csp->data.position );
	client->state.orientation = FQuaternion( csp->data.orientation );
	client->state.velocity = FVector3( csp->data.velocity );
	client->state.angularVelocity = FVector3( csp->data.angularVelocity );

	memcpy( client->state.vehicleSpecificBytes, csp->data.vehicleSpecificBytes, 16 );

	if( !this->properties.remote ){ // not a shadow server => notify clients
		ClientStatePacket p;
		p.data = csp->data;
		p.pack();

		this->sendToAllClients( &p );
	}
}


void Server::handleServerUpdateRequestPacket( ServerUpdateRequestPacket* surp ){
	// simply send server info and server properties
	this->spp.udpPacket->address = surp->udpPacket->address;
	this->socket->send( &this->spp );

	this->sip.udpPacket->address = surp->udpPacket->address;
	this->socket->send( &this->sip );
}


void Server::sendPackets(){
	unsigned int currentMillis = this->network->getSystem()->getTimer()->getMilliseconds();
	int i;

	// update master server
	if( this->properties.registerOnMasterServer
		&& currentMillis > this->lastMasterServerUpdateMillis + Server::MASTER_SERVER_UPDATE_INTERVAL_MILLIS
		){

		MSRegisterGamePacket rgp;
		rgp.data.currentPlayers = this->info.numClients;
		rgp.data.maxPlayers = this->properties.maxClients;
		Packet::stringToString256( rgp.data.serverName, this->properties.name );
		if( this->info.numClients == this->properties.maxClients ){
			Packet::stringToString32( rgp.data.status, "full" );
		}else{
			Packet::stringToString32( rgp.data.status, "open" );
		}
		
		rgp.pack( this->masterServerIpAddress );

		this->socket->send( &rgp );

		this->lastMasterServerUpdateMillis = currentMillis;
	}

	// update and send server info if time is right
	if( currentMillis > this->lastServerInfoMillis + Server::SERVER_INFO_INTERVAL_MILLIS ){
		this->updateServerInfoPacket();
		this->sendToAllClients( &this->sip );
		this->lastServerInfoMillis = currentMillis;
	}

	// ping connected clients
	if( currentMillis > this->lastPingMillis + Server::PING_INTERVAL_MILLIS ){
		PingPacket p;
		p.data.clientId = -1;
		p.data.startMillis = currentMillis;
		p.pack();

		this->sendToAllClients( &p );
	
		this->lastPingMillis = currentMillis;
	}

	// test for idle clients
	for( i=0; i<this->properties.maxClients; i++ ){
		Client* c = this->clients[i];

		if( c == NULL )
			continue;

		if( !c->connectionInterrupted && currentMillis > c->lastPingMillis + Server::MAX_CLIENT_IDLE_TIME/2 * 1000 ){ // warn
			c->connectionInterrupted = true;
			System::warn( "(in Server::sendPackets()): Client '%s' has been idle for over %i seconds. It will get disconnected after %i seconds."
				, c->properties.name.c_str(), Server::MAX_CLIENT_IDLE_TIME/2, Server::MAX_CLIENT_IDLE_TIME );
		}

		if( currentMillis > c->lastPingMillis + Server::MAX_CLIENT_IDLE_TIME * 1000 ){ // disconnect clients after max idle time
			string clientName = c->properties.name;
			System::warn( "(in Server::sendPackets()): Client '%s' has been idle for over %i seconds. Disconnecting it..."
				, clientName.c_str(), Server::MAX_CLIENT_IDLE_TIME );
			this->disconnect( c, "MAX_CLIENT_IDLE_TIME reached" );
			this->removeAndDestroyClient( c );

			System::log( "Server: Client '%s' was disconnected. Reason: MAX_CLIENT_IDLE_TIME reached."
				, clientName.c_str() );
		}
	}

}

void Server::sendToAllClients( Packet* packet ) const {
	for( int i=0; i<this->properties.maxClients; i++ ){
		if( this->clients[i] != NULL && this->clients[i]->ready ){
			packet->udpPacket->address = this->clients[i]->properties.ipAddress;
			this->socket->send( packet );
		}
	}
}

void Server::sendServerPropertiesPacket( const Packet::ipAddress_t& ipAddress ){
	this->spp.udpPacket->address = ipAddress;
	this->socket->send( &this->spp );
}

void Server::updateServerInfoPacket(){
	this->sip.data.numClients = this->info.numClients;
	this->sip.data.secondsRunning = this->info.secondsRunning;

	// update clientArray
	this->sip.data.clientArray.usedSlots = 0;
	for( int i=0; i<this->properties.maxClients; i++ ){
		if( this->clients[i] != NULL ){
			this->sip.data.clientArray.usedSlots |= (0x01 << i);

			Client* client = this->clients[i];
			ServerInfoPacket::data_s::clientArray_s::entry_s* entry = &(this->sip.data.clientArray.entries[i]);

			Packet::stringToString32( entry->name, client->properties.name );
			entry->damageInflicted = client->info.damageInflicted;
			entry->score = client->info.score;
			entry->ping = client->info.ping;
			entry->secondsOnServer = client->info.secondsOnServer;
			entry->team = client->info.team;
			entry->vehicle = client->info.vehicle;
			entry->weapons[0] = client->info.weapons[0];
			entry->weapons[1] = client->info.weapons[1];
			entry->weapons[2] = client->info.weapons[2];
			entry->weapons[3] = client->info.weapons[3];

//			System::log("updateServerInfo: i: %i, t: %i, v: %i", i, entry->team, entry->vehicle);
		}
	}

	this->sip.pack();
}


void Server::sendServerInfoPacket( const Packet::ipAddress_t& ipAddress ){
	this->sip.udpPacket->address = ipAddress;
	this->socket->send( &this->sip );
}



Client** Server::getClientArray() const {
	return this->clients;
}

Client* Server::getClient( int slot ) const {
	if( slot < 0 || slot >= this->properties.maxClients ){
		throw Exception( "clientId out of bounds", "Server::getClient()" );
	}

	return this->clients[slot];
}






const string& Server::lookUpVehicleName( int index ) const {
	const string& ret = this->properties.availableVehicles[index];

	if( ret.empty() ){
		System::error( "(in Server::lookUpVehicleName()): Index %i is invalid.", index );
		throw Exception( "Couldn't look up vehicle name.", "Server::lookUpVehicleName()" );
	}

	return ret;
}

const string& Server::lookUpWeaponName( int index ) const {
	const string& ret = this->properties.availableWeapons[index];

	if( ret.empty() ){
		System::error( "(in Server::lookUpWeaponName()): Index %i is invalid.", index );
		throw Exception( "Couldn't look up weapon name.", "Server::lookUpWeaponName()" );
	}

	return ret;
}

