#include "ShotFactory.h"

#include "System.h"
#include "Exception.h"

#include "ProjectileShot.h"
#include "BallisticShot.h"

ShotFactory::ShotFactory(Game* game){
	this->game = game;

	this->shotsByName.clear();
}

ShotFactory::~ShotFactory(){
	// delete shots
	for( shotsByName_t::iterator iter = shotsByName.begin(); iter != shotsByName.end(); ++iter ){
		delete iter->second;
	}

}

void ShotFactory::addShot(const string& shotFilePath){
//	System::log( "ShotFactory: Parsing shot '%s'...", shotFilePath.c_str() );

	try{
		TiXmlDocument doc( shotFilePath.c_str() );
		if( !doc.LoadFile() ){
			System::error( "(in ShotFactory::addShot()): Couldn't load file '%s'. (Maybe not a correct xml file?)", shotFilePath.c_str() );
			throw Exception("Couldn't add shot.", "ShotFactory::addShot()");
		}
//		doc.Print();

		TiXmlNode* rootElement = doc.RootElement();
		TiXmlElement* element = rootElement->ToElement();

		string rootElementName = element->Value();
		if( rootElementName != "shot" ){
			System::error( "(in ShotFactory::addShot()): File '%s' does not have a <shot> root node.", shotFilePath.c_str() );
			throw Exception("Couldn't add shot.", "ShotFactory::addShot()");
		}

		// check version
		const char* tmp = element->Attribute( "formatVersion" );
		if( tmp == NULL ){
			System::error( "(in ShotFactory::addShot()): File '%s' has no version information.", shotFilePath.c_str() );
			throw Exception("Couldn't add shot.", "ShotFactory::addShot()");
		}else{
			string version = tmp;

			if( version != System::getInstance()->getVersionString() ){
				System::warn("(in ShotFactory::addShot()): file '%s' has version %s. Trying to parse anyway.", shotFilePath.c_str(), version.c_str() );
			}
		}


		Shot* s = NULL;

		// check type and create a weapon of the proper subclass
		tmp = element->Attribute( "type" );
		if( tmp == NULL ){
			System::error( "(in ShotFactory::addShot()): File '%s' has no type information.", shotFilePath.c_str() );
			throw Exception("Couldn't add shot.", "ShotFactory::addShot()");
		}else{
			string type = tmp;

			if( type == string("projectile") ){
				s = new ProjectileShot();
			}else if( type == string("ballistic") ){
				s = new BallisticShot();

			}else{
				System::warn("(in ShotFactory::addShot()): Shot file '%s' references unknown type ('%s'). Ignoring shot.", shotFilePath.c_str(), type.c_str() );
			}
		}


		if( s != NULL ){
			s->fromXmlElement( element );
			s->calculateChecksum();

			shotsByName_t::const_iterator c_iter = this->shotsByName.find( s->getName() );
			if( c_iter != shotsByName.end() ){ // another shot with that name is already registered -> throw exception
				System::error( "(in ShotFactory::addShot()): A shot with name '%s' is already registered.", s->getName().c_str() );
				throw Exception("Couldn't add shot.", "ShotFactory::addShot()");
			}
			shotsByName.insert( shotsByName_t::value_type(s->getName(), s) );
			System::log("ShotFactory: Added shot '%s' to list.", s->getName().c_str());
		}

	}catch( Exception& e ){
		System::warn("(ShotFactory::addWeapon()): An exception occured while parsing shot file '%s'. Ignoring shot.", shotFilePath.c_str());
		System::log("ShotFactory: The exception was: %s", e.getDescription().c_str());
	}

}


void ShotFactory::removeShot(const string& name){
	shotsByName_t::iterator iter = shotsByName.find( name );
	if( iter == this->shotsByName.end() ){
		throw Exception("A shot with name '" + name +"' is not registered.", "ShotFactory::removeShot()");
	}
	this->shotsByName.erase(iter);
}





Shot* ShotFactory::createShot(const string& name) const {

	shotsByName_t::const_iterator c_iter = this->shotsByName.find( name );
	if( c_iter == this->shotsByName.end() ){
		throw Exception("A shot with name '" + name +"' is not registered.", "ShotFactory::createShot()");
	}

	Shot* ret = c_iter->second->clone();

//	ret->reset();

	return ret;
}

void ShotFactory::destroyShot(Shot* shot) const {
	delete shot;
}


bool ShotFactory::isShotAvailable( const string& name ) const {
	shotsByName_t::const_iterator c_iter = this->shotsByName.find( name );

	return (c_iter != this->shotsByName.end());

}

vector<string> ShotFactory::getShotNames() const {
	vector<string> ret;

	for( shotsByName_t::const_iterator c_iter = this->shotsByName.begin(); c_iter != this->shotsByName.end(); ++c_iter ){
		ret.push_back( c_iter->first );
	}

	return ret;
}
